/* ***************************************************************************************
* FILE:          ViewTraverser.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewTraverser.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "ViewTraverser.h"
#include "ScmlSerializer.h"
#include <sstream>
#include <iterator>

#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Core/TextNode2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Group2D.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Courier/Visualization/IViewHandler.h>
#include <Courier/Visualization/ViewScene2D.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDescriptor.h>

#include "View/CGI/CgiExtensions/AppViewHandler.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ViewTraverser.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


namespace hmibase {
namespace trace {

ViewTraverser::ViewTraverser() :
   _viewHandler(NULL),
   _timer(NULL),
   _currentViewIndex(0),
   _traverseInProgress(false)
{
}


ViewTraverser::~ViewTraverser()
{
   stopTimer();

   if (_timer != NULL)
   {
      FEATSTD_DELETE(_timer);
      _timer = NULL;
   }

   _viewHandler = NULL;
}


ViewTraverser& ViewTraverser::getInstance()
{
   static ViewTraverser _instance;
   return _instance;
}


bool ViewTraverser::onCourierMessage(const TraverseViewsReqMsg& reqMsg)
{
   ETG_TRACE_FATAL_THR(("TraverseViewsReqMsg visitors=%s", reqMsg.GetVisitors().GetCString()));

   // split the visitors param
   VisitorsType visitorIds;
   std::istringstream inputStream;
   inputStream.str(reqMsg.GetVisitors().GetCString());
   std::string visitorId;
   while (std::getline(inputStream, visitorId, ','))
   {
      visitorIds.push_back(visitorId);
   }

   std::string viewName = reqMsg.GetViewName().GetCString();
   beginTraverse(visitorIds, viewName);

   return true;
}


bool ViewTraverser::onCourierMessage(const TimerExpiredMsg& msg)
{
   if (_timer == msg.GetTimer())
   {
      checkCurrentViewIndex();
      return true;
   }
   return false;
}


bool ViewTraverser::onCourierMessage(const Courier::ActivationResMsg& msg)
{
   if (_traverseInProgress && msg.GetSuccess())
   {
      onViewActivated(msg.GetViewId());
   }
   //don't consume this broadcast message
   return false;
}


void ViewTraverser::beginTraverse(const VisitorsType& visitorIds, const std::string& viewName)
{
   ETG_TRACE_FATAL_THR(("beginTraverse"));

   // make sure default visitors are initialized
   DefaultVisitors::initialize();

   // check visitors
   _visitorIds.clear();
   for (VisitorsType::const_iterator it = visitorIds.begin(); it != visitorIds.end(); ++it)
   {
      if (ViewVisitorRegistry::getItem(*it) != NULL)
      {
         _visitorIds.push_back(*it);
      }
      else
      {
         ETG_TRACE_FATAL_THR(("no visitor registered for id '%s'", (*it).c_str()));
      }
   }

   //if no valid visitors are specified use the default ones
   if (_visitorIds.empty())
   {
      _visitorIds.insert(_visitorIds.end(), DefaultVisitors::getIds().begin(), DefaultVisitors::getIds().end());
   }

   std::ostringstream outputStream;
   std::copy(_visitorIds.begin(), _visitorIds.end(), std::ostream_iterator<std::string>(outputStream, ","));
   ETG_TRACE_FATAL_THR(("traverse using visitors %s", outputStream.str().c_str()));

   Courier::View* view = NULL;

   // update view names, set current index and mark traverse session as active
   if (viewName.empty())
   {
      ETG_TRACE_FATAL_THR(("Begin Traverse All Views"));
      updateViewNames();
   }
   else
   {
      ETG_TRACE_FATAL_THR(("Begin Traverse ViewName=%s", viewName.c_str()));
      _viewNames.clear();
      _viewNames.push_back(viewName);

      view = _viewHandler->FindView(Courier::ViewId(viewName.c_str()));
   }

   if ((view != NULL) && (view->IsRenderingEnabled()) && (view->IsActive()))
   {
      ETG_TRACE_FATAL_THR(("View already available and active"));

      notifyVisitors(*view);
   }
   else
   {
      _currentViewIndex = 0;
      _traverseInProgress = true;

      // prepare surface, bring app in foreground, etc
      unsigned int surfaceId = ScreenBrokerClient::GetInstance().SurfaceId();
      unsigned int displayId = ScreenBrokerClient::GetInstance().GetDisplayID(surfaceId);
      ScreenBrokerClient::GetInstance().Show(::hmibase::view::ViewIdentifier(), surfaceId);
      POST_MSG((COURIER_MESSAGE_NEW(Courier::RenderReqMsg)(true)));
      POST_MSG((COURIER_MESSAGE_NEW(ApplicationStateUpdMsg)(hmibase::FORCE_IN_FOREGROUND, displayId)));

      // start the timer which triggers view switch
      startTimer();
   }
}


void ViewTraverser::endTraverse()
{
   ETG_TRACE_FATAL_THR(("endTraverse"));

   _currentViewIndex = 0;
   _traverseInProgress = false;
   stopTimer();
}


void ViewTraverser::startTimer()
{
   ETG_TRACE_FATAL_THR(("startTimer"));

   // late initialization for the timer
   if (_timer == NULL)
   {
      _timer = CANDERA_NEW(Util::Timer);
      if (_timer != NULL)
      {
         _timer->setName("ViewTraverser", "");
         _timer->setTimeoutWithRepeat(1000, 1000);
      }
   }

   if (_timer != NULL)
   {
      _timer->stop();
      _timer->start();
   }
}


void ViewTraverser::stopTimer()
{
   if (_timer != NULL)
   {
      _timer->stop();
   }
}


void ViewTraverser::updateViewNames()
{
   _viewNames.clear();

   // read view names from the asset
   Candera::AssetDescriptor::AssetIdIterator sceneIdIterator = Candera::DefaultAssetProvider::GetInstance().GetAssetDescriptor().GetAssetIdIterator(Candera::Scene2DLib);
   while (sceneIdIterator.IsValid())
   {
      const char* viewName = Candera::DefaultAssetProvider::GetInstance().GetNameById(Candera::Scene2DLib, *sceneIdIterator, 0);
      _viewNames.push_back(viewName);
      ++sceneIdIterator;
   }
   std::sort(_viewNames.begin(), _viewNames.end());
   ETG_TRACE_FATAL_THR(("updateViewNames count=%u", _viewNames.size()));
}


void ViewTraverser::checkCurrentViewIndex()
{
   // if not all views were visited, switch to the new view
   if (_currentViewIndex < _viewNames.size())
   {
      const char* viewName = _viewNames.at(_currentViewIndex).c_str();
      ETG_TRACE_FATAL_THR(("checkCurrentViewIndex index=%u (%u), name=%s", _currentViewIndex, _viewNames.size(), viewName));

      switchView(Courier::ViewId(viewName));
      ++_currentViewIndex;
   }
   // if all the views were visited, end traverse
   else
   {
      endTraverse();
   }
}


void ViewTraverser::switchView(const Courier::ViewId& viewId)
{
   // request hide/destroy for the previous view
   if (_previousActiveViewId != Courier::ViewId())
   {
      POST_MSG((COURIER_MESSAGE_NEW(Courier::ViewPlaceholderReqMsg)(_previousActiveViewId, true)));
      POST_MSG((COURIER_MESSAGE_NEW(ExtendedViewReqMsg)(hmibase::views::HideAndDestroy, _previousActiveViewId, 0)));
   }

   // request create+show for the new view
   POST_MSG((COURIER_MESSAGE_NEW(ExtendedViewReqMsg)(hmibase::views::CreateAndShow, viewId, 0)));
   _previousActiveViewId = viewId;
}


void ViewTraverser::onViewActivated(const Courier::ViewId& viewId)
{
   if (_traverseInProgress && (_viewHandler != NULL))
   {
      ETG_TRACE_FATAL_THR(("onViewActivated name=%s", viewId.CStr()));

      // get the view and notify the registered visitors to check it
      Courier::View* view = _viewHandler->FindView(viewId);
      if (view != NULL)
      {
         notifyVisitors(*view);
      }
   }
}


void ViewTraverser::notifyVisitors(Courier::View& view)
{
   for (VisitorsType::iterator it = _visitorIds.begin(); it != _visitorIds.end(); ++it)
   {
      ViewVisitor* visitor = ViewVisitorRegistry::getItem(*it);
      if (visitor != NULL)
      {
         visitor->visitView(view);
      }
   }
}


bool TraceReportWriter::beginReport(const std::string& itemName)
{
   ETG_TRACE_FATAL_THR(("beginReport item=%s", itemName.c_str()));
   return true;
}


bool TraceReportWriter::endReport()
{
   ETG_TRACE_FATAL_THR(("endReport"));
   return true;
}


bool TraceReportWriter::abortReport(const std::string& err)
{
   ETG_TRACE_FATAL_THR(("abortReport err=%s", err.c_str()));
   return true;
}


bool TraceReportWriter::write(const std::string& line)
{
   ETG_TRACE_FATAL_THR((";;;%s;;;", line.c_str()));
   return true;
}


FileReportWriter::FileReportWriter(const std::string& fileNameFormat, const std::string& outputDirectory)
   : _fileNameFormat(fileNameFormat), _outputDirectory(outputDirectory)
{
}


FileReportWriter::~FileReportWriter()
{
   closeFile();
}


std::string& FileReportWriter::getDefaultOutputDirectory()
{
   static std::string _defaultOutputDirectory(
#ifdef WIN32
      "d:\\"
#else
      "/tmp/"
#endif
   );
   return _defaultOutputDirectory;
}


bool FileReportWriter::beginReport(const std::string& itemName)
{
   ETG_TRACE_FATAL_THR(("beginReport item=%s", itemName.c_str()));

   char buffer[256];
   SNPRINTF(buffer, sizeof(buffer), _fileNameFormat.c_str(), itemName.c_str());

   return openFile((getOutputDirectory().empty() ? getDefaultOutputDirectory() : getOutputDirectory()) + buffer);
}


bool FileReportWriter::endReport()
{
   ETG_TRACE_FATAL_THR(("endReport"));

   if (isFileOpen())
   {
      closeFile();
      return true;
   }
   return false;
}


bool FileReportWriter::abortReport(const std::string& err)
{
   ETG_TRACE_FATAL_THR(("abortReport err=%s", err.c_str()));

   closeFile();
   return true;
}


bool FileReportWriter::write(const std::string& line)
{
   if (isFileOpen())
   {
      _stream << line.c_str() << std::endl;
      return true;
   }
   return false;
}


bool FileReportWriter::isFileOpen() const
{
   return _stream.is_open();
}


bool FileReportWriter::openFile(const std::string& fileName)
{
   closeFile();

   ETG_TRACE_FATAL_THR(("openFile fileName=%s", fileName.c_str()));
   _stream.open(fileName.c_str());

   if (!isFileOpen())
   {
      ETG_TRACE_FATAL_THR(("failed to open file %s", fileName.c_str()));
   }
   return isFileOpen();
}


void FileReportWriter::closeFile()
{
   if (isFileOpen())
   {
      ETG_TRACE_FATAL_THR(("closeFile"));
      _stream.close();
   }
}


void NodeLayoutViewVisitor::visitView(Courier::View& view)
{
   ETG_TRACE_FATAL_THR(("visitView view=%s", view.GetId().CStr()));

   Courier::ViewScene2D* viewScene = view.ToViewScene2D();
   if ((viewScene != NULL)
         && (viewScene->GetScene2DContext() != NULL)
         && (viewScene->GetScene2DContext()->GetScene() != NULL))
   {
      if (getReportWriter() != NULL)
      {
         getReportWriter()->beginReport(view.GetId().CStr());
      }

      Candera::Scene2D* scene = viewScene->GetScene2DContext()->GetScene();
      printNodeLayout(*scene);

      if (getReportWriter() != NULL)
      {
         getReportWriter()->endReport();
      }
   }
}


static const char* getNodeTypeName(const Candera::Node2D& node)
{
   if (node.IsTypeOf(Candera::RenderNode::GetTypeId()))
   {
      return "RenderNode";
   }

   if (node.IsTypeOf(Candera::Scene2D::GetTypeId()))
   {
      return "Scene";
   }
   if (node.IsTypeOf(Candera::Group2D::GetTypeId()))
   {
      return "Group";
   }

   if (node.IsTypeOf(Candera::Camera::GetTypeId()))
   {
      return "Camera";
   }

   return "Node";
}


void NodeLayoutViewVisitor::printNodeLayout(Candera::Node2D& node, size_t level)
{
   if (level < 1024)
   {
      const char* typeName = getNodeTypeName(node);

      std::stringstream stream;
      for (size_t i = 0; i < level; ++i)
      {
         stream << ' ';
      }

      stream << '<' << typeName << " name=\"";
      if (node.GetName() != NULL)
      {
         stream << node.GetName();
      }
      else
      {
         stream << "(null)";
      }
      stream << "\" ";

      // for render nodes we write world position and scale
      if (node.IsTypeOf(Candera::RenderNode::GetTypeId()))
      {
         Candera::Rectangle bounds;
         node.GetWorldAxisAlignedBoundingRectangle(bounds);

         stream << "position=\"" << node.GetWorldPosition().GetX() << ';' << node.GetWorldPosition().GetY() << "\" "
                << "size=\"" << bounds.GetWidth() << ';' << bounds.GetHeight() << "\" "
                << "scale=\"" << node.GetScale().GetX() << ';' << node.GetScale().GetY() << "\" ";
      }
      if (node.GetChildCount() == 0)
      {
         stream << '/';
      }
      stream << '>';

      if (getReportWriter() != NULL)
      {
         getReportWriter()->write(stream.str());
      }

      Candera::Node2D* child = node.GetFirstChild();
      while (child != NULL)
      {
         printNodeLayout(*child, level + 1);
         child = child->GetNextSibling();
      }

      if (node.GetChildCount() > 0)
      {
         stream.str("");
         for (size_t i = 0; i < level; ++i)
         {
            stream << ' ';
         }
         stream << "</" << typeName << '>';

         if (getReportWriter() != NULL)
         {
            getReportWriter()->write(stream.str());
         }
      }
   }
}


void ScmlViewVisitor::visitView(Courier::View& view)
{
   ETG_TRACE_FATAL_THR(("visitView view=%s", view.GetId().CStr()));

   ReportWriter* itemWriter = getReportWriter();
   Courier::ViewScene2D* viewScene = view.ToViewScene2D();
   if ((itemWriter != NULL) && (viewScene != NULL)
         && (viewScene->GetScene2DContext() != NULL)
         && (viewScene->GetScene2DContext()->GetScene() != NULL))
   {
      itemWriter->beginReport(view.GetId().CStr());

      Candera::Scene2D* scene = viewScene->GetScene2DContext()->GetScene();

      if (itemWriter->getStream() != NULL)
      {
         hmibase::trace::ScmlSerializer::serialize(*scene, *itemWriter->getStream());
      }
      else
      {
         std::ostringstream stream;
         hmibase::trace::ScmlSerializer::serialize(*scene, stream);
         itemWriter->write(stream.str());
      }

      itemWriter->endReport();
   }
}


class WidgetReportCallback : public WidgetCheckCallback
{
   public:
      WidgetReportCallback(ReportWriter& itemWriter) : _itemWriter(itemWriter), _widgetCount(0) {}

      virtual bool CheckWidget(Candera::Widget2D* widget)
      {
         if (widget != NULL)
         {
            const char* widgetType = NULL;
            if (widget->GetMetaInfo() != NULL)
            {
               widgetType = widget->GetMetaInfo()->GetName();
            }
            if (widgetType == NULL)
            {
               widgetType = "<null>";
            }

            //BaseWidget2D* baseWidget = Candera::Dynamic_Cast<BaseWidget2D*>(widget);
            //const char* widgetName = (baseWidget != NULL) ? baseWidget->GetLegacyName() : widget->GetName();
            const char* widgetName = widget->GetName();
            if (widgetName == NULL)
            {
               widgetName = "<null>";
            }

            SNPRINTF(_buffer, sizeof(_buffer), "%u. %s:%s", _widgetCount, widgetType, widgetName);
            _itemWriter.write(_buffer);

            ++_widgetCount;
         }

         //continue searching through all the widgets to find the entire content
         return false;
      }

      unsigned int getWidgetCount() const
      {
         return _widgetCount;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(WidgetReportCallback);

      char _buffer[200];
      ReportWriter& _itemWriter;
      unsigned int _widgetCount;
};


void WidgetReportViewVisitor::visitView(Courier::View& view)
{
   ETG_TRACE_FATAL_THR(("visitView view=%s", view.GetId().CStr()));

   ReportWriter* itemWriter = getReportWriter();
   Courier::ViewScene2D* viewScene = view.ToViewScene2D();
   if ((itemWriter != NULL) && (viewScene != NULL)
         && (viewScene->GetScene2DContext() != NULL)
         && (viewScene->GetScene2DContext()->GetScene() != NULL))
   {
      itemWriter->beginReport(view.GetId().CStr());

      WidgetReportCallback finderCallback(*itemWriter);
      view.DistributeMessage(WidgetCheckReqMsg(&finderCallback));

      itemWriter->endReport();
   }
}


void DefaultVisitors::initialize()
{
   //this initializes the ids
   getIds();
}


const std::vector<std::string>& DefaultVisitors::getIds()
{
   static std::vector<std::string> visitorIds;

   static bool initialized = false;
   if (!initialized)
   {
      initialized = true;

      static FileReportWriter scmlFileWriter("Scene2D_%s.scml");
      static ScmlViewVisitor scmlVisitor(&scmlFileWriter);
      ViewVisitorRegistry::registerItem("Scml", &scmlVisitor);
      visitorIds.push_back("Scml");

      static FileReportWriter nodeLayoutFileWriter("Scene2D_%s.xml");
      static NodeLayoutViewVisitor nodeLayoutVisitor(&nodeLayoutFileWriter);
      ViewVisitorRegistry::registerItem("NodeLayout", &nodeLayoutVisitor);
      visitorIds.push_back("NodeLayout");

      static TraceReportWriter traceWriter;
      static WidgetReportViewVisitor widgetReportVisitor(&traceWriter);
      ViewVisitorRegistry::registerItem("WidgetReport", &widgetReportVisitor);
      visitorIds.push_back("WidgetReport");
   }

   return visitorIds;
}


}
}
