/* ***************************************************************************************
* FILE:          TraceQueryHelper.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TraceQueryHelper.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef TRACEQUERYHELPER_HPP
#define TRACEQUERYHELPER_HPP

#include "CanderaWidget/String/String.h"
#include "View/CGI/CgiExtensions/HMIRenderViewVisitor.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#include "hmibase/util/Singleton.h"
#include "hmibase/util/StringUtils.h"
#include "Courier/Visualization/ViewContainer.h"

#include "BaseContract/generated/BaseMsgs.h"

#include "hmi_trace_if.h"

namespace Candera {
class WidgetBase;
}


namespace hmibase {
namespace trace {


struct TMLCurrentViewInfo
{
   std::vector<std::string> mainScenes;
   std::vector<std::string> otherScenes;
};


struct TMLLoadedViewInfo
{
   std::vector<std::string> loadedScenes;
   std::vector<std::string> activeRenderedScenes;
   std::vector<std::string> activeNotRenderedScenes;
   std::vector<std::string> inactiveScenes;
};


class TraceQueryHelper
{
   public:
      struct stateInfo
      {
         unsigned int _activeStatemachineId;
         unsigned int _activeStateId;
         std::string _activeStateName;
         std::string _branch;

         stateInfo(unsigned int activeStatemachineNo, unsigned int aciveStateNo, std::string activeStateName, std::string branch):
            _activeStatemachineId(activeStatemachineNo),
            _activeStateId(aciveStateNo),
            _activeStateName(activeStateName),
            _branch(branch)
         {
         }
      };

      class IWidgetTraceQueryHelper
      {
         public:

            struct TextProperties
            {
               bool isValid;
               FeatStd::String text;
               Candera::Vector2 dimension;
               FeatStd::UInt32 maxNumOfLines;
               Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> style;
               Candera::Vector2 maxDimension;
               FeatStd::String referredTextId;
            };
            virtual void ProcessQueryTraceMsgToPrintVisibleText(const QueryTraceInfoReqMsg& msg, Courier::View* view, std::string& printData) = 0;
            virtual bool ProcessSimTouchRequest(const SimTouchReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor) = 0;
            virtual bool ProcessTraceQueryRequest(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor) = 0;
            virtual bool ProcessWidgetPropertySetRequest(const WidgetPropertySetterReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor) = 0;
            virtual void GetTextWidgetProperties(Candera::Widget2D* widget, TextProperties&) = 0;
            virtual void ProcessQueryTraceMsgToGetPossibleTouchPoints(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, Courier::View* view, const view::HMIRenderViewVisitor& hmiRenderViewVisitor) = 0;
      };
      static TraceQueryHelper& GetInstance()
      {
         static TraceQueryHelper theInstance;
         return theInstance;
      }

      void SetHook(IWidgetTraceQueryHelper* hook)
      {
         _hook = hook;
      }
      /** function declaration */
      bool setWidgetPropertyValue(Candera::WidgetBase* widgetBase, const char* propertyName, const char* propertyValue, std::string& printData);
      Candera::WidgetBase* getWidgetInstance(Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor, const char* widgetName, std::string& printData);
      bool postTouchMsg(const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector, Candera::Widget2D* widget2D);
      TMLCurrentViewInfo PrintActiveViewIDs(const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor, std::vector<std::string>& views);
      bool ProcessTraceQueryRequest(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor);
      bool ProcessSimTouchRequest(const SimTouchReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor);
      bool ProcessWidgetPropertySetRequest(const WidgetPropertySetterReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor);
      void GetTextWidgetProperties(Candera::Widget2D* widget, IWidgetTraceQueryHelper::TextProperties& properties);
      void GetAllLoadedScenes(const Courier::UInt32 QueryId, Courier::IViewHandler& viewHandler, ::hmibase::trace::TMLLoadedViewInfo&);
      void PrintLoadedScenes(const Courier::UInt32 QueryId, Courier::ViewContainer* vc, ::hmibase::trace::TMLLoadedViewInfo& viewsToReturn);

      template<class StateMachineType>
      std::string CollectSMInfo(StateMachineType& sm, std::string _myName = "")
      {
         std::string s;
         std::vector<stateInfo> activeStates;

         // get the currently active states of all statemachines, separated by linefeed
#ifdef VS_NOF_STATE_MACHINES
         SEM_STATE_TYPE StateList[VS_NOF_STATE_MACHINES];

         if (sm.SEM_StateAll(StateList, VS_NOF_STATE_MACHINES) != SES_FOUND)
         {
            char buf[50];
            SNPRINTF(buf, 50, "[%s]: Can not access states", _myName.c_str());
            s += std::string(buf);
         }
         else
         {
            s += "[";
            s += _myName.c_str();
            s += "] *** Detailed Statemachine info ***\n";

            for (unsigned i = 0; i < VS_NOF_STATE_MACHINES; i++)
            {
               const char* name = 0;

               if (sm.SEM_NameAbs(StateMachineType::STATE_TYPE, StateList[i], &name) == SES_TYPE_ERR)
               {
                  s += "type error";
               }
               else
               {
                  bool isNewState = true;
                  std::string n = name;
                  std::vector<stateInfo>::iterator it;
                  for (it = activeStates.begin(); it != activeStates.end(); it++)
                  {
                     // check for matching state name
                     if (n.find((*it)._activeStateName) == 0)
                     {
                        std::string branch = (*it)._activeStateName.substr(0, (*it)._activeStateName.rfind("."));
                        (*it) = stateInfo(i, StateList[i], n, branch);
                        isNewState = false;
                        break;
                     }
                     // check for matching branch name
                     else if (n.find((*it)._branch) == 0)
                     {
                        // state is in branch of active state, so it is inactive currently
                        isNewState = false;
                        break;
                     }
                  }

                  if (isNewState)
                  {
                     activeStates.push_back(stateInfo(i, StateList[i], n, n.substr(0, n.rfind("."))));
                  }

                  s += "\t\t\t[";
                  s += _myName.c_str();
                  s += "] StateMachine ";

                  char buf[10];
                  memset(buf, 0, 10);
                  sprintf(buf, "%d", i);
                  s += std::string(buf);
                  s += " : state ";
                  memset(buf, 0, 10);
                  sprintf(buf, "%d", StateList[i]);
                  s += std::string(buf);
                  s += ", name ";
                  if (strlen(name) > 150)
                  {
                     s += "...";
                     s += name + strlen(name) - 150;
                  }
                  else
                  {
                     s += name;
                  }
               }

               //vec.push_back(s);
               //s.clear();
               s += "\n";
            }

            std::string s2;
            s2 += "[";
            s2 += _myName.c_str();
            s2 += "] *** Currently active states ***\n";
            for (std::vector<stateInfo>::iterator it = activeStates.begin(); it != activeStates.end(); it++)
            {
               s2 += "[";
               s2 += _myName.c_str();
               s2 += "] active state ";
               if ((*it)._activeStateName.length() > 150)
               {
                  s2 += "...";
                  s2 += (*it)._activeStateName.substr((*it)._activeStateName.length() - 150);
               }
               else
               {
                  s2 += (*it)._activeStateName;
               }

               s2 += " : Statemachine ";
               char buf[10];
               memset(buf, 0, 10);
               sprintf(buf, "%d", (*it)._activeStatemachineId);
               s2 += std::string(buf);
               s2 += " : state id ";
               memset(buf, 0, 10);
               sprintf(buf, "%d", (*it)._activeStateId);
               s2 += std::string(buf);

               s2 += "\n";
            }

            s = s2 + s;
         }
#else
         s += "[";
         s += _myName.c_str();
         s += "] getStateInfo not supported, check generator settings";
#endif

         return s;
      }

      /****************************************************************************************************/

      /**************************************************************************************
      *FUNCTION:     printWidgetPropertyInfo
      *DESCRIPTION:  prints the property Value of the given widget's propertyName
      *PARAMETER:
      *RETURNVALUE:  void
      **************************************************************************************/
      template <class TBaseWidget, class TTextWidget, class TScrollableTextWidgetType>
      bool printWidgetPropertyInfo(Candera::WidgetBase* widgetBase, const char* propertyName, std::string& printData)
      {
         bool result = false;
         TBaseWidget /*BaseWidget2D*/* bw = dynamic_cast<TBaseWidget*>(widgetBase);
         TTextWidget /*TextWidget2D*/* textWidget = dynamic_cast<TTextWidget*>(bw);
         TScrollableTextWidgetType* scrollableTextWidget = dynamic_cast<TScrollableTextWidgetType*>(bw);

         if (widgetBase != 0)
         {
            if (scrollableTextWidget && strcmp(propertyName, "Text") == 0) // if we come across ScrollableTextWidget2D, then perform custom handling
            {
               Candera::String textdata = scrollableTextWidget->GetText();
               if (textdata.GetCharCount() != 0)
               {
                  TBaseWidget* bw = dynamic_cast<TBaseWidget*>(widgetBase);
                  if ((bw != 0) && (bw->GetNode() != 0) && (bw->GetNode()->IsEffectiveRenderingEnabled()))
                  {
                     SECURE_FEATSTD_STRING_ACCESS_BEGIN(textdata);
                     //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "PropertyName = %50s, Value = %150s", propertyName, textdata.GetCString());
                     printData += "PropertyName = ";
                     printData += propertyName;
                     printData += ", Value = ";
                     printData += textdata.GetCString();
                     printData += "\n";
                     SECURE_FEATSTD_STRING_ACCESS_END();
                  }
               }
               return true;
            }
            result = true;
            const FeatStd::Char* widgetType = bw != 0 ? bw->GetTypeName() : "Not BaseWidget2D type";
            //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "WidgetInstanceName = %s, WidgetTemplate = %s", widgetBase->GetName(), widgetType);
            printData += "WidgetInstanceName = ";
            printData += widgetBase->GetName();
            printData += ", WidgetTemplate = ";
            printData += widgetType;
            printData += "\n";

            Candera::MetaInfo::WidgetMetaInfo* metaInfo = widgetBase->GetMetaInfo();
            if (metaInfo != 0)
            {
               Candera::MetaInfo::PropertyMetaInfo<Candera::WidgetBase>* propertyMetaInfo;
               char buffer[100];

               propertyMetaInfo = metaInfo->LookupItem(propertyName);

               if (0 == propertyMetaInfo)
               {
                  //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "NULL pointer : propertyMetaInfo");
                  printData += "NULL pointer : propertyMetaInfo\n";
               }
               else if (strcmp(propertyMetaInfo->GetName(), propertyName) == 0)
               {
                  if (!propertyMetaInfo->Get(widgetBase, buffer, 100))
                  {
                     if (textWidget != 0 && (strcmp(propertyName, "Style") == 0))
                     {
                        Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> style = textWidget->GetStyle();
                        if (!style.PointsToNull())
                        {
                           Candera::TextRendering::Font fnt = style->GetDefaultFont();
                           //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "PropertyName = %s, FontName = %s, FontHeight = %d", propertyName, fnt.GetFaceName(), fnt.GetHeight());
                           printData += "PropertyName = ";
                           printData += propertyName;
                           printData += ", FontName = ";
                           printData += fnt.GetFaceName();
                           printData += " FontHeight = ";
                           std::stringstream ss;
                           ss << fnt.GetHeight();
                           printData += ss.str();
                           printData += "\n";
                        }
                     }
                     else
                     {
                        //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "PropertyName = %s, Property value for the given widget '%s' failed to transform to a string buffer", propertyName, widgetBase->GetName());
                        printData += "PropertyName = ";
                        printData += propertyName;
                        printData += ", Property value for the given widget ' ";
                        printData += widgetBase->GetName();
                        printData += "' failed to transform to a string buffer\n";
                     }
                  }
                  else
                  {
                     //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "PropertyName = %s, Value = %s", propertyName, buffer);
                     printData += "PropertyName = ";
                     printData += propertyName;
                     printData += ", Value = ";
                     printData += buffer;
                     printData += "\n";
                     memset(buffer, 0, sizeof(buffer));
                  }
               }
               else
               {
                  //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "Input PropertyName '%s' doesnot match the metaInfo '%s'", propertyName, propertyMetaInfo->GetName());
                  printData += "Input PropertyName '";
                  printData += propertyName;
                  printData += "' doesnot match the metaInfo '";
                  printData += propertyMetaInfo->GetName();
                  printData += "'\n";
               }
            }
         }
         return result;
      }

      /**************************************************************************************
      *FUNCTION:     printfWidgetInstanceInfo
      *DESCRIPTION:  prints the WidgetName and the values of its properties
      *PARAMETER:    WidgetBase*
      *RETURNVALUE:  void
      **************************************************************************************/
      template <class TBaseWidget, class TTextWidget, class TScrollableTextWidgetType>
      bool printfWidgetInstanceInfo(Candera::WidgetBase* widgetBase, std::string& printData)
      {
         bool result = false;
         TBaseWidget /*BaseWidget2D*/* bw = dynamic_cast<TBaseWidget*>(widgetBase);
         TTextWidget/*TextWidget2D*/* textWidget = dynamic_cast<TTextWidget*>(bw);
         TScrollableTextWidgetType* scrollableTextWidget = dynamic_cast<TScrollableTextWidgetType*>(bw);

         if (widgetBase != 0)
         {
            result = true;
            TBaseWidget* bw = dynamic_cast<TBaseWidget*>(widgetBase);
            const FeatStd::Char* widgetType = bw != 0 ? bw->GetTypeName() : "Not BaseWidget2D type";
            const FeatStd::Char* widgetName = bw != 0 ? bw->GetLegacyName() : widgetBase->GetName();

            //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "WidgetInstanceName = %s, WidgetTemplate = %s", widgetName, widgetType);
            printData += "WidgetInstanceName = ";
            printData += widgetName;
            printData += ", WidgetTemplate = ";
            printData += widgetType;
            printData += "\n";

            Candera::MetaInfo::WidgetMetaInfo* metaInfo = widgetBase->GetMetaInfo();
            if (metaInfo != 0)
            {
               //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "Widget Property Count %d", metaInfo->GetItemCount());
               printData += "Widget Property Count ";
               std::stringstream ss;
               ss << metaInfo->GetItemCount();
               printData += ss.str();
               printData += "\n";

               char buffer[100];
               for (int i = 0; i < metaInfo->GetItemCount(); i++)
               {
                  const char* propertyName = metaInfo->GetItem(i)->GetName();
                  Candera::MetaInfo::PropertyMetaInfo<Candera::WidgetBase>* propertyMetaInfo = metaInfo->LookupItem(propertyName);

                  if (0 == propertyMetaInfo)
                  {
                     //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "NULL pointer : propertyMetaInfo");
                     printData += "NULL pointer : propertyMetaInfo\n";
                  }
                  else
                  {
                     if (scrollableTextWidget && (strcmp(propertyName, "Text") == 0))
                     {
                        Candera::String textdata = scrollableTextWidget->GetText();
                        if (textdata.GetCharCount() != 0)
                        {
                           TBaseWidget* bw = dynamic_cast<TBaseWidget*>(widgetBase);
                           if ((bw != 0) && (bw->GetNode() != 0) && (bw->GetNode()->IsEffectiveRenderingEnabled()))
                           {
                              SECURE_FEATSTD_STRING_ACCESS_BEGIN(textdata);
                              //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "PropertyName = %50s, Value = %150s", propertyName, textdata.GetCString());
                              printData += "PropertyName = ";
                              printData += propertyName;
                              printData += ", Value = ";
                              printData += textdata.GetCString();
                              printData += "\n";
                              SECURE_FEATSTD_STRING_ACCESS_END();
                           }
                        }
                     }
                     else if (!propertyMetaInfo->Get(widgetBase, buffer, 100))
                     {
                        if ((strcmp(propertyName, "Style") == 0) && (textWidget != 0))
                        {
                           Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> style = textWidget->GetStyle();
                           if (!style.PointsToNull())
                           {
                              Candera::TextRendering::Font fnt = style->GetDefaultFont();
                              //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "PropertyName = %s, FontName = %s, FontHeight = %d", propertyName, fnt.GetFaceName(), fnt.GetHeight());
                              printData += "PropertyName = ";
                              printData += propertyName;
                              printData += ", FontName = ";
                              printData += fnt.GetFaceName();
                              printData += ", FontHeight = ";
                              std::stringstream ss;
                              ss << fnt.GetHeight();
                              printData += ss.str();
                              printData += "\n";
                           }
                        }
                        else if (strcmp(propertyName, "Node") == 0)
                        {
                           // do nothing
                        }
                        else
                        {
                           //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "PropertyName = %s, Property value for the given widget < %s > failed to transform to a string buffer", propertyName, widgetName);
                           printData += "PropertyName = ";
                           printData += propertyName;
                           printData += ", Property value for the given widget < ";
                           printData += widgetName;
                           printData += " > ";
                           printData += "failed to transform to a string buffer\n";
                        }
                     }
                     else
                     {
                        //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "PropertyName = %s, Value = %s", propertyName, buffer);
                        printData += "PropertyName = ";
                        printData += propertyName;
                        printData += ", Value = ";
                        printData += buffer;
                        printData += "\n";
                        memset(buffer, 0, sizeof(buffer));
                     }
                  }
               }
            }
         }
         return result;
      }

      /**************************************************************************************
      *FUNCTION:     getListWidget
      *DESCRIPTION:  Used to get the ListWidget based on the given ListId in a view.
      *PARAMETER:    listId, Courier::View
      *RETURNVALUE:  ListWidget2D
      **************************************************************************************/
      template<class TListWidget, class TWidgetFinder>
      TListWidget* /*ListWidget2D**/ getListWidget(const Courier::UInt32& listId, Courier::View* view)
      {
         TWidgetFinder /*ListWidgetFinderCallback*/ callback(listId);
         WidgetCheckReqMsg widgetCheckMsg(&callback);

         if (view != 0)
         {
            if (view->OnMessage(widgetCheckMsg))
            {
               return callback.GetFlexListWidget2D();
            }
         }
         return 0;
      }

      template<class TWidgetFinder, class TListWidget, class TListWidgetFinder>
      bool printAllVisibleTextInList(const Courier::UInt32& listId, Courier::View* view, std::string& printData)
      {
         /*ListWidget2D**/TListWidget* list = getListWidget<TListWidget, TListWidgetFinder>(listId, view);
         if (list != 0)
         {
            HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "LIST OF ALL THE TEXT IN CURRENT VIEW [%s]", view->GetId().CStr())
            TWidgetFinder /*PrintTextInListCallback*/ callback(&printData);
            WidgetCheckReqMsg msgToFindTextWidgets(&callback);
            //if (list->OnMessage(msgToFindTextWidgets))
            //{
            list->OnMessage(msgToFindTextWidgets);
            return true;
            //}
         }
         else
         {
            HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "ListWidget with Id %d not found", listId);
         }
         return false;
      }

      template<class TWidgetFinder>
      bool printAllVisibleTextInView(Courier::View* view, std::string& printData)
      {
         if (view != 0 && view->IsRenderingEnabled())
         {
            //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "LIST OF ALL THE TEXT IN CURRENT VIEW [%s]", view->GetId().CStr())
            printData += "LIST OF ALL THE TEXT IN CURRENT VIEW [";
            printData += view->GetId().CStr();
            printData += "]\n";
            TWidgetFinder /*PrintAllVisibleTextCallback*/ callback(&printData);
            WidgetCheckReqMsg msgToFindTextWidgets(&callback);
            //if (view->OnMessage(msgToFindTextWidgets))
            //{
            view->OnMessage(msgToFindTextWidgets);
            return true;
            //}
         }
         return false;
      }

      template<class TWidgetFinder>
      bool printAllVisibleTextInList(Courier::View* view, std::string& printData)
      {
         if (view != 0 && view->IsRenderingEnabled())
         {
            HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "LIST OF ALL THE TEXT IN ALL LISTWIDGETS");
            TWidgetFinder /*PrintTextInListCallback*/ callback(&printData);
            WidgetCheckReqMsg msgToFindTextWidgets(&callback);
            //if (view->OnMessage(msgToFindTextWidgets))
            //{
            view->OnMessage(msgToFindTextWidgets);
            return true;
            //}
         }
         return false;
      }

      /**************************************************************************************
      *  FUNCTION:    getDescendentWidgets
      *  DESCRIPTION: Returns a vector of child widgets with the given descenetWidgetName for the given ParentWidget.
      *               If childWidgetName is not specified, the function returns all the childWidgets for the given ParentWidget.
      *  PARAMETER:
      *  RETURNVALUE: vector of Candera::WidgetBase*
      **************************************************************************************/
      template<class TBaseWidget, class TWidgetFinder>
      FeatStd::Internal::Vector<Candera::WidgetBase*> getDescendentWidgets(std::string parentWidgetName, std::string descendentWidgetName, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor, std::string& printData)
      {
         Candera::WidgetBase* parentWidgetBasePtr = getWidgetInstance(viewHandler, hmiRenderViewVisitor, parentWidgetName.c_str(), printData);
         if (parentWidgetBasePtr != 0)
         {
            /*BaseWidget2D*/ TBaseWidget* parentWidgetPtr = static_cast<TBaseWidget*>(parentWidgetBasePtr);
            return  /*WidgetFinder*/ TWidgetFinder::FindDescendantWidget2D(/*WidgetFinder*/ TWidgetFinder::GetSceneContext(parentWidgetPtr), parentWidgetPtr->GetNode(), descendentWidgetName, true);
         }
         return FeatStd::Internal::Vector<Candera::WidgetBase*>();
      }

      /**************************************************************************************
      *  FUNCTION:    getWidgetPropertyInfo
      *  DESCRIPTION: Prints the property Value of the given widget's property name and Widget type.
      *               By default the Widget type considered is Candera::WidgetBase.
      *               For getting info about specific child widget types, use the template arguments.
      *  PARAMETER:
      *  RETURNVALUE:
      **************************************************************************************/
      template<class TChildWidgetType, class TBaseWidgetType, class TTextWidgetType, class TWidgetFinder, class TScrollableTextWidgetType>
      bool getWidgetPropertyInfo(const std::string& parentWidgetName, const std::string& childWidgetName, const char* propertyName , \
                                 Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor, std::string& printData)
      {
         bool result = false;
         if (parentWidgetName.empty())
         {
            Candera::WidgetBase* widgetBase = getWidgetInstance(viewHandler, hmiRenderViewVisitor, childWidgetName.c_str(), printData);
            if (widgetBase != 0)
            {
               result = true;
               (void)printWidgetPropertyInfo<TBaseWidgetType, TTextWidgetType, TScrollableTextWidgetType>(widgetBase, propertyName, printData);
            }
         }
         else
         {
            // parent widget name is specified
            FeatStd::Internal::Vector<Candera::WidgetBase*> childWidgets = getDescendentWidgets<TBaseWidgetType, TWidgetFinder>(parentWidgetName, childWidgetName, viewHandler, hmiRenderViewVisitor, printData);
            if (childWidgets.Size() == 0)
            {
               printData += "ChildWidget '";
               printData += childWidgetName.c_str();
               printData += "' not found\n";
               return false; // error condition will be printed independent of the return value
            }
            static size_t childWidgetsSize = 0;
            for (std::size_t i = 0; i < childWidgets.Size(); i++)
            {
               if (childWidgets[i] != 0)
               {
                  if (dynamic_cast<TChildWidgetType>(childWidgets[i]))
                  {
                     //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "------ print propertyInfo childWidget #%u ------", (++childWidgetsSize));
                     printData += "------ print propertyInfo childWidget #";
                     std::stringstream ss;
                     ss << ++childWidgetsSize;
                     printData += ss.str();
                     printData += " ------\n";
                     (void)printWidgetPropertyInfo<TBaseWidgetType, TTextWidgetType, TScrollableTextWidgetType>(childWidgets[i], propertyName, printData);
                     result = true;
                  }
               }
            }
            childWidgetsSize = 0; // reset the counter
         }
         return result;
      }

      /**************************************************************************************
      *FUNCTION:    getWidgetPropertyInfo
      *DESCRIPTION: prints the property Value of the given widget's property
      *PARAMETER:
      *RETURNVALUE:
      **************************************************************************************/
      template<class TBaseWidgetType, class TTextWidgetType, class TWidgetFinder, class TScrollableTextWidgetType>
      bool getWidgetPropertyInfo(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
      {
         std::vector<std::string> splitData = ::hmibase::util::split(msg.GetUserData3().GetCString(), '#');
         std::string parentWidgetName;
         std::string childWidgetName;

         std::string propertyNames[2];
         static bool propertyNameIndex = false;
         propertyNames[propertyNameIndex] = msg.GetUserData4().GetCString();

         bool result = false;
         std::string printData;

         if (splitData.size() == 2)
         {
            parentWidgetName = splitData[0];
            childWidgetName = splitData[1];
         }
         else
         {
            childWidgetName = msg.GetUserData3().GetCString();
         }

         printData += "###  WIDGET PROPERTIES AND VALUES START  ###\n";

         // Using default template type argument : Candera::WidgetBase
         result = getWidgetPropertyInfo<Candera::WidgetBase*, TBaseWidgetType, TTextWidgetType, TWidgetFinder, TScrollableTextWidgetType>(parentWidgetName, childWidgetName,
                  propertyNames[propertyNameIndex].c_str(), viewHandler, hmiRenderViewVisitor, printData);

         printData += "###  WIDGET PROPERTIES AND VALUES END  ###\n";

         if (result)
         {
            std::vector<std::string> vecData = ::hmibase::util::split(printData, '\n');
            std::vector<std::string>::iterator it = vecData.begin();

            while (it != vecData.end())
            {
               HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "%s", (*it).c_str());
               ++it;
            }
            propertyNameIndex = !propertyNameIndex; // toggle the buffers to be used only for positive result (widget found)
         }
         return result;
      }

      /**************************************************************************************
      *FUNCTION:    getTextIdsInListItem
      *DESCRIPTION: prints the text id of the given widget
      *PARAMETER:
      *RETURNVALUE:
      **************************************************************************************/
      template<class TBaseWidget, class TWidget, class TWidgetFinder, class TScrollableTextWidgetType>
      bool getTextIdsInListItem(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
      {
         HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "###  GET_ALL_VISIBLE_TEXTIDS_IN_LIST_ITEM START ###");
         std::string parentWidgetName;
         std::string printData;
         parentWidgetName = msg.GetUserData3().GetCString();

         getWidgetPropertyInfo < Candera::WidgetBase*, TBaseWidget, TWidget /*TextWidget2D**/, TWidgetFinder, TScrollableTextWidgetType > (parentWidgetName, "", "Text", viewHandler, hmiRenderViewVisitor, printData);
         HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "###  GET_ALL_VISIBLE_TEXTIDS_IN_LIST_ITEM END ###");
         return true;
      }

      /**************************************************************************************
      *  FUNCTION:    getWidgetInstanceInfo
      *  DESCRIPTION: Prints all the property Values of the given widget's name.
      If the childwidget name is not given, all the child widgets can be filtered by its type.
      *               By default the Widget type considered is Candera::WidgetBase.
      *               For getting info about specific child widget types, use the template arguments.
      *  PARAMETER:
      *  RETURNVALUE:
      **************************************************************************************/
      template<class TChildWidgetType, class TBaseWidgetType, class TTextWidgetType, class TWidgetFinder, class TScrollableTextWidgetType>
      bool getWidgetInstanceInfo(const std::string& parentWidgetName, const std::string& childWidgetName, Courier::IViewHandler& viewHandler,
                                 const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor, std::string& printData)
      {
         bool result = false;
         if (parentWidgetName.empty())
         {
            Candera::WidgetBase* widgetBase = getWidgetInstance(viewHandler, hmiRenderViewVisitor, childWidgetName.c_str(), printData);
            result = printfWidgetInstanceInfo<TBaseWidgetType, TTextWidgetType, TScrollableTextWidgetType>(widgetBase, printData);
         }
         else
         {
            // UserData3 => ParentWidgetName#ChildWidgetName
            FeatStd::Internal::Vector<Candera::WidgetBase*> childWidgets = getDescendentWidgets<TBaseWidgetType, TWidgetFinder>(parentWidgetName, childWidgetName, viewHandler, hmiRenderViewVisitor, printData);
            if (childWidgets.Size() == 0)
            {
               printData += "ChildWidget '";
               printData += childWidgetName.c_str();
               printData += "' not found\n";
               return true;
            }
            for (std::size_t i = 0; i < childWidgets.Size(); i++)
            {
               if (childWidgets[i] != 0)
               {
                  if (dynamic_cast<TChildWidgetType>(childWidgets[i]))
                  {
                     //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "------ print InstanceInfo ChildWidget #%d ------", i + 1);
                     printData += "------ print InstanceInfo ChildWidget #";
                     std::stringstream ss;
                     ss << (i + 1);
                     printData += ss.str();
                     printData += " ------\n";
                     result = true;
                     printfWidgetInstanceInfo<TBaseWidgetType, TTextWidgetType, TScrollableTextWidgetType>(childWidgets[i], printData);
                  }
               }
            }
         }
         return result;
      }

      /**************************************************************************************
      *  FUNCTION:     getWidgetInstanceInfo
      *  DESCRIPTION:
      *  PARAMETER:
      *  RETURNVALUE:
      **************************************************************************************/
      template<class TBaseWidgetType, class TTextWidgetType, class TWidgetFinder, class TScrollableTextWidgetType>
      bool getWidgetInstanceInfo(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
      {
         std::vector<std::string> splitData = ::hmibase::util::split(msg.GetUserData3().GetCString(), '#');
         std::string parentWidgetName;
         std::string childWidgetName;
         std::string printData;
         bool result = false;

         if (splitData.size() == 2)
         {
            parentWidgetName = splitData[0];
            childWidgetName = splitData[1];
         }
         else
         {
            childWidgetName = msg.GetUserData3().GetCString();
         }

         printData += "### WIDGET INSTANCE PROPERTIES AND THEIR VALUES START ###\n";

         result = getWidgetInstanceInfo<Candera::WidgetBase*, TBaseWidgetType, TTextWidgetType, TWidgetFinder, TScrollableTextWidgetType >(parentWidgetName, childWidgetName, viewHandler, hmiRenderViewVisitor, printData);

         printData += "### WIDGET INSTANCE PROPERTIES AND THEIR VALUES END ###\n";

         //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "### WIDGET INSTANCE PROPERTIES AND THEIR VALUES START ###");
         std::vector<std::string> vecData = ::hmibase::util::split(printData, '\n');
         std::vector<std::string>::iterator it = vecData.begin();

         while (it != vecData.end())
         {
            HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "%s", (*it).c_str());
            ++it;
         }
         //HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "### WIDGET INSTANCE PROPERTIES AND THEIR VALUES END ###");
         return result;
      }

      /**************************************************************************************
      *FUNCTION:     simTouchByText
      *DESCRIPTION:  simulate a touch message for a Touch surfaces. Enter the text on
      *touchable surface with TTFis command
      *DESCRIPTION:  Used for simulating touch message from the Ttfis command.
      *PARAMETER:    SimTouchByTextReqMsg, Courier::IViewHandler, HMIRenderViewVisitor
      *RETURNVALUE:
      **************************************************************************************/
      template <class TWidgetFinder>
      bool simTouchByText(const SimTouchReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
      {
         bool bReturn = false;
         Candera::Widget2D* pWidget;
         Courier::View* view = 0;

         const ::hmibase::view::HMIRenderViewVisitor::tViewInfoVector& viewInfoVec = hmiRenderViewVisitor.getActiveViewInfo();
         HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfoVec.begin();

         const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector = msg.GetData().getSurfaceInfoData();

         for (; cit != viewInfoVec.end(); ++cit)
         {
            std::vector<unsigned int>::const_iterator it = cit->surfaceIds.begin();
            for (; it != cit->surfaceIds.end(); ++it)
            {
               view = NULL;
               if (ScreenBrokerClient::GetInstance().IsMainSurface(*it))
               {
                  if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(*it) == ScreenBroker::SurfaceState::Mapped)
                  {
                     view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));
                  }
               }
               else if (static_cast<int>(hmibase::SURFACEID_POPUPBASE) <= *it && static_cast<int>(hmibase::SURFACEID_POPUPBASE_MAX) >= *it)
               {
                  view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));
               }
               else if (*it == ScreenBrokerClient::GetInstance().StatusBarSurfaceId())
               {
                  // this scene is the StatusBar surface scene. chk surface state
                  if (ScreenBrokerClient::GetInstance().GetStatusBarSurfaceState() == ScreenBroker::SurfaceState::Mapped)
                  {
                     view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));
                  }
               }
               else if ((*it == ScreenBrokerClient::GetInstance().FooterBarSurfaceId()) &&
                        (ScreenBrokerClient::GetInstance().GetFooterBarSurfaceState() == ScreenBroker::SurfaceState::Mapped))
               {
                  view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "SimTouchByText scene[%100s]", view->GetId().CStr());
               }

               if (view != NULL)
               {
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "### SIM TOUCHBY TEXT START ###");
                  SECURE_FEATSTD_STRING_ACCESS_BEGIN(msg.GetWidgetIdentifier());
                  TWidgetFinder /*TextWidgetFinderCallback*/ callback(msg.GetWidgetIdentifier().GetCString());
                  WidgetCheckReqMsg msg(&callback);
                  if (view->OnMessage(msg))
                  {
                     //Get searching widget name
                     pWidget = callback.GetTextWidget2D();

                     postTouchMsg(surfaceInfoVector, pWidget);
                  }
                  SECURE_FEATSTD_STRING_ACCESS_END();
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "### SIM TOUCHBY TEXT END ###");
               }
            }
         }
         return bReturn;
      }

      /**************************************************************************************
      *FUNCTION:     simTouchByTextId
      *DESCRIPTION:  simulate a touch message for a Touch surfaces. Enter the text on
      *touchable surface with TTFis command
      *DESCRIPTION:  Used for simulating touch message from the Ttfis command.
      *PARAMETER:    SimTouchByTextReqMsg, Courier::IViewHandler, HMIRenderViewVisitor
      *RETURNVALUE:
      **************************************************************************************/
      template <class TWidgetFinder>
      bool simTouchByTextId(const SimTouchReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
      {
         bool bReturn = false;
         Candera::Widget2D* pWidget;
         Courier::View* view = 0;

         const ::hmibase::view::HMIRenderViewVisitor::tViewInfoVector& viewInfoVec = hmiRenderViewVisitor.getActiveViewInfo();
         HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfoVec.begin();

         const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector = msg.GetData().getSurfaceInfoData();

         for (; cit != viewInfoVec.end(); ++cit)
         {
            std::vector<unsigned int>::const_iterator it = cit->surfaceIds.begin();
            for (; it != cit->surfaceIds.end(); ++it)
            {
               view = NULL;
               if (ScreenBrokerClient::GetInstance().IsMainSurface(*it))
               {
                  if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(*it) == ScreenBroker::SurfaceState::Mapped)
                  {
                     view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));
                  }
               }
               else if (static_cast<int>(hmibase::SURFACEID_POPUPBASE) <= *it && static_cast<int>(hmibase::SURFACEID_POPUPBASE_MAX) >= *it)
               {
                  view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));
               }
               else if (*it == ScreenBrokerClient::GetInstance().StatusBarSurfaceId())
               {
                  // this scene is the statusbar surface scene. chk surface state
                  if (ScreenBrokerClient::GetInstance().GetStatusBarSurfaceState() == ScreenBroker::SurfaceState::Mapped)
                  {
                     view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));
                  }
               }
               else if ((*it == ScreenBrokerClient::GetInstance().FooterBarSurfaceId()) &&
                        (ScreenBrokerClient::GetInstance().GetFooterBarSurfaceState() == ScreenBroker::SurfaceState::Mapped))
               {
                  view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "SimTouchByTextID scene[%100s]", view->GetId().CStr());
               }

               if (view != NULL)
               {
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "### SIM TOUCHBY TEXT ID START ###");
                  SECURE_FEATSTD_STRING_ACCESS_BEGIN(msg.GetWidgetIdentifier());

                  FeatStd::String convertedTextString = msg.GetWidgetIdentifier().GetCString();
                  TWidgetFinder /*TextIdWidgetFinderCallback*/ callback(msg.GetWidgetIdentifier().GetCString());
                  WidgetCheckReqMsg msg(&callback);
                  if (view->OnMessage(msg))
                  {
                     //Get searching widget name
                     pWidget = callback.GetTextWidget2D();

                     postTouchMsg(surfaceInfoVector, pWidget);
                  }
                  else
                  {
                     HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "Searching TextID [%s] is not found", convertedTextString.GetCString());
                  }
                  SECURE_FEATSTD_STRING_ACCESS_END();
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "### SIM TOUCHBY TEXT ID END ###");
               }
            }
         }
         return bReturn;
      }

      /**************************************************************************************
      *FUNCTION:     simTouchByIndex
      *DESCRIPTION:  simulate a touch message for a ListItem in the FlexListWidget2D.
      *DESCRIPTION:  Used for simulating touch message from the Ttfis command.
      *PARAMETER:    SimTouchByIndexReqMsg, Courier::IViewHandler, HMIRenderViewVisitor
      *RETURNVALUE:
      **************************************************************************************/
      template <class TWidgetFinder, class TTouchableWidgetType, class TListWidgetType, class TListWidgetFinder>
      bool simTouchByIndex(const SimTouchReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
      {
         const ::hmibase::view::HMIRenderViewVisitor::tViewInfoVector& viewInfoVec = hmiRenderViewVisitor.getActiveViewInfo();
         HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfoVec.begin();
         Courier::View* view = 0;
         TListWidgetType* flexListWidget2DPtr = 0;
         TTouchableWidgetType* baseButtonWidget2DPtr = 0;
         Courier::UInt32 buttonWidgetIndex = msg.GetButtonWidgetIndex();
         const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector = msg.GetData().getSurfaceInfoData();

         for (; cit != viewInfoVec.end(); ++cit)
         {
            std::vector<unsigned int>::const_iterator it = cit->surfaceIds.begin();
            for (; it != cit->surfaceIds.end(); ++it)
            {
               if (ScreenBrokerClient::GetInstance().IsMainSurface(*it))
               {
                  if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(*it) == ScreenBroker::SurfaceState::Mapped)
                  {
                     HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "ViewName = %s", cit->viewName.c_str());
                     view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));

                     if (view != NULL)
                     {
                        HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "### SIM TOUCHBY LIST ITEM INDEX START ###");

                        flexListWidget2DPtr = getListWidget<TListWidgetType, TListWidgetFinder>(msg.GetListId(), view);

                        if (flexListWidget2DPtr != 0)
                        {
                           TWidgetFinder /*IndexedWidgetFinderCallback<BaseButtonWidget2D>*/ callbackFindListItemButtonWidget(buttonWidgetIndex);
                           WidgetCheckReqMsg msgFindListItem(&callbackFindListItemButtonWidget);
                           if (flexListWidget2DPtr->OnMessage(msgFindListItem))
                           {
                              baseButtonWidget2DPtr = callbackFindListItemButtonWidget.GetWidgetPtr();

                              if (baseButtonWidget2DPtr != 0)
                              {
                                 // post Touch message
                                 postTouchMsg(surfaceInfoVector, baseButtonWidget2DPtr);
                                 return true;
                              }
                           }
                        }
                        HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "### SIM TOUCHBY LIST ITEM INDEX END ###");
                     }
                  }
               }
            }
         }

         return false;
      }

      template<class TBaseWidget, class TWidgetFinder>
      bool ProcessWidgetPropertySetRequest(const WidgetPropertySetterReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
      {
         std::vector<std::string> splitData = ::hmibase::util::split(msg.GetWidgetName().GetCString(), '#');
         std::string parentWidgetName;
         std::string childWidgetName;
         std::string printData;
         bool isParentWidgetNamePresent = false;
         bool result = true;

         if (splitData.size() == 2)
         {
            parentWidgetName = splitData[0];
            childWidgetName = splitData[1];
            isParentWidgetNamePresent = true;
         }

         if (!isParentWidgetNamePresent)
         {
            Candera::WidgetBase* widgetBase = getWidgetInstance(viewHandler, hmiRenderViewVisitor, msg.GetWidgetName().GetCString(), printData);
            result = setWidgetPropertyValue(widgetBase, msg.GetPropertyName().GetCString(), msg.GetPropertyValue().GetCString(), printData);
         }
         else
         {
            // ParentWidgetName is specified
            FeatStd::Internal::Vector<Candera::WidgetBase*> childWidgets = getDescendentWidgets<TBaseWidget, TWidgetFinder>(parentWidgetName, childWidgetName, viewHandler, hmiRenderViewVisitor, printData);
            for (std::size_t i = 0; i < childWidgets.Size(); i++)
            {
               if (childWidgets[i] != 0)
               {
                  result = setWidgetPropertyValue(childWidgets[i], msg.GetPropertyName().GetCString(), msg.GetPropertyValue().GetCString(), printData);
               }
            }
         }
         if (result)
         {
            std::vector<std::string> vecData = ::hmibase::util::split(printData, '\n');
            std::vector<std::string>::iterator it = vecData.begin();

            while (it != vecData.end())
            {
               HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "%s", (*it).c_str());
               ++it;
            }
         }
         return true;
      }

      template<class TChildWidgetType, class TBaseWidgetType, class TWidgetFinder>
      void getPossibleTouchPointsForElementWithName(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
      {
         const char* widgetName = msg.GetUserData4().GetCString();
         const char* viewName = msg.GetUserData3().GetCString();

         std::vector<std::string> splitData = ::hmibase::util::split(widgetName, '#');
         std::string parentWidgetName;
         std::string childWidgetName;
         std::string printData;
         FeatStd::Internal::Vector<Candera::WidgetBase*> widgets;

         if (splitData.size() == 2)
         {
            parentWidgetName = splitData[0];
            childWidgetName = splitData[1];
         }
         else
         {
            childWidgetName = widgetName;
         }

         if (parentWidgetName.empty())
         {
            Candera::WidgetBase* widget = FindWidget(viewHandler, viewName, widgetName, printData);
            if (!printPossibleTouchCoordinates(widget))
            {
               HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "Widget '%s' not found", widgetName);
            }
         }
         else
         {
            // parentwidget specified
            FeatStd::Internal::Vector<Candera::WidgetBase*> childWidgets = getDescendentWidgets<TBaseWidgetType, TWidgetFinder>(parentWidgetName, childWidgetName, viewHandler, hmiRenderViewVisitor, printData);

            for (std::size_t i = 0; i < childWidgets.Size(); i++)
            {
               if (childWidgets[i] != 0)
               {
                  if (dynamic_cast<TChildWidgetType>(childWidgets[i]))
                  {
                     HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "------ print coordinate info of ChildWidget #%d ------", i + 1);
                     if (!printPossibleTouchCoordinates(childWidgets[i]))
                     {
                        HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "Widget '%s' not found", childWidgets[i]->GetLegacyName());
                     }
                  }
               }
            }
         }
      }

      template < class TWidgetFinder>
      void getPossibleTouchPointsForElementsWithTextData(const QueryTraceInfoReqMsg& msg, Courier::View* view)
      {
         if (view != 0)
         {
            HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "Possibe touch points for Elements with Text");
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(msg.GetUserData3()); // get the Text/ TextId
            TWidgetFinder callback(msg.GetUserData3().GetCString());
            WidgetCheckReqMsg msg(&callback);
            if (view->OnMessage(msg))
            {
               printPossibleTouchCoordinates(callback.GetTextWidget2D());
            }
            SECURE_FEATSTD_STRING_ACCESS_END();
         }
      }

      template <class TWidgetFinder, class TTouchableWidgetType, class TListWidgetType, class TListWidgetFinder>
      void getPossibleTouchPointsForListWithIndex(const QueryTraceInfoReqMsg& msg, Courier::View* view)
      {
         TListWidgetType* flexListWidget2DPtr = 0;
         TTouchableWidgetType* widget = 0;
         flexListWidget2DPtr = getListWidget<TListWidgetType, TListWidgetFinder>(msg.GetUserData1(), view); // userData1 => ListId
         if (flexListWidget2DPtr != 0)
         {
            TWidgetFinder /*IndexedWidgetFinderCallback<BaseButtonWidget2D>*/ callbackFindListItemButtonWidget(msg.GetUserData2()); // userData2 => listItemIndex
            WidgetCheckReqMsg msgFindListItem(&callbackFindListItemButtonWidget);
            if (flexListWidget2DPtr->OnMessage(msgFindListItem))
            {
               widget = callbackFindListItemButtonWidget.GetWidgetPtr();
               if (widget != 0)
               {
                  printPossibleTouchCoordinates(widget);
               }
            }
            else
            {
               HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_FW,  "Widget not found at index '%d' in ListId '%d'", msg.GetUserData2(), msg.GetUserData1());
            }
         }
      }

   private:
      Candera::WidgetBase* FindWidget(Courier::IViewHandler& viewHandler, const Candera::Char* viewName, const char* widgetName, std::string& printData);
      bool isSurfaceTouchPointVisible(const hmibase::util::geometry::Point& pt, int surfaceId, const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector);
      bool simTouchByName(const SimTouchReqMsg& msg, Courier::ViewScene2D* viewScene2D, Courier::FrameworkWidget* fw);
      bool simTouchByName(const SimTouchReqMsg& msg, Courier::ViewScene3D* viewScene3D, Courier::FrameworkWidget* fw);
      bool simTouchByName(const SimTouchReqMsg& msg, Courier::View* view, Courier::FrameworkWidget* fw);
      bool simTouchByName(const SimTouchReqMsg& msg, Courier::IViewHandler& viewHandler);
      bool printAllVisibleText(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor, std::string& printData);
      std::string FindParentWidgetName(Candera::String widgetName);
      bool getScreenInfo(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor);
      void getFocusedWidgetInstanceInfo(Courier::IViewHandler& viewHandler);
      bool printPossibleTouchCoordinates(Candera::WidgetBase* wb);
      bool getPossibleTouchPoints(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const view::HMIRenderViewVisitor& hmiRenderViewVisitor);

      // To check if the given scene is currently displayed. Returns true if the given scene is visible
      bool isCurrentVisibleScene(const char* sceneName, const view::HMIRenderViewVisitor& hmiRenderViewVisitor);

      static IWidgetTraceQueryHelper* _hook;
      static Util::Timer& GetTimerInstance()
      {
         static Util::Timer _timer;
         _timer.setName("TraceQueryHelper", "SimTouch");
         return _timer;
      }
      HMIBASE_UNCOPYABLE_CLASS(TraceQueryHelper)
      /****************************************************************************************************/
};


}
}


#endif // TRACEQUERYHELPER_HPP
