/* ***************************************************************************************
* FILE:          TraceQueryHelper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TraceQueryHelper.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"

#include "TraceQueryHelper.hpp"
#include "Courier/Version.h"

#include "View/CGI/CgiExtensions/HMIRenderViewVisitor.h"

#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#ifndef WIN32
#include "ScreenBrokerSettings/Settings.h"
#endif

#include "AppUtils/SceneInfo.h"

#include "View/IMessageSystem.h"
#include "Trace/StartupInvestigation.h"
#include "Trace/MemoryStatistics.h"
#include "Trace/MessageStatistics.h"

#include "FeatStd/Monitor/PerformanceRecording/FileDumper.h"

#include "CanderaWidget/Widget2D/Widget2D.h"
#include "Courier/Visualization/ViewScene2D.h"
#include "CanderaWidget/Widget3D/Widget.h"
#include "Courier/Visualization/ViewScene3D.h"
#include <Courier/Messaging/MessagingMsgs.h>

#include "View/CGI/CgiExtensions/TouchInput.h"

#include "View/CGI/CgiExtensions/CourierMessageMapper.h"

#include "WidgetFinderCallback.h"
#include "WidgetInstanceFinderCallback.h"

#include "hmibase/util/StringUtils.h"

#include "FeatStd//Platform/PerfCounter.h"

#include <Focus/FManager.h>


#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TraceQueryHelper.cpp.trc.h"
#endif


namespace hmibase {
namespace trace {
TraceQueryHelper::IWidgetTraceQueryHelper* TraceQueryHelper::_hook = NULL;
/**************************************************************************************
*FUNCTION:    setWidgetPropertyValue
*DESCRIPTION: This function is used to set the widget property value given the below input parameters.
*PARAMETER:   WidgetBase pointer for which the property has to be set,
*PARAMETER:   Candera::String propertyName : Name of the widget property which has to be set.
*PARAMETER:   Candera::String propertyValue : Value of the property to be set.
*RETURNVALUE: void
**************************************************************************************/
bool TraceQueryHelper::setWidgetPropertyValue(Candera::WidgetBase* widgetBase, const char* propertyName, const char* propertyValue, std::string& printData)
{
   if (widgetBase != 0)
   {
      //ETG_TRACE_FATAL_THR_DCL((APP_TRACECLASS_ID(), "###  WIDGET PROPERTY VALUE SETTER START ###"));
      printData += "###  WIDGET PROPERTY VALUE SETTER START ###\n";
      //ETG_TRACE_FATAL_THR_DCL((APP_TRACECLASS_ID(), "WidgetInstanceName = %100s", widgetBase->GetName()));
      printData += "WidgetInstanceName = ";
      printData += widgetBase->GetName();
      printData += "\n";
      Candera::MetaInfo::WidgetMetaInfo* metaInfo = widgetBase->GetMetaInfo();
      if (metaInfo != 0)
      {
         Candera::MetaInfo::PropertyMetaInfo<Candera::WidgetBase>* propertyMetaInfo;

         std::string propertyNames[2];
         static bool propertyNameIndex = false;
         propertyNames[propertyNameIndex] = propertyName;

         propertyMetaInfo = metaInfo->LookupItem(propertyNames[propertyNameIndex].c_str());

         // temporary change to print the threadID, for debugging
         //ETG_TRACE_FATAL_THR_DCL((APP_TRACECLASS_ID(), "in TraceQueryHelper"));

         if (0 == propertyMetaInfo)
         {
            //ETG_TRACE_FATAL_THR_DCL((APP_TRACECLASS_ID(), "NULL pointer : propertyMetaInfo"));
            printData += "NULL pointer : propertyMetaInfo\n";
         }
         else if (strcmp(propertyMetaInfo->GetName(), propertyNames[propertyNameIndex].c_str()) == 0)
         {
            if (!propertyMetaInfo->Set(widgetBase, propertyValue))
            {
               //ETG_TRACE_FATAL_THR_DCL((APP_TRACECLASS_ID(), "PropertyName = %100s, Property value for the given widget < %100s > failed to transform to a string buffer", propertyName, widgetBase->GetName()));
               printData += "PropertyName = ";
               printData += propertyNames[propertyNameIndex].c_str();
               printData += ", Property value for the given widget <";
               printData += widgetBase->GetName();
               printData += "> failed to transform to a string buffer\n";
            }
         }
         else
         {
            //ETG_TRACE_FATAL_THR_DCL((APP_TRACECLASS_ID(), "Input PropertyName '%100s' doesnot match the metaInfo '%100s'", propertyName, propertyMetaInfo->GetName()));
            printData += "Input PropertyName '";
            printData += propertyNames[propertyNameIndex].c_str();
            printData += "' doesnot match the metaInfo '";
            printData += propertyMetaInfo->GetName();
            printData += "'\n";
         }

         propertyNameIndex = !propertyNameIndex;
      }
      //ETG_TRACE_FATAL_THR_DCL((APP_TRACECLASS_ID(), "###  WIDGET PROPERTY VALUE SETTER END ###"));
      printData += "###  WIDGET PROPERTY VALUE SETTER END ###\n";
      return true;
   }
   return false;
}


/**************************************************************************************
*   FUNCTION:     FindWidget
*   DESCRIPTION:  This function is used to find the widget pointer, given the below input parameters.
*   PARAMETER:    Courier::IViewHandler
*   PARAMETER:    const Candera::Char* : Name of the view which has the widget.
*   PARAMETER:    Candera::String widgetName : Name of the widget
*   RETURNVALUE:  Candera::WidgetBase
**************************************************************************************/
Candera::WidgetBase* TraceQueryHelper::FindWidget(Courier::IViewHandler& viewHandler, const Candera::Char* viewName, const char* widgetName, std::string& printData)
{
   //ETG_TRACE_FATAL_THR(("ViewName = %100s", viewName));
   printData += "ViewName = ";
   printData += viewName;
   printData += "\n";
   Courier::View* view = viewHandler.FindView(Courier::ViewId(viewName));

   if (view != NULL)
   {
      WidgetInstanceFinderCallback callback(widgetName);
      WidgetCheckReqMsg callBackMsg(&callback);

      if (view->OnMessage(callBackMsg))
      {
         // Widget Found
         return callback.getBaseWidget();
      }
   }

   //ETG_TRACE_FATAL_THR(("Widget = %100s not found", widgetName));
   printData += "Widget '";
   printData += widgetName;
   printData += "' not found\n";
   return NULL;
}


/**************************************************************************************
*FUNCTION:    getWidgetInstance
*DESCRIPTION: Function used to get the widget instance in the current scene, given the widgetName
*PARAMETER:   Candera::String WidgetName
*RETURNVALUE: Candera::WidgetBase*
**************************************************************************************/
Candera::WidgetBase* TraceQueryHelper::getWidgetInstance(Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor, const char* widgetName, std::string& printData)
{
   const HMIRenderViewVisitor::tViewInfoVector& viewInfoVec = hmiRenderViewVisitor.getActiveViewInfo();
   HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfoVec.begin();
   Candera::WidgetBase* widget = 0;

   for (; cit != viewInfoVec.end(); ++cit)
   {
      std::vector<unsigned int>::const_iterator it = cit->surfaceIds.begin();
      for (; it != cit->surfaceIds.end(); ++it)
      {
         if (ScreenBrokerClient::GetInstance().IsMainSurface(*it))
         {
            if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(*it) == ScreenBroker::SurfaceState::Mapped)
            {
               widget = FindWidget(viewHandler, cit->viewName.c_str(), widgetName, printData);
            }
         }
         else if (static_cast<int>(hmibase::SURFACEID_POPUPBASE) <= *it && static_cast<int>(hmibase::SURFACEID_POPUPBASE_MAX) >= *it)
         {
            widget = FindWidget(viewHandler, cit->viewName.c_str(), widgetName, printData);
         }
         else if (*it == ScreenBrokerClient::GetInstance().StatusBarSurfaceId())
         {
            // this scene is the statusbar surface scene. chk surface state
            if (ScreenBrokerClient::GetInstance().GetStatusBarSurfaceState() == ScreenBroker::SurfaceState::Mapped)
            {
               widget = FindWidget(viewHandler, cit->viewName.c_str(), widgetName, printData);
            }
         }
         else if ((*it == ScreenBrokerClient::GetInstance().FooterBarSurfaceId()) &&
                  (ScreenBrokerClient::GetInstance().GetFooterBarSurfaceState() == ScreenBroker::SurfaceState::Mapped))
         {
            widget = FindWidget(viewHandler, cit->viewName.c_str(), widgetName, printData);
         }
         if (widget != 0)
         {
            return widget;
         }
      }
   }
   return NULL;
}


/**************************************************************************************
*FUNCTION:     vPrintActiveViewIDs
*DESCRIPTION:  print the name of the current visible main screen if any
*PARAMETER:
*RETURNVALUE:
**************************************************************************************/
TMLCurrentViewInfo TraceQueryHelper::PrintActiveViewIDs(const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor, std::vector<std::string>& views)
{
   TMLCurrentViewInfo viewsToReturn;
   std::string s;
   // check if the application have only the popups
   if (ScreenBrokerClient::GetInstance().SurfaceId() == 0)
   {
      ETG_TRACE_FATAL_THR_DCL((APP_TRACECLASS_ID(), "No valid main surface id with this App"));
      return viewsToReturn;
   }

   const HMIRenderViewVisitor::tViewInfoVector& viewInfoVec = hmiRenderViewVisitor.getActiveViewInfo();
   ETG_TRACE_USR1_THR_DCL((APP_TRACECLASS_ID(), "vPrintActiveViewIDs()-TotalActiveViews[%u]", viewInfoVec.size()));

   for (HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfoVec.begin(); cit != viewInfoVec.end(); ++cit)
   {
      // check if view surface id is the main scene surface id
      std::vector<unsigned int>::const_iterator it = cit->surfaceIds.begin();
      for (; it != cit->surfaceIds.end(); ++it)
      {
         // this scene might be the main surface scene. chk surface state
         if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(*it) == ScreenBroker::SurfaceState::Mapped)
         {
            FeatStd::UInt32 displayId = ScreenBrokerClient::GetInstance().GetDisplayID(*it);
            views.push_back(cit->viewName);

            char buf[10];
            snprintf(buf, sizeof buf, "%u:%u", displayId, ScreenBrokerClient::GetInstance().SurfaceId());
            s += "M[";
            s += hmibase::trace::getAppName();
            s += "]:[";
            s += buf;
            s += "], '";
            s += cit->viewName;
            s += "'";

            viewsToReturn.mainScenes.push_back(s);
            // surface is currently visible
            ETG_TRACE_FATAL_THR(("Main scene[%16s]: displayId %u, surfaceId %u, '%100s'", hmibase::trace::getAppName().c_str(), displayId, ScreenBrokerClient::GetInstance().SurfaceId(), cit->viewName.c_str()));
         }
         else if (*it == ScreenBrokerClient::GetInstance().StatusBarSurfaceId())
         {
            // this scene is the statusbar surface scene. chk surface state
            if (ScreenBrokerClient::GetInstance().GetStatusBarSurfaceState() == ScreenBroker::SurfaceState::Mapped)
            {
               FeatStd::UInt32 displayId = ScreenBrokerClient::GetInstance().GetDisplayID(*it);
               views.push_back(cit->viewName);
               char buf[10];
               snprintf(buf, sizeof buf, "%u:%u", displayId, ScreenBrokerClient::GetInstance().StatusBarSurfaceId());
               s += "SB[";
               s += hmibase::trace::getAppName();
               s += "]:[";
               s += buf;
               s += "], '";
               s += cit->viewName;
               s += "'";

               viewsToReturn.otherScenes.push_back(s);
               // surface is currently visible
               ETG_TRACE_FATAL_THR(("Statusbar scene[%16s]: displayId %u, surfaceId %u, '%100s'", hmibase::trace::getAppName().c_str(), displayId, ScreenBrokerClient::GetInstance().StatusBarSurfaceId(), cit->viewName.c_str()));
            }
         }
         else if ((*it == ScreenBrokerClient::GetInstance().FooterBarSurfaceId()) &&
                  (ScreenBrokerClient::GetInstance().GetFooterBarSurfaceState() == ScreenBroker::SurfaceState::Mapped))
         {
            FeatStd::UInt32 displayId = ScreenBrokerClient::GetInstance().GetDisplayID(*it);
            views.push_back(cit->viewName);
            char buf[10];
            snprintf(buf, sizeof buf, "%u:%u", displayId, ScreenBrokerClient::GetInstance().FooterBarSurfaceId());
            s += "FB[";
            s += hmibase::trace::getAppName();
            s += "]:[";
            s += buf;
            s += "], '";
            s += cit->viewName;
            s += "'";

            viewsToReturn.otherScenes.push_back(s);
            ETG_TRACE_FATAL_THR(("Footerbar scene[%16s]: displayId %u, surfaceId %u, '%100s'", hmibase::trace::getAppName().c_str(), displayId, ScreenBrokerClient::GetInstance().FooterBarSurfaceId(), cit->viewName.c_str()));
         }

         s.clear();
      }
   }

   return viewsToReturn;
}


bool TraceQueryHelper::isSurfaceTouchPointVisible(const hmibase::util::geometry::Point& pt, int surfaceId, const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector)
{
   SurfaceInfoProvider::tSurfaceInfoVector::const_iterator it = surfaceInfoVector.begin();
   for (; it != surfaceInfoVector.end(); ++it)
   {
      if (surfaceId == it->surfaceId)
         // coordinate is not hidden by any other surface
      {
         return true;
      }

      if (it->touchableRect.contains(pt))
         // coordinate is part of surface, that is not the expected one, so it is hidden by this
      {
         return true; // allowing hidden surface also to receive touch input (temporary solution)
      }
   }
   return true; // allowing hidden surface also to receive touch input (temporary solution)
}


bool TraceQueryHelper::simTouchByName(const SimTouchReqMsg& msg, Courier::ViewScene2D* viewScene2D, Courier::FrameworkWidget* fw)
{
   if ((fw != 0) && (viewScene2D != 0))
   {
      // check for camera of the widget
      if (fw->IsTouchable())
      {
         // let's check if widget is part of a touchable rect
         Candera::WidgetBase* wb = static_cast<Candera::WidgetBase*>(fw);
         Candera::Widget2D* w = Candera::Dynamic_Cast<Candera::Widget2D*>(wb);
         if (w != 0)
         {
            Candera::Node2D* node = w->GetNode();
            if (node != 0)
            {
               const Courier::ViewScene2D::CameraPtrVector& cameras = viewScene2D->GetCameraPtrVector();
               for (FeatStd::SizeType idx = 0; idx < cameras.Size(); idx++)
               {
                  Candera::Camera2D* camera2D = cameras[idx];

                  const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector = msg.GetData().getSurfaceInfoData();
                  SurfaceInfoProvider::tSurfaceInfoVector::const_iterator it = surfaceInfoVector.begin();

                  for (; it != surfaceInfoVector.end(); ++it)
                  {
                     if (camera2D->IsRenderingEnabled() && hmibase::input::EqualsSurfaceId(camera2D->GetRenderTarget(), static_cast<Courier::UInt32>(it->surfaceId)))
                     {
                        Candera::Vector2 pt = node->GetWorldPosition();
                        Candera::Rectangle rec;
                        node->GetEffectiveBoundingRectangle(rec);

                        if (isSurfaceTouchPointVisible(hmibase::util::geometry::Point((hmibase::util::geometry::Coord)pt.GetX(), (hmibase::util::geometry::Coord)pt.GetY()), it->surfaceId, surfaceInfoVector))
                        {
                           Courier::Int32 X = static_cast<Courier::Int32>((pt.GetX() + (rec.GetWidth() / 2)));
                           Courier::Int32 Y = static_cast<Courier::Int32>((pt.GetY() + (rec.GetHeight() / 2)));
                           Candera::RenderTarget* rt = camera2D->GetRenderTarget();
                           if (rt != NULL)
                           {
                              Courier::Int32 MaxX = rt->GetWidth() - 1;
                              Courier::Int32 MaxY = rt->GetHeight() - 1;
                              Courier::Message* lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Down,
                                                       static_cast<Courier::XYDIM>(X >= MaxX ? MaxX : X),
                                                       static_cast<Courier::XYDIM>(Y >= MaxY ? MaxY : Y),
                                                       FeatStd::Internal::PerfCounter::Now(),
                                                       static_cast<Courier::PointerId>(0),
                                                       it->surfaceId);
                              POST_MSG_NOTRACE(lMsg);

                              Courier::Message* upMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Up,
                                                        static_cast<Courier::XYDIM>(X >= MaxX ? MaxX : X),
                                                        static_cast<Courier::XYDIM>(Y >= MaxY ? MaxY : Y),
                                                        FeatStd::Internal::PerfCounter::Now(),
                                                        static_cast<Courier::PointerId>(0),
                                                        it->surfaceId);
                              //Timer is used to have some delay in posting the TouchUp message so as to call the update correctly.
                              (void)GetTimerInstance().setTimeout(0, 100, upMsg);
                              GetTimerInstance().start();
                           }
                        }
                     }
                  }
               }
            }
         }
      }
   }

   return true;
}


bool TraceQueryHelper::simTouchByName(const SimTouchReqMsg& msg, Courier::ViewScene3D* viewScene3D, Courier::FrameworkWidget* fw)
{
   //Courier::FrameworkWidget* nearestWidget = 0;
   if (viewScene3D != 0 && viewScene3D->IsActive())
   {
      //Float currentDistance = 9999999999.0f;
      //Float minimalDistance = currentDistance;

      if (fw->IsTouchable())
      {
         Candera::WidgetBase* wb = static_cast<Candera::WidgetBase*>(fw);
         Candera::Widget* w = Candera::Dynamic_Cast<Candera::Widget*>(wb);
         if (w != 0)
         {
            Candera::Node* node = w->GetNode();
            if (node != 0)
            {
               const Courier::ViewScene3D::CameraPtrVector& cameras = viewScene3D->GetCameraPtrVector();
               for (FeatStd::SizeType idx = 0; idx < cameras.Size(); idx++)
               {
                  Candera::Camera* camera = cameras[idx];

                  const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector = msg.GetData().getSurfaceInfoData();
                  SurfaceInfoProvider::tSurfaceInfoVector::const_iterator it = surfaceInfoVector.begin();

                  for (; it != surfaceInfoVector.end(); ++it)
                  {
                     if (camera->IsRenderingEnabled() && hmibase::input::EqualsSurfaceId(camera->GetRenderTarget(), static_cast<Courier::UInt32>(it->surfaceId)))
                     {
                        Candera::Vector3 pt = node->GetWorldPosition();

                        if (isSurfaceTouchPointVisible(hmibase::util::geometry::Point((hmibase::util::geometry::Coord)pt.GetX(), (hmibase::util::geometry::Coord)pt.GetY()), it->surfaceId, surfaceInfoVector))
                        {
                           Courier::Message* lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Down,
                                                    static_cast<Courier::XYDIM>(pt.GetX()),
                                                    static_cast<Courier::XYDIM>(pt.GetY()),
                                                    FeatStd::Internal::PerfCounter::Now(),
                                                    static_cast<Courier::PointerId>(0),
                                                    it->surfaceId);
                           POST_MSG_NOTRACE(lMsg);

                           Courier::Message* upMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Up,
                                                     static_cast<Courier::XYDIM>(pt.GetX()),
                                                     static_cast<Courier::XYDIM>(pt.GetY()),
                                                     FeatStd::Internal::PerfCounter::Now(),
                                                     static_cast<Courier::PointerId>(0),
                                                     it->surfaceId);
                           //Timer is used to have some delay in posting the TouchUp message so as to call the update correctly.
                           (void)GetTimerInstance().setTimeout(0, 100, upMsg);
                           GetTimerInstance().start();
                        }
                     }
                  }
               }
            }
         }
      }
   }
   return false;
}


bool TraceQueryHelper::simTouchByName(const SimTouchReqMsg& msg, Courier::View* view, Courier::FrameworkWidget* fw)
{
   bool ret = false;
   if (view && view->Is2D())
   {
      ret = simTouchByName(msg, view->ToViewScene2D(), fw);
   }
   else if (view && view->Is3D())
   {
      ret = simTouchByName(msg, view->ToViewScene3D(), fw);
   }
   return ret;
}


bool TraceQueryHelper::simTouchByName(const SimTouchReqMsg& msg, Courier::IViewHandler& viewHandler)
{
   return simTouchByName(msg,
                         viewHandler.FindView(msg.GetViewId()),
                         viewHandler.FindWidget(msg.GetViewId(),

                               Courier::CompositePath(),

                               Courier::ItemId(msg.GetWidgetIdentifier().GetCString()))
                        );
}


/**************************************************************************************
*FUNCTION:     postTouchMsg
*DESCRIPTION:  post a TouchMsg for the given widget
*PARAMETER:    SurfaceInfoProvider::tSurfaceInfoVector&,Candera::Widget2D*
*RETURNVALUE:  bool
**************************************************************************************/
bool TraceQueryHelper::postTouchMsg(const SurfaceInfoProvider::tSurfaceInfoVector& surfaceInfoVector, Candera::Widget2D* widget2D)
{
   if (widget2D != 0)
   {
      Candera::Node2D* nodePtr = widget2D->GetNode();
      if (nodePtr != 0)
      {
         Courier::View* view = widget2D->GetParentView();
         Courier::ViewScene2D* viewScene2D = view->ToViewScene2D();
         if (viewScene2D != 0)
         {
            const Courier::ViewScene2D::CameraPtrVector& cameras = viewScene2D->GetCameraPtrVector();

            for (FeatStd::SizeType idx = 0; idx < cameras.Size(); idx++)
            {
               Candera::Camera2D* camera2D = cameras[idx];
               SurfaceInfoProvider::tSurfaceInfoVector::const_iterator it = surfaceInfoVector.begin();

               for (; it != surfaceInfoVector.end(); ++it)
               {
                  if (camera2D->IsRenderingEnabled() && hmibase::input::EqualsSurfaceId(camera2D->GetRenderTarget(), static_cast<Candera::UInt32>(it->surfaceId)))
                  {
                     Candera::Vector2 pt = nodePtr->GetWorldPosition();
                     Candera::Rectangle rec;
                     nodePtr->GetEffectiveBoundingRectangle(rec);

                     if (isSurfaceTouchPointVisible(hmibase::util::geometry::Point((hmibase::util::geometry::Coord)pt.GetX(), (hmibase::util::geometry::Coord)pt.GetY()), it->surfaceId, surfaceInfoVector))

                     {
                        Courier::Message* lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Down,
                                                 static_cast<Courier::XYDIM>(pt.GetX() + (rec.GetWidth() / 2)),
                                                 static_cast<Courier::XYDIM>(pt.GetY() + (rec.GetHeight() / 2)),
                                                 FeatStd::Internal::PerfCounter::Now(),
                                                 static_cast<Courier::PointerId>(0),
                                                 it->surfaceId);
                        POST_MSG_NOTRACE(lMsg);

                        lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Up,
                               static_cast<Courier::XYDIM>(pt.GetX() + (rec.GetWidth() / 2)),
                               static_cast<Courier::XYDIM>(pt.GetY() + (rec.GetHeight() / 2)),
                               FeatStd::Internal::PerfCounter::Now(),
                               static_cast<Courier::PointerId>(0),
                               it->surfaceId);

                        POST_MSG_NOTRACE(lMsg);

                        return true;
                     }
                  }
               }
            }
         }
      }
   }
   return false;
}


bool TraceQueryHelper::printAllVisibleText(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor, std::string& printData)
{
   bool result = false;
   if (_hook == NULL)
// todo: sync with implementation in WidgetTraceQueryHelper
   {
      ETG_TRACE_FATAL_THR(("PRINT VISIBLE TEXT failed due to missing processing unit in widget lib"));

      return false;
   }

   const HMIRenderViewVisitor::tViewInfoVector& viewInfoVec = hmiRenderViewVisitor.getActiveViewInfo();
   HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfoVec.begin();
   Courier::View* view = 0;

   //ETG_TRACE_FATAL_THR(("#########   PRINT VISIBLE TEXT START #########"));
   //printData += "#########   PRINT VISIBLE TEXT START #########\n";
   for (; cit != viewInfoVec.end(); ++cit)
   {
      std::vector<unsigned int>::const_iterator it = cit->surfaceIds.begin();
      for (; it != cit->surfaceIds.end(); ++it)
      {
         view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));

         if (ScreenBrokerClient::GetInstance().IsMainSurface(*it))
         {
            if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(*it) == ScreenBroker::SurfaceState::Mapped)
            {
               //ETG_TRACE_FATAL_THR(("Main SurfaceId = %d ", *it));
               printData += "Main SurfaceId = ";
               std::stringstream ss;
               ss << *it;
               printData += ss.str();
               printData += "\n";
               _hook->ProcessQueryTraceMsgToPrintVisibleText(msg, view, printData);
               result = true;
            }
         }
         else if (static_cast<unsigned int>(hmibase::SURFACEID_POPUPBASE) <= *it && static_cast<unsigned int>(hmibase::SURFACEID_POPUPBASE_MAX) >= *it)
         {
            //ETG_TRACE_FATAL_THR(("Popup SurfaceId = %d ", *it));
            printData += "Popup SurfaceId = ";
            std::stringstream ss;
            ss << *it;
            printData += ss.str();
            printData += "\n";
            _hook->ProcessQueryTraceMsgToPrintVisibleText(msg, view, printData);
            result = true;
         }
         else if (*it == ScreenBrokerClient::GetInstance().StatusBarSurfaceId())
         {
            // this scene is the main surface scene. chk surface state
            if (ScreenBrokerClient::GetInstance().GetStatusBarSurfaceState() == ScreenBroker::SurfaceState::Mapped)
            {
               //ETG_TRACE_FATAL_THR(("StatusBar SurfaceId = %d ", *it));
               printData += "StatusBar SurfaceId = ";
               std::stringstream ss;
               ss << *it;
               printData += ss.str();
               printData += "\n";
               _hook->ProcessQueryTraceMsgToPrintVisibleText(msg, view, printData);
               result = true;
            }
         }
         else if ((*it == ScreenBrokerClient::GetInstance().FooterBarSurfaceId()) &&
                  (ScreenBrokerClient::GetInstance().GetFooterBarSurfaceState() == ScreenBroker::SurfaceState::Mapped))
         {
            //ETG_TRACE_FATAL_THR(("FooterBar SurfaceId = %d ", *it));
            printData += "FooterBar SurfaceId = ";
            std::stringstream ss;
            ss << *it;
            printData += ss.str();
            printData += "\n";
            _hook->ProcessQueryTraceMsgToPrintVisibleText(msg, view, printData);
            result = true;
         }
      }
   }
   //ETG_TRACE_FATAL_THR(("#########   PRINT VISIBLE TEXT END #########"));
   //printData += "#########   PRINT VISIBLE TEXT END #########\n";
   return result;
}


//--------------------------------------------------------------------------------------
bool TraceQueryHelper::getPossibleTouchPoints(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   if (_hook == NULL)
   {
      ETG_TRACE_FATAL_THR(("GetPossibleTouchPoints failed due to missing processing unit in widget lib"));
      return false;
   }

   const HMIRenderViewVisitor::tViewInfoVector& viewInfoVec = hmiRenderViewVisitor.getActiveViewInfo();
   HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfoVec.begin();
   Courier::View* view = 0;

   ETG_TRACE_FATAL_THR(("#########   GET POSSIBLE TOUCH POINTS START #########"));
   for (; cit != viewInfoVec.end(); ++cit)
   {
      std::vector<unsigned int>::const_iterator it = cit->surfaceIds.begin();
      for (; it != cit->surfaceIds.end(); ++it)
      {
         view = viewHandler.FindView(Courier::ViewId(cit->viewName.c_str()));

         if (ScreenBrokerClient::GetInstance().IsMainSurface(*it))
         {
            if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(*it) == ScreenBroker::SurfaceState::Mapped)
            {
               ETG_TRACE_FATAL_THR(("Main SurfaceId = %d ", *it));
               _hook->ProcessQueryTraceMsgToGetPossibleTouchPoints(msg, viewHandler, view, hmiRenderViewVisitor);
            }
         }
         else if (static_cast<unsigned int>(hmibase::SURFACEID_POPUPBASE) <= *it && static_cast<unsigned int>(hmibase::SURFACEID_POPUPBASE_MAX) >= *it)
         {
            ETG_TRACE_FATAL_THR(("Popup SurfaceId = %d ", *it));
            _hook->ProcessQueryTraceMsgToGetPossibleTouchPoints(msg, viewHandler, view, hmiRenderViewVisitor);
         }
         else if (*it == ScreenBrokerClient::GetInstance().StatusBarSurfaceId())
         {
            ETG_TRACE_FATAL_THR(("StatusBar SurfaceId = %d ", *it));
            _hook->ProcessQueryTraceMsgToGetPossibleTouchPoints(msg, viewHandler, view, hmiRenderViewVisitor);
         }
         else if ((*it == ScreenBrokerClient::GetInstance().FooterBarSurfaceId()) &&
                  (ScreenBrokerClient::GetInstance().GetFooterBarSurfaceState() == ScreenBroker::SurfaceState::Mapped))
         {
            ETG_TRACE_FATAL_THR(("FooterBar SurfaceId = %d ", *it));
            _hook->ProcessQueryTraceMsgToGetPossibleTouchPoints(msg, viewHandler, view, hmiRenderViewVisitor);
         }
      }
   }

   return false;
}


//--------------------------------------------------------------------------------------
bool TraceQueryHelper::printPossibleTouchCoordinates(Candera::WidgetBase* wb)
{
   bool result = false;

   Candera::Widget2D* w = Candera::Dynamic_Cast<Candera::Widget2D*>(wb);
   if (w != 0)
   {
      if (w->GetNode() != 0 && w->GetNode()->IsEffectiveRenderingEnabled())
      {
         Candera::Vector2 pt = w->GetNode()->GetWorldPosition();
         Candera::Rectangle rec;
         w->GetNode()->GetEffectiveBoundingRectangle(rec);

         ETG_TRACE_FATAL_THR(("Coordinates of widget are [%d,%d]",
                              static_cast<Courier::XYDIM>(pt.GetX() + (rec.GetWidth() / 2)),
                              static_cast<Courier::XYDIM>(pt.GetY() + (rec.GetHeight() / 2))));

         return true;
      }
   }
   return result;
}


/**************************************************************************************
*FUNCTION:     FindParentWidgetName
*DESCRIPTION:  For a view which contains more than one child widget with the same name,
*DESCRIPTION:  then to point to a specific child widget, the parent name can be used as a
*DESCRIPTION:  distinguisher. The parentWidgetName and the childWidgetName are seperated by a
*DESCRIPTION:  delimieter (#). This function will seperate the ParentWidgetName from the
*DESCRIPTION:  ChildWidgetName.
*PARAMETER:    Candera::String, which may contain ParentWidgetName#ChildWidgetName
*RETURNVALUE:  string containning the ParentWidgetName
**************************************************************************************/
std::string TraceQueryHelper::FindParentWidgetName(Candera::String widgetName)
{
   // To find if the widgetName contains the ParentWidgetName
   size_t pos;
   std::string parentWidgetName;
   std::string delimiter = "#";
   std::string inputString = widgetName.GetCString();

   if ((pos = inputString.find(delimiter)) != std::string::npos)
   {
      parentWidgetName = inputString.substr(0, pos);

      inputString.erase(0, pos + delimiter.length());
      // Search for another Parent WidgetName
      if (inputString.find(delimiter) != std::string::npos)
      {
         // TODO TBD
      }

      return parentWidgetName;
   }
   return std::string();
}


/**************************************************************************************
*FUNCTION:     getScreenInfo
*DESCRIPTION:  This function prints the name of All the widgets present in the current scene.
*PARAMETER:
*RETURNVALUE:
**************************************************************************************/
bool TraceQueryHelper::getScreenInfo(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   if (isCurrentVisibleScene(msg.GetUserData3().GetCString(), hmiRenderViewVisitor))
   {
      Courier::View* view = viewHandler.FindView(Courier::ViewId(msg.GetUserData3().GetCString()));

      if (view != NULL && view->IsRenderingEnabled())
      {
         ETG_TRACE_FATAL_THR(("###  WIDGETS INFO START ###"));
         WidgetFinderCallback callback;
         WidgetCheckReqMsg msg(&callback);
         view->OnMessage(msg);
         ETG_TRACE_FATAL_THR(("###  WIDGETS INFO END ###"));
      }
   }
   return true;
}


void TraceQueryHelper::getFocusedWidgetInstanceInfo(Courier::IViewHandler& viewHandler)
{
   Courier::ViewId viewId = Focus::FManager::getInstance().getCurrentFocus().ViewId;
   std::string widgetId = Focus::FManager::getInstance().getCurrentFocus().WidgetId;
   if (!widgetId.empty())
   {
      ETG_TRACE_FATAL_THR(("FocusedWidget = %100s, ViewId = %100s", widgetId.c_str(), viewId.CStr()));
      Courier::FrameworkWidget* fw = viewHandler.FindWidget(viewId, Courier::CompositePath(), Courier::ItemId(widgetId.c_str()));
      if (fw != 0)
      {
         Candera::WidgetBase* widgetBase = static_cast<Candera::WidgetBase*>(fw);
         ETG_TRACE_FATAL_THR(("WidgetTemplate = %100s", widgetBase->GetTypeName()));
      }
   }
   else
   {
      ETG_TRACE_FATAL_THR(("No widget is focused"));
   }
}


void TraceQueryHelper::GetTextWidgetProperties(Candera::Widget2D* widget, IWidgetTraceQueryHelper::TextProperties& properties)
{
   if (_hook)
   {
      _hook->GetTextWidgetProperties(widget, properties);
   }
}


bool TraceQueryHelper::ProcessSimTouchRequest(const SimTouchReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   if (_hook && _hook->ProcessSimTouchRequest(msg, viewHandler, hmiRenderViewVisitor))
   {
      return true;
   }

   bool consumed = true;

   switch (msg.GetIdentifier())
   {
      case hmibase::SIM_TOUCH_BY_NAME:
         simTouchByName(msg, viewHandler);
         break;
      default:
         consumed = false;
         break;
   }

   return consumed;
}


bool TraceQueryHelper::ProcessTraceQueryRequest(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   if (_hook && _hook->ProcessTraceQueryRequest(msg, viewHandler, hmiRenderViewVisitor))
   {
      return true;
   }

   bool result = true;
   switch (msg.GetQueryId())
   {
#ifndef WIN32                                                                                   //lint -save -e1705
      case hmibase::GET_CURRENT_POPUP_VIEW:
         ScreenBrokerClient::GetInstance().RequestCurrentStatus(static_cast<Courier::UInt32>(ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActivePopup));
         break;
      case hmibase::GET_CURRENT_VISIBLE_POPUP_VIEW:
         ScreenBrokerClient::GetInstance().RequestCurrentStatus(static_cast<Courier::UInt32>(ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveVisiblePopup));
         break;
      case hmibase::GET_POPUP_STACK:
         ScreenBrokerClient::GetInstance().RequestCurrentStatus(static_cast<Courier::UInt32>(ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveVisiblePopup));
         break;
      case hmibase::GET_TOP_ACTIVE_SURFACE_NON_PERMANENT:
         ScreenBrokerClient::GetInstance().RequestCurrentStatus(static_cast<Courier::UInt32>(ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveSurfaceNonPermanent));
         break;
      case hmibase::GET_TOP_ACTIVE_SURFACE:
         ScreenBrokerClient::GetInstance().RequestCurrentStatus(static_cast<Courier::UInt32>(ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveSurface));
         break;
      case hmibase::SB_START_ANIMATION:
      {
         ScreenBrokerClient::GetInstance().StartAnimation(hmibase::view::ViewIdentifier(), msg.GetUserData1(), msg.GetUserData2());
      }
      break;
#endif                                                                                          //lint -restore
      case hmibase::ADD_STARTUP_INVESTIGATION:
      {
         Courier::UInt32 mp = msg.GetUserData1();
         Courier::UInt32 timestamp = msg.GetUserData2();
         HMI_STARTUP_INVESTIGATION_WITH_TIMESTAMP(mp, timestamp);
         break;
      }

      case hmibase::GET_STARTUP_INVESTIGATION:
         hmibase::trace::StartupInvestigation::s_getInstance().print();
         break;

      case hmibase::GET_FEAT_MEM_STAT:
         hmibase::trace::MemoryStatistics::print();
         break;

      case hmibase::GET_COURIER_MSG_STAT:
         hmibase::trace::MessageStatistics::print();
         break;

      case hmibase::ABORT_SB_ANIMS:
         ScreenBrokerClient::GetInstance().AbortAnimations();
         break;
      case hmibase::DUMP_CANDERA_PERF_MEASURES:
#ifdef FEATSTD_MONITOR_ENABLED
      {
         char f[100];
         static unsigned int count = 0;
#ifdef WIN32
         snprintf(f, sizeof(f), "CanderaPerfLog_%s_%02d_Win32.perflog", hmibase::trace::getAppName().c_str(), ++count);
#else
         snprintf(f, sizeof(f), "/tmp/CanderaPerfLog_%50s_%02d.perflog", hmibase::trace::getAppName().c_str(), ++count);
#endif
         FeatStd::PerfMon::FileDumper::Open(f);
         FeatStd::PerfMon::FileDumper::FlushAndClose();
         ETG_TRACE_FATAL_THR(("Candera performance log written to %100s", f));
      }
      break;
#else
      ETG_TRACE_FATAL_THR(("Candera performance monitor not enabled"));
      break;
#endif
      case hmibase::FORCE_REDRAW:
      {
         Courier::Message* lMsg = COURIER_MESSAGE_NEW(Courier::RenderReqMsg)();
         POST_MSG_NOTRACE(lMsg)
      }
      break;
      case hmibase::GET_TEXTMETADATA:
      {
         CreateTextMetaDataCollectionReqMsg createTextMetaDataCollectionReqMsg;
         viewHandler.OnMessage(createTextMetaDataCollectionReqMsg);
      }
      break;
      case hmibase::DUMP_SCREEN:
      {
         DumpScreenReqMsg dumpScreenReqMsg(msg.GetUserData3(), true);
         viewHandler.OnMessage(dumpScreenReqMsg);
      }
      break;
      case hmibase::DUMP_SCENE:
      {
         DumpScreenReqMsg dumpScreenReqMsg(msg.GetUserData3(), false);
         viewHandler.OnMessage(dumpScreenReqMsg);
      }
      break;
      case hmibase::SHOW_SCENE:
      case hmibase::HIDE_SCENE:
      {
         Courier::ViewId viewId(msg.GetUserData3().GetCString());
         ManualSceneActionReqMsg manualSceneActionReqMsg(viewId, msg.GetQueryId());
         viewHandler.OnMessage(manualSceneActionReqMsg);
      }
      break;
      case hmibase::TRAVERSE_VIEWS:
      {
         TraverseViewsReqMsg traverseViewsReqMsg(msg.GetUserData3(), msg.GetUserData4());
         viewHandler.OnMessage(traverseViewsReqMsg);
      }
      break;
      case hmibase::SET_LOCALE:
      {
         LocalizationReqMsg localizationReqMsg(msg.GetUserData3());
         viewHandler.OnMessage(localizationReqMsg);
      }
      break;

      case hmibase::GET_FOCUSED_WIDGET:
         getFocusedWidgetInstanceInfo(viewHandler);
         break;

      case hmibase::GET_SCREEN_INFO:
      {
         getScreenInfo(msg, viewHandler, hmiRenderViewVisitor);
      }
      break;

      /*******************************************************************/
      case hmibase::GET_ALL_VISIBLE_TEXT_IN_GIVEN_LISTWIDGET:
      case hmibase::GET_ALL_VISIBLE_TEXT_IN_VIEW:
      case hmibase::GET_ALL_VISIBLE_TEXT_IN_ALL_LISTWIDGETS:
      {
         std::string printData;
         if (printAllVisibleText(msg, viewHandler, hmiRenderViewVisitor, printData))
         {
            ETG_TRACE_FATAL_THR(("#########   PRINT VISIBLE TEXT START #########"));
            std::vector<std::string> vecData = ::hmibase::util::split(printData, '\n');
            std::vector<std::string>::iterator it = vecData.begin();

            while (it != vecData.end())
            {
               ETG_TRACE_FATAL_THR(("%s", (*it).c_str()));
               ++it;
            }
            ETG_TRACE_FATAL_THR(("#########   PRINT VISIBLE TEXT END #########"));
         }
      }
      break;

      case GET_POSSIBLE_TOUCHPOINT_ELEM_NAME:
      case GET_POSSIBLE_TOUCHPOINT_ELEM_TEXT:
      case GET_POSSIBLE_TOUCHPOINT_ELEM_TEXTID:
      case GET_POSSIBLE_TOUCHPOINT_LISTELEM:
      {
         getPossibleTouchPoints(msg, viewHandler, hmiRenderViewVisitor);
      }
      break;

      case hmibase::SET_FRAME_RATE:
      {
         std::stringstream str(msg.GetUserData3().GetCString());
         unsigned int fps = 0;
         str >> fps;
         if (0 != fps)
         {
            int cycleTime = 1000 / fps;
            Courier::ComponentId id = static_cast<Courier::ComponentId>(Courier::ComponentType::View);
            POST_MSG_NOTRACE(COURIER_MESSAGE_NEW(Courier::SetMessageReceiverCycleTimeMsg)(id, Courier::ComponentMessageReceiverTiming::MinResyncCycle, cycleTime));
         }
      }
      break;

      default:
         result = false;
         break;
   }

   return result;
}


bool TraceQueryHelper::ProcessWidgetPropertySetRequest(const WidgetPropertySetterReqMsg& msg, Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   if (_hook)
   {
      _hook->ProcessWidgetPropertySetRequest(msg, viewHandler, hmiRenderViewVisitor);
   }
   return false;
}


bool TraceQueryHelper::isCurrentVisibleScene(const char* sceneName, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   if (sceneName == 0)
   {
      return false;
   }
   const ::hmibase::view::HMIRenderViewVisitor::tViewInfoVector& viewInfoVec = hmiRenderViewVisitor.getActiveViewInfo();

   for (::hmibase::view::HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfoVec.begin(); cit != viewInfoVec.end(); ++cit)
   {
      // check if view surface id is the main scene surface id
      std::vector<unsigned int>::const_iterator it = cit->surfaceIds.begin();
      for (; it != cit->surfaceIds.end(); ++it)
      {
         if (ScreenBrokerClient::GetInstance().IsMainSurface(*it))
         {
            // this scene is the main surface scene. chk surface state
            if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(*it) == ScreenBroker::SurfaceState::Mapped)
            {
               if (!strcmp(sceneName, cit->viewName.c_str()))
               {
                  return true;
               }
            }
         }
         else if (static_cast<int>(hmibase::SURFACEID_POPUPBASE) <= *it && static_cast<int>(hmibase::SURFACEID_POPUPBASE_MAX) >= *it)
         {
            // this is a popup surface
            if (!strcmp(sceneName, cit->viewName.c_str()))
            {
               return true;
            }
         }
         else if (*it == ScreenBrokerClient::GetInstance().StatusBarSurfaceId())
         {
            // this scene is the statusbar surface scene. chk surface state
            if (ScreenBrokerClient::GetInstance().GetStatusBarSurfaceState() == ScreenBroker::SurfaceState::Mapped)
            {
               if (!strcmp(sceneName, cit->viewName.c_str()))
               {
                  return true;
               }
            }
         }
         else if ((*it == ScreenBrokerClient::GetInstance().FooterBarSurfaceId()) &&
                  (ScreenBrokerClient::GetInstance().GetFooterBarSurfaceState() == ScreenBroker::SurfaceState::Mapped))
         {
            if (!strcmp(sceneName, cit->viewName.c_str()))
            {
               return true;
            }
         }
      }
   }
   return false;
}


void TraceQueryHelper::PrintLoadedScenes(const Courier::UInt32 QueryId, Courier::ViewContainer* vc, ::hmibase::trace::TMLLoadedViewInfo& viewsToReturn)
{
   static std::string s;
   if (vc)
   {
      Courier::ViewLinkedList& views = vc->GetViews();
      if (views.GetSize() != 0)
      {
         Courier::View* viewscn;
         for (Courier::ViewLinkedList::Iterator it = views.Begin(); (it != views.End()); ++it)
         {
            bool IsContainerFlag = it->IsContainer();
            if (IsContainerFlag)
            {
               PrintLoadedScenes(QueryId, it->ToViewContainer(), viewsToReturn);
            }
            else
            {
               it->ToViewScene2D() ? viewscn = it->ToViewScene2D() : viewscn = it->ToViewScene3D();
               if (viewscn)
               {
                  if (QueryId == LOADED)   //ALL_VIEWS
                  {
                     ETG_TRACE_USR1_THR(("[%100s]  [%100s] - Status: LOADED = %6s; RENDERED = %6s; ACTIVE = %6s;", hmibase::trace::getAppName().c_str(), viewscn->GetId().CStr(), viewscn->IsContentLoaded() ? "True" : "False", viewscn->IsRenderingEnabled() ? "True" : "False", viewscn->IsActive() ? "True" : "False"));
                     s += viewscn->GetId().CStr();
                     viewsToReturn.loadedScenes.push_back(s);
                  }
                  else if (QueryId == ACTIVE_RENDERED)
                  {
                     if ((viewscn->IsActive()) && (viewscn->IsRenderingEnabled()))
                     {
                        ETG_TRACE_USR1_THR(("[%100s]  [%100s] - ACTIVE = TRUE; RENDERED = TRUE", hmibase::trace::getAppName().c_str(), viewscn->GetId().CStr()));
                        s += viewscn->GetId().CStr();
                        viewsToReturn.activeRenderedScenes.push_back(s);
                     }
                  }
                  else if (QueryId == ACTIVE_NOTRENDERED)
                  {
                     if ((viewscn->IsActive()) && (!viewscn->IsRenderingEnabled()))
                     {
                        ETG_TRACE_USR1_THR(("[%100s]  [%100s] - ACTIVE = TRUE; RENDERED = FALSE", hmibase::trace::getAppName().c_str(), viewscn->GetId().CStr()));
                        s += viewscn->GetId().CStr();
                        viewsToReturn.activeNotRenderedScenes.push_back(s);
                     }
                  }
                  else if (QueryId == INACTIVE)
                  {
                     if (!viewscn->IsActive() && (viewscn->IsRenderingEnabled()))
                     {
                        ETG_TRACE_USR1_THR(("[%100s]-  [%s100] - ACTIVE = FALSE", hmibase::trace::getAppName().c_str(), viewscn->GetId().CStr()));
                        s += viewscn->GetId().CStr();
                        viewsToReturn.inactiveScenes.push_back(s);
                     }
                  }
                  s.clear();
               }
            }
         }
      }
      else
      {
         ETG_TRACE_USR1_THR(("No Loaded scenes found for %s", hmibase::trace::getAppName().c_str()));
      }
   }
}


void TraceQueryHelper::GetAllLoadedScenes(const Courier::UInt32 QueryId, Courier::IViewHandler& viewHandler, ::hmibase::trace::TMLLoadedViewInfo& viewsToReturn)
{
   Courier::ViewContainer* vc = dynamic_cast<Courier::ViewContainer*>(viewHandler.FindView(Courier::ViewId()));
   PrintLoadedScenes(QueryId, vc, viewsToReturn);
}


/***************************************************************************************/
}  // namespace trace
}  // namespace hmibase
