/* ***************************************************************************************
* FILE:          ScmlSerializer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScmlSerializer.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "ScmlSerializer.h"

#include <FeatStd/Util/String.h>
#include <Candera/Engine2D/Core/BitmapImage2D.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Group2D.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/System/Mathematics/Math.h>

#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/SolidColorBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushColorBlend.h>
#include <CanderaPlatform/Device/Common/Effects/MirrorBitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/GlBitmapBrushMaskBlend.h>
#include <CanderaPlatform/Device/Common/Effects/GlBitmapBrushColorMaskBlend.h>


#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ScmlSerializer.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


namespace hmibase {
namespace trace {

class ValueWriter
{
   public:
      template <typename TValue>
      static void writePropertyValue(std::ostream& stream, TValue const& value, bool isDynamic = false);

      template <typename TValue>
      static const char* getValueText(TValue const& value);

      static void writeDynamicBegin(std::ostream& stream, bool isDynamic, const char* typeName)
      {
         if (isDynamic)
         {
            stream << '{';
            stream << typeName;
            stream << ' ';
         }
      }

      static void writeDynamicEnd(std::ostream& stream, bool isDynamic)
      {
         if (isDynamic)
         {
            stream << '}';
         }
      }

      static void replace(std::string& value, const char* oldToken, const char* newToken)
      {
         size_t oldTokenLength = strlen(oldToken);
         size_t newTokenLength = strlen(newToken);
         size_t pos = value.find(oldToken);
         while (pos != std::string::npos)
         {
            value.replace(pos, oldTokenLength, newToken);
            pos = value.find(oldToken, pos + newTokenLength);
         }
      }
};


template <> const char* ValueWriter::getValueText<bool>(bool const& value)
{
   return value ? "True" : "False";
}


template <> void ValueWriter::writePropertyValue<bool>(std::ostream& stream, bool const& value, bool)
{
   stream << getValueText(value);
}


template <> void ValueWriter::writePropertyValue<std::string>(std::ostream& stream, std::string const& value, bool)
{
   std::string s(value);
   replace(s, "&", "&amp;");
   replace(s, "<", "&lt;");
   replace(s, ">", "&gt;");
   replace(s, "\"", "&quot;");
   stream << s;
}


template <> void ValueWriter::writePropertyValue<const char*>(std::ostream& stream, const char* const& value, bool isDynamic)
{
   if (value != NULL)
   {
      writePropertyValue<std::string>(stream, value, isDynamic);
   }
}


template <> void ValueWriter::writePropertyValue<FeatStd::String>(std::ostream& stream, FeatStd::String const& value, bool isDynamic)
{
   writePropertyValue(stream, value.GetCString(), isDynamic);
}


template <> void ValueWriter::writePropertyValue<Candera::Vector2>(std::ostream& stream, Candera::Vector2 const& value, bool isDynamic)
{
   writeDynamicBegin(stream, isDynamic, "Vector2");
   stream << value.GetX() << ';' << value.GetY();
   writeDynamicEnd(stream, isDynamic);
}


template <> void ValueWriter::writePropertyValue<Candera::Vector3>(std::ostream& stream, Candera::Vector3 const& value, bool isDynamic)
{
   writeDynamicBegin(stream, isDynamic, "Vector3");
   stream << value.GetX() << ';' << value.GetY() << ';' << value.GetZ();
   writeDynamicEnd(stream, isDynamic);
}


template <> void ValueWriter::writePropertyValue<Candera::Rectangle>(std::ostream& stream, Candera::Rectangle const& value, bool isDynamic)
{
   writeDynamicBegin(stream, isDynamic, "Rectangle");
   stream << value.GetLeft() << ';' << value.GetTop() << ';' << value.GetWidth() << ';' << value.GetHeight();
   writeDynamicEnd(stream, isDynamic);
}


template <> void ValueWriter::writePropertyValue<Candera::Color::Data>(std::ostream& stream, Candera::Color::Data const& value, bool isDynamic)
{
   writeDynamicBegin(stream, isDynamic, "Color");
   stream << value.red << ';' << value.green << ';' << value.blue << ';' << value.alpha;
   writeDynamicEnd(stream, isDynamic);
}


template <> void ValueWriter::writePropertyValue<Candera::Color>(std::ostream& stream, Candera::Color const& value, bool isDynamic)
{
   writePropertyValue(stream, value.GetData(), isDynamic);
}


#define DEFAULT_WRITE_PROP_VALUE(valueType, typeName) \
   template <> void ValueWriter::writePropertyValue<valueType>(std::ostream& stream, valueType const & value, bool isDynamic) \
      { \
         writeDynamicBegin(stream, isDynamic, typeName); \
         stream << value; \
         writeDynamicEnd(stream, isDynamic); \
         }

DEFAULT_WRITE_PROP_VALUE(FeatStd::Float, "Float")
DEFAULT_WRITE_PROP_VALUE(FeatStd::Int, "Int")
DEFAULT_WRITE_PROP_VALUE(FeatStd::Int8, "Int8")
DEFAULT_WRITE_PROP_VALUE(FeatStd::Int16, "Int16")
//DEFAULT_WRITE_PROP_VALUE(FeatStd::Int32, "Int32")
DEFAULT_WRITE_PROP_VALUE(FeatStd::Int64, "Int64")
DEFAULT_WRITE_PROP_VALUE(FeatStd::UInt, "UInt")
DEFAULT_WRITE_PROP_VALUE(FeatStd::UInt8, "UInt8")
DEFAULT_WRITE_PROP_VALUE(FeatStd::UInt16, "UInt16")
//DEFAULT_WRITE_PROP_VALUE(FeatStd::UInt32, "UInt32")
DEFAULT_WRITE_PROP_VALUE(FeatStd::UInt64, "UInt64")

#define SWITCH_CASE_PROP_CANDERA_VALUE(value) case value: return #value;
#define SWITCH_CASE_PROP_CANDERA_VALUE2(className, value) case className::value: return #value;
#define SWITCH_DEFAULT_PROP_CANDERA_VALUE(value) default: return NULL;

#define SIMPLE_WRITE_PROP_VALUE(valueType) template <> void ValueWriter::writePropertyValue<valueType>(std::ostream& stream, valueType const & value, bool)\
   {\
      const char* str = getValueText(value);\
      if (str != NULL)\
      {\
         stream << getValueText(value); \
      }\
      else\
      {\
         stream << static_cast<int>(value); \
      }\
   }

template <> const char* ValueWriter::getValueText<Candera::HorizontalAlignment>(Candera::HorizontalAlignment const& value)
{
   using namespace Candera;
   switch (value)
   {
         SWITCH_CASE_PROP_CANDERA_VALUE(HLeft)
         SWITCH_CASE_PROP_CANDERA_VALUE(HRight)
         SWITCH_CASE_PROP_CANDERA_VALUE(HCenter)
         SWITCH_CASE_PROP_CANDERA_VALUE(HStretch)
         SWITCH_DEFAULT_PROP_CANDERA_VALUE(value)
   }
}


SIMPLE_WRITE_PROP_VALUE(Candera::HorizontalAlignment)

template <> const char* ValueWriter::getValueText<Candera::VerticalAlignment>(Candera::VerticalAlignment const& value)
{
   using namespace Candera;
   switch (value)
   {
         SWITCH_CASE_PROP_CANDERA_VALUE(VTop)
         SWITCH_CASE_PROP_CANDERA_VALUE(VBottom)
         SWITCH_CASE_PROP_CANDERA_VALUE(VCenter)
         SWITCH_CASE_PROP_CANDERA_VALUE(VStretch)
         SWITCH_DEFAULT_PROP_CANDERA_VALUE(value)
   }
}


SIMPLE_WRITE_PROP_VALUE(Candera::VerticalAlignment)

template <> const char* ValueWriter::getValueText<Candera::RenderDevice2D::BlendFactor>(Candera::RenderDevice2D::BlendFactor const& value)
{
   using namespace Candera;
   switch (value)
   {
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, Zero)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, One)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, SourceColor)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, InverseSourceColor)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, SourceAlpha)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, InverseSourceAlpha)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, DestColor)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, InverseDestColor)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, DestAlpha)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, InverseDestAlpha)
         SWITCH_DEFAULT_PROP_CANDERA_VALUE(value)
   }
}


SIMPLE_WRITE_PROP_VALUE(Candera::RenderDevice2D::BlendFactor)

template <> const char* ValueWriter::getValueText<Candera::RenderDevice2D::BlendOperation>(Candera::RenderDevice2D::BlendOperation const& value)
{
   using namespace Candera;
   switch (value)
   {
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, Add)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, Subtract)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, ReverseSubtract)
         SWITCH_CASE_PROP_CANDERA_VALUE2(RenderDevice2D, BitwiseAnd)
         SWITCH_DEFAULT_PROP_CANDERA_VALUE(value)
   }
}


SIMPLE_WRITE_PROP_VALUE(Candera::RenderDevice2D::BlendOperation)

template <> void ValueWriter::writePropertyValue<Candera::CanderaObject*>(std::ostream& stream, Candera::CanderaObject* const& value, bool isDynamic)
{
   if ((value != NULL) && (value->GetName() != NULL) && (value->GetName()[0] != '\0'))
   {
      writeDynamicBegin(stream, isDynamic, "Ref");

      std::string canderaName(value->GetName());

      size_t pos = canderaName.find('#');
      while (pos != std::string::npos)
      {
         canderaName.replace(pos, 1, 1, '/');
         pos = canderaName.find('#', pos);
      }

      size_t sclDelimiterPos = canderaName.find("//");
      if (sclDelimiterPos != std::string::npos)
      {
         canderaName.erase(sclDelimiterPos, 1);
         canderaName.insert(0, 1, '/');
      }
      canderaName.insert(0, 1, '/');

      stream << canderaName.c_str();

      writeDynamicEnd(stream, isDynamic);
   }
}


class ScmlWriter
{
   public:
      ScmlWriter(std::ostream& stream) : _stream(stream), _currentTagHasContent(false)
      {
      }

      void writeHeader()
      {
         _stream << "<?xml version=\"1.0\" encoding=\"utf - 8\"?>\n";
      }

      void writeTabs()
      {
         for (size_t i = 0; i < _tagStack.size(); ++i)
         {
            _stream << "   ";
         }
      }

      void writeBeginTag(const char* tag)
      {
         if ((_tagStack.size() > 0) && !_currentTagHasContent)
         {
            _stream << '>' << '\n';
         }

         writeTabs();
         _stream << '<';
         _stream << tag;

         _tagStack.push_back(tag);
         _currentTagHasContent = false;
      }

      void writeEndTag()
      {
         if (_tagStack.size() > 0)
         {
            if (_currentTagHasContent)
            {
               std::string tag = _tagStack.back();
               _tagStack.pop_back();

               writeTabs();
               _stream << '<' << '/';
               _stream << tag.c_str();
               _stream << '>';
            }
            else
            {
               _tagStack.pop_back();
               _stream << '/' << '>';
            }
            _stream << '\n';

            //parent always has content
            _currentTagHasContent = true;
         }
         else
         {
            //error: no current tag
         }
      }

      template <typename TValue>
      void writeProperty(const char* name, const TValue& value, bool isDynamic = false)
      {
         if (_currentTagHasContent)
         {
            //error: tag has content already, can't add properties
         }
         else
         {
            _stream << ' ';
            if (isDynamic)
            {
               _stream << "dyn:";
            }
            _stream << name;
            _stream << '=' << '\"';
            ValueWriter::writePropertyValue(_stream, value, isDynamic);
            _stream << '\"';
         }
      }

   private:
      ScmlWriter(const ScmlWriter&);
      ScmlWriter& operator=(const ScmlWriter&);

      std::ostream& _stream;
      std::vector<std::string> _tagStack;
      bool _currentTagHasContent;
};


class ItemIndex
{
   public:
      static int get()
      {
         return _index;
      }
      static void reset()
      {
         _index = 0;
      }
      static void increment()
      {
         ++_index;
      }

   private:
      static int _index;
};


int ItemIndex::_index = 0;

template <typename TItem>
class ItemSerializerBase
{
   public:
      virtual const char* getTag(TItem&) const
      {
         return NULL;
      }

      virtual TItem const* getDefaultItem() const
      {
         return NULL;
      }

      virtual void serialize(TItem& item, ScmlWriter& writer)
      {
         ItemIndex::increment();

         const char* tag = getTag(item);
         if (tag != NULL)
         {
            writer.writeBeginTag(tag);
            serializeProperties(item, writer);
            serializeContent(item, writer);
            writer.writeEndTag();
         }
         else
         {
            //error: item not supported as tag
         }
      }

      virtual void serializeProperties(TItem& item, ScmlWriter& writer)
      {
         if (item.GetName() != NULL && item.GetName()[0] != '\0')
         {
            writer.writeProperty("Name", item.GetName());
         }
         else
         {
            writer.writeProperty("Name", ItemIndex::get());
         }
      }

      virtual void serializeContent(TItem&, ScmlWriter&)
      {
      }
};


class SerializerFactory
{
   public:
      template <typename TItem>
      static ItemSerializerBase<TItem>* getSerializer(const TItem& item);
};


template<> ItemSerializerBase<Candera::Node2D>* SerializerFactory::getSerializer<Candera::Node2D>(const Candera::Node2D& item);
template<> ItemSerializerBase<Candera::Effect2D>* SerializerFactory::getSerializer<Candera::Effect2D>(const Candera::Effect2D& item);

#define CHECK_AND_WRITE_PROPERTY(item, defaultItem, propName)\
      if (item.Get##propName() != defaultItem->Get##propName()) \
            { \
         writer.writeProperty(#propName, item.Get##propName()); \
            }

#define CHECK_AND_WRITE_BOOL_PROPERTY(item, defaultItem, propName)\
      if (item.Is##propName() != defaultItem->Is##propName()) \
            { \
         writer.writeProperty("Is"#propName, item.Is##propName()); \
            }

#define CHECK_AND_WRITE_FLOAT_PROPERTY(item, defaultItem, propName)\
      if (!Candera::Math::FloatAlmostEqual(item.Get##propName(), defaultItem->Get##propName())) \
                  { \
         writer.writeProperty(#propName, item.Get##propName()); \
                  }

class Node2DSerializer : public ItemSerializerBase<Candera::Node2D>
{
      typedef ItemSerializerBase<Candera::Node2D> Base;

   public:
      virtual void serializeProperties(Candera::Node2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         const Candera::Node2D* defaultItem = getDefaultItem();
         if (defaultItem != NULL)
         {
            CHECK_AND_WRITE_PROPERTY(item, defaultItem, Position)
            CHECK_AND_WRITE_PROPERTY(item, defaultItem, PivotOffset)
            CHECK_AND_WRITE_PROPERTY(item, defaultItem, PivotPoint)
            CHECK_AND_WRITE_FLOAT_PROPERTY(item, defaultItem, Rotation)
            CHECK_AND_WRITE_PROPERTY(item, defaultItem, Scale)
            CHECK_AND_WRITE_FLOAT_PROPERTY(item, defaultItem, AlphaValue)
            CHECK_AND_WRITE_PROPERTY(item, defaultItem, RenderOrderRank)
            CHECK_AND_WRITE_BOOL_PROPERTY(item, defaultItem, RenderingEnabled)
         }
      }

      virtual void serializeContent(Candera::Node2D& item, ScmlWriter& writer)
      {
         Base::serializeContent(item, writer);

         Candera::Node2D* child = item.GetFirstChild();
         while (child != NULL)
         {
            ItemSerializerBase<Candera::Node2D>* serializer = SerializerFactory::getSerializer(*child);
            if (serializer != NULL)
            {
               serializer->serialize(*child, writer);
            }
            child = child->GetNextSibling();
         }
      }
};


class RenderNodeSerializer : public Node2DSerializer
{
      typedef Node2DSerializer Base;

   public:
      virtual const char* getTag(Candera::Node2D&) const
      {
         return "RenderNode2D";
      }

      virtual Candera::Node2D const* getDefaultItem() const
      {
         Candera::RenderNode* defaultItem = Candera::RenderNode::Create();
         return defaultItem;
      }

      virtual void serializeProperties(Candera::Node2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(&item);
         const Candera::RenderNode* defaultItem = Candera::Dynamic_Cast<const Candera::RenderNode*>(getDefaultItem());
         if ((renderNode != NULL) && (defaultItem != NULL))
         {
         }
      }

      virtual void serializeContent(Candera::Node2D& item, ScmlWriter& writer)
      {
         Base::serializeContent(item, writer);

         Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(&item);
         if (renderNode != NULL)
         {
            Candera::Effect2D* effect = renderNode->GetEffect(0);
            if (effect != NULL)
            {
               writer.writeBeginTag("RenderNode2D.Effects");

               ItemSerializerBase<Candera::Effect2D>* serializer = SerializerFactory::getSerializer(*effect);
               if (serializer != NULL)
               {
                  serializer->serialize(*effect, writer);
               }
               writer.writeEndTag();
            }
         }
      }
};


class Group2DSerializer : public Node2DSerializer
{
      //typedef Node2DSerializer Base;

   public:
      virtual const char* getTag(Candera::Node2D&) const
      {
         return "Group2D";
      }

      virtual Candera::Node2D const* getDefaultItem() const
      {
         static Candera::Node2D* defaultItem = Candera::Group2D::Create();
         return defaultItem;
      }
};


class Scene2DSerializer : public Node2DSerializer
{
      typedef Node2DSerializer Base;

   public:
      virtual const char* getTag(Candera::Node2D&) const
      {
         return "Scene2D";
      }

      virtual Candera::Node2D const* getDefaultItem() const
      {
         static Candera::Node2D* defaultItem = Candera::Scene2D::Create();
         return defaultItem;
      }

      virtual void serialize(Candera::Node2D& item, ScmlWriter& writer)
      {
         ItemIndex::reset();

         Base::serialize(item, writer);
      }
};


class Camera2DSerializer : public Node2DSerializer
{
      typedef Node2DSerializer Base;

   public:
      virtual const char* getTag(Candera::Node2D&) const
      {
         return "Camera2D";
      }

      virtual Candera::Node2D const* getDefaultItem() const
      {
         static Candera::Node2D* defaultItem = Candera::Camera2D::Create();
         return defaultItem;
      }

      virtual void serializeProperties(Candera::Node2D& item, ScmlWriter& writer)
      {
         bool isRenderingEnabled = item.IsRenderingEnabled();
         if (!isRenderingEnabled)
         {
            item.SetRenderingEnabled(true);
         }

         Base::serializeProperties(item, writer);

         item.SetRenderingEnabled(isRenderingEnabled);
      }
};


/*
#define WRITE_EFFECT_PROPERTY(item, propName)\
         writer.writeProperty(#propName, item.propName().Get(), true);
*/

#define WRITE_EFFECT_PROPERTY2(item, propName, scPropName)\
         writer.writeProperty(#scPropName, item.propName().Get(), true);

#define CHECK_AND_WRITE_EFFECT_PROPERTY(item, defaultItem, propName)\
      if (item.propName().Get() != defaultItem.propName().Get()) \
                                          { \
         writer.writeProperty(#propName, item.propName().Get(), true); \
                                          }

#define CHECK_AND_WRITE_EFFECT_PROPERTY2(item, defaultItem, propName, scPropName)\
      if (item.propName().Get() != defaultItem.propName().Get()) \
                              { \
         writer.writeProperty(#scPropName, item.propName().Get(), true); \
                              }

#define CHECK_AND_WRITE_EFFECT_REF_PROPERTY(item, propName)\
      if (!item. propName ().Get().PointsToNull()) \
                                                      { \
         writer.writeProperty<Candera::CanderaObject*>(#propName, item.propName().Get().GetPointerToSharedInstance(), true); \
                                                      }

class Effect2DSerializer : public ItemSerializerBase<Candera::Effect2D>
{
      typedef ItemSerializerBase<Candera::Effect2D> Base;

   public:
      virtual const char* getTag(Candera::Effect2D&) const
      {
         return "Effect";
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::MetaInfo::Effect2DMetaInfo* metaInfo = item.GetMetaInfo();
         if (metaInfo != NULL)
         {
            std::string temp;
            temp.append("/#Config/Effect:").append(metaInfo->GetName());
            writer.writeProperty("TemplateName", temp.c_str());
         }
      }
};


class BitmapBrushSerializer : public Effect2DSerializer
{
      typedef Effect2DSerializer Base;

   public:
      virtual Candera::Effect2D const* getDefaultItem() const
      {
         static Candera::Effect2D::SharedPointer defaultItem = Candera::BitmapBrush::Create();
         return defaultItem.GetPointerToSharedInstance();
      }

      static void serializeBitmapBrush(Candera::BitmapBrush& brush, Candera::BitmapBrush& /*defaultBrush*/, ScmlWriter& writer)
      {
         Candera::Image2D* image = brush.Image().Get();
         Candera::BitmapImage2D* bitmapImage = Candera::Dynamic_Cast<Candera::BitmapImage2D*>(image);
         if ((bitmapImage != NULL) && !bitmapImage->GetBitmap().PointsToNull())
         {
            writer.writeProperty<Candera::CanderaObject*>("Image", bitmapImage->GetBitmap().GetPointerToSharedInstance(), true);
         }
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::BitmapBrush* effect = Candera::Dynamic_Cast<Candera::BitmapBrush*>(&item);
         const Candera::BitmapBrush* defaultItem = Candera::Dynamic_Cast<const Candera::BitmapBrush*>(getDefaultItem());
         if ((effect != NULL) && (defaultItem != NULL))
         {
            serializeBitmapBrush(*effect, const_cast<Candera::BitmapBrush&>(*defaultItem), writer);
         }
      }
};


class BlendEffectSerializer : public Effect2DSerializer
{
      typedef Effect2DSerializer Base;

   public:
      virtual Candera::Effect2D const* getDefaultItem() const
      {
         static Candera::Effect2D::SharedPointer defaultItem = Candera::BlendEffect::Create();
         return defaultItem.GetPointerToSharedInstance();
      }

      static void serializeBlendEffect(Candera::BlendEffect& brush, Candera::BlendEffect& defaultBrush, ScmlWriter& writer)
      {
         CHECK_AND_WRITE_EFFECT_PROPERTY2(brush, defaultBrush, ColorSourceBlendFactor, ColorBlendFactorSrc)
         CHECK_AND_WRITE_EFFECT_PROPERTY2(brush, defaultBrush, ColorDestinationBlendFactor, ColorBlendFactorDst)
         CHECK_AND_WRITE_EFFECT_PROPERTY(brush, defaultBrush, ColorBlendOperation)
         CHECK_AND_WRITE_EFFECT_PROPERTY2(brush, defaultBrush, AlphaSourceBlendFactor, AlphaBlendFactorSrc)
         CHECK_AND_WRITE_EFFECT_PROPERTY2(brush, defaultBrush, AlphaDestinationBlendFactor, AlphaBlendFactorDst)
         CHECK_AND_WRITE_EFFECT_PROPERTY(brush, defaultBrush, AlphaBlendOperation)
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::BlendEffect* effect = Candera::Dynamic_Cast<Candera::BlendEffect*>(&item);
         const Candera::BlendEffect* defaultItem = Candera::Dynamic_Cast<const Candera::BlendEffect*>(getDefaultItem());
         if ((effect != NULL) && (defaultItem != NULL))
         {
            serializeBlendEffect(*effect, const_cast<Candera::BlendEffect&>(*defaultItem), writer);
         }
      }
};


class BitmapBrushBlendSerializer : public Effect2DSerializer
{
      typedef Effect2DSerializer Base;

   public:
      virtual Candera::Effect2D const* getDefaultItem() const
      {
         static Candera::Effect2D::SharedPointer defaultItem = Candera::BitmapBrushBlend::Create();
         return defaultItem.GetPointerToSharedInstance();
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::BitmapBrushBlend* effect = Candera::Dynamic_Cast<Candera::BitmapBrushBlend*>(&item);
         const Candera::BitmapBrushBlend* defaultItem = Candera::Dynamic_Cast<const Candera::BitmapBrushBlend*>(getDefaultItem());
         if ((effect != NULL) && (defaultItem != NULL))
         {
            BitmapBrushSerializer::serializeBitmapBrush(effect->GetBitmapBrush(), const_cast<Candera::BitmapBrush&>(defaultItem->GetBitmapBrush()), writer);
            BlendEffectSerializer::serializeBlendEffect(effect->GetBlendEffect(), const_cast<Candera::BlendEffect&>(defaultItem->GetBlendEffect()), writer);
         }
      }
};


class BitmapBrushColorBlendSerializer : public BitmapBrushBlendSerializer
{
      typedef BitmapBrushBlendSerializer Base;

   public:
      virtual Candera::Effect2D const* getDefaultItem() const
      {
         static Candera::Effect2D::SharedPointer defaultItem = Candera::BitmapBrushColorBlend::Create();
         return defaultItem.GetPointerToSharedInstance();
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         //Candera::BitmapBrushColorBlend* effect = Candera::Dynamic_Cast<Candera::BitmapBrushColorBlend*>(&item);
         //const Candera::BitmapBrushColorBlend* defaultItem = Candera::Dynamic_Cast<const Candera::BitmapBrushColorBlend*>(getDefaultItem());
         //if ((effect != NULL) && (defaultItem != NULL))
         //{
         //}
      }
};


class GlBitmapBrushMaskBlendSerializer : public BitmapBrushBlendSerializer
{
      typedef BitmapBrushBlendSerializer Base;

   public:
      virtual Candera::Effect2D const* getDefaultItem() const
      {
         static Candera::Effect2D::SharedPointer defaultItem = Candera::GlBitmapBrushMaskBlend::Create();
         return defaultItem.GetPointerToSharedInstance();
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::GlBitmapBrushMaskBlend* effect = Candera::Dynamic_Cast<Candera::GlBitmapBrushMaskBlend*>(&item);
         const Candera::GlBitmapBrushMaskBlend* defaultItem = Candera::Dynamic_Cast<const Candera::GlBitmapBrushMaskBlend*>(getDefaultItem());
         if ((effect != NULL) && (defaultItem != NULL))
         {
            Candera::Image2D* maskImage = effect->GetMaskEffect().Mask().Get();
            Candera::BitmapImage2D* maskBitmapImage = Candera::Dynamic_Cast<Candera::BitmapImage2D*>(maskImage);
            if ((maskBitmapImage != NULL) && !maskBitmapImage->GetBitmap().PointsToNull())
            {
               writer.writeProperty<Candera::CanderaObject*>("Mask", maskBitmapImage->GetBitmap().GetPointerToSharedInstance(), true);
            }
         }

         //Candera::BitmapBrushColorBlend* effect = Candera::Dynamic_Cast<Candera::BitmapBrushColorBlend*>(&item);
         //const Candera::BitmapBrushColorBlend* defaultItem = Candera::Dynamic_Cast<const Candera::BitmapBrushColorBlend*>(getDefaultItem());
         //if ((effect != NULL) && (defaultItem != NULL))
         //{
         //}
      }
};


class MirrorBitmapBrushBlendSerializer : public BitmapBrushBlendSerializer
{
      typedef BitmapBrushBlendSerializer Base;

   public:
      virtual Candera::Effect2D const* getDefaultItem() const
      {
         static Candera::Effect2D::SharedPointer defaultItem = Candera::MirrorBitmapBrushBlend::Create();
         return defaultItem.GetPointerToSharedInstance();
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::MirrorBitmapBrushBlend* effect = Candera::Dynamic_Cast<Candera::MirrorBitmapBrushBlend*>(&item);
         const Candera::MirrorBitmapBrushBlend* defaultItem = Candera::Dynamic_Cast<const Candera::MirrorBitmapBrushBlend*>(getDefaultItem());
         if ((effect != NULL) && (defaultItem != NULL))
         {
            BitmapBrushSerializer::serializeBitmapBrush(effect->GetBitmapBrush(), const_cast<Candera::BitmapBrush&>(defaultItem->GetBitmapBrush()), writer);
            BlendEffectSerializer::serializeBlendEffect(effect->GetBlendEffect(), const_cast<Candera::BlendEffect&>(defaultItem->GetBlendEffect()), writer);

            Candera::MirrorEffect& mirrorEffect = effect->GetMirrorEffect();
            Candera::MirrorEffect& defaultMirrorEffect = const_cast<Candera::MirrorEffect&>(defaultItem->GetMirrorEffect());

            CHECK_AND_WRITE_EFFECT_PROPERTY(mirrorEffect, defaultMirrorEffect, MirrorAxisFrom)
            CHECK_AND_WRITE_EFFECT_PROPERTY(mirrorEffect, defaultMirrorEffect, MirrorAxisTo)
            CHECK_AND_WRITE_EFFECT_PROPERTY(mirrorEffect, defaultMirrorEffect, Alpha)
         }
      }
};


class SolidColorBrushBlendSerializer : public Effect2DSerializer
{
      typedef Effect2DSerializer Base;

   public:
      virtual Candera::Effect2D const* getDefaultItem() const
      {
         static Candera::Effect2D::SharedPointer defaultItem = Candera::SolidColorBrushBlend::Create();
         return defaultItem.GetPointerToSharedInstance();
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::SolidColorBrushBlend* effect = Candera::Dynamic_Cast<Candera::SolidColorBrushBlend*>(&item);
         const Candera::SolidColorBrushBlend* defaultItem = Candera::Dynamic_Cast<const Candera::SolidColorBrushBlend*>(getDefaultItem());
         if ((effect != NULL) && (defaultItem != NULL))
         {
            BlendEffectSerializer::serializeBlendEffect(effect->GetBlendEffect(), const_cast<Candera::BlendEffect&>(defaultItem->GetBlendEffect()), writer);

            Candera::SolidColorBrush& solidColorBrush = effect->GetSolidColorBrush();
            Candera::SolidColorBrush& defaultSolidColorBrush = const_cast<Candera::SolidColorBrush&>(defaultItem->GetSolidColorBrush());

            WRITE_EFFECT_PROPERTY2(solidColorBrush, Color, FillColor)
            CHECK_AND_WRITE_EFFECT_PROPERTY(solidColorBrush, defaultSolidColorBrush, Size)
         }
      }
};


class TextBrushBlendSerializer : public Effect2DSerializer
{
      typedef Effect2DSerializer Base;

   public:
      virtual Candera::Effect2D const* getDefaultItem() const
      {
         static Candera::Effect2D::SharedPointer defaultItem = Candera::TextBrushBlend::Create();
         return defaultItem.GetPointerToSharedInstance();
      }

      virtual void serializeProperties(Candera::Effect2D& item, ScmlWriter& writer)
      {
         Base::serializeProperties(item, writer);

         Candera::TextBrushBlend* effect = Candera::Dynamic_Cast<Candera::TextBrushBlend*>(&item);
         const Candera::TextBrushBlend* defaultItem = Candera::Dynamic_Cast<const Candera::TextBrushBlend*>(getDefaultItem());
         if ((effect != NULL) && (defaultItem != NULL))
         {
            BlendEffectSerializer::serializeBlendEffect(effect->GetBlendEffect(), const_cast<Candera::BlendEffect&>(defaultItem->GetBlendEffect()), writer);

            Candera::TextBrush& textBrush = effect->GetTextBrush();
            Candera::TextBrush& defaultTextBrush = const_cast<Candera::TextBrush&>(defaultItem->GetTextBrush());

            CHECK_AND_WRITE_EFFECT_REF_PROPERTY(textBrush, Style)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, Text)
            WRITE_EFFECT_PROPERTY2(textBrush, Color, TextColor)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, HorizontalAlignment)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, VerticalAlignment)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, CacheArea)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, LayoutingArea)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, GlyphSpacing)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, LineSpacing)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, MultiLineLayouting)
            CHECK_AND_WRITE_EFFECT_PROPERTY(textBrush, defaultTextBrush, WordWrap)
         }
      }
};


template<> ItemSerializerBase<Candera::Node2D>* SerializerFactory::getSerializer<Candera::Node2D>(const Candera::Node2D& item)
{
   if (item.IsTypeOf(Candera::Scene2D::GetTypeId()))
   {
      static Scene2DSerializer serializer;
      return &serializer;
   }
   if (item.IsTypeOf(Candera::RenderNode::GetTypeId()))
   {
      static RenderNodeSerializer serializer;
      return &serializer;
   }
   if (item.IsTypeOf(Candera::Group2D::GetTypeId()))
   {
      static Group2DSerializer serializer;
      return &serializer;
   }
   if (item.IsTypeOf(Candera::Camera2D::GetTypeId()))
   {
      static Camera2DSerializer serializer;
      return &serializer;
   }
   return NULL;
}


template<> ItemSerializerBase<Candera::Effect2D>* SerializerFactory::getSerializer<Candera::Effect2D>(const Candera::Effect2D& item)
{
   if (item.IsTypeOf(Candera::BitmapBrushBlend::GetTypeId()))
   {
      static BitmapBrushBlendSerializer serializer;
      return &serializer;
   }
   if (item.IsTypeOf(Candera::BitmapBrushColorBlend::GetTypeId()))
   {
      static BitmapBrushColorBlendSerializer serializer;
      return &serializer;
   }
   if (item.IsTypeOf(Candera::GlBitmapBrushMaskBlend::GetTypeId()))
   {
      static GlBitmapBrushMaskBlendSerializer serializer;
      return &serializer;
   }
   if (item.IsTypeOf(Candera::MirrorBitmapBrushBlend::GetTypeId()))
   {
      static MirrorBitmapBrushBlendSerializer serializer;
      return &serializer;
   }
   if (item.IsTypeOf(Candera::SolidColorBrushBlend::GetTypeId()))
   {
      static SolidColorBrushBlendSerializer serializer;
      return &serializer;
   }
   if (item.IsTypeOf(Candera::TextBrushBlend::GetTypeId()))
   {
      static TextBrushBlendSerializer serializer;
      return &serializer;
   }
   return NULL;
}


bool ScmlSerializer::serialize(Candera::Scene2D& scene, std::ostream& stream)
{
   ScmlWriter writer(stream);
   ItemSerializerBase<Candera::Node2D>* serializer = SerializerFactory::getSerializer<Candera::Node2D>(scene);
   if (serializer != NULL)
   {
      serializer->serialize(scene, writer);
   }
   return true;
}


}
}
