/* ***************************************************************************************
* FILE:          MessageStatistics.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MessageStatistics.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "MessageStatistics.h"


#ifndef WIN32
#ifdef COURIER_MESSAGING_MONITOR_ENABLED
#include "FeatStd/Config.h"
#include "Courier/Messaging/MessagingMonitor.h"
using namespace FeatStd::Diagnostics;
#include "hmibase/util/Mutex.h"
#endif
#endif


#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MessageStatistics.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


namespace hmibase {
namespace trace {

MessageStatistics::MessageStatistics()
{
}


MessageStatistics::~MessageStatistics()
{
}


void MessageStatistics::print()
{
#if defined (COURIER_MESSAGING_MONITOR_ENABLED)
   hmibase::util::cCriticalSection cs;
   cs.Enter();
   ETG_TRACE_FATAL_THR((" #### [%25s] Courier message statistics ####", hmibase::util::trace::getAppName()));

   Courier::MessagingMonitor::RouterData rd;
   Courier::MessagingMonitor::GetRouterData(rd);
   ETG_TRACE_FATAL_THR(("  [%25s] RouterData: OverallPosts %u, OverallCorrupted %u, PostsPerSecond %u, PeakPostsPerSecond %u, AverageMessageSize %u, MaximumMessageSize %u",
                        hmibase::trace::getAppName().c_str(),
                        /// The overall count of message posts.
                        rd.mOverallPosts,
                        /// The overall number of corrupted messages
                        rd.mOverallCorrupted,
                        /// The average number of message posts per second.
                        rd.mPostsPerSecond,
                        /// The maximum number of message posts per second.
                        rd.mPeakPostsPerSecond,
                        /// The average size of posted messages (only available when IPC enabled).
                        rd.mAverageMessageSize,
                        /// The size of the largest posted message (only available when IPC enabled).
                        rd.mMaximumMessageSize));
   // Receiver
   for (int i = 0; i < Courier::MessagingMonitor::GetReceiverCount(); ++i)
   {
      Courier::MessagingMonitor::ReceiverData receiverData;
      Courier::MessagingMonitor::GetReceiverData(i, receiverData);

      ETG_TRACE_FATAL_THR(("  [%25s] Receiver [%25s]: Cycles %u, OverallReads %u, OverallPending %u",
                           hmibase::trace::getAppName().c_str(),
                           receiverData.GetName(),
                           /// The overall number of processing cycles of the ComponentMessageReceiver (thread).
                           receiverData.mCycles,
                           /// The overall number of reads from the message queue.
                           receiverData.mOverallReads,
                           /// The overall number of pending messages. Messages which have been newly queued in the message queue, while processing already queued messages.
                           receiverData.mOverallPending));
      ETG_TRACE_FATAL_THR(("  [%25s] Receiver [%25s]: AvgReadsPerCycle %u, PeakReadsPerCycle %u, AvgPendingPerCycle %u, PeakPendingPerCycle %u",
                           hmibase::trace::getAppName().c_str(),
                           receiverData.GetName(),
                           /// The average number of reads from the message queue per cycle.
                           receiverData.mAvgReadsPerCycle,
                           /// The peak number of reads from the message queue per cycle.
                           receiverData.mPeakReadsPerCycle,
                           /// The average number of pending messages.
                           receiverData.mAvgPendingPerCycle,
                           /// The peak number of pending messages.
                           receiverData.mPeakPendingPerCycle));
   }

   // Components
   Courier::ComponentId cids[] = { Courier::ComponentType::Controller,
                                   Courier::ComponentType::Model,
                                   Courier::ComponentType::View
                                 };

   unsigned int size = sizeof cids / sizeof(Courier::ComponentId);
   for (unsigned int i = 0; i < size; ++i)
   {
      Courier::MessagingMonitor::ComponentData cd;
      Courier::MessagingMonitor::GetComponentData(cids[i], cd);

      ETG_TRACE_FATAL_THR(("  [%25s] Component %i: OverallMsgs %u, MsgsConsumed %u, OverallTime %u ms, AvgTime %u ms",
                           hmibase::trace::getAppName().c_str(),
                           ETG_CENUM(Courier::ComponentType::Enum, cids[i]),
                           cd.mOverallMessages,
                           cd.mOverallMessagesConsumed,
                           cd.mOverallTimeMs,
                           cd.mAverageTimeMs));
   }
   ETG_TRACE_FATAL_THR((" ################"));

   cs.Leave();
#else
   ETG_TRACE_FATAL_THR(("[%25s] Courier Message statistics not enabled in the code, get in contact with HMIBase team", hmibase::trace::getAppName().c_str()));
#endif
}


}
}
