/* ***************************************************************************************
* FILE:          TimerObjectPool.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TimerObjectPool is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#ifndef _TimerObjectPool_H
#define _TimerObjectPool_H


class TimerObjectPool
{
   private:

      /// Integer constants
      enum PrivateIntegerConstants
      {
         ALIGNMENT = sizeof(void*)   ///< Minimal size of object in object pool (at least void pointer)
      };

      /// Element of singly-linked list
      struct Link
      {
         Link* next;                ///< Pointer to a next element of singly-linked list
      };
      struct Buffer
      {
         Buffer* next;
         char* data;
      };

      bool  _isInitialized;            ///< Flag indicating object pool initialization
      Link* _firstFree;             ///< Pointer to a first free element in the list
      Link* _lastFree;              ///< Pointer to a flast free element in the list
      int   _objectSize;              ///< Size of one object within object pool
      int   _numGrow;                 ///< Number of objects object pool should be extended with
      int   _numAllocated;            ///< Number of allocated objects
      int   _bytesAllocated;          ///< Overall size of allocated objects
      int   _numCurrentlyUsed;        ///< Number of currently used objects

      Buffer* _blocks;
      Buffer* _headBlk;
   public:
      /**
      *  @brief Initialize object.
      *
      *  Initialaze member variables with defauld values.
      *
      */
      TimerObjectPool();

      /**
      *  @brief Un-Initialize object.
      *
      *  Frees all allocated memory.
      *
      */
      virtual ~TimerObjectPool();

      /**
      *  @brief Return number of allocated object within object pool (used and unused)
      *
      *  @return _numAllocated Number of allocated objects
      *
      */
      int getNumAllocated() const
      {
         return _numAllocated;
      }

      /**
      *  @brief Return overall size of allocated objects in bytes (used and unused)
      *
      *  @return _bytesAllocated Overall size of allocated objects
      *
      */
      int getBytesAllocated() const
      {
         return _bytesAllocated;
      }

      /**
      *  @brief Return number of currently used objects within object pool
      *
      *  @return _numCurrentlyUsed Number of currently used objects
      *
      */
      int getNumCurrentlyUsed() const
      {
         return _numCurrentlyUsed;
      }

      /**
      *  @brief Initializes Object Pool
      *
      *  Function initializes object pool structure. The size of object pool is calculated
      *  from input parameters - number and size of object. Additionally when object pool is full
      *  it will be extended by getObject function with the value of numGrow.
      *
      *  @param[in]  objectSize size of one object within object pool
      *  @param[in]  numInitial initial number of objects
      *  @param[in]  numGrow    number of objects object pool should be extended with
      *
      *  @return true if initialization successful, false otherwise
      *
      */
      bool initialize(int objectSize, int numInitial, int numGrow);

      /**
      *  @brief Returns pointer to first free object within object pool
      *
      *  Function returns pointer to first element of the free objects list.
      *  If this is necessary it allocates more memory (value of m_nNumGrow)
      *
      *  @return pointer to next free object within object pool
      *
      */
      void* getObject();

      /**
      *  @brief Removes object from object pool
      *
      *  Basically function calls addFreeObject private function to turn removed object into a
      *  free object. So the object is not really removed but just added to a list of free objects.
      *
      *  @param[in]  object pointer to an object which needs to be removed from object pool
      *
      */
      void releaseObject(void* object);

   private:
      /**
      *  @brief Adds new object into a free objects list
      *
      *  Function adds object passed as an argument to the end of a free objects list.
      *
      *  @param[in]  object pointer to an object which needs to be removed from object pool
      *
      */
      void addFreeObject(Link* object);

      /**
      *  @brief Allocates memory for desired number of objects
      *
      *  Additionally function creates singly-linked list of free objects which is later used for
      *  managing objects within object pool
      *
      */
      void allocate(int num);
};


#endif //#ifndef _TimerObjectPool_H
