/* ***************************************************************************************
* FILE:          TimerManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TimerManager is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "sys_std_if.h"
#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TimerManager.cpp.trc.h"
#endif


#include  "TimerManager.h"

namespace hmibase {
namespace input {
namespace gesture {

TimerManager* TimerManager::_TimerMngInstance = 0;

bool TimerManager::initializeTimerManager()
{
   //init result
   bool result = false;
   // check for existing instance
   // when Instance is not existing create new one
   if (_TimerMngInstance == 0)
   {
      //create new instance
      _TimerMngInstance = new TimerManager();
      GESTURE_ASSERT(_TimerMngInstance != 0);
      //init instance
      result = _TimerMngInstance->initialize();
      _TimerMngInstance->createMasterthread();
   }
   return result;
}


int TimerManager::getWaitTime() const
{
   int result = -1;
   // if waiting time is long enough(greater than ExpireTime) Result is zero
   // else Result is -1
   if (_firstTimer != 0)
   {
      //get current time
      int currentTime = (int)clockGetElapsedTime();
      //calculate wait time
      result = _firstTimer->expireTime - currentTime;
      if (result < 0)
      {
         result = 0;
      }
   }
   return result;
}


bool TimerManager::startTimer(GestureDetector* receiver, int userId, int time)
{
   bool result = false;
   // first stop timer (if running)
   stopTimer(receiver, userId, false);
   //ETG_TRACE_USR4_THR(("[FUNC]--> TimerManager::startTimer"));
   // check parameters
   if ((receiver != 0) && (time > 0))
   {
      // get a timer info object
      TimerInfo* pInfo = reinterpret_cast<TimerInfo*>(_TimerPool.getObject());
      // set data
      pInfo->receiver = receiver;
      pInfo->userId = userId;
      int currentTime = (int)clockGetElapsedTime();
      pInfo->expireTime = currentTime + time;

      // look for correct place within list to insert new timer
      // (sorted by expire time)
      TimerInfo** ppInfo = &_firstTimer;
      while (*ppInfo != 0)
      {
         if (((*ppInfo)->expireTime - currentTime) > time)
         {
            break;
         }
         ppInfo = &(*ppInfo)->next;
      }

      // insert new timer
      pInfo->next = *ppInfo;
      *ppInfo = pInfo;
      result = true;
   }
   return result;
}


bool TimerManager::stopTimer(const GestureDetector* receiver, int userId, bool allUserIds)
{
   // go through list of timers, and remove all that match the criteria
   TimerInfo** ppInfo = &_firstTimer;
   bool isStopped = false;

   while (*ppInfo != 0)
   {
      TimerInfo* pInfo = *ppInfo;
      if ((pInfo->receiver == receiver) && (allUserIds || (pInfo->userId == userId)))
      {
         *ppInfo = pInfo->next;
         _TimerPool.releaseObject(pInfo);

         isStopped = true;
         if (!allUserIds)
         {
            // there can be only one with given receiver and user id -> break now
            break;
         }
      }
      else
      {
         // go to next entry
         ppInfo = &pInfo->next;
      }
   }
   return isStopped;
}


void TimerManager::notifyTimers(void* _pvArg)
{
   TimerInfo* pInfo = reinterpret_cast<TimerInfo*>(_pvArg);
   //get current time
   // int currentTime = (int)clockGetElapsedTime();
   if (pInfo != 0)
   {
      //send the Call back to Corresponding receiver
      pInfo->receiver->processTimerEvent(pInfo->userId);
      //printf("notifyTimers %d",pInfo->userId);
      TimerManager::s_getInstance()._TimerPool.releaseObject(pInfo);
   }
}


TimerManager::~TimerManager()
{
   // Only for lint, destructor is never called so there is no need to delete all timers
   delete _firstTimer;
   _firstTimer = 0;
#ifndef WIN32
   pthread_exit(0);
#endif
}


TimerManager::TimerManager() :
   _firstTimer(0)
{
   // empty
}


bool TimerManager::initialize()
{
   // initialize Timer Pool
   bool result = _TimerPool.initialize(sizeof(TimerInfo), static_cast<int>(NUM_INIT_TIMERS), static_cast<int>(NUM_GROW_TIMERS));
   return result;
}


void TimerManager::notifySoftTimersTimers()
{
   //get current time
   int currentTime = (int)clockGetElapsedTime();
   while (_firstTimer != 0)
   {
      //get first timer
      TimerInfo* pInfo = _firstTimer;
      if ((pInfo->expireTime - currentTime) > 0)
      {
         break;
      }
      _firstTimer = pInfo->next;

      //handle recievers
      //need to check how we can handle if the Gesture handler Removed before timer expiry may be Similar functionality Like MAGIC_VALUE
      pInfo->receiver->processTimerEvent(pInfo->userId);
      _TimerPool.releaseObject(pInfo);
   }
}


#ifdef WIN32
DWORD WINAPI TimerManager::timerThreadStart(LPVOID lpParam)
{
   TimerManager::s_getInstance().pollThread();
   return 0;
}


void TimerManager::createMasterthread()
{
   HANDLE  hThread;
   DWORD   dwThreadId;
   hThread = CreateThread(
                NULL,       // default security attributes
                0,          // use default stack size
                (LPTHREAD_START_ROUTINE)timerThreadStart, // thread function name
                (void*)this,          // argument to thread function
                0,             // use default creation flags
                &dwThreadId);  // returns the thread identifier

   if (hThread == NULL)
   {
      ETG_TRACE_USR4_THR(("TimerManager::createMasterthread()-->Insufficient resources to create thread"));
      DebugBreak();
   }
}


#else
void* TimerManager::timerThreadStart(void* /*pParam*/)
{
   TimerManager::s_getInstance().pollThread();
   return 0;
}


void TimerManager::createMasterthread()
{
   pthread_t _timeThread ;
   int rvalue = pthread_create(&_timeThread, NULL, timerThreadStart, (void*)const_cast<char*>("Timer_thread"));  //Create a Sleeping thread which wakeups after after every 10ms
   if (rvalue != 0)
   {
      switch (rvalue)
      {
         case EAGAIN :
            ETG_TRACE_USR4_THR(("TimerManager::createMasterthread()-->Insufficient resources to create another thread"));
            break;
         case EINVAL :
            ETG_TRACE_USR4_THR(("TimerManager::createMasterthread()-->Invalid settings in attr"));
            break;
         case EPERM :
            ETG_TRACE_USR4_THR(("TimerManager::createMasterthread()-->No permission to set the scheduling policy and parameters specified in attr"));
            break ;
         default:
            ETG_TRACE_USR4_THR(("TimerManager::createMasterthread()-->invalid"));
            break ;
      }
   }
}


#endif
void TimerManager::pollThread()
{
   for (;/*ever*/;)
   {
#ifdef WIN32
      Sleep(30);
#else
      usleep(30000); // 30 mili seconds
#endif

      int timeToWait = TimerManager::s_getInstance().getWaitTime();
      bool isSomethingToDo = (timeToWait == 0);

      if (isSomethingToDo)
      {
         // process timer events
         TimerManager::s_getInstance().notifySoftTimersTimers(); // notify all the Expired Timers
      }
   }
}


#ifdef WIN32
uint32_t TimerManager::clockGetElapsedTime()
{
   static bool s_bTimeBeginPeriodCalled = false;
   if (!s_bTimeBeginPeriodCalled)
   {
      s_bTimeBeginPeriodCalled = true;
      MMRESULT result = timeBeginPeriod(1);
      GESTURE_ASSERT(result == TIMERR_NOERROR);
   }
   return (uint32_t)timeGetTime();
}


#else
uint32_t TimerManager::clockGetElapsedTime()
{
   timeval oCur;
   struct timespec ts;
   uint32_t ms;
   if (clock_gettime(CLOCK_MONOTONIC, &ts))
   {
      gettimeofday(&oCur, NULL);
      ms = static_cast<uint32_t>(oCur.tv_sec * 1000 + oCur.tv_usec / 1000);
   }
   else
   {
      ms = static_cast<uint32_t>(ts.tv_sec * 1000 + ts.tv_nsec / 1000000);
   }
   return ms;
}


#endif
//check if we can implement pause and Timer Running states
}


}
}
