/* ***************************************************************************************
* FILE:          TapGestureDetector.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TapGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#ifndef _TapGestureDetector_H
#define _TapGestureDetector_H


#include "GestureBasetypes.h"
#include "GestureDetector.h"

namespace hmibase {
namespace input {
namespace gesture {

class TapGestureDetector : public hmibase::input::gesture::GestureDetector
{
   public:

      /**
      *  @brief Initialize object.
      *
      *  Initialaze member variables with default values.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] maxDistance maximum distance to move; when moving beyond this limit,
      *                          the gesture is aborted
      *  @param[in] holdTime time until hold is detected or repeat is started
      *  @param[in] repeatTime time for repeat after hold time has expired
      *
      */
      TapGestureDetector(IGestureListener* receiver, int gesturePriority, bool doubleTapEnabled, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType, unsigned int doubleTapDetectionTime, int maxDistance, unsigned int holdTime, unsigned int repeatTime);

      /**
      *  @brief Un-Initialize object.
      *
      *  Frees all allocated memory.
      *
      */
      virtual ~TapGestureDetector();

      /**
      *  @brief Get priority of the gesture
      *
      *  @return  Gesture priority (refers to GesturePriority enumeration)
      *
      */
      virtual int getPriority();

      /**
      *  @brief Cancel the gesture
      *
      *  Cancelling the gesture stops all internal timers (if any)
      *  and sends the abort command to gesture receiver (if needed).
      *
      */
      virtual void cancel();

      /**
      *  @brief Detects the gesture
      *
      *  Function process the input touchMsg and checks if all conditions
      *  needed to detect a gesture are fulfilled. If yes - function will
      *  inform registered receiver about detecting the gesture.
      *
      *  @param[in]
      *
      *  @return True if the gesture was detected, false otherwise
      *
      */
      virtual bool detect();

      /**
      *  @brief Register for the gesture detection.
      *
      *  This function adds object of this class to the list of gestures
      *  managed by GestureHandler.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] maxDistance maximum distance to move; when moving beyond this limit,
      *                          the gesture is aborted
      *  @param[in] timerType type of the timer, e.g long or repeat
      *
      *  @return true if the registration successful, false otherwise
      *
      */
      static bool registerGesture(IGestureListener* receiver,
                                  int gesturePriority = static_cast<int>(PRIORITY_TAP),
                                  bool doubleTapEnabled = true,
                                  bool parallelGestureRecognitionEnabled = false,
                                  GestureEvent::GestureType  parallelRecognizableGestureType = GestureEvent::GT_None,
                                  unsigned int doubleTapDetectionTime = 750,
                                  int maxDistance = 80,
                                  unsigned int holdTime = 1000,
                                  unsigned int repeatTime = 200
                                 );

      /**
      *  @brief  Process a timer event.
      *
      *  Function handles hold and repeat timers.
      *  When the hold timer expires receiver is informed about the tap hold gesture and the repeat timer is started.
      *  When the repeat timer expires, receiver is informed about the tap repeat gesture and repeat timer is re-started
      *
      *  @param[in]  userId  Identifier for this timer
      *
      */
      virtual void processTimerEvent(int userId);

   private:
      /**
      *  @brief  Sends Tap Gesture Start Event To Gesture Receiver.
      *
      *  Function Sends ET_START when Tap Gesture is Recognized
      *
      *  @param[in]  userId  Identifier for this timer
      *
      */
      void sendTapStartEvent();

      /**
      *  @brief Enumeration for timer identifiers of this class
      */
      enum TimerIds
      {
         TIMERID_TOUCH_LONG,                         ///< timer id: touch long-press detection
         TIMERID_TOUCH_REPEAT,                       ///< timer id: touch repeat detection
         TIMERID_TOUCH_STARTEVDELAY,                 /// to delay start event
         TIMERID_UNDEF
      };

      TimerIds _activeTimerId;
      unsigned int _tapHoldTime;
      unsigned int _tapPrimaryRepeatTime;

      //Tap Configuration parameters
      int  _tapMaxDistance; //< maximum distance to move; when moving beyond this limit, the gesture is aborted
      unsigned int  _doubleTapDetectionTime; //Time with in which Double tap Should be recognized
      int  _gesturePriority; // used to Determine the Order of Gesture Detection
      bool _isDoubleTapEnabled; // used to enable Double Tap Detection

      bool  _isTapDetected;        //< Flag indicating if the tap gesture was already detected
      bool  _isDoubleTapDetected;  //< Flag indicating if the Double tap gesture was already detected

      bool _isInitialActionsExecuted; //< Flag indicating if the initial action for the gesture was executed
      bool _isFirstTouch;  //< Flag to indicate first touch on screen
      bool _isStartEvSent; //< Flag To indicate ET_START event sent to

      static unsigned long _LastTapEndedTime;   //< static variable to keep track of the last known tap END
      static IGestureListener* _pLastGestureReceiver; // < static widget pointer

      bool _isTapHoldRepeatLogged;    //< flag if hold/repeat interaction has been logged for this touch

      Vector2D _previousPoint;     //< Position (coordinates) detected during previous touch event
};


}
}


}

#endif // #ifndef _TapGestureDetector_H
