/* ***************************************************************************************
* FILE:          SwipeGestureDetector.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SwipeGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#ifndef _SwipeGestureDetector_H
#define _SwipeGestureDetector_H


#include "GestureBasetypes.h"
#include "GestureDetector.h"

namespace hmibase {
namespace input {
namespace gesture {
class SwipeGestureDetector : public hmibase::input::gesture::GestureDetector
{
   public:
      /**
      *  @brief Initialize object.
      *
      *  Initialaze member variables with defauld values.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] minSpeed minimum movement speed in pixel/s
      *  @param[in] minDistance minimum distance of total movement
      *  @param[in] direction defines the direction in which the movement should
      *                       be detected: horizontal, vertical, 2D; this is relevant
      *                       for minDistance but not for holdMaxDistance
      *
      */
      SwipeGestureDetector(IGestureListener* receiver, int gesturePriority, GestureEvent::Direction direction, bool parallelGestureRecognitionEnabled,
                           GestureEvent::GestureType parallelRecognizableGestureType, int minSpeed, int minDistance);

      /**
      *  @brief Un-Initialize object.
      *
      *  Frees all allocated memory.
      *
      */
      virtual ~SwipeGestureDetector();

      /**
      *  @brief Get priority of the gesture
      *
      *  @return  Gesture priority (refers to GesturePriority enumeration)
      *
      */
      virtual int  getPriority();

      /**
      *  @brief Get the direction of the gesture
      *
      *  Direction is valid only for DragNudge and SwipeFling gestures
      *
      *  @return  Gesture direction (refers to Direction enumeration)
      *
      */
      virtual GestureEvent::Direction getDirection();

      /**
      *  @brief Cancel the gesture
      *
      *  Cancelling the gesture stops all internal timers (if any)
      *  and sends the abort command to gesture receiver (if needed).
      *
      */
      virtual void cancel();

      /**
      *  @brief Detects the gesture
      *
      *  Function process the input touchMsg and checks if all conditions
      *  needed to detect a gesture are fulfilled. If yes - function will
      *  inform registered receiver about detecting the gesture.
      *
      *
      *  @return True if the gesture was detected, false otherwise
      *
      */
      virtual bool detect();

      /**
      *  @brief Register for the gesture detection.
      *
      *  This function adds object of this class to the list of gestures
      *  managed by GestureHandler.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] minSpeed minimum movement speed in pixel/s
      *  @param[in] minDistance minimum distance of total movement
      *  @param[in] direction defines the direction in which the movement should
      *                       be detected: horizontal, vertical, 2D; this is relevant
      *                       for minDistance but not for holdMaxDistance
      *
      *  @return true if the registration successful, false otherwise
      *
      */
      static bool registerGesture(IGestureListener* receiver,
                                  int gesturePriority = static_cast<int>(PRIORITY_SWIPE),
                                  GestureEvent::Direction direction = GestureEvent::DIR_2D,
                                  bool parallelGestureRecognitionEnabled = false,
                                  GestureEvent::GestureType  parallelRecognizableGestureType = GestureEvent::GT_None,
                                  int  minSpeed = 50,
                                  int  minDistance = 60
                                 );

   private:
      int _swipeMinSpeed;				 //< Minimum movement speed in pixel/s
      int _swipeMinDistance;			 //< Minimum distance of total movement
      GestureEvent::Direction _swipeDirection;   ///< Defines the direction in which the movement should be detected:
      bool _isSwipeMinDistanceExceed;   //< true if minimum distance has been exceeded
      int _gesturePriority;    //< used to Determine the Order of Gesture Detection
};


}
}


}

#endif // #ifndef _SwipeGestureDetector_H
