/* ***************************************************************************************
* FILE:          SpreadGestureDetector.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SpreadGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#ifndef _SpreadGestureDetector_H
#define _SpreadGestureDetector_H


#include "GestureBasetypes.h"
#include "GestureDetector.h"

namespace hmibase {
namespace input {
namespace gesture {

class SpreadGestureDetector : public hmibase::input::gesture::GestureDetector
{
   public:
      /**
      *  @brief Initialize object.
      *
      *  Initialaze member variables with defauld values.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] doubleTouchMaxTime maximum time between first and second touch position
      *  @param[in] spreadMinFactor minimum factor (distance now divided by distance at
      *                                      press time) for detecting a spread; should be = 1.0
      *  @param[in] pinchMaxFactor maximum factor (distance now divided by distance at press
      *                                      time) for detecting a pinch; should be = 1.0
      *
      */
      SpreadGestureDetector(IGestureListener* receiver, int gesturePriority, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType, unsigned int doubleTouchMaxTime, float spreadMinFactor, float pinchMaxFactor);

      /**
      *  @brief Un-Initialize object.
      *
      *  Frees all allocated memory.
      *
      */
      virtual ~SpreadGestureDetector();

      /**
      *  @brief Get priority of the gesture
      *
      *  @return  Gesture priority (refers to GesturePriority enumeration)
      *
      */
      virtual int getPriority();

      /**
      *  @brief Cancel the gesture
      *
      *  Cancelling the gesture stops all internal timers (if any)
      *  and sends the abort command to gesture receiver (if needed).
      *
      */
      virtual void cancel();

      /**
      *  @brief Detects the gesture
      *
      *  Function process the input touchMsg and checks if all conditions
      *  needed to detect a gesture are fulfilled. If yes - function will
      *  inform registered receiver about detecting the gesture.
      *
      *
      *  @return True if the gesture was detected, false otherwise
      *
      */
      virtual bool detect();

      /**
      *  @brief Register for the gesture detection.
      *
      *  This function adds object of this class to the list of gestures
      *  managed by GestureHandler.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] doubleTouchMaxTime maximum time between first and second touch position
      *  @param[in] nSpreadMinFactor minimum factor (distance now divided by distance at
      *                                      press time) for detecting a spread; should be = 1.0
      *  @param[in] nPinchMaxFactor maximum factor (distance now divided by distance at press
      *                                      time) for detecting a pinch; should be = 1.0
      *
      *  @return true if the registration successful, false otherwise
      *
      */
      static bool registerGesture(IGestureListener* receiver,
                                  int gesturePriority = static_cast<int>(PRIORITY_SPREAD),
                                  bool parallelGestureRecognitionEnabled = false,
                                  GestureEvent::GestureType  parallelRecognizableGestureType = GestureEvent::GT_None,
                                  unsigned int   doubleTouchMaxTime = 1000,
                                  float nSpreadMinFactor = 125,
                                  float nPinchMaxFactor = 75
                                 );

      /**
      *  @brief  Process a timer event.
      *
      *  Handles the timer which counts the time between the first and the second touch. If the time difference between two touch points
      *  is greater than defined, gesture is cancelled.
      *
      *  @param[in]  userId  Identifier for this timer
      *
      */
      virtual void processTimerEvent(int userId);

   private:
      /**
      *  @brief Enumeration for timer identifiers of this class
      */
      enum TimerIds
      {
         TIMERID_DOUBLE_TOUCH_TIME   //< timer id: double touch detection
      };

      unsigned int   _doubleTouchMaxTime;      //< maximum time between first and second touch position
      int   _initialDistanceSquared;  //< Initial distance between two touch positions
      float _spreadMinFactor;         //< minimum factor (distance now divided by distance at press time) for detecting a spread; should be = 1.0
      float _pinchMaxFactor;          //< maximum factor (distance now divided by distance at press time) for detecting a pinch; should be = 1.0
      int   _gesturePriority;         //< used to Determine the Order of Gesture Detection

      bool _isDoubleTouchDetected;    //< Flag indicating if the double touch has been detected
      bool _isSpreadPinchDetected;    //< Flag indicating if the spread/pinch gesture has been detected
      bool _isInitialActionsExecuted; //< Flag indicating if the initial action for the gesture was executed
      bool _isGestureCancelled;       //< Flag indicating if the gesture is cancelled
      bool _isInitialTouchDetected;   //< Flag indicating if at least one valid touch point was detected
};


}
}


}
#endif // #ifndef _SpreadGestureDetector_H
