/* ***************************************************************************************
* FILE:          RotateGestureDetector.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RotateGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#ifndef _RotateGestureDetector_H
#define _RotateGestureDetector_H


#include "GestureBasetypes.h"
#include "GestureDetector.h"
#include <cmath>
namespace hmibase {
namespace input {
namespace gesture {

class RotateGestureDetector : public hmibase::input::gesture::GestureDetector
{
   public:
      /**
      *  @brief Initialize object.
      *
      *  Initialize member variables with default values.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] minSpeed minimum movement speed in pixel/s
      *  @param[in] minDistance minimum distance of total movement
      *  @param[in] direction defines the direction in which the movement should
      *                       be detected: horizontal, vertical, 2D; this is relevant
      *                       for minDistance but not for holdMaxDistance
      *
      */
      RotateGestureDetector(IGestureListener* receiver, int gesturePriority, bool parallelGestureRecognitionEnabled,
                            GestureEvent::GestureType parallelRecognizableGestureType, unsigned int doubleTouchMaxTime, int minRotationAngle);

      /**
      *  @brief Un-Initialize object.
      *
      *  Frees all allocated memory.
      *
      */
      virtual ~RotateGestureDetector();

      /**
      *  @brief Get priority of the gesture
      *
      *  @return  Gesture priority (refers to GesturePriority enumeration)
      *
      */
      virtual int  getPriority();

      /**
      *  @brief Get the direction of the gesture
      *
      *  Direction is valid only for DragNudge and SwipeFling gestures
      *
      *  @return  Gesture direction (refers to Direction enumeration)
      *
      */
      virtual GestureEvent::Direction getDirection();

      /**
      *  @brief Cancel the gesture
      *
      *  Cancelling the gesture stops all internal timers (if any)
      *  and sends the abort command to gesture receiver (if needed).
      *
      */
      virtual void cancel();

      /**
      *  @brief Detects the gesture
      *
      *  Function process the input touchMsg and checks if all conditions
      *  needed to detect a gesture are fulfilled. If yes - function will
      *  inform registered receiver about detecting the gesture.
      *
      *  @param[in] touchMsg Touch event data
      *
      *  @return True if the gesture was detected, false otherwise
      *
      */
      virtual bool detect();

      /**
      *  @brief Register for the gesture detection.
      *
      *  This function adds object of this class to the list of gestures
      *  managed by GestureHandler.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] minSpeed minimum movement speed in pixel/s
      *  @param[in] minDistance minimum distance of total movement
      *  @param[in] direction defines the direction in which the movement should
      *                       be detected: horizontal, vertical, 2D; this is relevant
      *                       for minDistance but not for holdMaxDistance
      *
      *  @return true if the registration successful, false otherwise
      *
      */
      static bool registerGesture(IGestureListener* receiver,
                                  int gesturePriority = static_cast<int>(PRIORITY_ROTATE),
                                  bool parallelGestureRecognitionEnabled = false,
                                  GestureEvent::GestureType  parallelRecognizableGestureType = GestureEvent::GT_None,
                                  unsigned int  doubleTouchMaxTime = 50,
                                  int  minRotationAngle = 60
                                 );
      /**
      *  @brief  Process a timer event.
      *
      *  Handles the timer which counts the time between the first and the second touch. If the time difference between two touch points
      *  is greater than defined, gesture is cancelled.
      *
      *  @param[in]  userId  Identifier for this timer
      *
      */
      virtual void processTimerEvent(int userId);
   private:
      float angleBetweenLines(const Vector2D& sPoint1, const Vector2D& sPoint2, const Vector2D& nPoint1, const Vector2D& nPoint2);

      enum TimerIds
      {
         TIMERID_DOUBLE_TOUCH_TIME   //< timer id: double touch detection
      };

      unsigned int _doubleTouchMaxTime;				 //< Minimum movement speed in pixel/s
      GestureEvent::Direction _rotateDirection;   ///< Defines the direction in which the movement should be detected:
      float _rotateMinAngle;   //Minimum Change in angle(in Radians) for a touch interaction to be considered as a rotation gesture
      int _gesturePriority;    //< used to Determine the Order of Gesture Detection
      bool _isDoubleTouchDetected;    //< Flag indicating if the double touch has been detected
      bool _isInitialActionsExecuted; //< Flag indicating if the initial action for the gesture was executed
      bool _isGestureCancelled;       //< Flag indicating if the gesture is cancelled
      bool _isInitialTouchDetected;   //< Flag indicating if at least one valid touch point was detected
      bool _isRotateDetected;        // < Flag indicating if the rotate gesture is detected
};


}
}


}
#endif // #ifndef _RotateGestureDetector_H
