/* ***************************************************************************************
* FILE:          RotateGestureDetector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RotateGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "sys_std_if.h"
#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RotateGestureDetector.cpp.trc.h"
#endif


#include "RotateGestureDetector.h"
#include "GestureHandler.h"
#ifdef GESTURE_SIMULATION
#include "TimerManager.h"
#else
#include "AppUtils/Timer.h"
#include "BaseContract/generated/BaseGesturesMsgs.h"
#endif
#include "IGestureListener.h"

namespace hmibase {
namespace input {
namespace gesture {
const float C_PI = 3.141593f;

RotateGestureDetector::RotateGestureDetector(IGestureListener* receiver, int gesturePriority, bool parallelGestureRecognitionEnabled,
      GestureEvent::GestureType parallelRecognizableGestureType, unsigned int doubleTouchMaxTime, int minRotationAngle) :
   GestureDetector(receiver, parallelGestureRecognitionEnabled, parallelRecognizableGestureType),
   _doubleTouchMaxTime(doubleTouchMaxTime),
   _rotateDirection(GestureEvent::DIR_2D),
   _rotateMinAngle(static_cast<float>(minRotationAngle)),//30 Degrees
   _gesturePriority(gesturePriority),
   _isDoubleTouchDetected(false),
   _isInitialActionsExecuted(false),
   _isGestureCancelled(false),
   _isInitialTouchDetected(false),
   _isRotateDetected(false)
{
   //set gesture
   setGestureType(GestureEvent::GT_Rotate);
   setDirection(GestureEvent::DIR_2D);

   setGestureDetectorType(GestureEvent::GT_Rotate);
}


RotateGestureDetector::~RotateGestureDetector()
{
   //empty
}


bool RotateGestureDetector::registerGesture(IGestureListener* receiver, int gesturePriority, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType, unsigned int doubleTouchMaxTime, int minRotationAngle)
{
   GESTURE_ASSERT(receiver != 0);
   //init result flag
   bool result = false;

   if (GestureHandler::s_getInstance().getRegistrationEnabled())
   {
      //get detector
      GestureDetector* detector = new RotateGestureDetector(receiver, gesturePriority, parallelGestureRecognitionEnabled, parallelRecognizableGestureType, doubleTouchMaxTime, minRotationAngle);
      GESTURE_ASSERT(detector != 0);
      //register detector
      result = GestureHandler::s_getInstance().insertDetector(detector);
      if (!result)
      {
         //free detector if registration failed
         delete detector;
      }
   }
   return result;
}


void RotateGestureDetector::cancel()
{
   if ((getReceiver() != 0) && _isRotateDetected)
   {
      _isRotateDetected = false;
      _isDoubleTouchDetected = false;
      //set parameter
      setEventType(GestureEvent::ET_ABORT);
      ETG_TRACE_USR4_THR(("RotateGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                          ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                          ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
      getReceiver()->processGestureEvent(getGesture());
   }
}


bool RotateGestureDetector::detect()
{
   bool isPoint1Press = GestureHandler::s_getInstance().getPosition1Valid();
   bool isPoint2Press = GestureHandler::s_getInstance().getPosition2Valid();
   bool detected = false;

   IGestureListener* receiver = getReceiver();

   if (receiver != 0)
   {
      // Set current touch possition and press information
      setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
      setPoint1Valid(isPoint1Press);
      setPoint2(GestureHandler::s_getInstance().getLastTouchPoint2());
      setPoint2Valid(isPoint2Press);

      if (_isGestureCancelled)
      {
         // Cancel the detection in case the time between the first and second touch is greater than configured
         // using _doubleTouchMaxTime member variable
         setAlive(false);//GestureHandler::s_getInstance().removeDetector(this);
         return false;
      }

      if (isPoint1Press && isPoint2Press)
      {
         //stop the timer on double touch detection.
#ifdef GESTURE_SIMULATION
         TimerManager::s_getInstance().stopTimer(this, static_cast<int>(TIMERID_DOUBLE_TOUCH_TIME), false);
#else
         _timer.stop();
#endif


         if (!_isInitialActionsExecuted)
         {
            _isDoubleTouchDetected = true;
            setEventType(GestureEvent::ET_HOLD);
            ETG_TRACE_USR4_THR(("RotateGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
            // Send the gesture hold event to the receiver
            receiver->processGestureEvent(getGesture());
            _isInitialActionsExecuted = true;
         }

         float rotateAngle = angleBetweenLines(GestureHandler::s_getInstance().getStartPoint1(), GestureHandler::s_getInstance().getStartPoint2(), GestureHandler::s_getInstance().getLastTouchPoint1(), GestureHandler::s_getInstance().getLastTouchPoint2());

         //set Direction
         if (rotateAngle > 0)
         {
            _rotateDirection = GestureEvent::DIR_CLOCKWISE;
         }
         else
         {
            _rotateDirection = GestureEvent::DIR_ANTICLOCKWISE;
         }

         if (!_isRotateDetected && (_rotateMinAngle <= static_cast<int16_t>(std::abs(rotateAngle))))
         {
            _isRotateDetected = true;

            setRotateAngle(static_cast<int16_t>(rotateAngle));
            setDirection(_rotateDirection);
#ifdef GESTURE_SIMULATION
            TimerManager::s_getInstance().stopTimer(this, static_cast<int>(TIMERID_DOUBLE_TOUCH_TIME), false);
#else
            _timer.stop();
#endif
            setEventType(GestureEvent::ET_START);
            ETG_TRACE_USR4_THR(("RotateGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
            // Send the gesture START event to the receiver
            receiver->processGestureEvent(getGesture());
            detected = true;
            return detected;
         }
         if (_isRotateDetected)
         {
            setDirection(_rotateDirection);
            setRotateAngle(static_cast<int16_t>(rotateAngle));
            setEventType(GestureEvent::ET_MOVE);
            ETG_TRACE_USR4_THR(("RotateGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
            // If rotate gesture was detected send the gesture MOVE event
            receiver->processGestureEvent(getGesture());
         }
      }

      if (_isDoubleTouchDetected)
      {
         // This block is executed in case the double touch was detected before
         // and a touch event with new coordinates is received now

         if (!isPoint1Press || !isPoint2Press)
         {
            // In case at least one touch release is detected, send the gesture END event and remove current
            // gesture detector
            setEventType(GestureEvent::ET_END);
            setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
            setPoint1Valid(true);
            setPoint2(GestureHandler::s_getInstance().getLastTouchPoint2());
            setPoint2Valid(true);
            ETG_TRACE_USR4_THR(("RotateGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
            receiver->processGestureEvent(getGesture());
            _isDoubleTouchDetected = false;
            _isRotateDetected = false;
            setAlive(false);//GestureHandler::s_getInstance().removeDetector(this);
            //send event to interaction logger only once after complete pinch/spread action
            //GUI_UTIL_InteractionLogger::s_logTouchInformation(GUI_EVENT_GESTURE_ROTATE, receiver,
            //                                                        getGesture().point1, getGesture().point2);
            return false;
         }
      }
      else
      {
         // In case the double touch was not detected
         if (isPoint1Press || isPoint2Press)
         {
            if (!_isInitialTouchDetected)
            {
               // In case only one touch point is valid, start timer to check if the next touch point will be received
               // in the defined time
#ifdef GESTURE_SIMULATION
               TimerManager::s_getInstance().startTimer(this, static_cast<int>(TIMERID_DOUBLE_TOUCH_TIME), _doubleTouchMaxTime);
#else
               _timer.setName("RotateGestureDetector", "");
               _timer.setTimerId(GestureEvent::GT_Rotate);
               _timer.setTimeout(TIMERID_DOUBLE_TOUCH_TIME, _doubleTouchMaxTime, &(GestureHandler::s_getInstance()));
               _timer.start();
#endif
               _isInitialTouchDetected = true;
            }
         }
      }
   }//(receiver != 0)

   return detected;
}


GestureEvent::Direction RotateGestureDetector::getDirection()
{
   //return direction
   return _rotateDirection;
}


int RotateGestureDetector::getPriority()
{
   //get priority
   return _gesturePriority;
}


float RotateGestureDetector::angleBetweenLines(const Vector2D& sPoint1, const Vector2D& sPoint2, const Vector2D& nPoint1, const Vector2D& nPoint2)
{
   float angle1 = atan2f((float)(sPoint2.y - sPoint1.y), (float)(sPoint2.x - sPoint1.x));
   float angle2 = atan2f((float)(nPoint2.y - nPoint1.y), (float)(nPoint2.x - nPoint1.x));

   float angle = (float)(((angle1 - angle2) * 180) / C_PI);
   angle = static_cast<float>((int)angle % 360);
   if (angle < -180.f)
   {
      angle += 360.0f;
   }
   if (angle > 180.f)
   {
      angle -= 360.0f;
   }
   return -angle;
}


void RotateGestureDetector::processTimerEvent(int userId)
{
   //case handling
   switch (userId)
   {
      case TIMERID_DOUBLE_TOUCH_TIME:
         //set cancel flag
         _isGestureCancelled = true;
         ETG_TRACE_USR4_THR(("RotateGestureDetector.processTimerEvent: cancel gesture due to timer"));
         break;

      default:
         break;
   }
}


}
}


}
