/* ***************************************************************************************
* FILE:          GestureHandler.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GestureHandler is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef _GestureHandler_H
#define _GestureHandler_H

#include "View/MessagingDefines.h"
#include "GestureBasetypes.h"
#include <algorithm>
#include <map>
#include <vector>
#include <limits>
#include <limits.h>

#ifdef GESTURE_SIMULATION
#define COURIER_MSG_MAP_BEGIN(x)
#define ON_COURIER_MESSAGE(x)
#define COURIER_MSG_MAP_END()
#else
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
#include "View/CGI/CgiExtensions/CourierMessageMapper.h"
#endif
#include "BaseContract/generated/BaseGesturesMsgs.h"
#include "hmibase/util/TimerThreaded.h"
#endif

namespace hmibase {
namespace input {
namespace gesture {

class IGestureListener;
class GestureDetector;

#define MAX_POINTER_ID 1

class GestureHandler
#ifndef GESTURE_SIMULATION
   : public hmibase::util::TimerThreaded::ITimerFunction
#endif
{
   public:
      /**
      *  @brief Get the one and only instance of this class.
      *
      *  @return instance of this class
      */
      static GestureHandler& s_getInstance();

      /**
      *  @brief Handle all kind of touch events
      *
      *  @param[in]  evTouch Touch event
      */
      bool processTouchData();

      /**
      *  @brief Reconfigures the Gesture Priority
      *
      *  @param[in] newGesturePriority new Gesture Priority Map
      *  @return True if Gesture Priority Configuration Successful
      */
      bool configureGesturePriority(std::map<GestureEvent::GestureType, char> newGesturePriority);
      /**
      *  @brief configures Gesture Priority to Default Values (e.g.: enum GesturePriority)
      */
      void setGesturePriorityToDefaults();

      //   protected:

      /**
      *  @brief Get the first gesture detector
      *
      *  @return First gesture detector in the list
      */
      GestureDetector* getFirstDetector()
      {
         return _firstDetector;
      }
      /**
      *  @brief Insert the gesture detector to the list
      *
      *  @param[in] detector Detector which should be added to the list
      *
      *  @return True if the detector was successfuly inserted
      */
      bool insertDetector(GestureDetector* detector);
      /**
      *  @brief Remove all lower priority detectors
      *
      *  @param[in] detector All detectors with lower priority then this one,
      *                       will be removed
      */
      void removeFurtherGestureDetectors(GestureDetector* detector);
      /**
      *  @brief Remove all lower priority detectors
      *
      *  @param[in] detector All detectors with lower priority then this one,
      *                       will be removed
      */
      void removeNonParallelGestureDetectors(GestureDetector* detector, GestureEvent::GestureType gestureType) const;
      /**
      *  @brief Remove specified gesture detector
      *
      *  @param[in] detector Detector which should be removed from the list
      */
      void removeDetector(GestureDetector* detector);
      /**
      *  @brief Remove all detectors of specified receiver
      *
      *  @param[in] handler Receiver of the gesture events
      */
      void removeDetectors(const IGestureListener* handler);
      /**
      *  @brief Remove all detectors
      */
      void removeAllGestures();
      /**
      *  @brief gets the Gesture Priority of the Specified Gesture
      *  @param[in] gestureType Type of the gesture e.g. GT_DragNudge
      *  @return  Priority of the Specified GestureType
      */

      char getGesturePriority(GestureEvent::GestureType gestureType);

      /**
      *  @brief Get the coordinates of the first touch event for point 1(within one gesture)
      *
      *  @return Coordinates of the first touch event
      */
      Vector2D  getStartPoint1()
      {
         if (_touchData._FirstTouchPointerList.size() > 0)
         {
            return  _touchData._FirstTouchPointerList[0]._coordinates;
         }
         return Vector2D(-1, -1); //send invalid Touch Point
      }

      /**
      *  @brief Get the coordinates of the first touch event for point 2(within one gesture)
      *
      *  @return Coordinates of the first touch event
      */
      Vector2D  getStartPoint2()
      {
         if (_touchData._FirstTouchPointerList.size() > 1)
         {
            return  _touchData._FirstTouchPointerList[1]._coordinates;
         }
         return Vector2D(-1, -1); //send invalid Touch Point
      }

      /**
      *  @brief Get the last coordinates of the first touch point (within one gesture)
      *
      *  @return Coordinates of the last touch point
      */
      Vector2D  getLastTouchPoint1()
      {
         if (_touchData._LastTouchPointerList.size() > 0)
         {
            return _touchData._LastTouchPointerList[0]._coordinates;
         }
         return Vector2D(-1, -1); //send invalid Touch Point
      }

      /**
      *  @brief Get the last coordinates of the second touch point (within one gesture)
      *
      *  @return Coordinates of the last touch point
      */
      Vector2D  getLastTouchPoint2()
      {
         if (_touchData._LastTouchPointerList.size() > 1)
         {
            return _touchData._LastTouchPointerList[1]._coordinates;
         }
         return Vector2D(-1, -1); //send invalid Touch Point
      }

      /**
      *  @brief Get the last velocity of the first touch point (within one gesture)
      *
      *  @return Velocity of the last touch point 1
      */
      Vector2D  getLastVelocity1()
      {
         if (_touchData._LastTouchPointerList.size() > 0)
         {
            return _touchData._LastTouchPointerList[0]._velocity;
         }
         return Vector2D(0, 0); //send invalid velocity
      }

      /**
      *  @brief Get the last velocity of the second touch point (within one gesture)
      *
      *  @return Velocity of the last touch point 2
      */
      Vector2D  getLastVelocity2()
      {
         if (_touchData._LastTouchPointerList.size() > 1)
         {
            return _touchData._LastTouchPointerList[1]._velocity;
         }
         return Vector2D(0, 0); //send invalid velocity
      }

      /**
      *  @brief Returns the validity of the first Touchpoint (within one gesture)
      *
      *  @return true if Touchpoint is valid else false
      */
      bool getPosition1Valid()
      {
         return _touchData._isTouchPos1Valid;
      }

      /**
      *  @brief Returns the validity of the second Touchpoint (within one gesture)
      *
      *  @return true if Touchpoint is valid else false
      */
      bool getPosition2Valid()
      {
         return _touchData._isTouchPos2Valid;
      }

      /**
      *  @brief Cancels all further gesture registrations
      *
      *  This function could be called by widgets to prevent other widgets
      *  to register for any gesture.
      *
      *  Gestures registration is disabled only for current gesture, flag
      *  is reset to true after the touch release.
      */
      void cancelFurtherRegistrations()
      {
         _isRegistrationEnabled = false;
      }

      /**
      *  @brief Check if gestures registration is enabled in the framework
      *
      *  @return True if gesture registration is enabled, false otherwise
      */
      bool getRegistrationEnabled()
      {
         return _isRegistrationEnabled;
      }

      bool onTouchEvent(const int x, const int y, const unsigned int state, const unsigned int pointerId, unsigned int timestamp = 0);
      void onTouchAbort();

      COURIER_MSG_MAP_BEGIN(0)
      ON_COURIER_MESSAGE(hmibase::input::gesture::GestureTimerExpiredMsg)
      COURIER_MSG_MAP_END()

      virtual void printInfo();

#ifndef GESTURE_SIMULATION
      /**
      *  @brief timer callback
      */
      virtual void timerExpired(hmibase::util::TimerThreaded::TimerId timerId, uint32_t timIndex);
#endif

   protected:
#ifndef GESTURE_SIMULATION
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
      virtual bool onCourierMessage(const hmibase::input::gesture::GestureTimerExpiredMsg& msg);
#endif
#endif

   private:
      /* each Touch point will be represented With its Coordinates and Velocity Direction */
      struct TouchPoint
      {
         TouchPoint() :
            _coordinates(INT_MIN, INT_MIN),
            _velocity(0, 0),
            _timestamp(0),
            _touchState(0)
         {
         }
         Vector2D _coordinates;
         Vector2D _velocity;
         uint32_t _timestamp;
         uint8_t  _touchState;
      };

      struct TouchSequenceData
      {
         TouchSequenceData() :
            _u32Timestamp(0),
            _isTouchPos1Valid(false),
            _isTouchPos2Valid(false)
         {
            TouchPoint temp;

            for (int i = 0; i <= MAX_POINTER_ID; i++)//Create a Vector With n Supported number of touch points mostly restricted to 2 Touch points
            {
               _FirstTouchPointerList.push_back(temp); //initialize Vector With Default Values
               _LastTouchPointerList.push_back(temp);  //initialize Vector With Default Values
            }
         }

         std::vector<TouchPoint>   _FirstTouchPointerList;  // To Store the Start point of  Current Gesture, use pointerId as vector index
         std::vector<TouchPoint>   _LastTouchPointerList;   // to Store Current Touchpoints of Current Gesture, use pointerId as vector index
         uint32_t _u32Timestamp;  //< Timestamp of the touch event
         bool _isTouchPos1Valid;  //< Flag indicating if the touch point 1 is valid (pressed)
         bool _isTouchPos2Valid;  //< Flag indicating if the touch point 2 is valid (pressed)
         //uint32_t _velocityCalcTimestamp;  //< Timestamp of the touch event
      };

      enum touchStates
      {
         /** The pointer is currently moved. Before the pointer has been moved it had the state State::Down. */
         TOUCHEVENT_MOTION,
         /** The pointer is currently put down. After this state either several State::Move states or the state State::Up are possible. */
         TOUCHEVENT_DOWN,
         /** The pointer is currently lifted. Before this the pointer had the State::Down or State::Moving state. */
         TOUCHEVENT_UP,
         /** The pointer is currently lifted. Indicates the end of a contact point list. */
         TOUCHEVENT_FRAME, //Current not send From HMI base
         /** Sent if the compositor decides the touch stream is a global gesture. No further events are sent to the clients from that particular gesture */
         TOUCHEVENT_CANCEL //Current not send From HMI base
      };

      struct RawTouchSequenceData
      {
         std::vector<Vector2D>  _currentlyTouchedPositions;  //< to Store all the Touchpoints, use pointerId as vector index
         uint32_t _u32Timestamp;  //< Timestamp of the touch event
         uint32_t  _touchState;    //< Flag is true in case of the proximity

         RawTouchSequenceData() : _u32Timestamp(0), _touchState(0)
         {
            for (unsigned int i = 0; i <= MAX_POINTER_ID; ++i)
            {
               _currentlyTouchedPositions.push_back(Vector2D(INT_MIN, INT_MIN));
            }
         }
      };

      GestureHandler();

      virtual ~GestureHandler();

      /**
      *  @brief The copy constructor shall never be called.
      *
      *  Therefore, it is defined private.
      *
      *  @param[in]  rhs  The object to copy
      */
      GestureHandler(const GestureHandler& rhs);

      /**
      *  @brief The assignment operator shall never be called.
      *
      *  Therefore, it is defined private.
      *
      *  @param[in]  rhs  The object to copy
      */
      GestureHandler& operator=(const GestureHandler& rhs);

      /**
      *  @brief Adapter To Convert Client touch Points To Library understandable Touchpoints formate
      *
      *  @return
      */
      void GetTouchRecord(TouchSequenceData&);

      /**
      *  @brief calculates Touch Velocity.
      *
      *  @return Vector2D
      */
      Vector2D calculateTouchVelocity(int32_t timediff, int32_t xDisplacement, int32_t yDisplacement, int32_t touchpoint);

      static GestureHandler* _gestureHandlerInstance;         //< pointer to the one and only instance of this class
      GestureDetector* _firstDetector;            //< pointer to the first element of the list of detectors
      bool _isRegistrationEnabled;                 //< Flag indicating if gestures registration is enabled
      std::map<GestureEvent::GestureType, char>  _gesturePriorityMap;

      bool _isTouchPressed;                        //< true if touch is currently pressed
      bool _isTouch2Pressed;                       //< true if touch point 2 is currently pressed

      TouchSequenceData _touchData;     //< used To store the Filtered Touch Data

      // touch position preparation
      RawTouchSequenceData _rawTouchSequenceData;  // object with touch data to be passed to GestureHandler
};


}
}


}

#endif // #ifndef _GestureHandler_H
