/* ***************************************************************************************
* FILE:          GestureHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GestureHandler is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "GestureHandler.h"
#include "GestureDetector.h"
#ifdef GESTURE_SIMULATION
#include "TimerManager.h"
#else
#include "hmibase/util/Ticker.h"
#endif

// including all the Gestures handlers for Priority handling Centralized in Gesture handler
#include "TapGestureDetector.h"
#include "SwipeGestureDetector.h"
#include "SpreadGestureDetector.h"
#include "DragGestureDetector.h"
#include "RotateGestureDetector.h"
#include "LegacyTouchDetector.h"

#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GestureHandler.cpp.trc.h"
#endif

//#include <stdio.h>
namespace hmibase {
namespace input {
namespace gesture {

//for Touch Velocity Calculation
const int VELOCITY_MIN = -32768;
const int VELOCITY_MAX = 32768;
//const int VELOCITY_SAMPLING_RATE_MS = 9;//should be Set to 9ms need to check on Target
// lag filter coefficients for calculating velocity
const int CA = 67;
const int CB = 189;

GestureHandler* GestureHandler::_gestureHandlerInstance = 0;

GestureHandler::GestureHandler() :
   _firstDetector(0),
   _isRegistrationEnabled(true),
   _isTouchPressed(false),
   _isTouch2Pressed(false),
   _touchData(),
   _rawTouchSequenceData()
{
   setGesturePriorityToDefaults();//set the Gesture Priority To Defualt Values
}


GestureHandler::~GestureHandler()
{
   _firstDetector = 0;
}


GestureHandler& GestureHandler::s_getInstance()
{
   if (_gestureHandlerInstance == 0)
   {
      //init instance
      _gestureHandlerInstance = new GestureHandler();
#ifdef GESTURE_SIMULATION
      TimerManager::initializeTimerManager();
#endif
      GESTURE_ASSERT(_gestureHandlerInstance != 0);
   }
   return *_gestureHandlerInstance;
}


Vector2D  GestureHandler::calculateTouchVelocity(int32_t timediff, int32_t xDisplacement, int32_t yDisplacement, int32_t touchpoint)
{
   Vector2D velocity;
   if (timediff > 0)
   {
      if (_touchData._LastTouchPointerList[static_cast<unsigned int>(touchpoint)]._velocity.x == 0)
      {
         velocity.x = ((1000 * xDisplacement) / timediff);
      }
      else
      {
         velocity.x = (((CA * ((1000 * xDisplacement) / timediff)) + (CB * _touchData._LastTouchPointerList[static_cast<unsigned int>(touchpoint)]._velocity.x))) >> 8;
      }
      if (_touchData._LastTouchPointerList[static_cast<unsigned int>(touchpoint)]._velocity.y == 0)
      {
         velocity.y = ((1000 * yDisplacement) / timediff);
      }
      else
      {
         velocity.y = (((CA * ((1000 * yDisplacement) / timediff)) + (CB * _touchData._LastTouchPointerList[static_cast<unsigned int>(touchpoint)]._velocity.y))) >> 8;
      }
   }

   //ETG_TRACE_USR4_THR(("CalculateTouchVelocity timediff=%d, point=%d, displacement=[%d,%d] , velocity=[%d,%d]", timediff, touchpoint, xDisplacement, yDisplacement, velocity.x, velocity.y));

   velocity.x = (velocity.x < VELOCITY_MIN) ? VELOCITY_MIN : ((velocity.x > VELOCITY_MAX) ? VELOCITY_MAX : velocity.x);
   velocity.y = (velocity.y < VELOCITY_MIN) ? VELOCITY_MIN : ((velocity.y > VELOCITY_MAX) ? VELOCITY_MAX : velocity.y);

   return velocity;
}


void GestureHandler::GetTouchRecord(TouchSequenceData& touchRecord)
{
   // calculate velocities
   // declare helper variables
   //set the Validity Flags
   touchRecord._isTouchPos1Valid = (_rawTouchSequenceData._touchState == static_cast<int>(TOUCHEVENT_DOWN)
                                    || _rawTouchSequenceData._touchState == static_cast<int>(TOUCHEVENT_MOTION))
                                   && (_rawTouchSequenceData._currentlyTouchedPositions.size() > 0)
                                   && (_rawTouchSequenceData._currentlyTouchedPositions[0].x > INT_MIN || _rawTouchSequenceData._currentlyTouchedPositions[0].y > INT_MIN);

   touchRecord._isTouchPos2Valid = (_rawTouchSequenceData._touchState == static_cast<int>(TOUCHEVENT_DOWN)
                                    || _rawTouchSequenceData._touchState == static_cast<int>(TOUCHEVENT_MOTION))
                                   && (_rawTouchSequenceData._currentlyTouchedPositions.size() > 1)
                                   && (_rawTouchSequenceData._currentlyTouchedPositions[1].x > INT_MIN || _rawTouchSequenceData._currentlyTouchedPositions[1].y > INT_MIN);

   int32_t timediff = (int32_t)(_rawTouchSequenceData._u32Timestamp - _touchData._u32Timestamp);

   for (uint8_t i = 0; i < _rawTouchSequenceData._currentlyTouchedPositions.size(); i++)
   {
      if (_rawTouchSequenceData._currentlyTouchedPositions[i] == Vector2D(INT_MIN, INT_MIN))
      {
         continue;
      }

      int32_t deltaX = (int32_t)_rawTouchSequenceData._currentlyTouchedPositions[i].x - (int32_t)_touchData._LastTouchPointerList[i]._coordinates.x;
      int32_t deltaY = (int32_t)_rawTouchSequenceData._currentlyTouchedPositions[i].y - (int32_t)_touchData._LastTouchPointerList[i]._coordinates.y;
      bool isTouchPosValid = (_rawTouchSequenceData._touchState == static_cast<int>(TOUCHEVENT_DOWN)
                              || _rawTouchSequenceData._touchState == static_cast<int>(TOUCHEVENT_MOTION))
                             && (_rawTouchSequenceData._currentlyTouchedPositions[i].x > INT_MIN || _rawTouchSequenceData._currentlyTouchedPositions[i].y > INT_MIN);
      if (isTouchPosValid)
      {
         touchRecord._LastTouchPointerList[i]._velocity = calculateTouchVelocity(timediff, deltaX, deltaY, i);
      }
      else
      {
         touchRecord._LastTouchPointerList[i]._velocity = Vector2D(0, 0);
      }

      //copy Touch points
      touchRecord._LastTouchPointerList[i]._coordinates = _rawTouchSequenceData._currentlyTouchedPositions[i];
   }
#ifdef GESTURE_SIMULATION
   touchRecord._u32Timestamp = TimerManager::s_getInstance().clockGetElapsedTime();       //lint !e1705
#else
   touchRecord._u32Timestamp = static_cast<uint32_t>(hmibase::util::Ticker::getTickCountMsec());
#endif
}


bool GestureHandler::onTouchEvent(const int x, const int y, const unsigned int state, const unsigned int pointerId, unsigned int timestamp)
{
   if (pointerId > MAX_POINTER_ID)
   {
      ETG_TRACE_FATAL_THR(("Can handle up to two pointers, but pointerId %d received", pointerId));
      return false;
   }

   _rawTouchSequenceData._touchState = state;
   _rawTouchSequenceData._u32Timestamp = timestamp;

   // add new touch point to list of currently active touch points
   if (state == 1) //TouchMsgState::Down
   {
      if (_rawTouchSequenceData._currentlyTouchedPositions[pointerId] != Vector2D(INT_MIN, INT_MIN))
      {
         //somehow a second touch is received on the same pointer
         ETG_TRACE_ERR_THR(("Press on pointerId %d before previous session ended", pointerId));
      }
      else
      {
         _rawTouchSequenceData._currentlyTouchedPositions[pointerId] = Vector2D(static_cast<TouchCoord>(x), static_cast<TouchCoord>(y));
      }
   }
   else if (state == 0) //TouchMsgState::Move
   {
      _rawTouchSequenceData._currentlyTouchedPositions[pointerId] = Vector2D(static_cast<TouchCoord>(x), static_cast<TouchCoord>(y));
   }
   else if (state == 2) //TouchMsgState::Up
   {
      _rawTouchSequenceData._currentlyTouchedPositions[pointerId] = Vector2D(INT_MIN, INT_MIN);
   }

   return processTouchData();
}


void GestureHandler::onTouchAbort()
{
   _isTouchPressed = false;
   _isTouch2Pressed = false;

   removeAllGestures();

   _touchData = TouchSequenceData(); //clear the old Touch Session Data
   _rawTouchSequenceData = RawTouchSequenceData();
}


bool GestureHandler::processTouchData()
{
#ifdef GESTURE_SIMULATION
   _rawTouchSequenceData._u32Timestamp = TimerManager::s_getInstance().clockGetElapsedTime();              //lint !e1705  if Client Doesn't Send the Time Stamp take the Current Timestamp
#else
   _rawTouchSequenceData._u32Timestamp = static_cast<uint32_t>(hmibase::util::Ticker::getTickCountMsec());
#endif

   TouchSequenceData touchRecord;
   GetTouchRecord(touchRecord);

   _touchData._isTouchPos1Valid = touchRecord._isTouchPos1Valid;
   _touchData._isTouchPos2Valid = touchRecord._isTouchPos2Valid;
   _touchData._u32Timestamp = touchRecord._u32Timestamp;

   bool isPoint1Press = touchRecord._isTouchPos1Valid || touchRecord._isTouchPos2Valid;

   // Store the last valid touch point and velocity
   if (touchRecord._isTouchPos2Valid)
   {
      _touchData._LastTouchPointerList[1]._coordinates = touchRecord._LastTouchPointerList[1]._coordinates; //Storing the Current Touch Points
      if (!_isTouch2Pressed)
      {
         _touchData._FirstTouchPointerList[1]._coordinates = touchRecord._LastTouchPointerList[1]._coordinates; //Storing the Start Point
         _isTouch2Pressed = true;
      }
   }
   else
   {
      _isTouch2Pressed = false;
   }

   if (touchRecord._isTouchPos1Valid)
   {
      _touchData._LastTouchPointerList[0]._velocity = touchRecord._LastTouchPointerList[0]._velocity; //Store Current  Velocity of Point 1
   }
   if (touchRecord._isTouchPos2Valid)
   {
      _touchData._LastTouchPointerList[1]._velocity = touchRecord._LastTouchPointerList[1]._velocity; //Store Current  Velocity of Point 2
   }

   if (isPoint1Press)
   {
      if (!_isTouchPressed)
      {
         // look if someone (some widget usually) wants to register to this touch event
         _touchData._FirstTouchPointerList[0]._coordinates = touchRecord._LastTouchPointerList[0]._coordinates; //Storing the Start Point
         _isTouchPressed = true;
      }

      _touchData._LastTouchPointerList[0]._coordinates = touchRecord._LastTouchPointerList[0]._coordinates; //Storing the  Current Point 1
   }
   else
   {
      // Enable the registration on touch release
      _isRegistrationEnabled = true;
      // finish this press
      _isTouchPressed = false;
   }

   for (int i = 0; i <= MAX_POINTER_ID; i++)
   {
      ETG_TRACE_USR1_THR(("\t\tTouchRecord: pointer %d, First: coordinate (%d, %d), velocity (%d, %d)",
                          i,
                          touchRecord._FirstTouchPointerList[i]._coordinates.x,
                          touchRecord._FirstTouchPointerList[i]._coordinates.y,
                          touchRecord._FirstTouchPointerList[i]._velocity.x,
                          touchRecord._FirstTouchPointerList[i]._velocity.y));

      ETG_TRACE_USR1_THR(("\t\tTouchRecord: pointer %d, Last: coordinate (%d, %d), velocity (%d, %d)",
                          i,
                          touchRecord._LastTouchPointerList[i]._coordinates.x,
                          touchRecord._LastTouchPointerList[i]._coordinates.y,
                          touchRecord._LastTouchPointerList[i]._velocity.x,
                          touchRecord._LastTouchPointerList[i]._velocity.y));
   }

   // Set the timestamp for all detectors
   GestureDetector* detector = _firstDetector;
   while (detector != 0)
   {
      detector->setTimestamp(static_cast<int>(touchRecord._u32Timestamp));
      detector = detector->getNextDetector();
   }

   // Detect gesture
   detector = _firstDetector;
   bool detected = false;
   bool processed = false;
   while (detector != 0)
   {
      GestureDetector* pNextDetector = detector->getNextDetector();
      detected = detector->detect();
      if (detected)
      {
         if (detector->getParallelGestureRecognitionEnabled() && detector->getSimultaneouslyRecognizableGestureType() != GestureEvent::GT_None)
         {
            removeNonParallelGestureDetectors(detector, detector->getSimultaneouslyRecognizableGestureType());
            pNextDetector = detector->getNextDetector(); //pNextDetector might be invalid Pointer as Gesture handler Might be Deleted so copy next Gesture From the List of Simultanious Gesture Detectors
         }
         else
         {
            // Cancel previous gestures
            removeFurtherGestureDetectors(detector);
            pNextDetector = 0;
         }
      }
      if ((detector != 0) && (false == detector->isAlive()))
      {
         removeDetector(detector);
      }

      detector = pNextDetector;
      processed = true;
   }

   if (!isPoint1Press)
   {
      // Remove all in case of press end
      while (_firstDetector != 0)
      {
         detector = _firstDetector;
         _firstDetector = detector->getNextDetector();
         detector->cancel();
         delete detector;
      }

      TouchSequenceData currentFilterdTouchData;
      _touchData = currentFilterdTouchData; //clear the old Touch Session Data
   }

   return processed;
}


bool GestureHandler::insertDetector(GestureDetector* newDetector)
{
   bool result = false;

   if (newDetector != 0)
   {
      // set default to 'true' to add gesture when reaching the end of the list
      result = true;

      // go through priority-sorted list of gestures
      int newPriority = newDetector->getPriority();
      GestureDetector** ppDetector = &_firstDetector;
      GestureDetector* pDetector = *ppDetector;
      while (pDetector != 0)
      {
         int  priority = pDetector->getPriority();
         if (newPriority > priority)
         {
            // add the new gesture before the current one because of priorities
            break;
         }
         if (priority == newPriority)
         {
            if (!newDetector->canBeCombined(pDetector))
            {
               // do not add the gesture because it cannot be combined with an existing one
               result = false;
               break;
            }
         }

         // go to next gesture
         ppDetector = &pDetector->getNextDetector();
         pDetector = *ppDetector;
      }

      if (result)
      {
         // insert into sorted list
         newDetector->getNextDetector() = *ppDetector;
         *ppDetector = newDetector;
      }
   }

   return result;
}


void GestureHandler::removeAllGestures()
{
   GestureDetector* detector = _firstDetector;

   while (_firstDetector != 0)
   {
      ETG_TRACE_USR4_THR(("GestureHandler::removeAllGesture delete Detector"));
      detector = _firstDetector;
      _firstDetector = detector->getNextDetector();
      detector->cancel();
      delete detector;
   }
}


void GestureHandler::removeDetectors(const IGestureListener* handler)
{
   if (handler != 0)
   {
      //get first detector
      GestureDetector** ppDetector = &_firstDetector;
      //cycle as long as detectors are available
      while (*ppDetector != 0)
      {
         //check if detector is assigned to given widget
         if ((*ppDetector)->getReceiver() == handler)
         {
            GestureDetector* detector = *ppDetector;
            //get next detector
            *ppDetector = detector->getNextDetector();
            //cancel detector
            detector->cancel();
            delete detector;
         }
         else
         {
            //skip to next detector if detector is not assigned to given widget
            ppDetector = &(*ppDetector)->getNextDetector();
         }
      }
   }
}


void GestureHandler::removeFurtherGestureDetectors(GestureDetector* detector)
{
   //write trace
   ETG_TRACE_USR4_THR(("GestureHandler.removeFurtherGestureDetectors"));

   if (detector != 0)
   {
      //get detector
      GestureDetector* pSurvivor = detector;
      detector = pSurvivor->getNextDetector();
      pSurvivor->getNextDetector() = 0;

      //remove all lower prio detectors
      while (detector != 0)
      {
         GestureDetector* pNextDetector = detector->getNextDetector();

         //cancel detector
         detector->cancel();
         delete detector;
         detector = pNextDetector;
      }

      //remove higher prio detectors for the same gesture type if parallel recognition is not enabled
      GestureDetector* pPrevious = 0;
      detector = _firstDetector;
      while ((detector != pSurvivor) && (detector != 0))
      {
         GestureDetector* pNextDetector = detector->getNextDetector();

         if ((detector->getGesture()._gestureType == pSurvivor->getGesture()._gestureType)
               && (!detector->getParallelGestureRecognitionEnabled())
            )
         {
            //we are removing one detector which is not the first one so we have to relink the previous detector to the next detector
            if (pPrevious != 0)
            {
               pPrevious->getNextDetector() = detector->getNextDetector();
            }
            //we are removing the first detector, so just reassign the first detector to the next one
            else
            {
               _firstDetector = _firstDetector->getNextDetector();
            }

            //cancel detector
            ETG_TRACE_USR4_THR(("cancel detector for the same gesture type %d as the active one", detector->getGesture()._gestureType));
            detector->getNextDetector() = 0;
            detector->cancel();
            delete detector;
         }
         else
         {
            pPrevious = detector;
         }

         detector = pNextDetector;
      }
   }
}


void GestureHandler::removeNonParallelGestureDetectors(GestureDetector* detector, GestureEvent::GestureType gestureType) const
{
   //write trace
   ETG_TRACE_USR4_THR(("GestureHandler.removeNonParallelGestureDetectors"));

   if (detector != 0)
   {
      //get detector
      GestureDetector* pSurvivor = detector;
      detector = pSurvivor->getNextDetector();
      GestureDetector* pPrevious = pSurvivor;
      while (detector != 0)
      {
         GestureDetector* pNextDetector = detector->getNextDetector();

         if (detector->getGesture()._gestureType != gestureType)
         {
            //cancel detector
            detector->cancel();
            delete detector;
            pPrevious->getNextDetector() = pNextDetector;
         }
         else
         {
            pPrevious = detector;
         }
         detector = pNextDetector;
      }
   }
}


void GestureHandler::removeDetector(GestureDetector* detector)
{
   if (detector != 0)
   {
      //get first detector
      GestureDetector** ppDetector = &_firstDetector;
      //look for given detector
      while ((*ppDetector != 0) && (*ppDetector != detector))
      {
         ppDetector = &(*ppDetector)->getNextDetector();
      }

      if (*ppDetector != 0)
      {
         *ppDetector = detector->getNextDetector();
         //cancel detector
         detector->cancel();
         delete detector;
      }
   }
}


void GestureHandler::setGesturePriorityToDefaults()
{
   _gesturePriorityMap[GestureEvent::GT_LegacyTouch] = static_cast<char>(GestureDetector::PRIORITY_LEGACY_TOUCH);
   _gesturePriorityMap[GestureEvent::GT_PressHoldRepeatTap] = static_cast<char>(GestureDetector::PRIORITY_TAP);
   _gesturePriorityMap[GestureEvent::GT_DragNudge] = static_cast<char>(GestureDetector::PRIORITY_DRAG);
   _gesturePriorityMap[GestureEvent::GT_SwipeFling] = static_cast<char>(GestureDetector::PRIORITY_SWIPE);
   _gesturePriorityMap[GestureEvent::GT_PinchSpread] = static_cast<char>(GestureDetector::PRIORITY_SPREAD);
   _gesturePriorityMap[GestureEvent::GT_Rotate] = static_cast<char>(GestureDetector::PRIORITY_ROTATE);
   _gesturePriorityMap[GestureEvent::GT_RawTouchData] = static_cast<char>(GestureDetector::PRIORITY_RAW_DATA);
}


bool GestureHandler::configureGesturePriority(std::map<GestureEvent::GestureType, char> mNewGesturePriority)
{
   bool configurationsucessfull = false;
   int key_to_be_found = static_cast<int>(GestureEvent::GT_PressHoldRepeatTap);
   int count = 0;
   for (std::map<GestureEvent::GestureType, char>::iterator iterator = mNewGesturePriority.begin(); iterator != mNewGesturePriority.end(); ++iterator)
   {
      if (iterator->first == static_cast<GestureEvent::GestureType>(key_to_be_found))
      {
         count = count + 1;
         ++key_to_be_found;
      }
   }

   if (mNewGesturePriority.size() > 0 && count == 4)
   {
      _gesturePriorityMap = mNewGesturePriority;
      configurationsucessfull = true;
   }
   else
   {
      //debug statement
      // the basic gesture priority is missing. hence configured priority is ignored and default priority is considered.
      setGesturePriorityToDefaults();
   }
   return configurationsucessfull;
}


char GestureHandler::getGesturePriority(GestureEvent::GestureType GestureType)
{
   if (_gesturePriorityMap.count(GestureType) > 0)
   {
      return _gesturePriorityMap[GestureType]; // need to Change //create iterators [if a wrong parameter passed] we should not create a new Map element
   }
   return GestureDetector::PRIORITY_INVALID;
}


#ifndef GESTURE_SIMULATION
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING

bool GestureHandler::onCourierMessage(const hmibase::input::gesture::GestureTimerExpiredMsg& msg)
{
   bool consumed = false;
   //get first detector
   GestureDetector** ppDetector = &_firstDetector;
   //cycle as long as detectors are available
   while (*ppDetector != 0)
   {
      //check if detector is assigned to given widget
      {
         GestureDetector* detector = *ppDetector;
         consumed = detector->OnGestureTimerExpired(msg.GetGestureDetectorId(), msg.GetTimerId());
         //skip to next detector if detector is not assigned to given widget
         ppDetector = &(*ppDetector)->getNextDetector();
      }
   }
   return consumed;
}


#endif
#endif


void GestureHandler::printInfo()
{
   GestureDetector** ppDetector = &_firstDetector;
   //cycle as long as detectors are available
   while (*ppDetector != 0)
   {
      GestureDetector* detector = *ppDetector;
      GestureEvent gesture = detector->getGesture();
      ETG_TRACE_FATAL_THR(("[%25s] active Gesture detector type %d", hmibase::trace::getAppName().c_str(), gesture._gestureType));
      ppDetector = &(*ppDetector)->getNextDetector();
   }
}


#ifndef GESTURE_SIMULATION
void GestureHandler::timerExpired(hmibase::util::TimerThreaded::TimerId timerId, uint32_t timIndex)
{
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
   Courier::Message* msg = COURIER_MESSAGE_NEW(GestureTimerExpiredMsg)(timerId, timIndex);
   if (msg)
   {
      msg->Post();
   }
#else
   PARAM_UNUSED(timerId);
   PARAM_UNUSED(timIndex);
#endif
}


#endif
}


}
}
