/* ***************************************************************************************
* FILE:          GestureDetector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "sys_std_if.h"
#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GestureDetector.cpp.trc.h"
#endif

#include "GestureDetector.h"
#ifdef GESTURE_SIMULATION
#include "TimerManager.h"
#endif
#include <string.h>

namespace hmibase {
namespace input {
namespace gesture {

GestureDetector::GestureDetector(IGestureListener* receiver, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType) :
   _nextGestureDetector(0),
   _receiver(receiver),
   _currentGesture(),
   _isAlive(true),
   _isParallelGestureRecognitionEnabled(parallelGestureRecognitionEnabled),
   _parallelRecognizableGestureType(parallelRecognizableGestureType)
{
#ifdef GESTURE_SIMULATION
#else
   _type = 0;
#endif
}


void GestureDetector::processTimerEvent(int /*userId*/)
{
   //never Called
}


GestureDetector::~GestureDetector()
{
#ifdef GESTURE_SIMULATION
   //stop all timers for this detector
   TimerManager::s_getInstance().stopTimer(this, 0, true);
#else
   _timer.stop();
#endif

   //reset reciever
   _receiver = 0;
   //reset pointer to the next gesture detector
   _nextGestureDetector = 0;
}


GestureEvent::Direction GestureDetector::getDirection()
{
   //return direction
   return GestureEvent::DIR_2D;
}


bool GestureDetector::canBeCombined(GestureDetector* detector)
{
   // by default, only horizontal and vertical gesture of the same type can be combined
   GestureEvent::Direction eDir1 = getDirection();
   GestureEvent::Direction eDir2 = detector->getDirection();

   switch (eDir1)
   {
   case GestureEvent::DIR_HORIZONTAL:
      return (eDir2 == GestureEvent::DIR_2D) || (eDir2 == GestureEvent::DIR_VERTICAL);

   case GestureEvent::DIR_VERTICAL:
      return (eDir2 == GestureEvent::DIR_2D) || (eDir2 == GestureEvent::DIR_HORIZONTAL);

   case GestureEvent::DIR_2D:
      return (eDir2 == GestureEvent::DIR_2D) || (eDir2 == GestureEvent::DIR_HORIZONTAL) || (eDir2 == GestureEvent::DIR_VERTICAL);

   default:
      return false;
   }
}


int GestureDetector::compareDistance(const Vector2D& point1, const Vector2D& point2, int distance)
{
   int distanceSquared = 0;
   switch (getDirection())
   {
      case GestureEvent::DIR_HORIZONTAL:
         // Calculate the horizontal distance
         distanceSquared = (point2.x - point1.x) * (point2.x - point1.x);
         break;
      case GestureEvent::DIR_VERTICAL:
         // Caluclate the vertical distance
         distanceSquared = (point2.y - point1.y) * (point2.y - point1.y);
         break;
      case GestureEvent::DIR_2D:
      default:
         // Calculate the 2D distance
         distanceSquared = (point2.x - point1.x) * (point2.x - point1.x) + (point2.y - point1.y) * (point2.y - point1.y);
         break;
   }

   return distanceSquared - (distance * distance);
}


bool GestureDetector::distanceExceeded(const Vector2D& point1, const Vector2D& point2, int distance)
{
   bool isDistanceExceeded = false;
   // Calculate the distance between two points (2D)
   int distanceSquared = (point2.x - point1.x) * (point2.x - point1.x) + (point2.y - point1.y) * (point2.y - point1.y);

   // Check if distance exceeds the limit
   if (distanceSquared > (distance * distance))
   {
      isDistanceExceeded = true;
   }

   return isDistanceExceeded;
}


}
}


}
