/* ***************************************************************************************
* FILE:          DragGestureDetector.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DragGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef _DragGestureDetector_H
#define _DragGestureDetector_H


#include "GestureBasetypes.h"
#include "GestureDetector.h"

namespace hmibase {
namespace input {
namespace gesture {

class DragGestureDetector : public hmibase::input::gesture::GestureDetector
{
   public:
      /**
      *  @brief Initialize object.
      *
      *  Initialaze member variables with defauld values.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] gesturePriority defines order in which Touch Points should be Analysed
      *  @param[in] direction defines the direction in which the movement should
      *                       be detected: horizontal, vertical, 2D; this is relevant
      *                       for minDistance but not for holdMaxDistance
      *  @param[in] parallelGestureRecognitionEnabled if set will make 2 gestures handlers can Analyse the Touch points Together
      *  @param[in] parallelRecognizablegestureType Gesture that can be Analysed with the Current gesture paralally
      *  @param[in] holdTime time to hold before dragging may start, or 0 to
      *                       disable that feature
      *  @param[in] holdMaxDistance maximum distance to move during holdTime
      *  @param[in] minDistance minimum distance to move to detect this gesture;
      *                       if holdTime is set, then this movement has to be
      *                       done after that time
      */
      DragGestureDetector(IGestureListener* receiver, int gesturePriority, GestureEvent::Direction direction, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizablegestureType, unsigned int holdTime, int holdMaxDistance, int minDistance);

      /**
      *  @brief Un-Initialize object.
      *
      *  Frees all allocated memory.
      */
      virtual ~DragGestureDetector();

      /**
      *  @brief Get priority of the gesture
      *
      *  @return  Gesture priority (refers to GesturePriority enumeration)
      */
      virtual int getPriority();

      /**
      *  @brief Get the direction of the gesture
      *
      *  Direction is valid only for DragNudge and SwipeFling gestures
      *
      *  @return  Gesture direction (refers to Direction enumeration)
      */
      virtual GestureEvent::Direction getDirection();

      /**
      *  @brief Cancel the gesture
      *
      *  Cancelling the gesture stops all internal timers (if any)
      *  and sends the abort command to gesture receiver (if needed).
      */
      virtual void cancel();

      /**
      *  @brief Detects the gesture
      *
      *  Function process the input touchMsg and checks if all conditions
      *  needed to detect a gesture are fulfilled. If yes - function will
      *  inform registered receiver about detecting the gesture.
      *
      *
      *
      *  @return True if the gesture was detected, false otherwise
      */
      virtual bool detect();

      /**
      *  @brief Register for the gesture detection.
      *
      *  This function adds object of this class to the list of gestures
      *  managed by GestureHandler.
      *
      *  @param[in] receiver Receiver of the gesture events
      *  @param[in] gesturePriority defines order in which Touch Points should be Analysed
      *  @param[in] direction defines the direction in which the movement should
      *			      be detected: horizontal, vertical, 2D; this is relevant
      *			      for minDistance but not for holdMaxDistance
      *  @param[in] parallelGestureRecognitionEnabled if set will make 2 gestures handlers can Analyse the Touch points Together
      *  @param[in] parallelRecognizablegestureType Gesture that can be Analysed with the Current gesture paralally
      *  @param[in] holdTime time to hold before dragging may start, or 0 to
      *                 disable that feature
      *  @param[in] holdMaxDistance maximum distance to move during holdTime
      *  @param[in] minDistance minimum distance to move to detect this gesture;
      *             if holdTime is set, then this movement has to be
      *             done after that time
      *  @param[in] detected Flag sets the gesture as automatically detected when set to true
      *
      *  @return true if the registration successful, false otherwise
      */
      static bool registerGesture(IGestureListener* receiver,
                                  int  gesturePriority = static_cast<int>(PRIORITY_DRAG),
                                  GestureEvent::Direction direction = GestureEvent::DIR_2D,
                                  bool parallelGestureRecognitionEnabled = false,
                                  GestureEvent::GestureType  parallelRecognizablegestureType = GestureEvent::GT_None,
                                  unsigned int  holdTime = 0,
                                  int  holdMaxDistance = 100,
                                  int  minDistance = 60,
                                  bool detected = true
                                 );

      /**
      *  @brief  Process a timer event.
      *
      *  @param[in]  userId  Identifier for this timer
      */
      virtual void processTimerEvent(int userId);

      /**
      *  @brief  Sets the drag gesture as detected.
      */
      void setAsDetected();

   private:
      /**
      *  @brief  Detect the drag gesture start.
      *
      *
      *  @return True if the gesture was detected, false otherwise
      */
      bool detectGestureStart();

      /**
      *  @brief Enumeration for timer identifiers of this class
      */
      enum TimerIds
      {
         TIMERID_DRAG_HOLD      ///< timer id: Hold time detection
      };

      unsigned int _dragHoldTime;            //< Time to hold before dragging may start, or 0 to disable that feature
      int _dragHoldMaxDistance;     //< Maximum distance to move during m_dragHoldTime
      int _dragMinDistance;         //< Minimum distance to move to detect this gesture; if holdTime is set, then this movement has to be done after that time
      GestureEvent::Direction _swipeDirection;   //< Defines the direction in which the movement should be detected:
      int  _gesturePriority; //< used to Determine the Order of Gesture Detection

      Vector2D _previousPoint;     //< Position (coordinates) detected during previous touch event
      bool _isWithinHoldTime;      //< Flag indicating if the timer for the holdTime is running
      bool _isHoldDetected;        //< Flag indicating if the timer for the holdTime has expired
      bool _isDragDetected;        //< Flag indicating if the drag gesture was already detected
      bool _isInitialActionsExecuted; //< Flag indicating if the initial action for the gesture was executed
};


}
}


}
#endif // #ifndef _DragGestureDetector_H
