/* ***************************************************************************************
* FILE:          DragGestureDetector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DragGestureDetector is part of HMI-Base Framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "sys_std_if.h"
#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DragGestureDetector.cpp.trc.h"
#endif


#include "DragGestureDetector.h"
#include "GestureHandler.h"
#ifdef GESTURE_SIMULATION
#include "TimerManager.h"
#else
#include "AppUtils/Timer.h"
#include "BaseContract/generated/BaseGesturesMsgs.h"
#endif
#include "IGestureListener.h"

namespace hmibase {
namespace input {
namespace gesture {

DragGestureDetector::DragGestureDetector(IGestureListener* receiver, int gesturePriority, GestureEvent::Direction direction, bool parallelGestureRecognitionEnabled, GestureEvent::GestureType parallelRecognizableGestureType, unsigned int holdTime, int holdMaxDistance, int minDistance) :
   GestureDetector(receiver, parallelGestureRecognitionEnabled, parallelRecognizableGestureType),
   _dragHoldTime(holdTime),
   _dragHoldMaxDistance(holdMaxDistance),
   _dragMinDistance(minDistance),
   _swipeDirection(direction),
   _gesturePriority(gesturePriority),
   _previousPoint(-1, -1),
   _isWithinHoldTime(false),
   _isHoldDetected(holdTime == 0),
   _isDragDetected(false),
   _isInitialActionsExecuted(false)
{
   //set gesture
   setGestureType(GestureEvent::GT_DragNudge);
   setDirection(direction);

   setGestureDetectorType(GestureEvent::GT_DragNudge);
}


DragGestureDetector::~DragGestureDetector()
{
   //empty
}


bool DragGestureDetector::registerGesture(IGestureListener* receiver,
      int  gesturePriority,
      GestureEvent::Direction direction,
      bool parallelGestureRecognitionEnabled,
      GestureEvent::GestureType  parallelRecognizableGestureType,
      unsigned int  holdTime,
      int  holdMaxDistance,
      int  minDistance,
      bool detected)
{
   GESTURE_ASSERT(receiver != 0);
   bool result = false;

   if (GestureHandler::s_getInstance().getRegistrationEnabled())
   {
      // Create the gesture detector and add it to the list of detectors handled by the GestureHandler
      GestureDetector* detector = new DragGestureDetector(receiver, gesturePriority, direction, parallelGestureRecognitionEnabled, parallelRecognizableGestureType, holdTime, holdMaxDistance, minDistance);
      GESTURE_ASSERT(detector != 0);
      result = GestureHandler::s_getInstance().insertDetector(detector);
      if (detected)
      {
         // In special case, set the gesture as detected immediately
         //  ((DragGestureDetector*)detector)->setAsDetected(); //lint !e613 GMNGGUILINT-613-1 asserted some lines before
      }

      if (!result)
      {
         delete detector;
      }
   }

   return result;
}


void DragGestureDetector::setAsDetected()
{
   //set detected
   _isWithinHoldTime = false;
   _isInitialActionsExecuted = true;
   _isHoldDetected = true;
   _isDragDetected = true;
}


void DragGestureDetector::cancel()
{
   // Just send cancel to registered component
   if ((getReceiver() != 0) && _isDragDetected)
   {
      setEventType(GestureEvent::ET_ABORT);
      setPoint1Valid(false);
      ETG_TRACE_USR4_THR(("DragGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                          ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                          ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
      getReceiver()->processGestureEvent(getGesture());
   }
}


bool DragGestureDetector::detectGestureStart()
{
   bool result = false;
   IGestureListener* receiver = getReceiver();
   if (receiver != 0)
   {
      // Check if current point reached the minimum distance to detect the drag
      int distance = compareDistance(GestureHandler::s_getInstance().getStartPoint1(), GestureHandler::s_getInstance().getLastTouchPoint1(), _dragMinDistance);
      if (distance >= 0)
      {
         // Send the gesture START event, and make sure start is send only once by setting _isDragDetected flag
         setEventType(GestureEvent::ET_START);
         ETG_TRACE_USR4_THR(("DragGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                             ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                             ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
         receiver->processGestureEvent(getGesture());
         _isDragDetected = true;
         result = true;
      }
   }

   return result;
}


bool DragGestureDetector::detect()
{
   bool isPoint1Press = GestureHandler::s_getInstance().getPosition1Valid(); // touchMsg._isTouchPos1Valid;
   bool isPoint2Press = GestureHandler::s_getInstance().getPosition2Valid(); //touchMsg._isTouchPos2Valid;
   bool detected = false;
   bool isDistanceExceeded = false;

   IGestureListener* receiver = getReceiver();

   if (receiver != 0)
   {
      // Set the default touch position and press information
      setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());
      setPoint1Valid(true);

      setPoint1Velocity(GestureHandler::s_getInstance().getLastVelocity1());
      setPoint1VelocityValid(true);

      if (isPoint1Press && !isPoint2Press)
      {
         setPoint1(GestureHandler::s_getInstance().getLastTouchPoint1());

         if (_isWithinHoldTime)
         {
            // If we are within he hold time, check if max distance is exceeded
            isDistanceExceeded = distanceExceeded(GestureHandler::s_getInstance().getStartPoint1(), GestureHandler::s_getInstance().getLastTouchPoint1(), _dragHoldMaxDistance);
         }

         if (isDistanceExceeded)
         {
            // Cancel gesture if the maximum move distance was exceeded
            setAlive(false);//GestureHandler::s_getInstance().removeDetector(this);
            return false;
         }
         else
         {
            // As an initial acation - start hold timer (previously running timers with same ID will automatically be stopped)
            if (!_isInitialActionsExecuted)
            {
               _previousPoint = GestureHandler::s_getInstance().getStartPoint1();

               if (_dragHoldTime > 0)
               {
#ifdef GESTURE_SIMULATION
                  TimerManager::s_getInstance().startTimer(this, static_cast<int>(TIMERID_DRAG_HOLD), _dragHoldTime);
#else
                  _timer.setName("DragDropDetector", "");
                  _timer.setTimerId(GestureEvent::GT_DragNudge);
                  _timer.setTimeout(TIMERID_DRAG_HOLD, _dragHoldTime, &(GestureHandler::s_getInstance()));
                  _timer.start();
#endif
                  _isWithinHoldTime = true;
               }

               _isInitialActionsExecuted = true;
            }

            if (_isHoldDetected)
            {
               setEventType(GestureEvent::ET_MOVE);
               bool isMoved = false;
               // When the hold time was reached, now start the drag detection
               if (!_isDragDetected)
               {
                  detected = detectGestureStart();
               }
               else
               {
                  // At this point drag was already detected, so check if there is a move
                  setEventType(GestureEvent::ET_MOVE);
                  switch (_swipeDirection)
                  {
                     case GestureEvent::DIR_HORIZONTAL:
                     {
                        // User registered for horizontal move, so check if current x coordinate differs from previous
                        setDirection(GestureEvent::DIR_HORIZONTAL);
                        if (GestureHandler::s_getInstance().getLastTouchPoint1().x != _previousPoint.x)
                        {
                           isMoved = true;
                           ETG_TRACE_USR4_THR(("DragGestureDetector.detect:  sending horizontal move"));
                        }
                     }
                     break;

                     case GestureEvent::DIR_VERTICAL:
                     {
                        // User registered for vertical move, so check if current y coordinate differs from previous
                        setDirection(GestureEvent::DIR_VERTICAL);
                        if (GestureHandler::s_getInstance().getLastTouchPoint1().y != _previousPoint.y)
                        {
                           isMoved = true;
                           ETG_TRACE_USR4_THR(("DragGestureDetector.detect:  sending vertical move"));
                        }
                     }
                     break;

                     case GestureEvent::DIR_2D:
                     {
                        // User register for 2D move, so check if current x or y coordinate differs from previous
                        setDirection(GestureEvent::DIR_2D);
                        if ((GestureHandler::s_getInstance().getLastTouchPoint1().x != _previousPoint.x) || (GestureHandler::s_getInstance().getLastTouchPoint1().y != _previousPoint.y))
                        {
                           isMoved = true;
                           ETG_TRACE_USR4_THR(("DragGestureDetector.detect:  sending 2D move"));
                        }
                     }
                     break;

                     default:
                        break;
                  }

                  if (isMoved)
                  {
                     ETG_TRACE_USR4_THR(("DragGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                         ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                         ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
                     // If move was detected, send the MOVE gesture
                     receiver->processGestureEvent(getGesture());
                  }
               }
            }
         }
      }

      if (!isPoint1Press || isPoint2Press)
      {
         // If the touch was released and the drag was detected before, send the gesture END
         if (_isDragDetected)
         {
            setEventType(GestureEvent::ET_END);
            ETG_TRACE_USR4_THR(("DragGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
            receiver->processGestureEvent(getGesture());
            //  GUI_UTIL_InteractionLogger::s_logTouchInformation(GUI_EVENT_GESTURE_DRAG, receiver,
            //                                                   getGesture().point1, Vector2D (-1, -1));
            _isDragDetected = false;
         }
         // Remove current detector at release (once released gesture cannot be continued)
         setAlive(false);//GestureHandler::s_getInstance().removeDetector(this);
         return false;
      }

      _previousPoint = GestureHandler::s_getInstance().getLastTouchPoint1();
   }

   return detected;
}


int  DragGestureDetector::getPriority()
{
   //return priority
   return _gesturePriority;
}


GestureEvent::Direction DragGestureDetector::getDirection()
{
   //return direction
   return _swipeDirection;
}


void DragGestureDetector::processTimerEvent(int userId)
{
   IGestureListener* receiver = getReceiver();

   if (receiver != 0)
   {
      switch (userId)
      {
         case TIMERID_DRAG_HOLD:
         {
            // When the hold timer expires, set the gesture HOLD event and the gesture detected flag
            _isWithinHoldTime = false;
            _isHoldDetected = true;

            setEventType(GestureEvent::ET_HOLD);
            ETG_TRACE_USR4_THR(("DragGestureDetector [%25s]: gesture event, type %d, state %d", hmibase::trace::getAppName().c_str(),
                                ETG_CENUM(GestureEvent::GestureType, getGesture()._gestureType),
                                ETG_CENUM(GestureEvent::GestureState, getGesture()._gestureState)));
            // Send the gesture HOLD event
            receiver->processGestureEvent(getGesture());

            if (detectGestureStart())
            {
               if (this->getParallelGestureRecognitionEnabled())
               {
                  GestureHandler::s_getInstance().removeNonParallelGestureDetectors(this, getSimultaneouslyRecognizableGestureType());
               }
               else
               {
                  GestureHandler::s_getInstance().removeFurtherGestureDetectors(this);
               }
            }
            break;
         }
         default:
            break;
      }
   }
}


}
}


}
