/* ***************************************************************************************
* FILE:          StubHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  StubHandler.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "StubHandler.h"

#include "hmibase/util/StringUtils.h"
#include "hmibase/util/Macros.h"
#include "View/IMessageSystem.h"
#include <algorithm>

#include "hmi_trace_if.h"

// need this here until only one hierarchy level if supported in ETG_I_CENUM
using namespace hmibase::services::hmiappctrl;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_APPCTRL_STUB
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/StubHandler.cpp.trc.h"
#endif

using namespace ::asf::core;
using namespace ::bosch::cm::ai::hmi::appctrl::HMIAppCtrl;

namespace hmibase {
namespace services {
namespace hmiappctrl {

StubHandler::StubHandler()
   : HMIAppCtrlStub("hmiAppCtrlPort_o")
   , _appsInitialized(false)
{
   _timer.start(*this, 5000, asf::core::Timer::Periodic);
}


StubHandler::StubHandler(unsigned int heartBeatTime)
   : HMIAppCtrlStub("hmiAppCtrlPort_o")
   , _appsInitialized(false)
{
   _timer.start(*this, heartBeatTime > 0 ? heartBeatTime : 5000, asf::core::Timer::Periodic);
}


StubHandler::~StubHandler()
{
   _timer.stop();
}


// get pid
unsigned int StubHandler::getPid(unsigned int appId)
{
   return static_cast<unsigned int>(_taskArray.getPid(appId));
}


void StubHandler::onExpired(Timer& /*timer*/, boost::shared_ptr<TimerPayload>/*data*/)
{
   // add check for correct timer if more than one are used
   {
      sendQueryClientSignal();
   }
}


void StubHandler::onUpdateClientStatusRequest(const boost::shared_ptr<UpdateClientStatusRequest>& data)
{
   ETG_TRACE_USR1_THR(("StubHandler onClientStatusRequest"));

   bool ret = false;
   int32 pid = static_cast<int>(data->getPid());

   SYSTEMD_LOG("StubHandler::------------\n");
   SYSTEMD_LOG("StubHandler::onUpdateClientStatusRequest pid=%d, name=%s\n", pid, data->getName().c_str());

   std::string app_name = data->getName();
   int32 new_app_state = static_cast<int>(data->getState_());
   //ETG_TRACE_USR4_THR(("SERVER RECEIVED: pid: %d,  App: %20s  State: %d ",pid , app_name.c_str(), new_app_state));
   // check at first for relevant PID
   AppTaskArray::tenError errorCode = _taskArray.setPid(app_name, pid);
   if (errorCode == AppTaskArray::NO_NAME_MATCH)
   {
      if (std::find(_unknownAppNames.begin(), _unknownAppNames.end(), app_name) == _unknownAppNames.end())
      {
         // this should never happen, so make an entry in err mem
         SYSTEMD_LOG("ERROR !!! onUpdateClientStatusRequest -- unknown hmi application %s\n", app_name.c_str());
         ETG_TRACE_ERRMEM_THR(("unknown hmi application %s", app_name.c_str()));
         _unknownAppNames.push_back(app_name);
      }
   }
   else
   {
      if (errorCode == AppTaskArray::PID_UPDATED)
      {
         ETG_TRACE_USR1_THR(("###>set PID by name:%d %s", pid, app_name.c_str()));
      }
      ret = true;
   }

   if (new_app_state == 1)
   {
      updateInitFlag(app_name);
   }

   // find data set in attribute
   std::vector< ApplicationState >& clients = HMIAppCtrlStub::getClientsMutable();
   EnumAppState state  = new_app_state == 1 ? EnumAppState__APP_STATE_UP_AND_READY : EnumAppState__APP_STATE_UNKNOWN;

   std::vector< ApplicationState >::iterator clientIterator = clients.begin();
   while (clientIterator != clients.end())
   {
      if (clientIterator->getPid() == pid)
      {
         break;
      }
      ++clientIterator;
   }

   // Update attribute if required
   if (clientIterator == clients.end())
   {
      clients.push_back(ApplicationState(pid, state));
      sendClientsUpdate();
   }
   else
   {
      if (clientIterator->getAppState() != state)
      {
         ApplicationState newDataSet(pid, state);
         *clientIterator = newDataSet;
         sendClientsUpdate();
      }
   }

   if (ret)
   {
      SYSTEMD_LOG("StubHandler::send ApplicationInitialization for %s\n", data->getName().c_str());
      sendApplicationInitialization();
   }
   else
   {
      SYSTEMD_LOG("ERROR !!!! StubHandler::###no running application found for %s\n", data->getName().c_str());
      ETG_TRACE_ERR_THR(("###>no running application found"));
   }
}


void StubHandler::sendApplicationInitialization()
{
   if (!_appsInitialized && (checkInitFlag()))
   {
#ifdef not_used
      Courier::Message* msg = COURIER_MESSAGE_NEW(HMIAppsInitializedUpdMsg)(true);
      if ((msg != 0) && (msg->Post()))
      {
         SYSTEMD_LOG("StubHandler::send HMIAppsInitializedUpdMsg\n");
         ETG_TRACE_ERR_THR(("HMI Applications initialized"));
      }
      else
      {
         SYSTEMD_LOG("ERROR StubHandler::send HMIAppsInitializedUpdMsg\n");
         ETG_TRACE_ERR_THR(("Unable to post initialization message"));
      }
#else
      SYSTEMD_LOG("StubHandler::send HMIAppsInitializedUpdMsg\n");
      ETG_TRACE_ERR_THR(("HMI Applications initialized"));
      postMsg(*newMessage<HMIAppsInitializedUpdMsg>(true));
#endif
      _appsInitialized = true;
   }
}


void StubHandler::onDisplayStatusEvalRequest(const boost::shared_ptr<DisplayStatusEvalRequest>& data)
{
   ETG_TRACE_USR1_THR(("StubHandler onDisplayStatusEvalRequest"));

   int32 pid = _taskArray.getPid(data->getData().getReceiverName());
   if (pid != 0)
   {
      MultidisplayData d = data->getData();
      d.setPidReceiver(pid);
      // send event to proxies for specific use (pid)
      sendEvDisplayStatusEvalSignal(d);
   }
}


void StubHandler::onDisplayCmdRequest(const boost::shared_ptr<DisplayCmdRequest>& data)
{
   ETG_TRACE_USR1_THR(("StubHandler onDisplayCmdRequest"));

   int32 pid = _taskArray.getPid(data->getData().getReceiverName());
   if (pid != 0)
   {
      displayData d = data->getData();
      d.setPidReceiver(pid);
      // send event to proxies for specific use (pid)
      SYSTEMD_LOG("GUISTARTUP(1): sendSetDisplaySignal Master to App pid=%d\n", pid);
      sendSetDisplaySignal(d);
   }
}


void StubHandler::onMultiDisplayCmdRequest(const boost::shared_ptr<MultiDisplayCmdRequest>& data)
{
   ETG_TRACE_USR1_THR(("StubHandler onMultiDisplayCmdRequest"));

   int32 pid = _taskArray.getPid(data->getData().getReceiverName());
   if (pid != 0)
   {
      MultidisplayData d = data->getData();
      d.setPidReceiver(pid);
      // send event to proxies for specific use (pid)
      sendSetMultiDisplaySignal(d);
   }

   //sendMultiDisplayCmdResponse(data->getAct());
}


unsigned int StubHandler::getPidOfMasterApp()
{
   return _taskArray.getPidOfMasterApp();
}


void StubHandler::updateInitFlag(std::string sAppName)
{
   _taskArray.updateInitFlag(sAppName);
}


bool StubHandler::checkInitFlag()
{
   return _taskArray.traceInitializedStatus();
}


void StubHandler::onKeyForwardingRequest(const boost::shared_ptr<KeyForwardingRequest>& data)
{
#ifdef not_used
   Courier::Message* msg = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(data->getKeyRequest().getKeyCode(), (hmibase::HardKeyStateEnum)data->getKeyRequest().getKeyState(), data->getKeyRequest().getDisplayId(), (data->getKeyRequest().getUserData()));
   ETG_TRACE_USR1_THR(("[DisplayCtrlStrub] onKeyForwardingRequest KeyCode %d, KeyState %d DisplayId %d", data->getKeyRequest().getKeyCode(), ETG_CENUM(hmibase::HardKeyStateEnum, data->getKeyRequest().getKeyState()), data->getKeyRequest().getDisplayId()));
   if (msg != 0)
   {
      msg->Post();
   }
#else
   ETG_TRACE_USR1_THR(("[DisplayCtrlStrub] onKeyForwardingRequest KeyCode %d, KeyState %d DisplayId %d", data->getKeyRequest().getKeyCode(), ETG_CENUM(hmibase::HardKeyStateEnum, data->getKeyRequest().getKeyState()), data->getKeyRequest().getDisplayId()));
   postMsg(*newMessage<HKStatusChangedUpdMsg>(
              data->getKeyRequest().getKeyCode(),
              (hmibase::HardKeyStateEnum)data->getKeyRequest().getKeyState(),
              data->getKeyRequest().getDisplayId(),
              data->getKeyRequest().getUserData()
           ));
#endif
}


void StubHandler::onEncoderForwardingRequest(const boost::shared_ptr<EncoderForwardingRequest>& data)
{
#ifdef not_uswed
   Courier::Message* msg = COURIER_MESSAGE_NEW(EncoderStatusChangedUpdMsg)(data->getEncoderRequest().getEncCode(), data->getEncoderRequest().getEncValue(),  data->getEncoderRequest().getSurfaceId(),  data->getEncoderRequest().getUserData());
   ETG_TRACE_USR1_THR(("[DisplayCtrlStrub] onEncoderForwardingRequest EncCode %d, value %d, userData %d", data->getEncoderRequest().getEncCode(), data->getEncoderRequest().getEncValue() ,  data->getEncoderRequest().getUserData()));
   if (msg != 0)
   {
      msg->Post();
   }
#else
   ETG_TRACE_USR1_THR(("[DisplayCtrlStrub] onEncoderForwardingRequest EncCode %d, value %d, userData %d", data->getEncoderRequest().getEncCode(), data->getEncoderRequest().getEncValue() , data->getEncoderRequest().getUserData()));
   postMsg(*newMessage<EncoderStatusChangedUpdMsg>(data->getEncoderRequest().getEncCode(), data->getEncoderRequest().getEncValue(),  data->getEncoderRequest().getSurfaceId(), data->getEncoderRequest().getUserData()));
#endif
}


void StubHandler::onApplicationRenderedUpdateRequest(const boost::shared_ptr<ApplicationRenderedUpdateRequest>& data)
{
#ifdef not_used
   Courier::Message* msg = COURIER_MESSAGE_NEW(ApplicationRenderedUpdMsg)(data->getSurfaceid());
   if (msg != 0)
   {
      msg->Post();
   }
#else
   postMsg(*newMessage<ApplicationRenderedUpdMsg>(data->getSurfaceid()));
#endif
}


void StubHandler::onApplicationSwitchCompleteRequest(const boost::shared_ptr<ApplicationSwitchCompleteRequest>& data)
{
#ifdef not_used
   Courier::Message* msg = COURIER_MESSAGE_NEW(ApplicationSwitchCompleteUpdMsg)(data->getSurfaceid(), data->getSurfaceState());
   if (msg != 0)
   {
      msg->Post();
   }
#else
   postMsg(*newMessage<ApplicationSwitchCompleteUpdMsg>(data->getSurfaceid(), data->getSurfaceState()));
#endif
}


void StubHandler::onBeepForwardingRequest(const boost::shared_ptr<BeepForwardingRequest>& data)
{
#ifdef not_used
   Courier::Message* msg = COURIER_MESSAGE_NEW(PlayBeepReqMsg)(data->getBeepType());
   ETG_TRACE_USR1_THR(("[DisplayCtrlStrub] onBeepForwardingRequest BeepType %d", data->getBeepType()));
   if (msg != 0)
   {
      msg->Post();
   }
#else
   ETG_TRACE_USR1_THR(("[DisplayCtrlStrub] onBeepForwardingRequest BeepType %d", data->getBeepType()));
   postMsg(*newMessage<PlayBeepReqMsg>(data->getBeepType()));
#endif
}


void StubHandler::onSubSurfaceSwitchCompleteRequest(const boost::shared_ptr<SubSurfaceSwitchCompleteRequest>& data)
{
   sendSubSurfaceSwitchUpdateSignal(data->getSurfaceid(), data->getSurfaceState());
}


void StubHandler::onRequestSlaveSurfaceRepositioningRequest(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::RequestSlaveSurfaceRepositioningRequest >& request)
{
   sendSlaveSurfaceRepositioningSignal(request->getPidSender(), request->getActivate(), request->getSurfaceId());
}


void StubHandler::onResponseSlaveSurfaceRepositioningRequest(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::ResponseSlaveSurfaceRepositioningRequest >& request)
{
   sendSlaveSurfaceRepositioningAckSignal(request->getPidSender(), request->getActivate(), request->getSurfaceId(), request->getStatus());
}


void StubHandler::onRequestDirectTextureConsumerPositionInfoRequest(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::RequestDirectTextureConsumerPositionInfoRequest >& request)
{
   sendDirectTextureConsumerPositionInfoRequestSignal(request->getPidSender(), request->getInstanceId());
}


void StubHandler::onResponseDirectTextureConsumerPositionInfoRequest(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::ResponseDirectTextureConsumerPositionInfoRequest >& request)
{
   sendDirectTextureConsumerPositionInfoResponseSignal(request->getPidSender(), request->getInstanceId(), request->getXPos(), request->getYPos());
}


void StubHandler::onFollowUpActionRequest(const ::boost::shared_ptr<  bosch::cm::ai::hmi::appctrl::HMIAppCtrl::FollowUpActionRequest >& request)
{
   sendFollowUpActionRequestSignal(request->getActionType(), request->getUserData());
}


void StubHandler::onResponseFollowUpActionRequest(const ::boost::shared_ptr<  bosch::cm::ai::hmi::appctrl::HMIAppCtrl::ResponseFollowUpActionRequest >& request)
{
   sendFollowUpActionResponseSignal(request->getActionType(), request->getUserData());
}


void StubHandler::onRequestExternalImageProviderActivationRequest(const ::boost::shared_ptr< RequestExternalImageProviderActivationRequest >& request)
{
   std::vector< ExternalImageData >& data = HMIAppCtrlStub::getExternalImagesStatusMutable();
   for (std::vector< int32 >::const_iterator id = request->getInstanceIds().begin(); id != request->getInstanceIds().end(); ++id)
   {
      bool known = false;
      // update attribute
      for (std::vector< ExternalImageData >::iterator dataset = data.begin(); dataset != data.end(); ++dataset)
      {
         if (dataset->getInstanceId() == *id)
         {
            known = true;
            if (dataset->getGadgetState() != request->getTargetState())
            {
               ExternalImageData newDataSet(request->getTargetState(), *id, request->getConsumerId());
               *dataset = newDataSet;
               break;
            }
         }
      }

      if (known == false)
      {
         data.push_back(ExternalImageData(request->getTargetState(), *id, request->getConsumerId()));
      }
   }

   // don't send updates on attribute even if there are changes.
   // signal should be used to get updates, attribute is only for pull requests

   // broadcast to all clients
   sendActivateExternalImageProviderSignal(request->getTargetState(), request->getInstanceIds(), request->getConsumerId());
}


void StubHandler::onRequestExternalImageProviderTouchRequest(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::HMIAppCtrl::RequestExternalImageProviderTouchRequest >& request)
{
   // broadcast to all clients
   sendExternalImageProviderTouchSignal(request->getPidSender(), request->getData(), request->getInstanceId());
}


void StubHandler::onExternalImagesStatusGet(const ::boost::shared_ptr< ExternalImagesStatusGet >& /*payload*/)
{
   sendExternalImagesStatusGetUpdate(getExternalImagesStatus());
}


void StubHandler::onDrmBufferAllocationStatusRequest(const ::boost::shared_ptr< DrmBufferAllocationStatusRequest >& request)
{
   pid_t pid = request->getPid();

   drmBufferAllocationStatus status = request->getStatus();
   uint64_t size = request->getBufferSize();

   if (size > 0)
   {
      std::map<pid_t, uint64_t>::iterator it = _drmBufferStatistics.find(pid);

      if (it != _drmBufferStatistics.end())
      {
         if (status == drmBufferAllocationStatus__DRM_BUFFER_ALLOCATION_SUCCESS)
         {
            (*it).second += size;
         }
         else if (status == drmBufferAllocationStatus__DRM_BUFFER_DEALLOCATION_SUCCESS)
         {
            if (size <= (*it).second)
            {
               (*it).second -= size;
            }
            else
            {
               // this should never happen
               ETG_TRACE_ERR_THR(("onDrmBufferAllocationStatusRequest pid %d freed %d > allocated %d ???", pid, size, (*it).second));
            }
         }
         else
         {
            ETG_TRACE_ERR_THR(("onDrmBufferAllocationStatusRequest pid %d reports size %d for status %d ???", pid, size, status));
         }
      }
      else
      {
         _drmBufferStatistics[pid] = size;
      }
   }

   for (std::map<pid_t, uint64_t>::iterator it = _drmBufferStatistics.begin(); it != _drmBufferStatistics.end(); ++it)
   {
      ETG_TRACE_USR4_THR(("onDrmBufferAllocationStatusRequest Allocation status %d bytes pid %d", (*it).second, (*it).first));
   }

   if (request->getStatus() == drmBufferAllocationStatus__DRM_BUFFER_ALLOCATION_FAILED)
   {
      ETG_TRACE_USR4_THR(("onDrmBufferAllocationStatusRequest request garbage collection"));
      // some allocation failed, so request all clients to clean up their caches
      sendDrmGarbageCollectionRequestSignal();
   }
}


} // namespace
} // namespace
} // namespace
