/* ***************************************************************************************
* FILE:          AppTaskArray.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppTaskArray.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef APPTASKARRAY_H_
#define APPTASKARRAY_H_

#include <vector>

namespace hmibase {
typedef unsigned int appid_t;
}


class AppTask
{
   private:
      std::string       _sAppIdentName;   // "APP::XXXXXX" ...
      hmibase::appid_t  _appId;           // Super StateMachine ID
      bool              _initialized;     //
      bool              _masterApp;       // first entry should be the master
      bool              _mandatory;
      int               _pid;             // linux proc id
      // extended info
      unsigned          _surface;         // main surface
      bool              _switchableGroup; // application is member of main group like playground, tuner, media, ...
      bool              _needsAudio;      // if audi channel has to be activated
      bool              _externalStart;   // TEST apps
      std::string       _serviceName;     // service name for external start
      bool              _isFallbackApp;   // e.g. if start fails

      unsigned int      _hkCode;          // keyCode for direct activation

   public:

      AppTask() : _appId(0),
         _initialized(false),
         _masterApp(false),
         _mandatory(false),
         _pid(0),
         _surface(0),
         _switchableGroup(false),
         _needsAudio(false),
         _externalStart(false),
         _serviceName(),
         _isFallbackApp(false),
         _hkCode(0)
      {
      }

      explicit AppTask(const char* appName, hmibase::appid_t appId, bool mandatory, bool masterApp) :
         _sAppIdentName(appName),
         _appId(appId),
         _initialized(false),
         _masterApp(masterApp),
         _mandatory(mandatory),
         _pid(0),
         _surface(0),
         _switchableGroup(false),
         _needsAudio(false),
         _externalStart(false),
         _serviceName(),
         _isFallbackApp(false),
         _hkCode(0)
      {
      }

      explicit AppTask(const char* appName, hmibase::appid_t appId, bool mandatory,
                       unsigned surface, bool switchableGroup, bool needsAudio, bool externalStart, const char* service, bool isFallbackApp, unsigned int hkCode = 0) :
         _sAppIdentName(appName),
         _appId(appId),
         _initialized(false),
         _masterApp(false),
         _mandatory(mandatory),
         _pid(0),
         _surface(surface),
         _switchableGroup(switchableGroup),
         _needsAudio(needsAudio),
         _externalStart(externalStart),
         _serviceName(service),
         _isFallbackApp(isFallbackApp),
         _hkCode(hkCode)
      {
      }

      virtual ~AppTask() {}

      const std::string& getAppIdentName() const
      {
         return _sAppIdentName;
      }
      void setAppId(hmibase::appid_t appId)
      {
         _appId = appId;
      }
      hmibase::appid_t getAppId(void) const
      {
         return _appId;
      }
      void setPid(int pid)
      {
         _pid = pid;
      }
      int getPid(void) const
      {
         return _pid;
      }
      void setInitialized(bool initialized)
      {
         _initialized = initialized;
      }
      bool isInitialized() const
      {
         return _initialized;
      }
      bool isRunning() const
      {
         return _initialized; // has to be checked
      }
      void setAsMasterApp(bool masterApp)
      {
         _masterApp = masterApp;
      }
      bool isMasterApp() const
      {
         return _masterApp;
      }
      void setSurface(unsigned surface)
      {
         _surface = surface;
      }
      unsigned getSurface() const
      {
         return _surface;
      }
      void setSwitchableGroup()
      {
         _switchableGroup = true;
      }
      bool isSwitchable() const
      {
         return _switchableGroup && _surface > 0 ;
      }
      void setNeedsAudio()
      {
         _needsAudio = true;
      }
      bool getNeedsAudio() const
      {
         return _needsAudio;
      }
      void setExternalStart()
      {
         _externalStart = true;
      }
      bool getExternalStart() const
      {
         return _externalStart;
      }
      void setServiceName(const char* s)
      {
         _serviceName = s;
      }
      const std::string& getServiceName() const
      {
         return _serviceName;
      }
      void setFallbackApp()
      {
         _isFallbackApp = true;
      }
      bool isFallbackApp() const
      {
         return _isFallbackApp;
      }
      void setMandatory(bool value)
      {
         _mandatory = value;
      }
      bool isMandatory() const
      {
         return _mandatory;
      }
      void setHkCode(unsigned int hkCode)
      {
         _hkCode = hkCode;
      }
      unsigned int getHkCode()
      {
         return _hkCode;
      }

      bool operator==(const AppTask& appTask) const
      {
         return appTask.getAppIdentName() == _sAppIdentName;
      }
      bool operator!=(const AppTask& appTask) const
      {
         return appTask.getAppIdentName() != _sAppIdentName;
      }
      bool operator==(const std::string& name) const
      {
         return name == _sAppIdentName;
      }
      bool operator!=(const std::string& name) const
      {
         return name != _sAppIdentName;
      }
};


class AppTaskArray
{
   public:
      enum tenError
      {
         PID_ALREADY_UP_TO_DATE = 0,
         PID_UPDATED,
         NO_NAME_MATCH
      };

      typedef std::vector<AppTask*> AppTasks;

      AppTaskArray();
      ~AppTaskArray();

      bool addTask(AppTask* app);

      const AppTask* getTaskByAppId(hmibase::appid_t appId) const;
      const AppTask* getTaskByName(const std::string& sAppIdentName) const;
      const AppTask* getTaskByPid(int pid) const;
      const AppTask* getFollowerInList(hmibase::appid_t appId) const;

      const AppTask* getTaskByHkCode(unsigned int hkCode) const;
      const AppTask* getFollowerByHkCode(hmibase::appid_t appId, unsigned int hkCode) const;

      int  getPid(const std::string& sAppIdentName) const;
      int  getPid(unsigned int smid) const;
      int  getPidOfMasterApp() const;
      bool isAppPid(int pid) const;
      tenError setPid(const std::string& sAppIdentName, int pid);

      void setMasterApp(const std::string& name);
      void updateInitFlag(const std::string& appName);
      void setMandatory(const std::string& sAppIdentName, bool value);

      bool traceInitializedStatus() const;

      // for legacy reason
      const AppTasks& getAppTasks() const
      {
         return _appTasks;
      }

      static AppTaskArray* getInstance();
   private:
      static AppTaskArray* _theInstance;
      AppTasks             _appTasks;
};


#endif /* APPTASKARRAY_H_ */
