/* ***************************************************************************************
* FILE:          AppTaskArray.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppTaskArray.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "AppTaskArray.h"
#include "hmibase/util/Macros.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_APPCTRL_STUB
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AppTaskArray.cpp.trc.h"
#endif

AppTaskArray* AppTaskArray::_theInstance = 0;

AppTaskArray::AppTaskArray()
{
   _theInstance = this;
}


AppTaskArray::~AppTaskArray()
{
}


// static
AppTaskArray* AppTaskArray::getInstance()
{
   return _theInstance;
}


bool AppTaskArray::addTask(AppTask* app)
{
   if (_appTasks.size() == 0)
   {
      app->setAsMasterApp(true);
   }
   _appTasks.push_back(app);
   return true;
}


const AppTask* AppTaskArray::getTaskByAppId(hmibase::appid_t appId) const
{
   for (AppTasks::const_iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if ((*it)->getAppId() == appId)
      {
         return *it;
      }
   }
   return 0;
}


const AppTask* AppTaskArray::getTaskByPid(int pid) const
{
   for (AppTasks::const_iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if ((*it)->getPid() == pid)
      {
         return *it;
      }
   }
   return 0;
}


const AppTask* AppTaskArray::getTaskByName(const std::string& sAppIdentName) const
{
   for (AppTasks::const_iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if ((*it)->getAppIdentName() == sAppIdentName)
      {
         return *it;
      }
   }
   return 0;
}


const AppTask* AppTaskArray::getFollowerInList(hmibase::appid_t appId) const
{
   for (AppTasks::const_iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if (appId == 0)
      {
         return *it;
      }
      if ((*it)->getAppId() == appId)
      {
         ++it;
         if (it == _appTasks.end())
         {
            break;
         }
         return *it;
      }
   }
   return 0;
}


const AppTask* AppTaskArray::getTaskByHkCode(unsigned int hkCode) const
{
   if (hkCode != 0)
   {
      for (AppTasks::const_iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
      {
         if ((*it)->getHkCode() == hkCode)
         {
            return *it;
         }
      }
   }
   return 0;
}


const AppTask* AppTaskArray::getFollowerByHkCode(hmibase::appid_t appId, unsigned int hkCode) const
{
   AppTask* firstAppAssignedToHkCode = 0;
   bool returnNextMatch = false;

   for (AppTasks::const_iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if ((firstAppAssignedToHkCode == 0) && (hkCode == (*it)->getHkCode()))
      {
         firstAppAssignedToHkCode = *it;
      }

      if (appId == 0)
      {
         // return first app assigned to hardkey
         if (firstAppAssignedToHkCode != 0)
         {
            return firstAppAssignedToHkCode;
         }
      }
      else
      {
         if (returnNextMatch)
         {
            if (hkCode == (*it)->getHkCode())
            {
               return *it;
            }
         }
         else if ((*it)->getAppId() == appId)
         {
            returnNextMatch = true;
         }
      }
   }

   return returnNextMatch ? firstAppAssignedToHkCode : 0;
}


AppTaskArray::tenError AppTaskArray::setPid(const std::string& sAppIdentName, int pid)
{
   for (AppTasks::iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if ((*it)->getAppIdentName() == sAppIdentName)
      {
         if ((*it)->getPid() == pid)
         {
            return PID_ALREADY_UP_TO_DATE;
         }
         else
         {
            (*it)->setPid(pid);
            return PID_UPDATED;
         }
      }
   }
   SYSTEMD_LOG("ERROR AppTaskArray::setPid %s\n", sAppIdentName.c_str());
   return NO_NAME_MATCH;
}


int AppTaskArray::getPid(const std::string& sAppIdentName) const
{
   const AppTask* it = getTaskByName(sAppIdentName);
   if (it)
   {
      return it->getPid();
   }
   return 0;
}


int AppTaskArray::getPid(hmibase::appid_t appId) const
{
   const AppTask* it = getTaskByAppId(appId);
   if (it)
   {
      return it->getPid();
   }
   return 0;
}


bool AppTaskArray::isAppPid(int pid) const
{
   const AppTask* it = getTaskByPid(pid);
   if (it)
   {
      return true;
   }
   return false;
}


void AppTaskArray::setMasterApp(const std::string& appName)
{
   for (AppTasks::iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      // there is only one master application, by default it's the first one in the vector
      if ((*it)->getAppIdentName() == appName)
      {
         (*it)->setAsMasterApp(true);
      }
      else
      {
         (*it)->setAsMasterApp(false);
      }
   }
}


int AppTaskArray::getPidOfMasterApp() const
{
   for (AppTasks::const_iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if ((*it)->isMasterApp())
      {
         return (*it)->getPid();
      }
   }
   return 0;
}


void AppTaskArray::updateInitFlag(const std::string& appName)
{
   for (AppTasks::iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if ((*it)->getAppIdentName() == appName)
      {
         (*it)->setInitialized(true);
      }
      // SYSTEMD_LOG("AppTaskArray::updateInitFlag %s = %d\n",(*it)->getAppIdentName().c_str(),(*it)->isInitialized() ? 1:0);
   }
}


void AppTaskArray::setMandatory(const std::string& appName, bool value)
{
   for (AppTasks::iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      if ((*it)->getAppIdentName() == appName)
      {
         (*it)->setMandatory(value);
      }
   }
}


bool AppTaskArray::traceInitializedStatus() const
{
   bool result(true);
   for (AppTasks::const_iterator it = _appTasks.begin(); it != _appTasks.end(); ++it)
   {
      SYSTEMD_LOG("AppTaskArray::traceInitializedStatus - Mandatory App %s [man=%s] [initdone=%s]\n",
                  (*it)->getAppIdentName().c_str(),
                  (*it)->isMandatory() ? "yes" : "no",
                  (*it)->isInitialized() ? "yes" : "no");
      if ((*it)->isMandatory() && !(*it)->isInitialized())
      {
         ETG_TRACE_USR4_THR(("checkInitFlag:: Mandatory App %s not initialized", (*it)->getAppIdentName().c_str()));
         result = false;
      }
   }
   return result;
}
