/* ***************************************************************************************
* FILE:          SyncBlockProducerFactory.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SyncBlockProducerFactory.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef SYNCBLOCKPRODUCERFACTORY_H
#define SYNCBLOCKPRODUCERFACTORY_H

#include <map>
#include <set>

#include "Gadget/SyncBlock.h"
#include "AppUtils/HmiMacros.h"
#include "AppUtils/Timer.h"

namespace hmibase {
namespace gadget {
namespace videobuffer {
class VideoBufferType;
} // namespace
} // namespace
} // namespace


namespace hmibase {
namespace gadget {


class SyncBlockProducerFactory
{
   public:
      typedef std::pair<unsigned int, unsigned int> BufferIdentifier;

      class Accessor
      {
         public:

            unsigned int GetId(BufferIdentifier key)
            {
               std::map<BufferIdentifier, unsigned int>::iterator it = _ids.find(key);
               if (it != _ids.end())
               {
                  return it->second;
               }
               return 0;
            }

            void SetId(BufferIdentifier key, unsigned int id)
            {
               _ids[key] = id;
            }

         private:
            std::map<BufferIdentifier, unsigned int> _ids;
      };

      class DrmBufferContainer
      {
         public:
            DrmBufferContainer(unsigned int syncBlockId, unsigned int instanceId);
            ~DrmBufferContainer();

            bool createDrmBuffers(int w, int h, int depth, int bpp);
            bool terminateProducer();
            bool checkTermination();
            hmibase::gadget::videobuffer::VideoBufferType* getCurrentBuffer();
            int getCurrentBufferId();

            int exchange();

            bool isEmpty()
            {
               return _drmBuffers.empty();
            }

            unsigned int getRefCount()
            {
               return _refCount;
            }

            void increaseRefCount()
            {
               ++_refCount;
            }

            void decreaseRefCount()
            {
               if (_refCount > 0)
               {
                  --_refCount;
               }
            }

            void setOutOfCacheTimestampMs(unsigned int timestampMs)
            {
               _outOfCacheTimestampMs = timestampMs;
            }

            unsigned int getOutOfCacheTimestampMs()
            {
               return _outOfCacheTimestampMs;
            }

            bool isGarbage();

         private:
            ProducerType* _producer;
            std::map<int, hmibase::gadget::videobuffer::VideoBufferType*> _drmBuffers;

            DrmBufferContainer();

            unsigned int _syncBlockId;
            unsigned int _instanceId;
            bool _attached;
            int _bufHandle;
            unsigned int _refCount;
            unsigned int _outOfCacheTimestampMs;

            HMIBASE_UNCOPYABLE(DrmBufferContainer)
      };

      typedef std::set<unsigned int> SyncBlockIdSet;
      typedef std::map<unsigned int, SyncBlockIdSet> SyncBlockConnectionsMap;

      static bool Init();
      static void Deinit();

      DrmBufferContainer* GetSyncBlockProducer(unsigned int syncBlockId, unsigned int instanceId, Accessor* accessor);

      SyncBlockIdSet GetSyncBlockIDForInstance(unsigned int producerInstanceId);

      void AddIdTuple(unsigned int syncBlockId, unsigned int instanceId);

      static SyncBlockProducerFactory& GetInstance()
      {
         Init(); // initialize if required
         return *_theInstance;
      }

      void ReleaseSyncBlockProducer(unsigned int syncBlockId, unsigned int instanceId, Accessor* accessor);

      bool OnMessage(const Courier::Message& msg);

      bool setGarbageColletionIntervalMs(uint32_t intervalMs);

   private:

      enum GCReason
      {
         GC_TIMER,
         GC_FORCE
      };

      /** clean up everything that needs no async communication */
      bool GarbageCollect(GCReason reason = GC_FORCE);

      bool setNextGCTimestamp(uint32_t timeStampMs);

      SyncBlockProducerFactory();
      ~SyncBlockProducerFactory();
      SyncBlockProducerFactory(const SyncBlockProducerFactory&);
      SyncBlockProducerFactory& operator=(const SyncBlockProducerFactory&);

      std::map<BufferIdentifier, DrmBufferContainer*>_syncBlockProvider;
      SyncBlockConnectionsMap _syncBlockConnections;

      uint32_t _garbageCollectionIntervalMs;
      Util::Timer _garbageCollectionTimer;

      static SyncBlockProducerFactory* _theInstance;
};


}
}


#endif // SYNCBLOCKPRODUCERFACTORY_H
