/* ***************************************************************************************
* FILE:          FTrace.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FTrace.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FCommon.h"
#include "FContainer.h"
#include "FDataSet.h"
#include "FData.h"
#include "FActiveViewGroup.h"
#include "FController.h"
#include "FSession.h"
#include "FManager.h"
#include "FStateInfo.h"
#include "FConfigInfo.h"
#include "FUtils.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FTrace.cpp.trc.h"
#endif

namespace Focus {
bool FUtils::isDetailedTraceActive()
{
#if defined(VARIANT_S_FTR_ENABLE_TRC_GEN)
   return etg_bIsTraceActive(TR_CLASS_HMI_FW_FOCUS, 8); //lint !e1786 intended
#else
   return true;
#endif
}


void FUtils::printTextTrace(char* text, unsigned int length)
{
#if !defined(ETG_C_TRACE_MAX)
#define ETG_C_TRACE_MAX 200
#endif

#if defined(ETG_C_TRACE_MAX)
   ETG_TRACE_USR4_THR(("                                                            "));
   ETG_TRACE_USR4_THR(("<<<<<<<<<<<<<<<<<<<<<<<<<FOCUS INFO>>>>>>>>>>>>>>>>>>>>>>>>>"));
   const unsigned int maxLineLength = (ETG_C_TRACE_MAX) - 20/*overhead*/;
   unsigned int currentLineLength;
   for (unsigned int pos = 0; pos < length; pos += currentLineLength)
   {
      currentLineLength = 0;
      while (pos + currentLineLength < length)
      {
         //if new line is found, replace it with null termination char
         if ((text[pos + currentLineLength] == '\n') || (currentLineLength >= maxLineLength))
         {
            text[pos + currentLineLength] = '\0';
            ++currentLineLength;
            break;
         }
         ++currentLineLength;
      }

      const char* temp = text + pos;
      ETG_TRACE_USR4_THR(("%s", temp));
   }
   ETG_TRACE_USR4_THR(("<<<<<<<<<<<<<<<<<<<<<<<<<FOCUS INFO>>>>>>>>>>>>>>>>>>>>>>>>>"));
#else
   (void)length;
   ETG_TRACE_USR4_THR(("FUtils::printTextTrace %s", text));
#endif
}


void FManager::printSessionDebugInfo()
{
   FSession* session = getSession();
   if (session != NULL)
   {
      FUtils::printItemTrace(*session);
   }
}


}

using namespace Focus;

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {

static ::FeatStd::UInt32 AppendCurrentAppId(::FeatStd::StringBuffer& stringBuffer)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("<");
   tcharCount += stringBuffer.AppendObject(FManager::getInstance().getCurrentAppId());
   tcharCount += stringBuffer.Append(">");

   return tcharCount;
}


template<typename TPtr>
::FeatStd::UInt32 AppendPtr(::FeatStd::StringBuffer& stringBuffer, TPtr const& object)
{
   char buffer[11];
   SNPRINTF(buffer, sizeof(buffer), "0x%p", object);
   return stringBuffer.Append(buffer);
}


template<> ::FeatStd::UInt32 StringBufferAppender<FSequenceId>::Append(::FeatStd::StringBuffer& stringBuffer, FSequenceId const& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.AppendObject(val.Id);

   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FGroupCurrentFocusInfo>::Append(::FeatStd::StringBuffer& stringBuffer, FGroupCurrentFocusInfo const& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append(FID_STR(val.Id));

   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FAppCurrentFocusInfo>::Append(::FeatStd::StringBuffer& stringBuffer, FAppCurrentFocusInfo const& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append(FVIEWID_STR(val.ViewId));
   tcharCount += stringBuffer.Append("/");
   tcharCount += stringBuffer.Append(FID_STR(val.WidgetId));

   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FCameraData>::Append(::FeatStd::StringBuffer& stringBuffer, FCameraData const& val)
{
   ::FeatStd::UInt32 tcharCount = 0;
   //tcharCount += stringBuffer.Append("FCameraData { ");

   tcharCount += stringBuffer.Append("Display=");
   tcharCount += stringBuffer.AppendObject(val.DisplayId);

   tcharCount += stringBuffer.Append(", Offset={");
   tcharCount += stringBuffer.AppendObject(static_cast<int>(val.DisplayOffset.GetX()));
   tcharCount += stringBuffer.Append(",");
   tcharCount += stringBuffer.AppendObject(static_cast<int>(val.DisplayOffset.GetY()));
   tcharCount += stringBuffer.Append("}");

   //tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FCameraDataVector>::Append(::FeatStd::StringBuffer& stringBuffer, const FCameraDataVector& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   //tcharCount += stringBuffer.Append("{");
   for (FCameraDataVector::const_iterator it = val.begin(); it != val.end(); ++it)
   {
      if (it != val.begin())
      {
         tcharCount += stringBuffer.Append(", ");
      }

      FCameraData cameraData = *it;
      tcharCount += stringBuffer.AppendObject(cameraData);
   }
   //tcharCount += stringBuffer.Append(" }");

   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FWidgetData>::Append(::FeatStd::StringBuffer& stringBuffer, FWidgetData const& val)
{
   ::FeatStd::UInt32 tcharCount = 0;
   //tcharCount += stringBuffer.Append("FWidgetData { ");

   tcharCount += stringBuffer.Append(", Enabled=");
   tcharCount += stringBuffer.AppendObject(val.Enabled);

   tcharCount += stringBuffer.Append(", SeqNr=");
   tcharCount += stringBuffer.AppendObject(val.SequenceNr);

   tcharCount += stringBuffer.Append(", Order=");
   tcharCount += stringBuffer.AppendObject(val.Order);

   if (val.Order != val.EffectiveOrder)
   {
      tcharCount += stringBuffer.Append(", EffOrder=");
      tcharCount += stringBuffer.AppendObject(val.EffectiveOrder);
   }

   if (val.ControllerSetId != Constants::InvalidControllerSetId)
   {
      tcharCount += stringBuffer.Append(", ControllerSet=");
      tcharCount += stringBuffer.AppendObject(val.ControllerSetId);
   }

   //tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FGroupData>::Append(::FeatStd::StringBuffer& stringBuffer, FGroupData const& val)
{
   ::FeatStd::UInt32 tcharCount = 0;
   //tcharCount += stringBuffer.Append("FGroupData { ");

   tcharCount += stringBuffer.Append(", Configured=");
   tcharCount += stringBuffer.AppendObject(val.Configured);

   tcharCount += stringBuffer.Append(", WrapAround=");
   tcharCount += stringBuffer.AppendObject(val.WrapAround);

   tcharCount += stringBuffer.Append(", PreserveFocus=");
   tcharCount += stringBuffer.AppendObject(val.PreserveFocus);

   tcharCount += stringBuffer.Append(", DefaultOrder=");
   tcharCount += stringBuffer.AppendObject(val.DefaultOrder);

   tcharCount += stringBuffer.Append(", Layer=");
   tcharCount += stringBuffer.AppendObject(val.Layer);

   tcharCount += stringBuffer.Append(", AnchorMode=");
   tcharCount += stringBuffer.AppendObject(val.AnchorMode);

   if (val.AnchorMode != Focus::FAnchorMode::None)
   {
      tcharCount += stringBuffer.Append(", AnchorId=");
      tcharCount += stringBuffer.AppendObject(val.AnchorId);
   }

   //tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FListData>::Append(::FeatStd::StringBuffer& stringBuffer, FListData const& val)
{
   ::FeatStd::UInt32 tcharCount = 0;
   //tcharCount += stringBuffer.Append("FListData { ");

   tcharCount += stringBuffer.Append(", FirstVisibleIndex=");
   tcharCount += stringBuffer.AppendObject(val.FirstVisibleIndex);

   tcharCount += stringBuffer.Append(", VisibleItemCount=");
   tcharCount += stringBuffer.AppendObject(val.VisibleItemCount);

   tcharCount += stringBuffer.Append(", TotalItemCount=");
   tcharCount += stringBuffer.AppendObject(val.TotalItemCount);

   //tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FListItemData>::Append(::FeatStd::StringBuffer& stringBuffer, FListItemData const& val)
{
   ::FeatStd::UInt32 tcharCount = 0;
   //tcharCount += stringBuffer.Append("FListItemData { ");

   tcharCount += stringBuffer.Append(", ListId=");
   tcharCount += stringBuffer.AppendObject(val.ListId);

   tcharCount += stringBuffer.Append(", Index=");
   tcharCount += stringBuffer.AppendObject(val.Index);

   //tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FData>::Append(::FeatStd::StringBuffer& stringBuffer, FData const& data)
{
   (void)data;
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FData { ");
   //tcharCount += stringBuffer.AppendObject(data._widgetData);
   //tcharCount += stringBuffer.AppendObject(data._groupData);
   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<Focusable>::Append(::FeatStd::StringBuffer& stringBuffer, Focusable const& f)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("Focusable { ");

   tcharCount += stringBuffer.Append("Id=");
   tcharCount += stringBuffer.Append(FID_STR(f.getId()));

   FGroup* parent = f.getParent();
   if (parent != NULL)
   {
      tcharCount += stringBuffer.Append(", Parent=");
      tcharCount += stringBuffer.Append(FID_STR(parent->getId()));
   }

   //FWidgetData* widgetData = (const_cast<Focusable&>(f)).getData<FWidgetData>();
   //if (widgetData != NULL)
   //{
   //   tcharCount += stringBuffer.AppendObject(*widgetData);
   //}

   //FGroupData* groupData = (const_cast<Focusable&>(f)).getData<FGroupData>();
   //if ((groupData != NULL) && (groupData->getConfigured()))
   //{
   //   tcharCount += stringBuffer.AppendObject(*groupData);
   //}

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FGroup>::Append(::FeatStd::StringBuffer& stringBuffer, const FGroup& group)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FGroup { ");

   tcharCount += stringBuffer.Append("Id=");
   tcharCount += stringBuffer.Append(FID_STR(group.getId()));

   FGroupCurrentFocusInfo* crtFocusInfo = (const_cast<FGroup&>(group)).Data.get<FGroupCurrentFocusInfo>();
   if (crtFocusInfo != NULL)
   {
      tcharCount += stringBuffer.Append(", CurrentFocus=");
      tcharCount += stringBuffer.AppendObject(*crtFocusInfo);
   }

   tcharCount += stringBuffer.Append(", Children={ ");
   for (size_t index = 0; index < group.Children.count(); ++index)
   {
      Focusable* f = group.Children.get(index);
      if (f != NULL)
      {
         if (index > 0)
         {
            tcharCount += stringBuffer.Append(", ");
         }
         tcharCount += stringBuffer.Append(FID_STR(f->getId()));
      }
   }
   tcharCount += stringBuffer.Append(" }");

   //tcharCount += stringBuffer.Append(", currentFocus=");
   //tcharCount += stringBuffer.Append(group.getCurrentFocusInfo().c_str());

   //tcharCount += stringBuffer.Append(", children={ ");
   //for (size_t index = 0; index < group.count(); ++index)
   //{
   //   Focusable* f = group.get(index);
   //   if (f != NULL)
   //   {
   //      if (index > 0)
   //      {
   //         tcharCount += stringBuffer.Append(", ");
   //      }
   //      tcharCount += stringBuffer.Append(f->getId().c_str());
   //   }
   //}
   //tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FActiveViewGroup>::Append(::FeatStd::StringBuffer& stringBuffer, FActiveViewGroup const& viewGroup)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FActiveViewGroup { ");

   tcharCount += stringBuffer.Append(", Groups={");
   for (size_t index = 0; index < viewGroup.Groups.count(); ++index)
   {
      FGroup* f = viewGroup.Groups.getAt(index);
      if (f != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n  ");
         tcharCount += stringBuffer.AppendObject(*f);
      }
   }
   tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(", Widgets={");
   for (size_t index = 0; index < viewGroup.Widgets.count(); ++index)
   {
      Focusable* f = viewGroup.Widgets.getAt(index);
      if (f != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n  ");
         tcharCount += stringBuffer.AppendObject(*f);
      }
   }
   tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(", Anchors={");
   for (size_t index = 0; index < viewGroup.Anchors.count(); ++index)
   {
      Focusable* f = viewGroup.Anchors.getAt(index);
      if (f != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n  {");
         tcharCount += stringBuffer.AppendObject(viewGroup.Anchors.getKeyAt(index));
         tcharCount += stringBuffer.Append(":");
         tcharCount += stringBuffer.Append(FID_STR(f->getId()));
         tcharCount += stringBuffer.Append("}");
      }
   }
   tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FWidgetConfig>::Append(::FeatStd::StringBuffer& stringBuffer, const FWidgetConfig& f)
{
   FWidgetConfig& val = const_cast<FWidgetConfig&>(f);

   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FWidgetConfig { ");

   tcharCount += stringBuffer.Append("Id=");
   tcharCount += stringBuffer.Append(FID_STR(val.getId()));

   tcharCount += stringBuffer.Append(", View=");
   tcharCount += stringBuffer.Append(FVIEWID_STR(val.View.getId()));

   FWidgetConfig* parent = val.getParent();
   if (parent != NULL)
   {
      tcharCount += stringBuffer.Append(", Parent=");
      tcharCount += stringBuffer.Append(FID_STR(parent->getId()));
   }

   FWidgetData* widgetData = val.Data.get<FWidgetData>();
   if (widgetData != NULL)
   {
      tcharCount += stringBuffer.AppendObject(*widgetData);
   }

   FGroupData* groupData = val.Data.get<FGroupData>();
   if (groupData != NULL)
   {
      tcharCount += stringBuffer.AppendObject(*groupData);
   }

   FListData* listData = val.Data.get<FListData>();
   if (listData != NULL)
   {
      tcharCount += stringBuffer.AppendObject(*listData);
   }

   FListItemData* listItemData = val.Data.get<FListItemData>();
   if (listItemData != NULL)
   {
      tcharCount += stringBuffer.AppendObject(*listItemData);
   }

   FRectangle* rectangle = val.Data.get<FRectangle>();
   if (rectangle != NULL)
   {
      tcharCount += stringBuffer.Append(", Bounds={");
      tcharCount += stringBuffer.AppendObject(static_cast<int>(rectangle->GetLeft()));
      tcharCount += stringBuffer.Append(",");
      tcharCount += stringBuffer.AppendObject(static_cast<int>(rectangle->GetTop()));
      tcharCount += stringBuffer.Append(",");
      tcharCount += stringBuffer.AppendObject(static_cast<int>(rectangle->GetWidth()));
      tcharCount += stringBuffer.Append(",");
      tcharCount += stringBuffer.AppendObject(static_cast<int>(rectangle->GetHeight()));
      tcharCount += stringBuffer.Append("}");
   }

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FGroupState>::Append(::FeatStd::StringBuffer& stringBuffer, const FGroupState& f)
{
   FGroupState& val = const_cast<FGroupState&>(f);

   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FGroupState { ");

   tcharCount += stringBuffer.Append("Id=");
   tcharCount += stringBuffer.Append(FID_STR(val.getId()));

   if (val.UpdateSequenceId != FSequenceId())
   {
      tcharCount += stringBuffer.Append(", SeqId=");
      tcharCount += stringBuffer.AppendObject(val.UpdateSequenceId);
   }

   tcharCount += stringBuffer.Append(", Crt=");
   tcharCount += stringBuffer.AppendObject(val.CurrentFocusInfo);

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FViewConfig>::Append(::FeatStd::StringBuffer& stringBuffer, const FViewConfig& view)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FViewConfig { ");

   tcharCount += stringBuffer.Append("Id=");
   tcharCount += stringBuffer.Append(FVIEWID_STR(view.getId()));

   FCameraDataVector* cameraDataVector = (const_cast<FViewConfig&>(view)).Data.get<FCameraDataVector>();
   if (cameraDataVector != NULL)
   {
      tcharCount += stringBuffer.Append(", Cameras={");
      tcharCount += stringBuffer.AppendObject(*cameraDataVector);
      tcharCount += stringBuffer.Append(" }");
   }

   tcharCount += stringBuffer.Append(", Widgets={ ");
   for (size_t index = 0; index < view.getChildCount(); ++index)
   {
      FWidgetConfig* f = view.getChild(index);
      if (f != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n  ");
         tcharCount += stringBuffer.AppendObject(*f);
      }
   }
   tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FViewState>::Append(::FeatStd::StringBuffer& stringBuffer, const FViewState& viewState)
{
   FViewState& view = const_cast<FViewState&>(viewState);

   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FViewState { ");

   tcharCount += stringBuffer.Append("Id=");
   tcharCount += stringBuffer.Append(FVIEWID_STR(view.getId()));

   FViewFocusVisible* viewFocusVisible = view.getData<FViewFocusVisible>();
   if (viewFocusVisible != NULL)
   {
      tcharCount += stringBuffer.Append(", FocusVisible=");
      tcharCount += stringBuffer.AppendObject(viewFocusVisible->Visible);
   }

   tcharCount += stringBuffer.Append(", Groups={ ");
   for (size_t index = 0; index < view.getChildCount(); ++index)
   {
      FGroupState* f = view.getChild(index);
      if (f != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n  ");
         tcharCount += stringBuffer.AppendObject(*f);
      }
   }
   tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FAppConfig>::Append(::FeatStd::StringBuffer& stringBuffer, const FAppConfig& appConfig)
{
   FAppConfig& val = const_cast<FAppConfig&>(appConfig);

   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FAppConfig { ");

   tcharCount += stringBuffer.Append("Id=");
   tcharCount += stringBuffer.AppendObject(val.getId());

   FAppCurrentFocusInfo* crtFocus = val.Data.get<FAppCurrentFocusInfo>();
   if (crtFocus != NULL)
   {
      tcharCount += stringBuffer.Append(", CrtFocus=");
      tcharCount += stringBuffer.AppendObject(*crtFocus);
   }

   tcharCount += stringBuffer.Append(", Views={");
   for (size_t index = 0; index < val.getChildCount(); ++index)
   {
      FViewConfig* view = val.getChild(index);
      if (view != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n");
         tcharCount += stringBuffer.AppendObject(*view);
      }
   }
   tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FAppState>::Append(::FeatStd::StringBuffer& stringBuffer, const FAppState& appState)
{
   FAppState& val = const_cast<FAppState&>(appState);

   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FAppState { ");

   tcharCount += stringBuffer.Append("Id=");
   tcharCount += stringBuffer.AppendObject(val.getId());

   tcharCount += stringBuffer.Append(", CrtFocus=");
   tcharCount += stringBuffer.AppendObject(val.CurrentFocusInfo);

   tcharCount += stringBuffer.Append(", Views={");
   for (size_t index = 0; index < val.getChildCount(); ++index)
   {
      FViewState* view = val.getChild(index);
      if (view != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n");
         tcharCount += stringBuffer.AppendObject(*view);
      }
   }
   tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FSession>::Append(::FeatStd::StringBuffer& stringBuffer, FSession const& session)
{
   FSession& val = const_cast<FSession&>(session);

   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("FSession { ");

   tcharCount += stringBuffer.Append("State=");
   tcharCount += stringBuffer.AppendObject(static_cast<unsigned int>(val.getStatus()));

   tcharCount += stringBuffer.Append(", ActiveViewGroup=");
   tcharCount += stringBuffer.AppendObject(val.ActiveViewGroup);

   tcharCount += stringBuffer.Append(", Config={");
   for (size_t i = 0; i < val.ConfigInfo.count(); ++i)
   {
      FAppConfig* appConfig = val.ConfigInfo.getAt(i).GetPointerToSharedInstance();
      if (appConfig != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n");
         tcharCount += stringBuffer.AppendObject(*appConfig);
      }
   }
   tcharCount += stringBuffer.Append(" }");

   tcharCount += stringBuffer.Append(", State={");
   for (size_t i = 0; i < val.StateInfo.count(); ++i)
   {
      FAppState* appState = val.StateInfo.getAt(i).GetPointerToSharedInstance();
      if (appState != NULL)
      {
         tcharCount += AppendCurrentAppId(stringBuffer);
         tcharCount += stringBuffer.Append("\n");
         tcharCount += stringBuffer.AppendObject(*appState);
      }
   }
   tcharCount += stringBuffer.Append(" }");

   //if (session.CurrentAppConfig != NULL)
   //{
   //   tcharCount += stringBuffer.AppendObject(*session.CurrentAppConfig);
   //}

   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 StringBufferAppender<FSession*>::Append(::FeatStd::StringBuffer& stringBuffer, FSession* const& session)
{
   if (session == NULL)
   {
      return 0;
   }
   return stringBuffer.AppendObject(*session);
}


template<> ::FeatStd::UInt32 StringBufferAppender<Focus::FAnchorMode::Enum >::Append(::FeatStd::StringBuffer& stringBuffer, Focus::FAnchorMode::Enum const& mode)
{
   ::FeatStd::UInt32 tcharCount = 0;
   switch (mode)
   {
      case Focus::FAnchorMode::None:
         tcharCount += stringBuffer.Append("None");
         break;
      case Focus::FAnchorMode::Container:
         tcharCount += stringBuffer.Append("Container");
         break;
      case Focus::FAnchorMode::Content:
         tcharCount += stringBuffer.Append("Content");
         break;
      default:
         tcharCount += stringBuffer.Append("##unknown##");
         break;
   }
   return tcharCount;
}


}
#endif
