/* ***************************************************************************************
* FILE:          FStateInfo.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FStateInfo.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_STATE_H__
#define __FOCUS_STATE_H__

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"
#include "FContainer.h"
#include "FDataSet.h"
#include "FSharedPointer.h"

namespace Focus {

class FGroupState;
class FViewState;
class FAppState;
class FStateInfo;

//////////////////////////////////////////////////////////////////////////////////////////////
template<typename TId, typename TOwner, typename TChildId, typename TChild>
class FContainerState : public FComponentBase<TId, TOwner>
{
      typedef  FComponentBase<TId, TOwner> Base;
   public:
      typedef FMapContainer<TChildId, TChild> ChildrenType;

      FContainerState(const TId& id) : Base(id), _children(true) {}
      virtual ~FContainerState() {}

      size_t getChildCount() const
      {
         return _children.count();
      }
      TChild* getChild(size_t index) const
      {
         return _children.getAt(index);
      }

      TChild* getChild(const TChildId& id) const
      {
         return _children.get(id);
      }
      TChild* getOrCreateChild(const TChildId& id)
      {
         TChild* child = _children.get(id);
         if (child == NULL)
         {
            child = FOCUS_NEW(TChild)(id);
            _children.add(id, child);
         }
         return child;
      }

      // removes and deletes the child with the specified id if it is found
      void removeChild(const TChildId& id, bool alsoDelete = true)
      {
         _children.remove(id, !alsoDelete);
      }

   private:
      FContainerState(const FContainerState&);
      FContainerState& operator=(const FContainerState&);

      ChildrenType _children;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FStateInfo : public FSharedPtrMapContainer<FAppId, FAppState>
{
   public:
      FStateInfo() {}
      ~FStateInfo() {}

   private:
      FStateInfo(const FStateInfo&);
      FStateInfo& operator=(const FStateInfo&);
};


//////////////////////////////////////////////////////////////////////////////////////////////
struct FAppCurrentFocusInfo
{
   FAppCurrentFocusInfo(const FViewId& viewId = Constants::InvalidViewId, const FId& widgetId = Constants::InvalidId) : ViewId(viewId), WidgetId(widgetId) {}

   FViewId ViewId;
   FId WidgetId;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FAppState : public FContainerState<FAppId, FStateInfo, FViewId, FViewState>
{
      typedef FContainerState<FAppId, FStateInfo, FViewId, FViewState> Base;

   public:
      FAppState(const FAppId& id) : Base(id), CurrentFocusInfo() {}
      virtual ~FAppState() {}

      FAppCurrentFocusInfo CurrentFocusInfo;

   private:
      FAppState(const FAppState&);
      FAppState& operator=(const FAppState&);

      FOCUS_SHARED_POINTER_DECLARATION();
};


typedef FSharedPtr<FAppState> FAppStateSharedPtr;

//////////////////////////////////////////////////////////////////////////////////////////////
struct FViewFocusVisible
{
   FViewFocusVisible(bool visible = false) : Visible(visible) {}

   bool Visible;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FViewState : public FContainerState<FViewId, FAppState, FId, FGroupState>
{
      typedef FContainerState<FViewId, FAppState, FId, FGroupState> Base;

   public:
      FViewState(const FViewId& id) : Base(id)/*, _active(false)*/ {}
      virtual ~FViewState() {}

      //bool isActive() const { return _active; }
      //void setActive(bool active) { _active = active; }

   private:
      FViewState(const FViewState&);
      FViewState& operator=(const FViewState&);

      //bool _active;
};


//////////////////////////////////////////////////////////////////////////////////////////////
struct FGroupCurrentFocusInfo
{
   FGroupCurrentFocusInfo(const FId& currentFocusInfo = Constants::InvalidId) : Id(currentFocusInfo) {}

   FId Id;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FGroupState : public FComponentBase<FId, FViewState>
{
      typedef FComponentBase<FId, FViewState> Base;
   public:
      FGroupState(const FId& id) : Base(id) {}

      FGroupState(const FId& id, const FId& currentFocusId) : Base(id), CurrentFocusInfo(currentFocusId) {}

      FGroupCurrentFocusInfo CurrentFocusInfo;
      //sequence id when the current focus info of this group has been updated
      FSequenceId UpdateSequenceId;
};


}
#endif
