/* ***************************************************************************************
* FILE:          FSharedPointer.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FSharedPointer.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_SHARED_POINTER_H__
#define __FOCUS_SHARED_POINTER_H__

#include <map>
#include <FeatStd/MemoryManagement/SharedPointer.h>

#define FOCUS_SHARED_POINTER_DECLARATION() FEATSTD_SHARED_POINTER_DECLARATION()
#define FSharedPtr FeatStd::MemoryManagement::SharedPointer

namespace Focus {
//   using FeatStd::MemoryManagement::SharedPointer;

//////////////////////////////////////////////////////////////////////////////////////////////
template <typename TChild>
class FSharedPtrVectorContainer
{
      typedef std::vector<FSharedPtr<TChild> > ChildrenType;

   public:
      FSharedPtrVectorContainer() {}
      virtual ~FSharedPtrVectorContainer() {}

      void add(FSharedPtr<TChild> child)
      {
         _children.push_back(child);
      }

      void removeAt(size_t index)
      {
         _children.erase(_children.begin() + index);
      }

      void remove(FSharedPtr<TChild> child)
      {
         FContainerUtils::remove(_children, child);
      }

      void removeAll()
      {
         _children.clear();
      }

      FSharedPtr<TChild> get(size_t index) const
      {
         return (index < count()) ? _children.at(index) : NULL;
      }

      size_t count() const
      {
         return _children.size();
      }

      size_t indexOf(FSharedPtr<TChild> child)
      {
         typename ChildrenType::iterator it = std::find(_children.begin(), _children.end(), child);
         if (it != _children.end())
         {
            return it - _children.begin();
         }
         return Constants::InvalidIndex;
      }

      template <typename TVal>
      size_t indexOf(bool (*func)(FSharedPtr<TChild>, const TVal& val), const TVal& val)
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            if (func(*it, val))
            {
               return it - _children.begin();
            }
         }
         return Constants::InvalidIndex;
      }

      void sort(bool (*compareFunction)(FSharedPtr<TChild>, FSharedPtr<TChild>))
      {
         std::stable_sort(_children.begin(), _children.end(), compareFunction);
      }

      void execute(void (*func)(FSharedPtr<TChild>))
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            func(*it);
         }
      }

      template <typename TVal>
      void execute(void (*func)(FSharedPtr<TChild>, TVal* val), TVal* val)
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            func(*it, val);
         }
      }

   private:
      ChildrenType _children;
};


//////////////////////////////////////////////////////////////////////////////////////////////
template <typename TKey, typename TChild>
class FSharedPtrMapContainer
{
      typedef std::map<TKey, FSharedPtr<TChild> > ChildrenType;

   public:
      FSharedPtrMapContainer() {}
      virtual ~FSharedPtrMapContainer() {}

      size_t count() const
      {
         return _children.size();
      }

      FSharedPtr<TChild> getAt(size_t index) const
      {
         if (index < _children.size())
         {
            typename ChildrenType::const_iterator it = _children.begin();
            std::advance(it, index);
            return it->second;
         }
         return FSharedPtr<TChild>();
      }

      FSharedPtr<TChild> get(const TKey& id) const
      {
         typename ChildrenType::const_iterator it = _children.find(id);
         if (it != _children.end())
         {
            return it->second;
         }
         return FSharedPtr<TChild>();
      }

      void add(const TKey& id, FSharedPtr<TChild> child)
      {
         _children[id] = child;
      }

      void remove(const TKey& id)
      {
         typename ChildrenType::iterator it = _children.find(id);
         if (it != _children.end())
         {
            _children.erase(it);
         }
      }

      void removeAll()
      {
         _children.clear();
      }

      void removeWhere(bool (*func)(const TKey&, FSharedPtr<TChild>))
      {
         typename ChildrenType::iterator it = _children.begin();
         while (it != _children.end())
         {
            if (func(it->first, it->second))
            {
               _children.erase(it++);//advance the iterator than erase using the old iterator
            }
            else
            {
               ++it;
            }
         }
      }

      void execute(void (*func)(const TKey&, FSharedPtr<TChild>))
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            func(it->first, it->second);
         }
      }

      template <typename TVal>
      void execute(void (*func)(const TKey&, FSharedPtr<TChild>, TVal* val), TVal* val)
      {
         for (typename ChildrenType::iterator it = _children.begin(); it != _children.end(); ++it)
         {
            func(it->first, it->second, val);
         }
      }

   private:
      ChildrenType _children;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FSharedPtrUtils
{
   public:
      template <typename TId, typename TItem>
      static FSharedPtr<TItem> get(std::map<TId, FSharedPtr<TItem> >& map, const TId& id)
      {
         typename std::map<TId, FSharedPtr<TItem> >::iterator it = map.find(id);
         if (it != map.end())
         {
            return it->second;
         }
         FSharedPtr<TItem> sharedPtr;
         return sharedPtr;
      }

      template <typename TId, typename TItem>
      static FSharedPtr<TItem> getOrCreate(std::map<TId, FSharedPtr<TItem> >& map, const TId& id)
      {
         typename std::map<TId, FSharedPtr<TItem> >::iterator it = map.find(id);
         if (it != map.end())
         {
            return it->second;
         }
         FSharedPtr<TItem> sharedPtr(FOCUS_NEW(TItem)(id));
         map[id] = sharedPtr;
         return sharedPtr;
      }

      template <typename TId, typename TItem>
      static FSharedPtr<TItem> getAt(std::map<TId, FSharedPtr<TItem> >& map, size_t index)
      {
         if (index < map.size())
         {
            typename std::map<TId, FSharedPtr<TItem> >::const_iterator it = map.begin();
            std::advance(it, index);
            return it->second;
         }
         return FSharedPtr<TItem>();
      }
};


}

#endif //__FOCUS_SHARED_POINTER_H__
