/* ***************************************************************************************
* FILE:          FConfigInfo.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FConfigInfo.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_CONFIG_H__
#define __FOCUS_CONFIG_H__

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"
#include "FContainer.h"
#include "FDataSet.h"
#include "FSharedPointer.h"

namespace Focus {
class FWidgetConfig;
class FViewConfig;
class FAppConfig;
class FConfigInfo;

//////////////////////////////////////////////////////////////////////////////////////////////
template<typename TId, typename TOwner, typename TChildId, typename TChild>
class FContainerConfig : public FComponentBase<TId, TOwner>
{
      typedef FComponentBase<TId, TOwner> Base;

   public:
      typedef FMapContainer<TChildId, TChild> ChildrenType;

      FContainerConfig(const TId& id) : Base(id), _children(true) {}
      virtual ~FContainerConfig() {}

      size_t getChildCount() const
      {
         return _children.count();
      }
      TChild* getChild(size_t index) const
      {
         return _children.getAt(index);
      }

      TChild* getChild(const TChildId& id) const
      {
         return _children.get(id);
      }
      TChild* getOrCreateChild(const TChildId& id)
      {
         TChild* child = _children.get(id);
         if (child == NULL)
         {
            child = createChild(id);
            if (child != NULL)
            {
               _children.add(id, child);
            }
         }
         return child;
      }

   private:
      FContainerConfig(const FContainerConfig&);
      FContainerConfig& operator=(const FContainerConfig&);

      virtual TChild* createChild(const TChildId& id) = 0;

      ChildrenType _children;
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FConfigInfo : public FSharedPtrMapContainer<FAppId, FAppConfig>
{
   public:
      FConfigInfo() {}
      ~FConfigInfo() {}

   private:
      FConfigInfo(const FConfigInfo&);
      FConfigInfo& operator=(const FConfigInfo&);
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FAppConfig : public FContainerConfig<FAppId, FConfigInfo, FViewId, FViewConfig>
{
      typedef FContainerConfig<FAppId, FConfigInfo, FViewId, FViewConfig> Base;

   public:
      FAppConfig(const FAppId& id) : Base(id) {}
      ~FAppConfig() {}

   private:
      FAppConfig(const FAppConfig&);
      FAppConfig& operator=(const FAppConfig&);

      virtual FViewConfig* createChild(const FViewId& id);

      FOCUS_SHARED_POINTER_DECLARATION();
};


typedef FSharedPtr<FAppConfig> FAppConfigSharedPtr;

//////////////////////////////////////////////////////////////////////////////////////////////
class FViewConfig : public FContainerConfig<FViewId, FAppConfig, FId, FWidgetConfig>
{
      typedef FContainerConfig<FViewId, FAppConfig, FId, FWidgetConfig> Base;

   public:
      FViewConfig(const FViewId& id, FAppConfig& app);
      ~FViewConfig() {}

      FAppConfig& App;

   private:
      FViewConfig(const FViewConfig&);
      FViewConfig& operator=(const FViewConfig&);

      virtual FWidgetConfig* createChild(const FId& id);
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FWidgetConfig : public FComponentBase<FId, FViewConfig>
{
      typedef FComponentBase<FId, FViewConfig> Base;

   public:
      FWidgetConfig(const FId& id, FViewConfig& view) : Base(id), View(view), _parent(NULL) {}
      virtual ~FWidgetConfig()
      {
         _parent = NULL;
      }

      FWidgetConfig* getParent() const
      {
         return _parent;
      }
      void setParent(FWidgetConfig* parent)
      {
         _parent = parent;
      }

      FViewConfig& View;

   private:
      FWidgetConfig(const FWidgetConfig&);
      FWidgetConfig& operator=(const FWidgetConfig&);

      FWidgetConfig* _parent;
};


}

#endif
