/* ***************************************************************************************
* FILE:          FActiveViewGroup.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FActiveViewGroup.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_ACTIVE_VIEW_GROUP_H__
#define __FOCUS_ACTIVE_VIEW_GROUP_H__

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/
#include "FCommon.h"
#include "FContainer.h"
#include "FDataSet.h"
#include "FConfigInfo.h"

namespace Focus {
////////////////////////////////////////////////////////////////////////////
class FGroup;
class FocusableVisitor;
class Focusable
{
   public:
      Focusable(const FId& id);
      virtual ~Focusable();

      const FId& getId() const
      {
         return _id;
      }

      FGroup* getParent() const
      {
         return _parent;
      }
      void setParent(FGroup* parent);

      virtual void visit(FocusableVisitor& visitor) = 0;
      virtual FWidgetConfig* getConfig() = 0;

   private:
      Focusable(const Focusable&);
      Focusable& operator=(const Focusable&);

      FId _id;
      FGroup* _parent;
};


////////////////////////////////////////////////////////////////////////////
class FWidgetConfig;
class FWidget : public Focusable
{
      typedef Focusable Base;

   public:
      FWidget(const FId& id, FWidgetConfig& config) : Base(id), Config(config) {}
      virtual ~FWidget() {}

      virtual void visit(FocusableVisitor& visitor);

      virtual FWidgetConfig* getConfig()
      {
         return &Config;
      }

      template <typename TData>
      TData* getData()
      {
         return Config.Data.get<TData>();
      }

      FWidgetConfig& Config;

   private:
      FWidget(const FWidget&);
      FWidget& operator=(const FWidget&);
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FGroup : public Focusable
{
      typedef Focusable Base;

   public:
      FGroup(const FId& id) : Base(id), Children(false), ReachableChildren(false), Config(false) {}
      virtual ~FGroup() {}

      void refreshReachableChildren(FSession& session);

      virtual void visit(FocusableVisitor& visitor);

      virtual FWidgetConfig* getConfig()
      {
         return (Config.count() == 0) ? NULL : Config.get(0);
      }

      template <typename TData>
      TData* getData()
      {
         TData* data = Data.get<TData>();
         if (data != NULL)
         {
            return data;
         }

         for (size_t i = 0; i < Config.count(); ++i)
         {
            FWidgetConfig* config = Config.get(i);
            if (config != NULL)
            {
               data = config->getData<TData>();
               if (data != NULL)
               {
                  return data;
               }
            }
         }

         return NULL;
      }

      FDataSet Data;

      typedef FVectorContainer<Focusable> ChildrenType;
      ChildrenType Children;
      ChildrenType ReachableChildren;

      typedef FVectorContainer<FWidgetConfig> ConfigType;
      ConfigType Config;

   private:
      FGroup(const FGroup&);
      FGroup& operator=(const FGroup&);
};


//////////////////////////////////////////////////////////////////////////////////////////////
class FViewConfig;
class FActiveViewGroup
{
   public:
      FActiveViewGroup();
      ~FActiveViewGroup() {}

      void reset();

      FGroup* getRootGroup()
      {
         return Groups.get(Constants::InvalidId);
      }

      typedef std::vector<FViewId> ViewsType;
      ViewsType Views;

      typedef FMapContainer<FId, FGroup> GroupsType;
      GroupsType Groups;

      typedef FMapContainer<FAppViewWidgetId, FWidget> WidgetsType;
      WidgetsType Widgets;

      typedef std::vector<Focusable*> FocusableListType;
      FocusableListType FocusableList;

      typedef FMapContainer<FAnchorId, FGroup> AnchorsType;
      AnchorsType Anchors;

   private:
      FActiveViewGroup(const FActiveViewGroup&);
      FActiveViewGroup& operator=(const FActiveViewGroup&);
};


}
#endif
