/* ***************************************************************************************
* FILE:          FDefaultSerializer.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultSerializer.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __FOCUS_DEFAULT_SERIALIZER_H__
#define __FOCUS_DEFAULT_SERIALIZER_H__

#include <vector>
#include "Focus/FSerialization.h"

/*****************************************************************************/
/* FOCUS MANAGER INCLUDES                                                    */
/*****************************************************************************/

namespace Focus {
class FAppConfig;
class FAppState;
class FConfigInfo;
class FStateInfo;

namespace Serialize {

/*
 * ItemSer is used to serialize focus info and contains several vectors
 * with simple structures (no pointers, references or complex types but just int, bool).
 * ConfigInfo is serialized using XyzConfigSer items.
 * StateInfo is serialized using XyzStateSer items.
 * Data is serialized using XyzDataSer items.
 * XyzConfigSer, XyzStateSer and XyzDataSer they are not referencing each other.
 * They are only connected to XyzIndexSer items via the ItemIndex field.
 * These index items form a tree similar to the tree formed by the Config/State info.
 * The parent (owner) is indicated by the OwnerIndex.
 *
 *
 * AppConfigSer          ->     AppIndexSer          <-          AppStateSer
 *                  (ItemIndex)      ^           (ItemIndex)
 *                                   | (OwnerIndex)
 *                                   |
 * ViewConfigSer         ->     ViewIndexSer         <-          ViewStateSer
 *                  (ItemIndex)      ^           (ItemIndex)
 *                                   | (OwnerIndex)
 *                                   |
 * WidgetConfigSer       ->    WidgetIndexSer        <-          GroupStateSer
 *                  (ItemIndex)      ^           (ItemIndex)
 *                                   | (OwnerIndex)
 *                                   |
 *       WidgetDataSer, GroupDataSer, ListDataSer, ListItemDataSer
 */

/***********************************************************************/
/* Base structure for index items. */
template <typename TOwnerSer>
struct ItemIndexSerBase
{
   typedef TOwnerSer OwnerSerType;

   ItemIndexSerBase() : OwnerIndex(-1) {}

   /* Index of the parent (owner) index item. App is owner for View and View is owner for Widget. */
   int OwnerIndex;
};


/* Base structure for config and state items. */
template <typename TItemIndexSer>
struct ItemSerBase
{
   typedef TItemIndexSer ItemIndexSerType;

   ItemSerBase() : ItemIndex(-1) {}

   int ItemIndex;
};


/* Base structure for named items. */
template <typename TId>
struct NamedItemSerBase
{
   typedef TId IdType;

   NamedItemSerBase() : NameIndex(-1) {}

   int NameIndex;
};


/* Base structure for data items. */
template <typename TData>
struct DataItemSerBase
{
   typedef TData DataType;

   DataItemSerBase() : Data(TData()) {}

   TData Data;
};


/***********************************************************************/
struct AppIndexSer : public ItemIndexSerBase < void >, public NamedItemSerBase < FAppId > {};
struct AppConfigSer : public ItemSerBase < AppIndexSer > {};


struct AppStateSer : public ItemSerBase < AppIndexSer >
{
   AppStateSer() : CurrentFocusViewNameIndex(-1), CurrentFocusWidgetNameIndex(-1) {}

   int CurrentFocusViewNameIndex;
   int CurrentFocusWidgetNameIndex;
};


/***********************************************************************/
struct ViewIndexSer : public ItemIndexSerBase < AppIndexSer >, public NamedItemSerBase < FViewId > {};
struct ViewConfigSer : public ItemSerBase < ViewIndexSer > {};


struct ViewStateSer : public ItemSerBase < ViewIndexSer >
{
   ViewStateSer() : FocusVisible(false) {}

   bool FocusVisible;
};


/***********************************************************************/
struct WidgetIndexSer : public ItemIndexSerBase < ViewIndexSer >, public NamedItemSerBase < FId > {};


struct WidgetConfigSer : public ItemSerBase < WidgetIndexSer >
{
   WidgetConfigSer() : ParentGroupViewNameIndex(-1), ParentGroupWidgetNameIndex(-1) {}

   int ParentGroupViewNameIndex;
   int ParentGroupWidgetNameIndex;
};


struct GroupStateSer : public ItemSerBase < WidgetIndexSer >
{
   GroupStateSer() : CurrentFocusWidgetNameIndex(-1), SequenceId(-1) {}

   //for now we use widget name index instead of widget index
   int CurrentFocusWidgetNameIndex;
   int SequenceId;
};


/***********************************************************************/
struct CameraDataSer : public ItemSerBase < ViewIndexSer >, public DataItemSerBase < FCameraData > {};
struct WidgetDataSer : public ItemSerBase < WidgetIndexSer >, public DataItemSerBase < FWidgetData > {};
struct GroupDataSer : public ItemSerBase < WidgetIndexSer >, public DataItemSerBase < FGroupData > {};
struct ListDataSer : public ItemSerBase < WidgetIndexSer >, public DataItemSerBase < FListData > {};
struct ListItemDataSer : public ItemSerBase < WidgetIndexSer >, public DataItemSerBase < FListItemData > {};
struct BoundsDataSer : public ItemSerBase < WidgetIndexSer >, public DataItemSerBase < FRectangle > {};

/***********************************************************************/
struct InfoSer
{
   ~InfoSer();

   typedef std::vector<std::string> NamesType;
   NamesType Names;

   /***********************************************************************/
   typedef std::vector<AppIndexSer*> AppIndexType;
   AppIndexType AppIndex;

   typedef std::vector<AppConfigSer*> AppConfigType;
   AppConfigType AppConfig;

   typedef std::vector<AppStateSer*> AppStateType;
   AppStateType AppState;

   /***********************************************************************/
   typedef std::vector<ViewIndexSer*> ViewIndexType;
   ViewIndexType ViewIndex;

   typedef std::vector<ViewConfigSer*> ViewConfigType;
   ViewConfigType ViewConfig;

   typedef std::vector<ViewStateSer*> ViewStateType;
   ViewStateType ViewState;

   /***********************************************************************/
   typedef std::vector<WidgetIndexSer*> WidgetIndexType;
   WidgetIndexType WidgetIndex;

   typedef std::vector<WidgetConfigSer*> WidgetConfigType;
   WidgetConfigType WidgetConfig;

   typedef std::vector<GroupStateSer*> GroupStateType;
   GroupStateType GroupState;

   /***********************************************************************/
   typedef std::vector<CameraDataSer*> CameraDataType;
   CameraDataType CameraData;

   typedef std::vector<WidgetDataSer*> WidgetDataType;
   WidgetDataType WidgetData;

   typedef std::vector<GroupDataSer*> GroupDataType;
   GroupDataType GroupData;

   typedef std::vector<ListDataSer*> ListDataType;
   ListDataType ListData;

   typedef std::vector<ListItemDataSer*> ListItemDataType;
   ListItemDataType ListItemData;

   typedef std::vector<BoundsDataSer*> BoundsDataType;
   BoundsDataType BoundsData;

   /***********************************************************************/
   template <typename TItemSer>
   std::vector<TItemSer*>& getTable();

   template<typename TId>
   TId getIdFromIndex(int index);

   template<typename TItemSer>
   typename TItemSer::IdType getId(TItemSer& itemSer)
   {
      return getIdFromIndex<typename TItemSer::IdType>(itemSer.NameIndex);
   }

   template <typename TItemSer>
   TItemSer* getItem(int index)
   {
      std::vector<TItemSer*>& table = getTable<TItemSer>();
      return ((index >= 0) && (static_cast<size_t>(index) < table.size())) ? table[index] : NULL;
   }

   void clear();
   void printDebug() const;
};


/***********************************************************************/
class FDefaultSerializer : public FSerializer
{
   public:
      FDefaultSerializer(InfoSer* info);
      virtual ~FDefaultSerializer();

      InfoSer* getInfo() const
      {
         return _infoSer;
      }
      void setInfo(InfoSer* info);

      virtual bool serialize(FConfigInfo& configInfo, FStateInfo& stateInfo);
      virtual bool deserialize(FConfigInfo& configInfo, FStateInfo& stateInfo);

   private:
      InfoSer* _infoSer;
};


}
}


#endif
