/* ***************************************************************************************
* FILE:          FDefaultSerializer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultSerializer.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if defined(_lint) && (_lint <= 912) // Versions up to and including 9.00L
//lint -save -e754  PQM_authorized_multi_558  Reason: False positive for template method
#elif defined(_lint)
#error Check if above problems still exist with new Lint version
#endif

#include "gui_std_if.h"
//lint -restore

#include <map>
#include <set>

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "Focus/FConfigInfo.h"
#include "Focus/FStateInfo.h"
#include "Focus/FData.h"
#include "Focus/FUtils.h"
#include "FDefaultSerializer.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultSerializer.cpp.trc.h"
#endif


namespace Focus {
namespace Serialize {

/*****************************************************************************/
template <> FAppId InfoSer::getIdFromIndex(int nameIndex)
{
   if ((nameIndex > 0) && (nameIndex < static_cast<int>(this->Names.size())))
   {
      int appId;
      if (sscanf(this->Names[nameIndex].c_str(), "%d", &appId) > 0)            //lint !e732
      {
         return appId;
      }
   }
   return Constants::InvalidAppId;
}


/*****************************************************************************/
template <> FViewId InfoSer::getIdFromIndex(int nameIndex)
{
   if ((nameIndex > 0) && (nameIndex < static_cast<int>(this->Names.size())))
   {
      //view id does not allocate the string, it just keeps a copy of the pointer
      //for this reason we can not use the temporary strings
      static std::set<std::string> viewNames;
      std::set<std::string>::iterator it = viewNames.insert(this->Names[nameIndex]).first; //lint !e732
      if (it != viewNames.end())
      {
         return FViewId((*it).c_str());
      }
   }
   return Constants::InvalidViewId;
}


/*****************************************************************************/
template <> FId InfoSer::getIdFromIndex(int nameIndex)
{
   if ((nameIndex > 0) && (nameIndex < static_cast<int>(this->Names.size())))
   {
      return this->Names[nameIndex];                                           //lint !e732
   }
   return Constants::InvalidId;
}


/*****************************************************************************/
#define INFO_SER_GET_TABLE(table)\
template <> std::vector<table##Ser*>& InfoSer::getTable<table##Ser>()\
{\
   return this->table;\
}


INFO_SER_GET_TABLE(AppIndex)
INFO_SER_GET_TABLE(AppConfig)
INFO_SER_GET_TABLE(AppState)
INFO_SER_GET_TABLE(ViewIndex)
INFO_SER_GET_TABLE(ViewConfig)
INFO_SER_GET_TABLE(ViewState)
INFO_SER_GET_TABLE(WidgetIndex)
INFO_SER_GET_TABLE(WidgetConfig)
INFO_SER_GET_TABLE(GroupState)
INFO_SER_GET_TABLE(CameraData)
INFO_SER_GET_TABLE(WidgetData)
INFO_SER_GET_TABLE(GroupData)
INFO_SER_GET_TABLE(ListData)
INFO_SER_GET_TABLE(ListItemData)
INFO_SER_GET_TABLE(BoundsData)


/*****************************************************************************/
class FNameTable
{
   public:
      FNameTable() : _needsIndicesUpdate(false) {}

      void clear();
      int indexOf(const std::string& name);
      void copy(std::vector<std::string>& vector);
      void addName(const std::string& name);

   private:
      FNameTable(const FNameTable&);
      FNameTable& operator=(const FNameTable&);

      void updateIndices();

      typedef std::map<std::string, int> NamesType;
      NamesType _names;
      bool _needsIndicesUpdate;
};


/*****************************************************************************/
void FNameTable::clear()
{
   _names.clear();
}


/*****************************************************************************/
int FNameTable::indexOf(const std::string& name)
{
   if (_needsIndicesUpdate)
   {
      updateIndices();
   }

   NamesType::const_iterator it = _names.find(name);
   return (it != _names.end()) ? (*it).second : -1;
}


/*****************************************************************************/
void FNameTable::copy(std::vector<std::string>& vector)
{
   if (_needsIndicesUpdate)
   {
      updateIndices();
   }

   for (NamesType::const_iterator it = _names.begin(); it != _names.end(); ++it)
   {
      vector.push_back((*it).first);
   }
}


/*****************************************************************************/
void FNameTable::addName(const std::string& name)
{
   _names[name] = -1;
   _needsIndicesUpdate = true;
}


/*****************************************************************************/
void FNameTable::updateIndices()
{
   int index = 0;
   for (NamesType::iterator it = _names.begin(); it != _names.end(); ++it)
   {
      (*it).second = index++;
   }
   _needsIndicesUpdate = false;
}


/*****************************************************************************/
class FSerializeUtils
{
   public:
      /*****************************************************************************/
      static bool serialize(InfoSer& infoSer, FAppConfig& appConfig, FAppState& appState);

      /*****************************************************************************/
      static bool serialize(InfoSer& infoSer, const FConfigInfo& configInfo, const FStateInfo& stateInfo);

   protected:
      /*****************************************************************************/
      template <typename TId>
      static std::string getName(const TId& id);

      /***********************************************************************/
      /* COLLECT NAMES                                                       */
      /***********************************************************************/
      template <typename TItemConfigOrState>
      static void collectNames(FNameTable& nameTable, TItemConfigOrState& itemConfigOrState)
      {
         collectSelfNames(nameTable, itemConfigOrState);
         collectChildrenNames(nameTable, itemConfigOrState);
      }

      /*****************************************************************************/
      template <typename TItemConfigOrState>
      static void collectSelfNames(FNameTable& nameTable, TItemConfigOrState& itemConfigOrState)
      {
         collectSelfNamesBase(nameTable, itemConfigOrState);
      }

      /*****************************************************************************/
      template <typename TItemConfigOrState>
      static void collectSelfNamesBase(FNameTable& nameTable, TItemConfigOrState& itemConfigOrState)
      {
         nameTable.addName(getName(itemConfigOrState.getId()));
      }

      /*****************************************************************************/
      template <typename TItemConfigOrState>
      static void collectChildrenNames(FNameTable& nameTable, TItemConfigOrState& itemConfigOrState)
      {
         for (size_t i = 0; i < itemConfigOrState.getChildCount(); ++i)
         {
            collectNames(nameTable, itemConfigOrState.getChild(i));
         }
      }

      /*****************************************************************************/
      template <typename TItemConfigOrState>
      static void collectNames(FNameTable& nameTable, TItemConfigOrState* itemConfigOrState)
      {
         if (itemConfigOrState != NULL)
         {
            collectNames(nameTable, *itemConfigOrState);
         }
      }

      /***********************************************************************/
      /* SERIALIZE ITEM CONFIG                                               */
      /***********************************************************************/
      template <typename TItemConfigOrState>
      static void serialize(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, TItemConfigOrState& itemConfigOrState)
      {
         int selfIndex = serializeSelf(infoSer, nameTable, ownerIndex, itemConfigOrState);
         serializeChildren(infoSer, nameTable, selfIndex, itemConfigOrState);
      }

      /*****************************************************************************/
      template <typename TItemIndexSer, typename TItemConfigOrState>
      static int serializeIndex(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, TItemConfigOrState& itemConfigOrState)
      {
         int selfIndex = -1;
         TItemIndexSer* itemIndexSer = FOCUS_NEW(TItemIndexSer);
         if (itemIndexSer != NULL)
         {
            itemIndexSer->NameIndex = nameTable.indexOf(getName(itemConfigOrState.getId()));
            itemIndexSer->OwnerIndex = ownerIndex;

            selfIndex = static_cast<int>(infoSer.getTable<TItemIndexSer>().size());
            infoSer.getTable<TItemIndexSer>().push_back(itemIndexSer);
         }

         return selfIndex;
      }

      /*****************************************************************************/
      template <typename TItemSer, typename TItemConfigOrState>
      static TItemSer* serializeSelfBase(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, TItemConfigOrState& itemConfigOrState, int& selfIndex)
      {
         TItemSer* itemSer = NULL;
         selfIndex = serializeIndex<typename TItemSer::ItemIndexSerType>(infoSer, nameTable, ownerIndex, itemConfigOrState);
         if (selfIndex != -1)
         {
            itemSer = FOCUS_NEW(TItemSer);
            if (itemSer != NULL)
            {
               infoSer.getTable<TItemSer>().push_back(itemSer);
               itemSer->ItemIndex = selfIndex;
            }
         }
         return itemSer;
      }

      /*****************************************************************************/
      template <typename TItemConfigOrState>
      static int serializeSelf(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, TItemConfigOrState& itemConfigOrState);

      /*****************************************************************************/
      template <typename TItemConfigOrState>
      static void serializeChildren(InfoSer& infoSer, FNameTable& nameTable, int selfIndex, TItemConfigOrState& itemConfigOrState)
      {
         for (size_t i = 0; i < itemConfigOrState.getChildCount(); ++i)
         {
            serialize(infoSer, nameTable, selfIndex, itemConfigOrState.getChild(i));
         }
      }

      /*****************************************************************************/
      template <typename TItemConfigOrState>
      static void serialize(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, TItemConfigOrState* itemConfigOrState)
      {
         if (itemConfigOrState != NULL)
         {
            serialize(infoSer, nameTable, ownerIndex, *itemConfigOrState);
         }
      }

      /***********************************************************************/
      /* SERIALIZE DATA                                                      */
      /***********************************************************************/
      template <typename TItemDataSer>
      static void serializeData(std::vector<TItemDataSer*>& table, FDataSet& dataSet, int selfIndex)
      {
         typename TItemDataSer::DataType* data = dataSet.get<typename TItemDataSer::DataType>();
         serializeData(table, data, selfIndex);
      }

      /*****************************************************************************/
      template <typename TItemDataSer>
      static void serializeData(std::vector<TItemDataSer*>& table, typename TItemDataSer::DataType* data, int selfIndex)
      {
         if (data != NULL)
         {
            TItemDataSer* dataSer = FOCUS_NEW(TItemDataSer);
            if (dataSer != NULL)
            {
               dataSer->ItemIndex = selfIndex;
               table.push_back(dataSer);

               updateDataSer(*dataSer, *data);
            }
         }
      }

      /*****************************************************************************/
      template <typename TItemDataSer, typename TData>
      static void updateDataSer(TItemDataSer& itemDataSer, TData& data)
      {
         itemDataSer.Data = data;
      }
};


/*****************************************************************************/
template <> std::string FSerializeUtils::getName<FAppId>(const FAppId& id)
{
   char buffer[sizeof(FAppId) * 3 + 1];

   int result = SNPRINTF(buffer, sizeof(buffer), "%d", id);
   if ((result < 0) || (result >= static_cast<int>(sizeof(buffer))))
   {
      buffer[0] = 0;
   }
   return buffer;
}


/*****************************************************************************/
template <> std::string FSerializeUtils::getName<FViewId>(const FViewId& id)
{
   return id.CStr();
}


/*****************************************************************************/
template <> std::string FSerializeUtils::getName<FId>(const FId& id)
{
   return id;
}


/*****************************************************************************/
template <> void FSerializeUtils::collectSelfNames<FAppState>(FNameTable& nameTable, FAppState& appState)
{
   collectSelfNamesBase(nameTable, appState);
   nameTable.addName(getName(appState.CurrentFocusInfo.ViewId));
   nameTable.addName(getName(appState.CurrentFocusInfo.WidgetId));
}


/*****************************************************************************/
template <> void FSerializeUtils::collectSelfNames<FGroupState>(FNameTable& nameTable, FGroupState& groupState)
{
   collectSelfNamesBase(nameTable, groupState);
   nameTable.addName(getName(groupState.CurrentFocusInfo.Id));
}


/*****************************************************************************/
template <> void FSerializeUtils::collectChildrenNames<FWidgetConfig>(FNameTable&, FWidgetConfig&)
{
   //no children for widget config
}


/*****************************************************************************/
template <> void FSerializeUtils::collectChildrenNames<FGroupState>(FNameTable&, FGroupState&)
{
   //no children for group state
}


/*****************************************************************************/
template <> int FSerializeUtils::serializeSelf<FAppConfig>(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, FAppConfig& itemConfigOrState)
{
   int selfIndex;
   AppConfigSer* itemConfigOrStateSer = serializeSelfBase<AppConfigSer>(infoSer, nameTable, ownerIndex, itemConfigOrState, selfIndex);
   if (itemConfigOrStateSer != NULL)
   {
      //nothing yet
   }
   return selfIndex;
}


/*****************************************************************************/
template <> int FSerializeUtils::serializeSelf<FAppState>(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, FAppState& itemConfigOrState)
{
   int selfIndex;
   AppStateSer* itemConfigOrStateSer = serializeSelfBase<AppStateSer>(infoSer, nameTable, ownerIndex, itemConfigOrState, selfIndex);
   if (itemConfigOrStateSer != NULL)
   {
      itemConfigOrStateSer->CurrentFocusViewNameIndex = nameTable.indexOf(getName(itemConfigOrState.CurrentFocusInfo.ViewId));
      itemConfigOrStateSer->CurrentFocusWidgetNameIndex = nameTable.indexOf(getName(itemConfigOrState.CurrentFocusInfo.WidgetId));
   }
   return selfIndex;
}


/*****************************************************************************/
template <> int FSerializeUtils::serializeSelf<FViewConfig>(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, FViewConfig& itemConfigOrState)
{
   int selfIndex;
   ViewConfigSer* itemConfigOrStateSer = serializeSelfBase<ViewConfigSer>(infoSer, nameTable, ownerIndex, itemConfigOrState, selfIndex);
   if (itemConfigOrStateSer != NULL)
   {
      FCameraDataVector* cameraDataVector = itemConfigOrState.getData<FCameraDataVector>();
      if (cameraDataVector != NULL)
      {
         for (FCameraDataVector::iterator it = cameraDataVector->begin(); it != cameraDataVector->end(); ++it)
         {
            serializeData(infoSer.CameraData, &(*it), selfIndex);
         }
      }
   }
   return selfIndex;
}


/*****************************************************************************/
template <> int FSerializeUtils::serializeSelf<FViewState>(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, FViewState& itemConfigOrState)
{
   int selfIndex;
   ViewStateSer* itemConfigOrStateSer = serializeSelfBase<ViewStateSer>(infoSer, nameTable, ownerIndex, itemConfigOrState, selfIndex);
   if (itemConfigOrStateSer != NULL)
   {
      FViewFocusVisible* focusVisible = itemConfigOrState.Data.get<FViewFocusVisible>();
      itemConfigOrStateSer->FocusVisible = (focusVisible != NULL) && focusVisible->Visible;
   }
   return selfIndex;
}


/*****************************************************************************/
template <> int FSerializeUtils::serializeSelf<FWidgetConfig>(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, FWidgetConfig& itemConfigOrState)
{
   int selfIndex;
   WidgetConfigSer* itemConfigOrStateSer = serializeSelfBase<WidgetConfigSer>(infoSer, nameTable, ownerIndex, itemConfigOrState, selfIndex);
   if (itemConfigOrStateSer != NULL)
   {
      FWidgetConfig* parentGroup = itemConfigOrState.getParent();
      if (parentGroup != NULL)
      {
         itemConfigOrStateSer->ParentGroupViewNameIndex = nameTable.indexOf(getName(parentGroup->View.getId()));
         itemConfigOrStateSer->ParentGroupWidgetNameIndex = nameTable.indexOf(getName(parentGroup->getId()));
      }

      serializeData(infoSer.WidgetData, itemConfigOrState.Data, selfIndex);
      serializeData(infoSer.GroupData, itemConfigOrState.Data, selfIndex);
      serializeData(infoSer.ListData, itemConfigOrState.Data, selfIndex);
      serializeData(infoSer.ListItemData, itemConfigOrState.Data, selfIndex);
      serializeData(infoSer.BoundsData, itemConfigOrState.Data, selfIndex);
   }
   return selfIndex;
}


/*****************************************************************************/
template <> void FSerializeUtils::serializeChildren<FWidgetConfig>(InfoSer&, FNameTable&, int, FWidgetConfig&)
{
   //no children for widget config
}


/*****************************************************************************/
template <> int FSerializeUtils::serializeSelf<FGroupState>(InfoSer& infoSer, FNameTable& nameTable, int ownerIndex, FGroupState& itemConfigOrState)
{
   int selfIndex;
   GroupStateSer* itemConfigOrStateSer = serializeSelfBase<GroupStateSer>(infoSer, nameTable, ownerIndex, itemConfigOrState, selfIndex);
   if (itemConfigOrStateSer != NULL)
   {
      itemConfigOrStateSer->SequenceId = itemConfigOrState.UpdateSequenceId.Id;
      //for now we use widget name instead of widget index
      itemConfigOrStateSer->CurrentFocusWidgetNameIndex = nameTable.indexOf(getName(itemConfigOrState.CurrentFocusInfo.Id));
   }
   return selfIndex;
}


/*****************************************************************************/
template <> void FSerializeUtils::serializeChildren<FGroupState>(InfoSer&, FNameTable&, int, FGroupState&)
{
   //no children for group state
}


/*****************************************************************************/
bool FSerializeUtils::serialize(InfoSer& infoSer, FAppConfig& appConfig, FAppState& appState)
{
   infoSer.clear();

   FNameTable nameTable;
   collectNames(nameTable, appConfig);
   collectNames(nameTable, appState);
   nameTable.copy(infoSer.Names);

   serialize(infoSer, nameTable, -1, appConfig);
   serialize(infoSer, nameTable, -1, appState);

   return true;
}


/*****************************************************************************/
bool FSerializeUtils::serialize(InfoSer& infoSer, const FConfigInfo& configInfo, const FStateInfo& stateInfo)
{
   infoSer.clear();

   //prepare the name table first
   FNameTable nameTable;
   for (size_t i = 0; i < configInfo.count(); ++i)
   {
      FAppConfigSharedPtr appConfig = configInfo.getAt(i);
      if (!appConfig.PointsToNull())
      {
         collectNames(nameTable, *appConfig);
      }
   }
   for (size_t i = 0; i < stateInfo.count(); ++i)
   {
      FAppStateSharedPtr appState = stateInfo.getAt(i);
      if (!appState.PointsToNull())
      {
         collectNames(nameTable, *appState);
      }
   }
   nameTable.copy(infoSer.Names);

   //proceed with the serialization of the items
   for (size_t i = 0; i < configInfo.count(); ++i)
   {
      FAppConfigSharedPtr appConfig = configInfo.getAt(i);
      if (!appConfig.PointsToNull())
      {
         serialize(infoSer, nameTable, -1, *appConfig);
      }
   }
   for (size_t i = 0; i < stateInfo.count(); ++i)
   {
      FAppStateSharedPtr appState = stateInfo.getAt(i);
      if (!appState.PointsToNull())
      {
         serialize(infoSer, nameTable, -1, *appState);
      }
   }
   return true;
}


/*****************************************************************************/
class FDeserializeUtils
{
   public:
      /*****************************************************************************/
      static bool deserialize(InfoSer& infoSer, FConfigInfo& configInfo, FStateInfo& stateInfo);

   private:
      /*****************************************************************************/
      template<typename TItemConfigOrState, typename TItemIndexSer, typename TConfigOrStateTable>
      static TItemConfigOrState* getItemConfigOrState(InfoSer& infoSer, TItemIndexSer& itemIndexSer, TConfigOrStateTable& configOrStateInfo)
      {
         typename TItemIndexSer::OwnerSerType* ownerSer = infoSer.getItem<typename TItemIndexSer::OwnerSerType>(itemIndexSer.OwnerIndex);
         if (ownerSer != NULL)
         {
            typename TItemConfigOrState::OwnerType* ownerConfigOrState = getItemConfigOrState<typename TItemConfigOrState::OwnerType, typename TItemIndexSer::OwnerSerType>(infoSer, *ownerSer, configOrStateInfo);
            if (ownerConfigOrState != NULL)
            {
               typename TItemIndexSer::IdType itemId = infoSer.getId(itemIndexSer);
               TItemConfigOrState* itemConfigOrState = ownerConfigOrState->getOrCreateChild(itemId);
               return itemConfigOrState;
            }
         }
         return NULL;
      }

      /*****************************************************************************/
      template<typename TOwnerConfigOrState, typename TItemSer, typename TConfigOrStateTable>
      static TOwnerConfigOrState* getOwnerConfigOrStateBase(InfoSer& infoSer, TItemSer& itemSer, TConfigOrStateTable& configOrStateInfo)
      {
         typename TItemSer::ItemIndexSerType* itemIndexSer = infoSer.getItem<typename TItemSer::ItemIndexSerType>(itemSer.ItemIndex);
         if (itemIndexSer != NULL)
         {
            return getItemConfigOrState<TOwnerConfigOrState>(infoSer, *itemIndexSer, configOrStateInfo);
         }
         return NULL;
      }

      /*****************************************************************************/
      template <typename TItemConfigOrState, typename TItemSer, typename TConfigOrStateTable>
      static void updateItem(InfoSer& infoSer, TItemConfigOrState& itemConfigOrState, const TItemSer& itemSer, TConfigOrStateTable& configOrStateInfo)
      {
         //no default update
         (void)infoSer;
         (void)itemConfigOrState;
         (void)itemSer;
         (void)configOrStateInfo;
      }

      /*****************************************************************************/
      template <typename TItemSer, typename TItemConfigOrState, typename TConfigOrStateTable>
      static void deserializeItemTable(InfoSer& infoSer, TConfigOrStateTable& configOrStateInfo)
      {
         typename std::vector<TItemSer*>& v = infoSer.getTable<TItemSer>();
         for (typename std::vector<TItemSer*>::iterator it = v.begin(); it != v.end(); ++it)
         {
            TItemSer* itemSer = *it;
            if (itemSer != NULL)
            {
               TItemConfigOrState* itemConfigOrState = getOwnerConfigOrStateBase<TItemConfigOrState>(infoSer, *itemSer, configOrStateInfo);
               if (itemConfigOrState != NULL)
               {
                  updateItem(infoSer, *itemConfigOrState, *itemSer, configOrStateInfo);
               }
            }
         }
      }
};


/*****************************************************************************/
template<typename TAppConfigOrState, typename TConfigOrStateTable>
static TAppConfigOrState* getAppConfigOrStateBase(InfoSer& infoSer, AppIndexSer& appSer, TConfigOrStateTable& configOrStateInfo)
{
   FAppId appId = infoSer.getId(appSer);
   FSharedPtr<TAppConfigOrState> appConfigOrState = configOrStateInfo.get(appId);
   if (appConfigOrState.PointsToNull())
   {
      appConfigOrState = FSharedPtr<TAppConfigOrState>(FOCUS_NEW(TAppConfigOrState)(appId));
      if (!appConfigOrState.PointsToNull())
      {
         configOrStateInfo.add(appId, appConfigOrState);
      }
   }
   return appConfigOrState.GetPointerToSharedInstance();
}


/*****************************************************************************/
template<>
FAppConfig* FDeserializeUtils::getItemConfigOrState<FAppConfig, AppIndexSer, FConfigInfo>(InfoSer& infoSer, AppIndexSer& appSer, FConfigInfo& configInfo)
{
   return getAppConfigOrStateBase<FAppConfig>(infoSer, appSer, configInfo);
}


/*****************************************************************************/
template<>
FAppState* FDeserializeUtils::getItemConfigOrState<FAppState, AppIndexSer, FStateInfo>(InfoSer& infoSer, AppIndexSer& appSer, FStateInfo& stateInfo)
{
   return getAppConfigOrStateBase<FAppState>(infoSer, appSer, stateInfo);
}


/*****************************************************************************/
//lint -esym(1764, stateInfo) TODO: Remove this line as soon as parameter stateInfo is used in this function
template <>
void FDeserializeUtils::updateItem<FAppState, AppStateSer, FStateInfo>(InfoSer& infoSer, FAppState& appState, const AppStateSer& appStateSer, FStateInfo& stateInfo)
{
   (void)stateInfo;

   appState.CurrentFocusInfo.ViewId = infoSer.getIdFromIndex<FViewId>(appStateSer.CurrentFocusViewNameIndex);
   appState.CurrentFocusInfo.WidgetId = infoSer.getIdFromIndex<FId>(appStateSer.CurrentFocusWidgetNameIndex);
}


/*****************************************************************************/
//lint -esym(1764, infoSer) TODO: Remove this line as soon as parameter infoSer is used in functions below
//lint -esym(1764, configInfo) TODO: Remove this line as soon as parameter configInfo is used in functions below
//lint -esym(1764, itemConfig) TODO: Remove this line as soon as parameter itemConfig is used in functions below
template <>
void FDeserializeUtils::updateItem<FViewState, ViewStateSer, FStateInfo>(InfoSer& infoSer, FViewState& viewState, const ViewStateSer& viewStateSer, FStateInfo& stateInfo)
{
   (void)infoSer;
   (void)stateInfo;

   viewState.Data.set(FViewFocusVisible(viewStateSer.FocusVisible));
}


/*****************************************************************************/
template <>
void FDeserializeUtils::updateItem<FViewConfig, CameraDataSer, FConfigInfo>(InfoSer& infoSer, FViewConfig& itemConfig, const CameraDataSer& dataSer, FConfigInfo& configInfo)
{
   (void)infoSer;
   (void)configInfo;
   FCameraDataVector* cameraDataVector = itemConfig.getOrCreateData<FCameraDataVector>();
   if (cameraDataVector != NULL)
   {
      cameraDataVector->push_back(dataSer.Data);
   }
}


/*****************************************************************************/
template <>
void FDeserializeUtils::updateItem<FWidgetConfig, WidgetConfigSer, FConfigInfo>(InfoSer& infoSer, FWidgetConfig& itemConfig, const WidgetConfigSer& itemSer, FConfigInfo& configInfo)
{
   (void)infoSer;
   (void)itemConfig;
   (void)itemSer;
   (void)configInfo;

   //we can't update the parent now because it may be deserialized after this item
}


/*****************************************************************************/
template <>
void FDeserializeUtils::updateItem<FGroupState, GroupStateSer, FStateInfo>(InfoSer& infoSer, FGroupState& groupState, const GroupStateSer& groupStateSer, FStateInfo& stateInfo)
{
   (void)stateInfo;
   groupState.UpdateSequenceId.Id = groupStateSer.SequenceId;

   //for now we use widget name instead of widget index
   groupState.CurrentFocusInfo.Id = infoSer.getIdFromIndex<FId>(groupStateSer.CurrentFocusWidgetNameIndex);
}


/*****************************************************************************/
template <>
void FDeserializeUtils::updateItem<FWidgetConfig, WidgetDataSer, FConfigInfo>(InfoSer& infoSer, FWidgetConfig& itemConfig, const WidgetDataSer& dataSer, FConfigInfo& configInfo)
{
   (void)infoSer;
   (void)configInfo;
   itemConfig.setData(dataSer.Data);
}


/*****************************************************************************/
template <>
void FDeserializeUtils::updateItem<FWidgetConfig, GroupDataSer, FConfigInfo>(InfoSer& infoSer, FWidgetConfig& itemConfig, const GroupDataSer& dataSer, FConfigInfo& configInfo)
{
   (void)infoSer;
   (void)configInfo;
   itemConfig.setData(dataSer.Data);
}


/*****************************************************************************/
template <>
void FDeserializeUtils::updateItem<FWidgetConfig, ListDataSer, FConfigInfo>(InfoSer& infoSer, FWidgetConfig& itemConfig, const ListDataSer& dataSer, FConfigInfo& configInfo)
{
   (void)infoSer;
   (void)configInfo;
   itemConfig.setData(dataSer.Data);
}


/*****************************************************************************/
template <>
void FDeserializeUtils::updateItem<FWidgetConfig, ListItemDataSer, FConfigInfo>(InfoSer& infoSer, FWidgetConfig& itemConfig, const ListItemDataSer& dataSer, FConfigInfo& configInfo)
{
   (void)infoSer;
   (void)configInfo;
   itemConfig.setData(dataSer.Data);
}


/*****************************************************************************/
template <>
void FDeserializeUtils::updateItem<FWidgetConfig, BoundsDataSer, FConfigInfo>(InfoSer& infoSer, FWidgetConfig& itemConfig, const BoundsDataSer& dataSer, FConfigInfo& configInfo)
{
   (void)infoSer;
   (void)configInfo;
   itemConfig.setData(dataSer.Data);
}


/*****************************************************************************/
bool FDeserializeUtils::deserialize(InfoSer& infoSer, FConfigInfo& configInfo, FStateInfo& stateInfo)
{
   deserializeItemTable<AppConfigSer, FAppConfig>(infoSer, configInfo);
   deserializeItemTable<ViewConfigSer, FViewConfig>(infoSer, configInfo);
   deserializeItemTable<WidgetConfigSer, FWidgetConfig>(infoSer, configInfo);

   for (InfoSer::WidgetConfigType::iterator it = infoSer.WidgetConfig.begin(); it != infoSer.WidgetConfig.end(); ++it)
   {
      WidgetConfigSer* widgetSer = *it;
      if ((widgetSer != NULL) && (widgetSer->ParentGroupViewNameIndex != -1) && ((widgetSer->ParentGroupWidgetNameIndex != -1)))
      {
         FWidgetConfig* widgetConfig = getOwnerConfigOrStateBase<FWidgetConfig>(infoSer, *widgetSer, configInfo);
         if (widgetConfig != NULL)
         {
            ViewIndexSer parentGroupViewIndexSer;
            parentGroupViewIndexSer.NameIndex = widgetSer->ParentGroupViewNameIndex;
            FViewConfig* parentGroupView = widgetConfig->View.App.getChild(infoSer.getId(parentGroupViewIndexSer));
            if (parentGroupView != NULL)
            {
               WidgetIndexSer parentGroupWidgetIndexSer;
               parentGroupWidgetIndexSer.NameIndex = widgetSer->ParentGroupWidgetNameIndex;
               FWidgetConfig* parentGroup = parentGroupView->getChild(infoSer.getId(parentGroupWidgetIndexSer));
               if ((parentGroup != NULL) && (parentGroup != widgetConfig))
               {
                  widgetConfig->setParent(parentGroup);
               }
            }
         }
      }
   }

   deserializeItemTable<CameraDataSer, FViewConfig>(infoSer, configInfo);

   deserializeItemTable<WidgetDataSer, FWidgetConfig>(infoSer, configInfo);
   deserializeItemTable<GroupDataSer, FWidgetConfig>(infoSer, configInfo);
   deserializeItemTable<ListDataSer, FWidgetConfig>(infoSer, configInfo);
   deserializeItemTable<ListItemDataSer, FWidgetConfig>(infoSer, configInfo);
   deserializeItemTable<BoundsDataSer, FWidgetConfig>(infoSer, configInfo);

   deserializeItemTable<AppStateSer, FAppState>(infoSer, stateInfo);
   deserializeItemTable<ViewStateSer, FViewState>(infoSer, stateInfo);
   deserializeItemTable<GroupStateSer, FGroupState>(infoSer, stateInfo);

   return true;
}


/*****************************************************************************/
template <typename TItem>
static void clearVector(std::vector<TItem*>& v)
{
   for (typename std::vector<TItem*>::iterator it = v.begin(); it != v.end(); ++it)
   {
      TItem* item = *it;
      if (item != NULL)
      {
         FOCUS_DELETE(item);
      }
   }
   v.clear();
}


/*****************************************************************************/
InfoSer::~InfoSer()
{
   clear();
}


/*****************************************************************************/
void InfoSer::clear()
{
   Names.clear();

   clearVector(AppIndex);
   clearVector(AppConfig);
   clearVector(AppState);

   clearVector(ViewIndex);
   clearVector(ViewConfig);
   clearVector(ViewState);

   clearVector(WidgetIndex);
   clearVector(WidgetConfig);
   clearVector(GroupState);

   clearVector(CameraData);
   clearVector(WidgetData);
   clearVector(GroupData);
   clearVector(ListData);
   clearVector(ListItemData);
   clearVector(BoundsData);
}


/*****************************************************************************/
void InfoSer::printDebug() const
{
   FUtils::printItemTrace(*this);
}


/*****************************************************************************/
FDefaultSerializer::FDefaultSerializer(InfoSer* info) : _infoSer(info)
{
}


/*****************************************************************************/
FDefaultSerializer::~FDefaultSerializer()
{
   _infoSer = NULL;
}


/*****************************************************************************/
void FDefaultSerializer::setInfo(InfoSer* info)
{
   _infoSer = info;
}


/*****************************************************************************/
bool FDefaultSerializer::serialize(FConfigInfo& configInfo, FStateInfo& stateInfo)
{
   bool result = false;
   if (_infoSer != NULL)
   {
      result = FSerializeUtils::serialize(*_infoSer, configInfo, stateInfo);
      _infoSer->printDebug();
   }
   return result;
}


/*****************************************************************************/
bool FDefaultSerializer::deserialize(FConfigInfo& configInfo, FStateInfo& stateInfo)
{
   bool result = false;
   if (_infoSer != NULL)
   {
      _infoSer->printDebug();
      result = FDeserializeUtils::deserialize(*_infoSer, configInfo, stateInfo);
   }
   return result;
}


}//namespace Serialize
}//namespace Focus
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

using namespace Focus;
using namespace Focus::Serialize;

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
/*****************************************************************************/
template <typename TItemSer>
static ::FeatStd::UInt32 appendVector(::FeatStd::StringBuffer& stringBuffer, const std::vector<TItemSer*>& table, const char* tableName)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append(tableName);
   tcharCount += stringBuffer.Append(" { ");
   for (typename std::vector<TItemSer*>::const_iterator it = table.begin(); it != table.end(); ++it)
   {
      tcharCount += stringBuffer.Append(" { ");
      TItemSer* item = *it;
      if (item != NULL)
      {
         tcharCount += stringBuffer.AppendObject(*item);
      }
      else
      {
         tcharCount += stringBuffer.Append("<null>");
      }
      tcharCount += stringBuffer.Append(" },\n");
   }
   tcharCount += stringBuffer.Append(" },\n");
   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<InfoSer>::Append(::FeatStd::StringBuffer& stringBuffer, const InfoSer& valConst)
{
   InfoSer& val = const_cast<InfoSer&>(valConst);

   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("InfoSer { ");

   tcharCount += stringBuffer.Append(" Names { ");
   for (std::vector<std::string>::const_iterator it = val.Names.begin(); it != val.Names.end(); ++it)
   {
      tcharCount += stringBuffer.Append((*it).c_str());
      tcharCount += stringBuffer.Append(", ");
   }
   tcharCount += stringBuffer.Append(" },\n");

   tcharCount += appendVector(stringBuffer, val.AppIndex, "AppIndex");
   tcharCount += appendVector(stringBuffer, val.ViewIndex, "ViewIndex");
   tcharCount += appendVector(stringBuffer, val.WidgetIndex, "WidgetIndex");

   tcharCount += appendVector(stringBuffer, val.AppConfig, "AppConfig");
   tcharCount += appendVector(stringBuffer, val.ViewConfig, "ViewConfig");
   tcharCount += appendVector(stringBuffer, val.WidgetConfig, "WidgetConfig");

   tcharCount += appendVector(stringBuffer, val.CameraData, "CameraData");
   tcharCount += appendVector(stringBuffer, val.WidgetData, "WidgetData");
   tcharCount += appendVector(stringBuffer, val.GroupData, "GroupData");
   tcharCount += appendVector(stringBuffer, val.ListData, "ListData");
   tcharCount += appendVector(stringBuffer, val.ListItemData, "ListItemData");
   tcharCount += appendVector(stringBuffer, val.BoundsData, "BoundsData");

   tcharCount += appendVector(stringBuffer, val.AppState, "AppState");
   tcharCount += appendVector(stringBuffer, val.ViewState, "ViewState");
   tcharCount += appendVector(stringBuffer, val.GroupState, "GroupState");

   tcharCount += stringBuffer.Append("}");

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<AppIndexSer>::Append(::FeatStd::StringBuffer& stringBuffer, const AppIndexSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Name=");
   tcharCount += stringBuffer.AppendObject(val.NameIndex);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<AppConfigSer>::Append(::FeatStd::StringBuffer& stringBuffer, const AppConfigSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("App=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<AppStateSer>::Append(::FeatStd::StringBuffer& stringBuffer, const AppStateSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("App=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.Append(", CurrentFocusView=");
   tcharCount += stringBuffer.AppendObject(val.CurrentFocusViewNameIndex);
   tcharCount += stringBuffer.Append(", CurrentFocusWidget=");
   tcharCount += stringBuffer.AppendObject(val.CurrentFocusWidgetNameIndex);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<ViewIndexSer>::Append(::FeatStd::StringBuffer& stringBuffer, const ViewIndexSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Name=");
   tcharCount += stringBuffer.AppendObject(val.NameIndex);
   tcharCount += stringBuffer.Append(", App=");
   tcharCount += stringBuffer.AppendObject(val.OwnerIndex);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<ViewConfigSer>::Append(::FeatStd::StringBuffer& stringBuffer, const ViewConfigSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("View=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<CameraDataSer>::Append(::FeatStd::StringBuffer& stringBuffer, const CameraDataSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("View=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.Append(", ");
   tcharCount += stringBuffer.AppendObject(val.Data);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<ViewStateSer>::Append(::FeatStd::StringBuffer& stringBuffer, const ViewStateSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("View=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.Append(", FocusVisible=");
   tcharCount += stringBuffer.AppendObject(val.FocusVisible);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<WidgetIndexSer>::Append(::FeatStd::StringBuffer& stringBuffer, const WidgetIndexSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Name=");
   tcharCount += stringBuffer.AppendObject(val.NameIndex);
   tcharCount += stringBuffer.Append(", View=");
   tcharCount += stringBuffer.AppendObject(val.OwnerIndex);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<WidgetConfigSer>::Append(::FeatStd::StringBuffer& stringBuffer, const WidgetConfigSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Widget=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.Append(", ParentGroupViewName=");
   tcharCount += stringBuffer.AppendObject(val.ParentGroupViewNameIndex);
   tcharCount += stringBuffer.Append(", ParentGroupWidgetName=");
   tcharCount += stringBuffer.AppendObject(val.ParentGroupWidgetNameIndex);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<GroupStateSer>::Append(::FeatStd::StringBuffer& stringBuffer, const GroupStateSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Group=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.Append(", CurrentFocusWidgetName=");
   tcharCount += stringBuffer.AppendObject(val.CurrentFocusWidgetNameIndex);
   tcharCount += stringBuffer.Append(", SequenceId=");
   tcharCount += stringBuffer.AppendObject(val.SequenceId);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<WidgetDataSer>::Append(::FeatStd::StringBuffer& stringBuffer, const WidgetDataSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Widget=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.AppendObject(val.Data);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<GroupDataSer>::Append(::FeatStd::StringBuffer& stringBuffer, const GroupDataSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Widget=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.AppendObject(val.Data);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<ListDataSer>::Append(::FeatStd::StringBuffer& stringBuffer, const ListDataSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Widget=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.AppendObject(val.Data);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<ListItemDataSer>::Append(::FeatStd::StringBuffer& stringBuffer, const ListItemDataSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Widget=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.AppendObject(val.Data);

   return tcharCount;
}


/*****************************************************************************/
template<> ::FeatStd::UInt32 StringBufferAppender<BoundsDataSer>::Append(::FeatStd::StringBuffer& stringBuffer, const BoundsDataSer& val)
{
   ::FeatStd::UInt32 tcharCount = 0;

   tcharCount += stringBuffer.Append("Widget=");
   tcharCount += stringBuffer.AppendObject(val.ItemIndex);
   tcharCount += stringBuffer.Append(", {");
   tcharCount += stringBuffer.AppendObject(static_cast<int>(val.Data.GetLeft()));
   tcharCount += stringBuffer.Append(",");
   tcharCount += stringBuffer.AppendObject(static_cast<int>(val.Data.GetTop()));
   tcharCount += stringBuffer.Append(",");
   tcharCount += stringBuffer.AppendObject(static_cast<int>(val.Data.GetWidth()));
   tcharCount += stringBuffer.Append(",");
   tcharCount += stringBuffer.AppendObject(static_cast<int>(val.Data.GetHeight()));
   tcharCount += stringBuffer.Append("}");

   return tcharCount;
}


}
#endif

#if defined(_lint) && (_lint <= 912) // Versions up to and including 9.00L
//lint -restore  reactivate due to above deactivation.
#elif defined(_lint)
#error Check if deactivation / reactivation is necessary with new Lint version
#endif
///////////////////////////////////////////////////////////////////////////////
// <EOF>
