/* ***************************************************************************************
* FILE:          FDefaultRotaryController.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultRotaryController.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultRotaryController.h"
#include "Focus/FCommon.h"
#include "Focus/FContainer.h"
#include "Focus/FDataSet.h"
#include "Focus/FData.h"
#include "Focus/FActiveViewGroup.h"
#include "Focus/FGroupTraverser.h"
#include "Focus/FManagerConfig.h"
#include "Focus/FManager.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultRotaryController.cpp.trc.h"
#endif

namespace Focus {

bool FDefaultRotaryController::onMessage(FSession& session, FGroup& group, const FMessage& msg)
{
   if (msg.GetId() == EncoderStatusChangedUpdMsg::ID)
   {
      const EncoderStatusChangedUpdMsg* encoderMsg = Courier::message_cast<const EncoderStatusChangedUpdMsg*>(&msg);
      if (encoderMsg != NULL)//encode code is checked in focus input checker component
      {
         ETG_TRACE_USR1_THR(("FDefaultRotaryController::onMessage() app=%d, steps=%d, src=%u, visible=%u",
                             _manager.getCurrentAppId(), encoderMsg->GetEncSteps(), encoderMsg->GetEncCode(), _manager.isFocusVisible()));

         if (_manager.getActivityTimer() != NULL)
         {
            _manager.getActivityTimer()->restart();
         }

         bool crtFocusVisibleAndValid = _manager.isFocusVisible() && _manager.getAvgManager()->isCurrentFocusAvailable(session, &group);

         //if focus is not visible, generate a new sequence so that preserve focus can correctly reuse the current focus
         if (!crtFocusVisibleAndValid)
         {
            _manager.generateNewSequenceId();
         }

         bool result = moveFocus(session, group, encoderMsg->GetEncSteps());
         if (!crtFocusVisibleAndValid)
         {
            //if no element was found try the other direction
            if (!result)
            {
               result = moveFocus(session, group, -encoderMsg->GetEncSteps());
            }
            //new focus found
            if (result)
            {
               _manager.setFocusVisible(true);
            }
         }

         //message is consumed even if focus was not changed, this could change if necessary
         return true;
      }
   }

   return false;
}


bool FDefaultRotaryController::moveFocus(FSession& session, FGroup& group, int steps)
{
   ETG_TRACE_USR1_THR(("FDefaultRotaryController::moveFocus app=%d, steps=%d, group=%s",
                       _manager.getCurrentAppId(), steps, FID_STR(group.getId())));

   FWidget* newFocus = NULL;
   if (_manager.getAvgManager() != NULL)
   {
      Focusable* f = _manager.getAvgManager()->getFocus(session);
      FWidgetVisitor widgetVisitor;

      bool ascending = true;
      if (steps < 0)
      {
         ascending = false;
         steps = -steps;
      }

      FDefaultGroupTraverser traverser(session, widgetVisitor, group, f, ascending);

      //prevents infinite loops with wrap around
      size_t lastResetCount = Constants::InvalidIndex;

      bool crtFocusVisibleAndValid = _manager.isFocusVisible() && _manager.getAvgManager()->isCurrentFocusAvailable(session, &group);

      //if focus is not visible check old focus
      if (!crtFocusVisibleAndValid)
      {
         //also set steps to 1 and check the current/default focus
         steps = 1;
         if (f != NULL)
         {
            f->visit(widgetVisitor);
         }
      }

      while (static_cast<int>(widgetVisitor.getCount()) < steps)
      {
         //advanced to next
         if (traverser.advance())
         {
            //nothing to do
         }
         //no more elements
         else
         {
            //if wrap around reset the traverser
            FGroupData* groupData = group.getData<FGroupData>();
            if ((groupData != NULL) && (groupData->WrapAround) && (lastResetCount != widgetVisitor.getCount()))
            {
               //prevent infinite loops
               lastResetCount = widgetVisitor.getCount();

               traverser.reset();
            }
            else
            {
               break;
            }
         }
      }

      newFocus = widgetVisitor.getFocusable();
      if (newFocus != NULL)
      {
         _manager.getAvgManager()->setFocus(session, newFocus);
      }
   }
   return newFocus != NULL;
}


}
