/* ***************************************************************************************
* FILE:          FDefaultManagerConfig.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultManagerConfig.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultManagerConfig.h"

#include "FDefaultAvgManager.h"
#include "FDefaultAnimationManager.h"
#include "FDefaultSessionManager.h"
#include "FDefaultTaskFactory.h"
#include "FDefaultTaskManager.h"
#include "FDefaultConsistencyChecker.h"
#include "FDefaultVisibilityManager.h"
#include "FDefaultEnterKeyController.h"
#include "FDefaultJoystickController.h"
#include "FDefaultRotaryController.h"
#include "FDefaultReqController.h"

#include "Focus/FCourierAdapter.h"
#include "Focus/FManager.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultManagerConfig.cpp.trc.h"
#endif

namespace Focus {

FDefaultManagerConfigurator::FDefaultManagerConfigurator(FManager& manager, AppViewHandler* viewHandler) :
   Base(manager),
   _manager(manager),
   _activityTimer(NULL),
   _watchdogTimer(NULL),
   _viewHandler(viewHandler)
{
}


FDefaultManagerConfigurator::~FDefaultManagerConfigurator()
{
   _activityTimer = NULL;
   _watchdogTimer = NULL;
   _viewHandler = NULL;
}


FCourierTimer* FDefaultManagerConfigurator::getActivityTimer() const
{
   return _activityTimer;
}


FCourierTimer* FDefaultManagerConfigurator::getWatchdogTimer() const
{
   return _watchdogTimer;
}


AppViewHandler* FDefaultManagerConfigurator::getViewHandler() const
{
   return _viewHandler;
}


bool FDefaultManagerConfigurator::initialize()
{
   initializeTimers();
   initializeAvgManager();
   initializeAnimationManager();
   initializeVisibilityManager();
   initializeInputMsgQueue();
   initializeInputMsgChecker();
   initializeIpcManager();
   initializeOutputMsgHandler();
   initializeSessionManager();
   initializeTaskManager();
   initializeTaskFactory();
   initializeControllers();
   initializeConsistencyChecker();
   return true;
}


bool FDefaultManagerConfigurator::finalize()
{
   finalizeConsistencyChecker();
   finalizeControllers();
   finalizeTaskFactory();
   finalizeTaskManager();
   finalizeSessionManager();
   finalizeOutputMsgHandler();
   finalizeIpcManager();
   finalizeInputMsgChecker();
   finalizeInputMsgQueue();
   finalizeVisibilityManager();
   finalizeAnimationManager();
   finalizeAvgManager();
   finalizeTimers();
   return true;
}


bool FDefaultManagerConfigurator::initializeTimers()
{
   if (_activityTimer == NULL)
   {
      _activityTimer = FOCUS_NEW(FCourierTimer)(10000);
      if (_activityTimer != NULL)
      {
         _manager.addMsgReceiver(*_activityTimer);
         setActivityTimer(_activityTimer);
      }
   }

   if (_watchdogTimer == NULL)
   {
      _watchdogTimer = FOCUS_NEW(FCourierTimer)(1000);
      if (_watchdogTimer != NULL)
      {
         _manager.addMsgReceiver(*_watchdogTimer);
         setWatchdogTimer(_watchdogTimer);
      }
   }

   return true;
}


bool FDefaultManagerConfigurator::finalizeTimers()
{
   if (_activityTimer != NULL)
   {
      if (_manager.getActivityTimer() == _activityTimer)
      {
         setActivityTimer(NULL);
      }
      _manager.removeMsgReceiver(*_activityTimer);
      _activityTimer->stop();
      FOCUS_DELETE(_activityTimer);
      _activityTimer = NULL;
   }

   if (_watchdogTimer != NULL)
   {
      if (_manager.getWatchdogTimer() == _watchdogTimer)
      {
         setWatchdogTimer(NULL);
      }
      _manager.removeMsgReceiver(*_watchdogTimer);
      _watchdogTimer->stop();
      FOCUS_DELETE(_watchdogTimer);
      _watchdogTimer = NULL;
   }

   return true;
}


bool FDefaultManagerConfigurator::initializeControllers()
{
   setRootGroupControllerSetId(DefaultRootGroupControllerSetId);
   registerController(DefaultRootGroupControllerSetId, FOCUS_NEW(FDefaultEnterKeyController)(_manager));
   registerController(DefaultRootGroupControllerSetId, FOCUS_NEW(FDefaultRotaryController)(_manager));
   registerController(DefaultRootGroupControllerSetId, FOCUS_NEW(FDefaultJoystickController)(_manager));
   registerController(DefaultRootGroupControllerSetId, FOCUS_NEW(FDefaultReqController)(_manager));

   return true;
}


bool FDefaultManagerConfigurator::finalizeControllers()
{
   clearControllerSet(DefaultRootGroupControllerSetId, true);

   return true;
}


void FDefaultManagerConfigurator::clearControllerSet(FControllerSetId setId, bool performDelete)
{
   const FControllerList* defaultControllers = _configuration.getControllerSet(setId);
   if (defaultControllers != NULL)
   {
      if (performDelete)
      {
         for (size_t i = 0; i < defaultControllers->count(); ++i)
         {
            FController* controller = (*defaultControllers)[i];
            if (controller != NULL)
            {
               FOCUS_DELETE(controller);
            }
         }
      }
      const_cast<FControllerList*>(defaultControllers)->removeAll();
   }
}


bool FDefaultManagerConfigurator::initializeAvgManager()
{
   setAvgManager(FOCUS_NEW(FDefaultAvgManager)(_manager));
   return true;
}


bool FDefaultManagerConfigurator::finalizeAvgManager()
{
   if (_manager.getAvgManager() != NULL)
   {
      FOCUS_DELETE(_manager.getAvgManager());
      setAvgManager(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeAnimationManager()
{
   if (_viewHandler == NULL)
   {
      return false;
   }

   //don't enable yet the animation manager in the default configuration
   //setAnimationManager(FOCUS_NEW(FDefaultAnimationManager)(_manager, *_viewHandler));
   return true;
}


bool FDefaultManagerConfigurator::finalizeAnimationManager()
{
   if (_manager.getAnimationManager() != NULL)
   {
      FOCUS_DELETE(_manager.getAnimationManager());
      setAnimationManager(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeConsistencyChecker()
{
   if (_viewHandler == NULL)
   {
      return false;
   }

   setConsistencyChecker(FOCUS_NEW(FDefaultConsistencyChecker)(_manager, *_viewHandler));
   return true;
}


bool FDefaultManagerConfigurator::finalizeConsistencyChecker()
{
   if (_manager.getConsistencyChecker() != NULL)
   {
      FOCUS_DELETE(_manager.getConsistencyChecker());
      setConsistencyChecker(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeVisibilityManager()
{
   setVisibilityManager(FOCUS_NEW(FDefaultVisibilityManager)(_manager));
   //setVisibilityManager(FOCUS_NEW(FViewSpecificVisibilityManager)(_manager));
   return true;
}


bool FDefaultManagerConfigurator::finalizeVisibilityManager()
{
   if (_manager.getVisibilityManager() != NULL)
   {
      FOCUS_DELETE(_manager.getVisibilityManager());
      setVisibilityManager(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeInputMsgQueue()
{
   setInputMsgQueue(FOCUS_NEW(FCourierMessageQueue));
   return true;
}


bool FDefaultManagerConfigurator::finalizeInputMsgQueue()
{
   if (_manager.getInputMsgQueue() != NULL)
   {
      FOCUS_DELETE(_manager.getInputMsgQueue());
      setInputMsgQueue(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeInputMsgChecker()
{
   setInputMsgChecker(FOCUS_NEW(FCourierInputMsgChecker));
   return true;
}


bool FDefaultManagerConfigurator::finalizeInputMsgChecker()
{
   if (_manager.getInputMsgChecker() != NULL)
   {
      FOCUS_DELETE(_manager.getInputMsgChecker());
      setInputMsgChecker(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeIpcManager()
{
   //no ipc by default
   ETG_TRACE_USR1_THR(("FDefaultManagerConfigurator::initializeIpcManager Focus IPC is not configured"));
   return true;
}


bool FDefaultManagerConfigurator::finalizeIpcManager()
{
   return true;
}


bool FDefaultManagerConfigurator::initializeOutputMsgHandler()
{
   if (_viewHandler == NULL)
   {
      ETG_TRACE_ERR_THR(("FDefaultManagerConfigurator::initializeOutputMsgHandler viewHandler is NULL"));
      return false;
   }

   setOutputMsgHandler(FOCUS_NEW(FCourierOutputMsgHandler)(*_viewHandler));
   return true;
}


bool FDefaultManagerConfigurator::finalizeOutputMsgHandler()
{
   if (_manager.getOutputMsgHandler() != NULL)
   {
      FOCUS_DELETE(_manager.getOutputMsgHandler());
      setOutputMsgHandler(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeSessionManager()
{
   setSessionManager(FOCUS_NEW(FDefaultSessionManager)(_manager));
   return true;
}


bool FDefaultManagerConfigurator::finalizeSessionManager()
{
   if (_manager.getSessionManager() != NULL)
   {
      FOCUS_DELETE(_manager.getSessionManager());
      setSessionManager(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeTaskFactory()
{
   setTaskFactory(FOCUS_NEW(FDefaultTaskFactory)(_manager));
   return true;
}


bool FDefaultManagerConfigurator::finalizeTaskFactory()
{
   if (_manager.getTaskFactory() != NULL)
   {
      FOCUS_DELETE(_manager.getTaskFactory());
      setTaskFactory(NULL);
   }
   return true;
}


bool FDefaultManagerConfigurator::initializeTaskManager()
{
   setTaskManager(FOCUS_NEW(FDefaultTaskManager)(_manager));
   return true;
}


bool FDefaultManagerConfigurator::finalizeTaskManager()
{
   if (_manager.getTaskManager() != NULL)
   {
      FOCUS_DELETE(_manager.getTaskManager());
      setTaskManager(NULL);
   }
   return true;
}


}
