/* ***************************************************************************************
* FILE:          FDefaultIpcManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultIpcManager.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultIpcManager.h"
#include "Focus/FData.h"
#include "Focus/FManager.h"
#include "Focus/FSession.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultIpcManager.cpp.trc.h"
#endif

namespace Focus {

FDefaultIpcManager::FDefaultIpcManager(FManager& manager, FIpcAdapter& ipcAdapter) :
   _manager(manager),
   _ipcAdapter(ipcAdapter),
   _otherAppsInfoRequestInProgress(false)
{
   _ipcAdapter.addReceiver(*this);
}


FDefaultIpcManager::~FDefaultIpcManager()
{
   _ipcAdapter.removeReceiver(*this);
}


bool FDefaultIpcManager::isOtherAppsInfoRequestInProgress() const
{
   return _otherAppsInfoRequestInProgress;
};


bool FDefaultIpcManager::sendOtherAppsInfoRequest()
{
   ETG_TRACE_USR1_THR(("FDefaultIpcManager::sendOtherAppsInfoRequest app=%d", _manager.getCurrentAppId()));

   bool result = false;
   if (_otherAppsInfoRequestInProgress)
   {
      ETG_TRACE_USR1_THR(("FDefaultIpcManager::sendOtherAppsInfoRequest request in progress!"));
   }
   else
   {
      _otherAppsInfoRequestInProgress = true;
      result = _ipcAdapter.sendOtherAppsInfoRequest();

      //if sending of request failed we will not get a completed callback
      if (!result)
      {
         _otherAppsInfoRequestInProgress = false;
      }
   }
   return result;
}


//sends a message to the application identified by appState.getId()
//the Ipc receivers in that application should get an onCurrentAppStateReceived call
bool FDefaultIpcManager::sendOtherAppState(FAppStateSharedPtr appState)
{
   ETG_TRACE_USR1_THR(("FDefaultIpcManager::sendOtherAppState app=%d, otherAppId=%d",
                       _manager.getCurrentAppId(), appState.PointsToNull() ? -1 : appState->getId()));

   return _ipcAdapter.sendOtherAppState(appState);
}


//sends a message to the application which previously requested info from applications with visible surfaces
//the Ipc receivers in that application should get an onOtherAppInfoReceived call
bool FDefaultIpcManager::sendCurrentAppInfo(FAppConfigSharedPtr appConfig, FAppStateSharedPtr appState)
{
   return _ipcAdapter.sendCurrentAppInfo(appConfig, appState);
}


//queried slave
void FDefaultIpcManager::onCurrentAppInfoRequestReceived()
{
   ETG_TRACE_USR1_THR(("FDefaultIpcManager::onCurrentAppInfoRequestReceived app=%d", _manager.getCurrentAppId()));

   //start session
   //collect courier info
   //merge crt app state
   //send crt app info

   if (_manager.beginSession(FSession::QueryingSlave))
   {
      _manager.updateSession();
   }
}


//updated slave
void FDefaultIpcManager::onCurrentAppStateReceived(FAppStateSharedPtr appState)
{
   ETG_TRACE_USR1_THR(("FDefaultIpcManager::onCurrentAppStateReceived app=%d, crtAppId=%d",
                       _manager.getCurrentAppId(), appState.PointsToNull() ? -1 : appState->getId()));

   //start session
   //collect courier info
   //merge crt app state
   //merge received app state
   //merge back app state
   //publish focus

   if (!appState.PointsToNull() && (appState->getId() == _manager.getCurrentAppId()) && _manager.beginSession(FSession::UpdatingSlave))
   {
      FSession* session = _manager.getSession();
      if (session != NULL)
      {
         session->StateInfo.add(_manager.getCurrentAppId(), appState);
      }
      _manager.updateSession();
   }
}


//master
void FDefaultIpcManager::onOtherAppInfoReceived(FAppConfigSharedPtr appConfig, FAppStateSharedPtr appState)
{
   FSession* session = _manager.getSession();

   ETG_TRACE_USR1_THR(("FDefaultIpcManager::onOtherAppInfoReceived app=%d, otherAppId=%d, session=%d",
                       _manager.getCurrentAppId(),
                       appConfig.PointsToNull() ? -1 : appConfig->getId(),
                       session == NULL ? -1 : static_cast<int>(session->getStatus())));

   if (!_otherAppsInfoRequestInProgress)
   {
      ETG_TRACE_USR1_THR(("FDefaultIpcManager::onOtherAppInfoReceived no request in progress!"));
   }
   else
   {
      //notify listeners
      //merge received info into session

      //todo: check if session is in the correct state (collecting)
      if (session != NULL)
      {
         if (!appConfig.PointsToNull())
         {
            session->ConfigInfo.add(appConfig->getId(), appConfig);
         }
         if (!appState.PointsToNull())
         {
            session->StateInfo.add(appState->getId(), appState);
         }
      }
   }
}


//master
void FDefaultIpcManager::onOtherAppsInfoReceiveCompleted()
{
   ETG_TRACE_USR1_THR(("FDefaultIpcManager::onOtherAppsInfoReceiveCompleted app=%d", _manager.getCurrentAppId()));

   if (!_otherAppsInfoRequestInProgress)
   {
      ETG_TRACE_USR1_THR(("FDefaultIpcManager::onOtherAppInfoReceived no request in progress!"));
   }
   else
   {
      //notify listeners
      //mark collect task as completed

      _otherAppsInfoRequestInProgress = false;

      if (_manager.getOutputMsgHandler() != NULL)
      {
         UpdateFocusSessionReqMsg* msg = COURIER_MESSAGE_NEW(UpdateFocusSessionReqMsg)();
         if (msg != NULL)
         {
            _manager.getOutputMsgHandler()->postMessage(msg);
         }
      }
   }
}


}
