/* ***************************************************************************************
* FILE:          FDefaultAvgManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultAvgManager.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultAvgManager.h"
#include "Focus/FController.h"
#include "Focus/FData.h"
#include "Focus/FGroupTraverser.h"
#include "Focus/FManager.h"
#include "Focus/FSession.h"
#include "Focus/FStateInfo.h"
#include "Focus/FUtils.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultAvgManager.cpp.trc.h"
#endif

namespace Focus {
static const FControllerList* getControllerList(const FManager& manager, FWidgetConfig& config)
{
   FWidgetData* widgetData = config.getData<FWidgetData>();
   if (widgetData != NULL)
   {
      return manager.getControllerSet(widgetData->ControllerSetId);
   }
   return NULL;
}


bool FDefaultAvgManager::isCurrentFocusAvailable(FSession& session, FGroup* g)
{
   if (g == NULL)
   {
      g = session.ActiveViewGroup.getRootGroup();
   }

   Focusable* result = NULL;
   while (g != NULL)
   {
      result = getCurrentFocusChild(session, *g);
      g = dynamic_cast<FGroup*>(result);
   }
   return (result != NULL);
}


Focusable* FDefaultAvgManager::getFocus(FSession& session, FGroup* g)
{
   if (g == NULL)
   {
      g = session.ActiveViewGroup.getRootGroup();
   }

   Focusable* result = g;
   while (g != NULL)
   {
      Focusable* temp = getFocusImpl(session, *g);
      if (temp != NULL)
      {
         result = temp;
      }
      g = dynamic_cast<FGroup*>(temp);
   }
   return result;
}


Focusable* FDefaultAvgManager::getFocusImpl(FSession& session, FGroup& group)
{
   ETG_TRACE_USR1_THR(("FDefaultAvgManager::getFocusImpl app=%d, group=%s",
                       _manager.getCurrentAppId(), FID_STR(group.getId())));

   //controllers have the first chance to decide what is the current focus
   for (size_t i = 0; i < group.Config.count(); ++i)
   {
      FWidgetConfig* config = group.Config.get(i);
      if (config != NULL)
      {
         const FControllerList* controllerList = getControllerList(_manager, *config);
         if (controllerList != NULL)
         {
            for (size_t i = 0; i < controllerList->count(); ++i)
            {
               FController* controller = (*controllerList)[i];
               if (controller != NULL)
               {
                  Focusable* child = controller->getFocus(session, group);
                  if (child != NULL)
                  {
                     ETG_TRACE_USR1_THR(("FDefaultAvgManager::getFocusImpl Found a controller focus child=%s", FID_STR(child->getId())));
                     return child;
                  }
               }
            }
         }
      }
   }

   //group is empty
   if (group.Children.count() == 0)
   {
      return NULL;
   }

   //check current focus
   Focusable* child = getCurrentFocusChild(session, group);
   if (child != NULL)
   {
      ETG_TRACE_USR1_THR(("FDefaultAvgManager::getFocusImpl Found current focus child=%s", FID_STR(child->getId())));
      return child;
   }

   //check default focus
   child = getDefaultFocusChild(session, group);
   if (child != NULL)
   {
      ETG_TRACE_USR1_THR(("FDefaultAvgManager::getFocusImpl Found default focus child=%s", FID_STR(child->getId())));
      return child;
   }

   //get first child
   child = group.Children.get(0);
   ETG_TRACE_USR1_THR(("FDefaultAvgManager::getFocusImpl Found first child child=%s", FID_STR(child->getId())));
   return child;
}


Focusable* FDefaultAvgManager::getCurrentFocusChild(FSession& session, FGroup& group)
{
   (void)session;
   if (group.Children.count() > 0)
   {
      FGroupData* groupData = group.getData<FGroupData>();
      FGroupCurrentFocusInfo* currentFocusInfo = group.getData<FGroupCurrentFocusInfo>();
      if ((groupData != NULL) && (currentFocusInfo != NULL) && (currentFocusInfo->Id != Constants::InvalidId))
      {
         ETG_TRACE_USR1_THR(("FDefaultAvgManager::getFocusImpl Checking current focus info (preserve=%u) currentFocusInfo=%s",
                             groupData->PreserveFocus, FID_STR(currentFocusInfo->Id)));
         bool useCurrentFocusInfo = groupData->PreserveFocus;
         if (!useCurrentFocusInfo)
         {
            FSequenceId* updateSequenceId = group.getData<FSequenceId>();
            if ((updateSequenceId != NULL) && ((*updateSequenceId) == _manager.getCurrentSequenceId()))
            {
               useCurrentFocusInfo = true;
            }
            else
            {
               ETG_TRACE_USR1_THR(("FDefaultAvgManager::getFocusImpl Current focus info is expired groupSeqId=%d, managerSeqId=%d",
                                   updateSequenceId == NULL ? -1 : updateSequenceId->Id, _manager.getCurrentSequenceId().Id));
            }
         }
         if (useCurrentFocusInfo)
         {
            for (size_t i = 0; i < group.Children.count(); ++i)
            {
               Focusable* child = group.Children.get(i);
               if ((child != NULL) && (child->getId() == currentFocusInfo->Id))
               {
                  return child;
               }
            }
            ETG_TRACE_USR1_THR(("FDefaultAvgManager::getFocusImpl No child found to match the current focus info"));
         }
      }
   }
   return NULL;
}


Focusable* FDefaultAvgManager::getDefaultFocusChild(FSession& session, FGroup& group)
{
   (void)session;
   if (group.Children.count() > 0)
   {
      FGroupData* groupData = group.getData<FGroupData>();
      if ((groupData != NULL) && (groupData->Configured))
      {
         for (size_t i = 0; i < group.Children.count(); ++i)
         {
            Focusable* child = group.Children.get(i);
            if (child != NULL)
            {
               FWidgetData* widgetData = FUtils::getData<FWidgetData>(child);
               if ((widgetData != NULL) && (widgetData->Order == groupData->DefaultOrder))
               {
                  return child;
               }
            }
         }
      }
      return group.Children.get(0);
   }
   return NULL;
}


void FDefaultAvgManager::setFocus(FSession& session, Focusable* f)
{
   if (f != NULL)
   {
      ETG_TRACE_USR1_THR(("FDefaultAvgManager::setFocus app=%d, focusable=%s",
                          _manager.getCurrentAppId(), FID_STR(f->getId())));

      _manager.notifyInternalSetFocus(session, f);

      for (FGroup* p = f->getParent(); p != NULL; p = p->getParent())
      {
         setFocusImpl(session, *p, f);
         f = p;
      }
   }
}


//lint -esym(1764, session) TODO: Remove this line as soon as parameter session is used in this function
void FDefaultAvgManager::setFocusImpl(FSession& session, FGroup& group, const Focusable* f)
{
   (void)session;
   if (f != NULL)
   {
      group.Data.set(FGroupCurrentFocusInfo(f->getId()));
   }
   else
   {
      group.Data.set(FGroupCurrentFocusInfo());
   }
   group.Data.set(_manager.getCurrentSequenceId());
}


int FDefaultAvgManager::moveFocus(FSession& session, FGroup& group, int steps)
{
   ETG_TRACE_USR1_THR(("FDefaultAvgManager::moveFocus app=%d, steps=%d, children=%u, group=%s",
                       _manager.getCurrentAppId(), steps, group.Children.count(), FID_STR(group.getId())));

   if ((group.Children.count() == 0) || (steps == 0))
   {
      return 0;
   }

   Focus::Focusable* oldFocus = getFocus(session, &group);
   if (oldFocus != NULL)
   {
      ETG_TRACE_USR1_THR(("FDefaultAvgManager::moveFocus oldFocus=%s", FID_STR(oldFocus->getId())));
   }

   FWidgetVisitor widgetVisitor;
   FDefaultGroupTraverser traverser(session, widgetVisitor, group, oldFocus, steps > 0);

   int sign = ((steps < 0) ? -1 : 1);

   while ((static_cast<int>(widgetVisitor.getCount()) < sign * steps) && traverser.advance())
   {
      //nothing to do, just iterate until visitor count reaches the required value or until there are no more children
   }

   int actualSteps = static_cast<int>(widgetVisitor.getCount()) * sign;
   Focus::FWidget* newFocus = widgetVisitor.getFocusable();
   if (newFocus != NULL)
   {
      ETG_TRACE_USR1_THR(("FDefaultAvgManager::moveFocus actualSteps=%d, newFocus=%s",
                          actualSteps, FID_STR(newFocus->getId())));

      setFocus(session, newFocus);
   }

   return actualSteps;
}


//static size_t adjustIndex(size_t index, size_t count, int steps, bool wrapAround)
//{
//   if (count == 0)
//   {
//      return 0;
//   }
//   if (steps == 0)
//   {
//      return index;
//   }
//
//   int temp = static_cast<int>(index) + steps;
//   if (wrapAround)
//   {
//      temp = temp % static_cast<int>(count);
//      if (temp < 0)
//      {
//         temp += static_cast<int>(count);
//      }
//   }
//   else
//   {
//      if (temp < 0)
//      {
//         temp = 0;
//      }
//      else if (temp >= static_cast<int>(count))
//      {
//         temp = count  - 1;
//      }
//      else
//      {
//         //nothing to do
//      }
//   }
//   return static_cast<size_t>(temp);
//}
//
//void FDefaultAvgManager::moveFocus(FSession& session, FGroup& group, int steps)
//{
//   if (steps == 0)
//   {
//      ETG_TRACE_USR1_THR(("FocusManager::moveFocus steps is 0"));
//      return;
//   }
//
//   if (group.Children.count() == 0)
//   {
//      ETG_TRACE_USR1_THR(("FocusManager::moveFocus Current group %s has no children", FID_STR(group.getId())));
//      return;
//   }
//
//   FGroupData* groupData = group.Data.getOrCreate<FGroupData>();
//   if (groupData == NULL)
//   {
//      return;
//   }
//
//   size_t currentindex = getFocusIndex(session, group);
//
//   ETG_TRACE_USR1_THR(("FocusManager::moveFocus steps=%d, wrap=%u, count=%u, current=%u, group=%s",
//                       steps, groupData->getWrapAround(), group.Children.count(), currentindex, FID_STR(group.getId())));
//
//   if (currentindex == Constants::InvalidIndex)
//   {
//      currentindex = 0;
//   }
//
//   size_t index = adjustIndex(currentindex, group.Children.count(), steps, groupData->getWrapAround());
//
//   Focusable* newFocus = group.Children.get(index);
//   ETG_TRACE_USR1_THR(("FocusManager::moveFocus currentIndex=%u, index=%u, f=%s",
//                        currentindex, index, (newFocus != NULL ? FID_STR(newFocus->getId()) : "<null>")));
//   if (newFocus != NULL)
//   {
//      setFocus(session, newFocus);
//   }
//}
}
