/* ***************************************************************************************
* FILE:          FDefaultActiveGroupController.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FDefaultActiveGroupController.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

///////////////////////////////////////////////////////////////////////////////
//focus manager includes
#include "FDefaultActiveGroupController.h"
#include <Focus/FManager.h>

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_FOCUS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FDefaultActiveGroupController.cpp.trc.h"
#endif

namespace Focus {

FDefaultActiveGroupController::FDefaultActiveGroupController(FManager& manager, const FControllerSetId& delegateControllerId)
   : _manager(manager), _delegateControllerId(delegateControllerId)
{
}


const FControllerSetId& FDefaultActiveGroupController::getDelegateControllerId() const
{
   return _delegateControllerId;
}


void FDefaultActiveGroupController::setDelegateControllerId(const FControllerSetId& delegateControllerId)
{
   ETG_TRACE_USR1_THR(("FDefaultActiveGroupController::setDelegateControllerId delegateControllerId=%u, app=%d", delegateControllerId, _manager.getCurrentAppId()));

   _delegateControllerId = delegateControllerId;
}


Focusable* FDefaultActiveGroupController::getPlaceholder(FSession&, FGroup& group) const
{
   //by default the placeholder is the first child
   return (group.Children.count() > 0) ? group.Children.get(0) : NULL;
}


bool FDefaultActiveGroupController::isGroupActive(FSession& session, FGroup& group) const
{
   Focusable* placeholder = getPlaceholder(session, group);
   Focusable* currentFocus = (FManager::getInstance().getAvgManager() != NULL) ? FManager::getInstance().getAvgManager()->getFocus(session) : NULL;

   //group is active if it contains the focus and that is not the placeholder
   return ((currentFocus != NULL) && (currentFocus != placeholder) && (group.Children.indexOf(currentFocus) != Constants::InvalidIndex));
}


bool FDefaultActiveGroupController::refreshReachableChildren(FSession& session, FGroup& group)
{
   Focusable* placeholder = getPlaceholder(session, group);

   ETG_TRACE_USR4_THR(("FDefaultActiveGroupController::refreshReachableChildren placeholder=%50s groupActive=%u group=%s",
                       placeholder != NULL ? FID_STR(placeholder->getId()) : "<null>",
                       isGroupActive(session, group),
                       FID_STR(group.getId())));

   //group is active => add all its children except for the placeholder
   if ((placeholder == NULL) || isGroupActive(session, group))
   {
      for (size_t i = 0; i < group.Children.count(); ++i)
      {
         Focusable* child = group.Children.get(i);
         if (child != placeholder)
         {
            group.ReachableChildren.add(child);
         }
      }
   }
   //group is not active => add only the placeholder
   else
   {
      group.ReachableChildren.add(placeholder);
   }

   return true;
}


Focusable* FDefaultActiveGroupController::getDefaultFocus(FSession& session, FGroup& group) const
{
   if (FManager::getInstance().getAvgManager() == NULL)
   {
      ETG_TRACE_ERR_THR(("FDefaultActiveGroupController::getDefaultFocus null avg manager!"));
      return NULL;
   }

   Focusable* placeholder = getPlaceholder(session, group);
   Focusable* newFocus = FManager::getInstance().getAvgManager()->getDefaultFocusChild(session, group);

   //default focus is the placeholder => search for another child
   for (size_t i = 0; ((newFocus == NULL) || (newFocus == placeholder)) && (i < group.Children.count()); ++i)
   {
      newFocus = group.Children.get(i);
   }

   return newFocus;
}


bool FDefaultActiveGroupController::activateGroup(FSession& session, FGroup& group, bool activate)
{
   if (FManager::getInstance().getAvgManager() == NULL)
   {
      ETG_TRACE_ERR_THR(("FDefaultActiveGroupController::activateGroup null avg manager!"));
      return false;
   }

   FAvgManager& avgManager = *(FManager::getInstance().getAvgManager());
   Focusable* placeholder = getPlaceholder(session, group);

   ETG_TRACE_USR1_THR(("FDefaultActiveGroupController::activateGroup activate=%u isGroupActive=%u placeholder=%50s group=%s",
                       activate,
                       isGroupActive(session, group),
                       placeholder != NULL ? FID_STR(placeholder->getId()) : "<null>",
                       FID_STR(group.getId())));

   bool result = false;

   //group is already active or not active => no change
   if (activate == isGroupActive(session, group))
   {
      ETG_TRACE_USR1_THR(("FDefaultActiveGroupController::activateGroup already active or not active"));
      result = true;
   }
   else
   {
      //activate group by setting the focus on a child which is not the placeholder
      //deactivate group by setting the focus on the placeholder
      Focusable* newFocus = activate ? getDefaultFocus(session, group) : placeholder;

      if (newFocus != NULL)
      {
         ETG_TRACE_USR1_THR(("FDefaultActiveGroupController::activateGroup newFocus=%s",
                             newFocus != NULL ? FID_STR(newFocus->getId()) : "<null>"));

         avgManager.setFocus(session, newFocus);
         result = true;
      }
   }

   return result;
}


bool FDefaultActiveGroupController::onFocusGroupReqMsg(FSession& session, FGroup& group, const FocusGroupReqMsg& msg)
{
   FId reqGroupId(msg.GetGroup().GetCString());

   ETG_TRACE_USR1_THR(("FDefaultActiveGroupController::onFocusGroupReqMsg activate=%u reqGroup=%50s group=%s",
                       msg.GetActivate(), FID_STR(reqGroupId), FID_STR(group.getId())));

   bool consumed = false;

   //request is intended for this group => consume it
   if (group.getId() == reqGroupId)
   {
      consumed = activateGroup(session, group, msg.GetActivate());
   }
   else
   {
      //broadcast deactivation request => don't consume it
      if (!msg.GetActivate())
      {
         activateGroup(session, group, false);
      }
   }

   return consumed;
}


bool FDefaultActiveGroupController::onEnterKeyStatusChangedUpdMsg(FSession& session, FGroup& group, const EnterKeyStatusChangedUpdMsg& msg)
{
   ETG_TRACE_USR1_THR(("FDefaultActiveGroupController::onEnterKeyStatusChangedUpdMsg state=%u group=%s",
                       msg.GetState(), FID_STR(group.getId())));

   bool consumed = false;

   switch (msg.GetState())
   {
      //enter key up activates the group
      case hmibase::HARDKEYSTATE_UP:
      case hmibase::HARDKEYSTATE_LONGUP:
         consumed = activateGroup(session, group, true);
         break;

      default:
         break;
   }

   return consumed;
}


bool FDefaultActiveGroupController::onMessage(FSession& session, FGroup& group, const FMessage& msg)
{
   bool consumed = false;
   switch (msg.GetId())
   {
      case EnterKeyStatusChangedUpdMsg::ID:
      {
         const EnterKeyStatusChangedUpdMsg* updMsg = Courier::message_cast<const EnterKeyStatusChangedUpdMsg*>(&msg);
         if (updMsg != NULL)
         {
            consumed = onEnterKeyStatusChangedUpdMsg(session, group, *updMsg);
         }
         break;
      }

      case FocusGroupReqMsg::ID:
      {
         const FocusGroupReqMsg* reqMsg = Courier::message_cast<const FocusGroupReqMsg*>(&msg);
         if (reqMsg != NULL)
         {
            consumed = onFocusGroupReqMsg(session, group, *reqMsg);
         }
         break;
      }

      default:
         break;
   }

   //delegate unconsumed messages if the group is active
   if (!consumed && isGroupActive(session, group))
   {
      consumed = dispatchToDelegateControllers(session, group, msg);
   }

   return consumed;
}


bool FDefaultActiveGroupController::dispatchToDelegateControllers(FSession& session, FGroup& group, const FMessage& msg)
{
   bool consumed = false;

   const FControllerList* controllerList = FManager::getInstance().getControllerSet(_delegateControllerId);
   if (controllerList != NULL)
   {
      ETG_TRACE_USR4_THR(("FDefaultActiveGroupController::dispatchToDelegateControllers msg=%30s controllers=%u group=%s",
                          msg.GetName(), _delegateControllerId, FID_STR(group.getId())));

      for (size_t i = 0; !consumed && (i < controllerList->count()); ++i)
      {
         FController* controller = (*controllerList)[i];
         if (controller != NULL)
         {
            consumed = controller->onMessage(session, group, msg);
         }
      }
   }

   return consumed;
}


}
