/* ***************************************************************************************
* FILE:          Timer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Timer.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "View/IMessageSystem.h"
#include "Timer.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_TIMER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Timer.cpp.trc.h"
#endif

namespace Util {

/******************************************************************************
*  Constructor
******************************************************************************/
Timer::Timer() : _callback(0)
{
   memset(_msgRefs, 0, sizeof(_msgRefs));
}


Timer::Timer(uint32_t timeout) : _callback(0)
{
   ::hmibase::util::TimerThreaded::setTimeout(timeout, this);
   memset(_msgRefs, 0, sizeof(_msgRefs));
}


/******************************************************************************
*  Destructor
******************************************************************************/
Timer::~Timer()
{
   stop();
   resetTimeouts();
   memset(_msgRefs, 0, sizeof(_msgRefs));
   _callback = 0;
}


/******************************************************************************
*  ownMessage
******************************************************************************/
bool Timer::ownMessage(const TimerExpiredMsg& oMsg) const
{
   return (this == oMsg.GetTimer());
}


/******************************************************************************
*  setTimeout
******************************************************************************/
bool Timer::setTimeout(uint32_t timIndex, uint32_t timeout, Courier::Message* pNotifyMsg)
{
   ETG_TRACE_USR4_THR(("TimerThreaded::setTimeout() timIndex=%d t=%d ->'%s'", timIndex, timeout, getName().c_str()));
   bool rc = TimerThreaded::setTimeout(timIndex, timeout, this);
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
   if (timIndex < TIMER_MAX_TIME_OUTS)
   {
      _msgRefs[timIndex + 0] = pNotifyMsg ? new Courier::MessageReferrer(pNotifyMsg) : 0;
      _msgRefs[timIndex + 1] = 0;
   }
   else
   {
      ETG_TRACE_ERR_THR(("TimerThreaded::setTimeout() message ref skipped due to too big index (%d > %d) ->'%s'", timIndex, TIMER_MAX_TIME_OUTS, getName().c_str()));
   }
#else
   PARAM_UNUSED(pNotifyMsg);
#endif
   return rc;
}


/******************************************************************************
*  setTimeoutWithRepeat
******************************************************************************/
bool Timer::setTimeoutWithRepeat(uint32_t timeout, uint32_t repeatTime, Courier::Message* pNotifyMsg, TimeOutCallBack callback)
{
   ETG_TRACE_USR4_THR(("TimerThreaded::setTimeoutWithRepeat() t=%d  repeat=%d ->'%s'", timeout, repeatTime, getName().c_str()));
   _callback = callback;
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
   _msgRefs[0] = pNotifyMsg ? new Courier::MessageReferrer(pNotifyMsg) : 0;
   _msgRefs[1] = 0;
#else
   PARAM_UNUSED(pNotifyMsg);
#endif
   return TimerThreaded::setTimeoutWithRepeat(timeout, repeatTime, this);
}


/******************************************************************************
*  resetTimeouts
******************************************************************************/
void Timer::resetTimeouts()
{
   TimerThreaded::resetTimeouts();
   for (unsigned int i = 0; i < TABSIZE(_msgRefs); ++i)
   {
      if (_msgRefs[i])
      {
         // mRefCount=1; after this operation the messages is deleted
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
         delete _msgRefs[i];
#endif
      }
      _msgRefs[i] = 0;
   }
}


/******************************************************************************
*  timerExpired - don't block in this context !!
******************************************************************************/
void Timer::timerExpired(::hmibase::util::TimerThreaded::TimerId timerId, uint32_t timIndex)
{
   PARAM_UNUSED(timerId);
#ifdef VARIANT_S_FTR_ENABLE_COURIERMESSAGING
   Courier::MessageReferrer* ref = _msgRefs[timIndex];
   if (ref != NULL && ref->GetMessage() != NULL)
   {
      if (_callback != NULL)
      {
         ETG_TRACE_USR4_THR(("TimerThreaded::expire _callback custom msg (timIndex=%u) -> Name='%s'", timIndex, getName().c_str()));
         _callback(ref->GetMessage());
      }
      else
      {
         ETG_TRACE_USR4_THR(("TimerThreaded::expire custom msg post (timIndex=%u) -> Name='%s'", timIndex, getName().c_str()));
         ref->GetMessage()->Post();
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("TimerThreaded::expire post TimerExpiredMsg (timIndex=%u, repeatCounter=%u) ->'%s'", timIndex, getRepeatCounter(), getName().c_str()));
      TimerExpiredMsg& msg = *newMessage<TimerExpiredMsg>(this, timIndex, getRepeatCounter());
      msg.Post();
   }
#else
   ETG_TRACE_USR4_THR(("TimerThreaded::expire post TimerExpiredMsg (timIndex=%u, repeatCounter=%u) ->'%s'", timIndex, getRepeatCounter(), getName().c_str()));
   TimerExpiredMsg& msg = *newMessage<TimerExpiredMsg>(this, timIndex, getRepeatCounter());
   /// msg.Post();
#endif
}


}  // namespace Util
