/* ***************************************************************************************
* FILE:          ObjectGuard.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ObjectGuard.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef HMIBASE_UTIL_OBJECT_GUARD_H
#define HMIBASE_UTIL_OBJECT_GUARD_H


#include <FeatStd/MemoryManagement/SharedPointer.h>


namespace hmibase {
   namespace util {


      /*****************************************************************************/
      /** Provides support to check if an object instance has been destroyed before using it.

      // extend the class MyClass for which we need safe access from GuardedObject and define its accessor.
      class MyClass : public hmibase::util::GuardedObject
      {
      public:
         HMIBASE_TYPEDEF_OBJECT_ACCESSOR(MyClass)

         void doSomething();
      };

      // in a client class define a member variable of type accessor
      class MyOtherClass
      {
      public:
         // initialize the accessor from a valid instance
         void onObjectCreated(MyClass* obj)
         {
            _objAccessor = Accessor(obj);
         }

         //the object referred by the accessor could be deleted in the meanwhile
         void useObject()
         {
            if (_objAccessor.isValid())
            {
               _objAccessor->doSomething();
            }
         }

      private:
         // have a variable or a member of type accessor
         MyClass::Accessor _objAccessor;
      };
      */
      /*****************************************************************************/


      /*****************************************************************************/
      /** Call guardedObject.getGuard() on a valid object and store the returned value into a local or member variable.
       * Call guard.isValid() later to check if the object is still valid.
       */
      /*****************************************************************************/
      class ObjectGuard
      {
         friend class GuardedObject;

      public:
         ObjectGuard()
         {
            _data = GuardData::SharedPointer(FEATSTD_NEW(GuardData));
         }

         /* the guarded object is valid */
         bool isValid() const
         {
            return !_data.PointsToNull() && _data->_valid;
         }

      private:
         /* called only by the guarded object */
         void setValid(bool valid)
         {
            if (!_data.PointsToNull())
            {
               _data->_valid = valid;
            }
         }

         /*****************************************************************************/
         class GuardData
         {
         public:
            FEATSTD_TYPEDEF_SHARED_POINTER(GuardData);

            GuardData() : _valid(false)
            {
            }

            bool _valid;

         private:
            FEATSTD_SHARED_POINTER_DECLARATION();
         };

         GuardData::SharedPointer _data;
      };


      /*****************************************************************************/
      /** Base class for guarded objects.
       */
      /*****************************************************************************/
      class GuardedObject
      {
      public:
         GuardedObject() : _guard(NULL)
         {
         }

         /* when the guarded object is destroyed its guard is marked as invalid */
         ~GuardedObject()
         {
            if (_guard != NULL)
            {
               _guard->setValid(false);
               FEATSTD_DELETE(_guard);
               _guard = NULL;
            }
         }

         /* copy constructor, this object has its own guard */
         GuardedObject(const GuardedObject&) : _guard(NULL)
         {
         }

         /* assignment operator, this object's guard is not affected by the other object */
         GuardedObject& operator=(const GuardedObject&)
         {
            return *this;
         }

         /* creates the guard if not done already and returns it */
         ObjectGuard getGuard() const
         {
            if (_guard == NULL)
            {
               _guard = FEATSTD_NEW(ObjectGuard);
               if (_guard == NULL)
               {
                  return ObjectGuard();
               }
               _guard->setValid(true);
            }
            return *_guard;
         }

      private:
         mutable ObjectGuard* _guard;
      };


      /*****************************************************************************/
      /** Encapsulates a pointer to an object and its guard and helps to access the object safely.
       * In case the object is deleted then isValid will return false.
       */
      /*****************************************************************************/
      template <typename TObject>
      class ObjectAccessor
      {
      public:
         ObjectAccessor(TObject* obj = NULL) : _object(obj)
         {
            if (_object != NULL)
            {
               _objectGuard = _object->getGuard();
            }
         }

         ~ObjectAccessor()
         {
            _object = NULL;
         }

         /* resets the widget and its guard */
         void release()
         {
            _object = NULL;
            _objectGuard = ObjectGuard();
         }

         /* widget is not null and guard is valid */
         bool isValid() const
         {
            return (_object != NULL) && _objectGuard.isValid();
         }

         /* returns the widget if the guard is valid */
         TObject* getObjectSafely() const
         {
            return _objectGuard.isValid() ? _object : NULL;
         }

         /* returns the pointer to the object */
         TObject* operator->() const
         {
            return _object;
         }

         /* call this only if the guard is valid */
         TObject& operator*() const
         {
            return *_object;
         }

      private:
         ObjectGuard _objectGuard;
         TObject* _object;
      };


      /*****************************************************************************/
#define HMIBASE_TYPEDEF_OBJECT_ACCESSOR( CLASS ) \
      FEATSTD_LINT_SYMBOL(1516, *::Accessor, "Typedef to Accessor is also available in derived class for consistent usage.") \
      typedef hmibase::util::ObjectAccessor<CLASS> Accessor
   }
}


#endif
