/* ***************************************************************************************
* FILE:          ResourceController.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ResourceController.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

//#include "hall_std_if.h"

#include "AppHmi_MasterBase/AudioInterface/ResourceController.h"

#include "AppHmi_MasterBase/AudioInterface/Source.h"
#include "AppHmi_MasterBase/AudioInterface/Sink.h"
#include "AppHmi_MasterBase/AudioInterface/IHmiAudioServiceStubHandler.h"

#include "AppHmi_MasterBase/AudioInterface/AudioControllerObjectManager.h"
#include "AppHmi_MasterBase/AudioInterface/SoundController.h"
#include "AppUtils/HmiMacros.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_AUDIO_IF
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ResourceController.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;

namespace hmibase {
namespace apphmi_master {
namespace audio {

ResourceController::ResourceController() : m_pHmiAudioServiceStubHandler(0)
{
   for (int nID = static_cast<int>(GROUP_TUNER); nID < static_cast<int>(GROUP_COUNT); nID++)
   {
      aGroupLastSrc[nID] = NULL;
   }
}


ResourceController::~ResourceController()
{
   ::std::map<int, Source*>::iterator iter;
   for (iter = m_SrcList.begin(); iter != m_SrcList.end(); ++iter)
   {
      delete iter->second;
   }

   ::std::map<int, Sink*>::iterator iter2;
   for (iter2 = m_SinkList.begin(); iter2 != m_SinkList.end(); ++iter2)
   {
      delete iter2->second;
   }

   m_SrcList.clear();
   m_SinkList.clear();

   m_pHmiAudioServiceStubHandler = 0;
}


void ResourceController::AddSink(int nSinkID, std::string sSinkName, int nSinkAvailability, int nVolume, int nMuteState, int nClassID)
{
   ETG_TRACE_USR4_THR(("AddSink : %s", sSinkName.c_str()));
   ETG_TRACE_USR4_THR(("AddSink : Sink Id %d", nSinkID));
   Sink* pSink = new Sink(sSinkName);
   ::std::map<int, Sink*>::iterator iter = m_SinkList.find(nSinkID);
   if (iter != m_SinkList.end())
   {
      ETG_TRACE_USR4_THR(("AddSink : Sink already present with Availability %d, Volume %d, MuteState %d", nSinkAvailability, nVolume, nMuteState));
      delete pSink;
      iter->second->SetID(nSinkID);
      iter->second->SetAvailability(nSinkAvailability);
      iter->second->SetVolume(nVolume);
      iter->second->SetMuteState(nMuteState);
      iter->second->SetClassID(nClassID);
      OnUpdateSink(GetIndexOnSinkID(iter->second->GetID()));
   }
   else if (pSink != 0)
   {
      ETG_TRACE_USR4_THR(("AddSink : Sink created with Availability %d, Volume %d, MuteState %d", nSinkAvailability, nVolume, nMuteState));
      pSink->SetID(nSinkID);
      pSink->SetAvailability(nSinkAvailability);
      pSink->SetVolume(nVolume);
      pSink->SetMuteState(nMuteState);
      pSink->SetClassID(nClassID);
      m_SinkList[nSinkID] = pSink;
      AudioControllerObjectManager::getInstance().getSoundController<SoundController>()->onSinkMuteStateChangeUpdateMute(nSinkID, nMuteState);
      AudioControllerObjectManager::getInstance().getSoundController<SoundController>()->onVolumeValueChange(nSinkID, nVolume);
      OnUpdateSink(GetIndexOnSinkID(pSink->GetID()));
   }
   else
   {
      HMI_APP_ASSERT_ALWAYS();
   }
}


void ResourceController::RemoveSink(int nSinkID)
{
   ::std::map<int, Sink*>::iterator iter = m_SinkList.find(nSinkID);
   if (iter != m_SinkList.end())
   {
      int removedSinkIdx = GetIndexOnSinkID(iter->second->GetID());
      ETG_TRACE_USR4_THR(("AddSink : Sink %s removed", iter->second->GetName().c_str()));
      delete iter->second;
      m_SinkList.erase(iter);
      OnUpdateSink(removedSinkIdx);
   }
}

void ResourceController::SetGroupLastSrc(Source* pSource)
{
   if (pSource == NULL)
   {
      return;
   }
   int nSrcIndex = pSource->GetIndex();
   int nGroupId = nSrcIndex / SRC_GROUP_SIZE;
   if (nGroupId < static_cast<int>(GROUP_COUNT) && nSrcIndex < static_cast<int>(SRC_INVALID))
   {
      aGroupLastSrc[nGroupId] = pSource;
   }
}


Source* ResourceController::GetGroupLastSrc(int nGroupId)
{
   if (nGroupId < static_cast<int>(GROUP_COUNT))
   {
      return aGroupLastSrc[nGroupId];
   }
   else
   {
      return NULL;
   }
}


Sink* ResourceController::GetSinkOnID(int nSinkID)
{
   ETG_TRACE_USR4_THR(("GetSinkOnID : %d", nSinkID));
   ::std::map<int, Sink*>::iterator iter = m_SinkList.find(nSinkID);
   if (iter != m_SinkList.end())
   {
      return iter->second;
   }
   else
   {
      return NULL;
   }
}


void ResourceController::AddSource(int nSrcID, std::string sSrcName, int nSrcAvailability, int nSrcAvailabilityReason, int nClassID)
{
   ETG_TRACE_USR4_THR(("AddSource : %s", sSrcName.c_str()));
   ETG_TRACE_USR4_THR(("AddSource : Source Id %d", nSrcID));
   Source* pSource = new Source(sSrcName);
   if (pSource == NULL || !pSource->IsSourceValid())
   {
      ETG_TRACE_USR4_THR(("AddSource failed: Not a registered source %d", nSrcID));
      if (pSource != 0)
      {
         delete pSource;
      }
      return;
   }
   ::std::map<int, Source*>::iterator iter;
   iter = m_SrcList.find(nSrcID);
   if (iter != m_SrcList.end())
   {
      ETG_TRACE_USR4_THR(("AddSource : Source already present with Availability %d", nSrcAvailability));
      delete pSource;
      int srcAvailability = iter->second->GetAvailability();
      int srcAvailabilityReason = iter->second->GetAvailabilityReason();
      iter->second->SetAvailability(nSrcAvailability);
      iter->second->SetAvailabilityReason(nSrcAvailabilityReason);
      iter->second->SetClassID(nClassID);
      if ((srcAvailability != nSrcAvailability) || (srcAvailabilityReason != nSrcAvailabilityReason))
      {
         UpdateSourceListChange(iter->second->GetIndex());
         UpdateSourceAvailabilityChange(iter->second->GetIndex(), iter->second->GetDeviceId(), nSrcAvailability, nSrcAvailabilityReason);
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("AddSource : Source created with Availability %d", nSrcAvailability));
      pSource->SetID(nSrcID);
      pSource->SetAvailability(nSrcAvailability);
      pSource->SetAvailabilityReason(nSrcAvailabilityReason);
      pSource->SetClassID(nClassID);
      m_SrcList[nSrcID] = pSource;
      UpdateSourceListChange(pSource->GetIndex());
      UpdateSourceAvailabilityChange(pSource->GetIndex(), pSource->GetDeviceId(), nSrcAvailability, nSrcAvailabilityReason);
   }
}


void ResourceController::RemoveSource(int nSrcID)
{
   ::std::map<int, Source*>::iterator iter;
   iter = m_SrcList.find(nSrcID);
   if (iter != m_SrcList.end())
   {
      ETG_TRACE_USR4_THR(("RemoveSource : Source %s removed", iter->second->GetName().c_str()));
      int srcIndex = iter->second->GetIndex();
      removeGroupLastSrc(iter->second);
      delete iter->second;
      m_SrcList.erase(iter);
      UpdateSourceListChange(srcIndex);
   }
}


Source* ResourceController::GetSourceOnID(int nSrcID)
{
   ::std::map<int, Source*>::iterator iter;
   iter = m_SrcList.find(nSrcID);
   if (iter != m_SrcList.end())
   {
      return (iter->second);
   }
   else
   {
      return NULL;
   }
}


void ResourceController::mapSinkIdx(uint16_t nSinkIndex)
{
   switch (nSinkIndex)
   {
      case COCKPIT_SINK_ID :
      {
         m_sink_name = COCKPIT_SINK;
         break;
      }
      case PRIVATE_SPEAKER_SINK_ID :
      {
         m_sink_name = PRIVATE_SPEAKER_SINK;
         break;
      }
      case CABIN_A_SINK_ID :
      {
         m_sink_name = CABIN_A_SINK;
         break;
      }
      case CABIN_B_SINK_ID :
      {
         m_sink_name = CABIN_B_SINK;
         break;
      }
      case CABIN_A_MEDIAGLASS1_SINK_ID :
      {
         m_sink_name = CABIN_A_MEDIAGLASS1_SINK;
         break;
      }
      case CABIN_A_MEDIAGLASS2_SINK_ID :
      {
         m_sink_name = CABIN_A_MEDIAGLASS2_SINK;
         break;
      }
      case CABIN_B_MEDIAGLASS1_SINK_ID :
      {
         m_sink_name = CABIN_B_MEDIAGLASS1_SINK;
         break;
      }
      case CABIN_B_MEDIAGLASS2_SINK_ID :
      {
         m_sink_name = CABIN_B_MEDIAGLASS2_SINK;
         break;
      }
      default :
      {
         m_sink_name = "";
         break;
      }
   }
   ETG_TRACE_USR4_THR(("mapSinkIdx : Sink index %d mapped to sink name %s", nSinkIndex, m_sink_name.c_str()));
}


uint16_t ResourceController::mapSinkName(const std::string& nSinkName)
{
   uint16_t nsinkIdx = SINK_INVALID;
   if (nSinkName == COCKPIT_SINK)
   {
      nsinkIdx = COCKPIT_SINK_ID;
   }
   else if (nSinkName == PRIVATE_SPEAKER_SINK)
   {
      nsinkIdx = PRIVATE_SPEAKER_SINK_ID;
   }
   else if (nSinkName == CABIN_A_SINK)
   {
      nsinkIdx = CABIN_A_SINK_ID;
   }
   else if (nSinkName == CABIN_B_SINK)
   {
      nsinkIdx = CABIN_B_SINK_ID;
   }
   else if (nSinkName == CABIN_A_MEDIAGLASS1_SINK)
   {
      nsinkIdx = CABIN_A_MEDIAGLASS1_SINK_ID;
   }
   else if (nSinkName == CABIN_A_MEDIAGLASS2_SINK)
   {
      nsinkIdx = CABIN_A_MEDIAGLASS2_SINK_ID;
   }
   else if (nSinkName == CABIN_B_MEDIAGLASS1_SINK)
   {
      nsinkIdx = CABIN_B_MEDIAGLASS1_SINK_ID;
   }
   else if (nSinkName == CABIN_B_MEDIAGLASS2_SINK)
   {
      nsinkIdx = CABIN_B_MEDIAGLASS2_SINK_ID;
   }
   else
   {}
   ETG_TRACE_USR4_THR(("mapSinkName : Sink index %d mapped to sink name %s", nsinkIdx, nSinkName.c_str()));
   return nsinkIdx;
}


Sink* ResourceController::GetSinkOnIdx()
{
   ::std::map<int, Sink*>::iterator iter;

   Sink* sink = NULL;
   for (iter = m_SinkList.begin(); iter != m_SinkList.end(); ++iter)
   {
      if (m_sink_name == iter->second->GetName())
      {
         sink = (iter->second);
      }
   }
   return sink;
}


uint16_t ResourceController::GetIndexOnSinkID(int nSinkID)
{
   uint16_t nSinkIdx = SINK_INVALID;
   ::std::map<int, Sink*>::iterator iter;

   Sink* sink = NULL;
   for (iter = m_SinkList.begin(); iter != m_SinkList.end(); ++iter)
   {
      if (nSinkID == iter->second->GetID())
      {
         sink = (iter->second);
      }
   }
   if (NULL != sink)
   {
      nSinkIdx = mapSinkName(sink->GetName());
   }

   return nSinkIdx;
}


Source* ResourceController::GetSourceOnIdx(int nSrcIndex, int nDevID)
{
   ::std::map<int, Source*>::iterator iter;
   for (iter = m_SrcList.begin(); iter != m_SrcList.end(); ++iter)
   {
      if (iter->second->GetIndex() == nSrcIndex && iter->second->GetDeviceId() == nDevID)
      {
         break;
      }
   }
   if (iter != m_SrcList.end())
   {
      return (iter->second);
   }
   else
   {
      return NULL;
   }
}


void ResourceController::GetGroupSrcList(int32 nGrpId, std::vector< sourceDetails >* pSrcList)
{
   pSrcList->clear();
   ::std::map<int, Source*>::iterator iter;
   for (iter = m_SrcList.begin(); iter != m_SrcList.end(); ++iter)
   {
      if (iter->second->GetIndex() / SRC_GROUP_SIZE == nGrpId)
      {
         sourceDetails srcDetails(iter->second->GetIndex(), iter->second->GetDeviceId(), iter->second->GetAvailability(), iter->second->GetAvailabilityReason());
#ifdef XXX
         pSrcList->push_back(boost::new_clone(srcDetails));
#else
         pSrcList->push_back(srcDetails);
#endif
      }
   }
}


Source* ResourceController::GetAvailableSrc(int nGrpId)
{
   int srcIdx = static_cast<int>(SRC_INVALID);
   Source* pSource = NULL;
   ::std::map<int, Source*>::iterator iter;
   for (iter = m_SrcList.begin(); iter != m_SrcList.end(); ++iter)
   {
      srcIdx = iter->second->GetIndex();
      if ((srcIdx / SRC_GROUP_SIZE == nGrpId) && (iter->second->GetAvailability() == RES_AVAILABLE) && (iter->second->GetType() == static_cast<int>(SOURCE_MAIN)))
      {
         pSource = iter->second;
         break;
      }
   }
   return pSource;
}


void ResourceController::UpdateSinkAvailability(int nSinkID, int nAvailability)
{
   ::std::map<int, Sink*>::iterator iter = m_SinkList.find(nSinkID);
   if (iter != m_SinkList.end())
   {
      iter->second->SetAvailability(nAvailability);
   }
}


void ResourceController::UpdateSourceAvailability(int nSourceID, int nAvailability, int nAvailabilityReason)
{
   ::std::map<int, Source*>::iterator iter = m_SrcList.find(nSourceID);
   if (iter != m_SrcList.end())
   {
      iter->second->SetAvailabilityReason(nAvailabilityReason);
      if (iter->second->GetAvailability() != nAvailability)
      {
         iter->second->SetAvailability(nAvailability);
         UpdateSourceListChange(iter->second->GetIndex());
         UpdateSourceAvailabilityChange(iter->second->GetIndex(), iter->second->GetDeviceId(), nAvailability, nAvailabilityReason);
      }
   }
}


void ResourceController::UpdateSourceListChange(int32 nSrcIndex)
{
   if (m_pHmiAudioServiceStubHandler)
   {
      m_pHmiAudioServiceStubHandler->sendSourceListUpdateRequest(nSrcIndex / SRC_GROUP_SIZE);
   }
}


void ResourceController::UpdateSourceAvailabilityChange(int32 nSrcIndex, int32 nSubSrcId, int32 nAvailability, int32 nAvailabilityReason)
{
   if (m_pHmiAudioServiceStubHandler)
   {
      m_pHmiAudioServiceStubHandler->sendSourceAvailabilityUpdateRequest(nSrcIndex, nSubSrcId, nAvailability, nAvailabilityReason);
   }
}


void ResourceController::removeGroupLastSrc(const hmibase::apphmi_master::audio::Source* pSource)
{
   if (pSource != NULL)
   {
      int nGrpId = pSource->GetIndex() / SRC_GROUP_SIZE;
      if ((nGrpId < static_cast<int>(GROUP_COUNT)) && (pSource == aGroupLastSrc[nGrpId]))
      {
         aGroupLastSrc[nGrpId] = NULL;
      }
   }
}


}
}


}
