/* ***************************************************************************************
* FILE:          ScreenBrokerClient.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScreenBrokerClient.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(ScreenBrokerClient_h)
#define ScreenBrokerClient_h

#include "AppBase/ScreenBrokerClient/IAppViewSettings.h"
#include <ScreenBroker/Client/ClientApi.h>
#include <ScreenBroker/PopupPresentationArg.h>
#include <ScreenBroker/SurfaceState.h>
#include <ScreenBroker/LayerState.h>
#include "BaseContract/generated/BaseMsgs.h"
#include "BaseContract/generated/BaseTypes_SurfaceLayerIds.h"
#include "AppBase/SystemConfiguration.h"
#include "hmibase/util/Semaphore.h"
#include "hmibase/util/Mutex.h"
#include "hmibase/util/ItemRegistry.h"
#include <map>
#include <vector>
#include <string>
#ifdef VARIANT_S_FTR_ENABLE_SBCLIENT_MOCK
#include "AppUtils/Timer.h"
#endif


#ifndef GEN3ARM
// workaround for single display setup. So an application can run on any device independent from real display id
// can be removed when final solution for CMG3G-16020 is available
#define VARIANT_S_FTR_ENABLE_IGNORE_DISPLAYID_IN_SINGLEDISPLAY_SETUP
#endif

namespace asf {
namespace core {
class BaseComponent;
}


}

class WindowInputEventHook;

class ScreenBrokerClient :
   public ScreenBroker::ClientApi
{
   public:

      ///
      bool InitClient(::asf::core::BaseComponent* baseComponent);

      ///
      static ScreenBrokerClient& GetInstance();

      ///
      ScreenBroker::UInt32 AppId() const;

      ///
      ScreenBroker::UInt32 SurfaceId() const;

      ///
      ScreenBroker::UInt32 StatusBarSurfaceId() const;

      ScreenBroker::UInt32 FooterBarSurfaceId() const;

      ///
      ScreenBroker::UInt32 VideoSurfaceId() const;

      ///
      ScreenBroker::Scaling::Enum Scaling() const;

      ///
      bool Init(ScreenBroker::UInt32 appId,
                std::vector<unsigned int> MainSurfaces,
                std::vector<unsigned int> VideoSurfaces,
                std::vector<unsigned int> PopupSurfaces,
                ScreenBroker::Scaling::Enum scaling,
                const IAppViewSettings* appViewSettings,
                WindowInputEventHook* inputEventHook = 0);

      ///
      void Terminate() const;
      ///
      void Synchronize() const;

      void Show(const ::hmibase::view::ViewIdentifier& viewId,
                ScreenBroker::UInt32 surfaceId,
                bool bIsPopup = false,
                bool bForce = false,
                ScreenBroker::UInt32 displayId = static_cast<ScreenBroker::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY),
                ScreenBroker::UInt32 entryCustomAnimationType = 0,//AnimationType::None
                ScreenBroker::UInt32 exitCustomAnimationType = 0);//AnimationType::None

      void Hide(const ::hmibase::view::ViewIdentifier& viewId,
                ScreenBroker::UInt32 surfaceId,
                bool bIsPopup = false,
                ScreenBroker::UInt32 displayId = static_cast<ScreenBroker::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY),
                ScreenBroker::UInt32 entryCustomAnimationType = 0,//AnimationType::None
                ScreenBroker::UInt32 exitCustomAnimationType = 0);//AnimationType::None);

      void StartAnimation(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 animationType);
      void StartLayerAnimation(ScreenBroker::UInt32 layerId, std::vector<ScreenBroker::UInt32> surfaceList , ScreenBroker::UInt32 animationType);
#ifndef VARIANT_S_FTR_ENABLE_SBCLIENT_MOCK

      //This API is not thread-safe by intend. Reason is that it can only be called by Screen Broker. It is a kind of
      //callback called from Screen Broker once the state of a surface changed. No other component may call this method.
      //The same applies for SetTouchAcceptance() which is called from OnSurfaceStateChanged() via DisableTouch() in case
      //of animations.
      virtual void OnSurfaceStateChanged(ScreenBroker::UInt32 userData,
                                         ScreenBroker::UInt32 surfaceId,
                                         ScreenBroker::UInt32 displayId,
                                         ScreenBroker::UInt32 displayAlias,
                                         ScreenBroker::SurfaceState::Enum surfaceState);
      ///
      virtual void OnCurrentStatus(ScreenBroker::UInt32 requestId,
                                   ScreenBroker::UInt32 surfaceId,
                                   ScreenBroker::UInt32 status,
                                   ScreenBroker::UInt32 userData);
      ///
      virtual void OnLayerStateChanged(ScreenBroker::UInt32 requestId,
                                       ScreenBroker::UInt32 layerId,
                                       ScreenBroker::LayerState::Enum layerState);

      ///
      virtual void OnCloseOnExternalTouch(bool status);
#endif

      void SetPopupFilterAll(ScreenBroker::UInt32 displayID = static_cast<ScreenBroker::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY));

      void SetPopupFilterPrio(ScreenBroker::UInt32 prio, ScreenBroker::UInt32 displayID = static_cast<ScreenBroker::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY));

      // use suffix 'Public'for the following method,
      // as the same names are already used for static methods in the base class
      ///
      void ClearPopupFilterPublic(ScreenBroker::UInt32 displayID = static_cast<ScreenBroker::UInt32>(hmibase::DEFAULT_UNKNOWN_DISPLAY));

      /// Abort all running/queued animations
      void AbortAnimations(void) const;

      /// ExplicitCommit
      void ExplicitSBCommit(void) const;

      /// Set input focus for a certain surface of this HMI-App with focus priority
      void SetInputFocusPublic(ScreenBroker::UInt32 surfaceId,
                               bool keyboardFocus,
                               bool pointerFocus,
                               ScreenBroker::Int32 priority,
                               ScreenBroker::UInt32 focusData) const;

      /// Set input focus for a certain surface of this HMI-App without focus priority
      void SetInputFocusPublic(ScreenBroker::UInt32 surfaceId,
                               bool keyboardFocus,
                               bool pointerFocus) const;

      ///
      void SendClosePopupOnExternalTouch(ScreenBroker::UInt32 surfaceId);

      inline ScreenBroker::SurfaceState::Enum GetStatusBarSurfaceState() const
      {
         return _mStatusBarSurfaceState;
      }
      inline ScreenBroker::SurfaceState::Enum GetVideoSurfaceState() const
      {
         return _mVideoSurfaceState;
      }
      inline ScreenBroker::SurfaceState::Enum GetFooterBarSurfaceState() const
      {
         return _mFooterBarSurfaceState;
      }

      // Returns the surface state of the corresponding popup surface
      ScreenBroker::SurfaceState::Enum GetPopupSurfaceState(ScreenBroker::UInt32 surfaceId);

      ScreenBroker::SurfaceState::Enum GetMainSurfaceState(ScreenBroker::UInt32 surfaceId);
      bool IsMainSurface(ScreenBroker::UInt32 surfaceId);
      ScreenBroker::UInt32 GetDisplayID(ScreenBroker::UInt32 surfaceId);

      void GetSurfaces(const ScreenBroker::UInt32& displayId, std::list<ScreenBroker::UInt32>* surfaceList);

      bool CheckSurfaceIdDisplayIdMapping(ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 displayId);
      typedef hmibase::util::ItemValRegistry<ScreenBroker::UInt32, ScreenBroker::UInt32> SurfaceDisplayMap;

   private:

      typedef std::map<uint32_t, ::hmibase::view::ViewIdentifier> ReverseViewHashMap;
      //stores mapping of hard key states (comming from input driver) to hmi key states

      typedef std::map<ScreenBroker::KeyState::Enum, hmibase::HardKeyStateEnum> KeyStateMap;
      //stores surface that are under animation and had no touch acceptance set when anim was started
      typedef std::vector<ScreenBroker::UInt32> NoTouchSurfaceVec;
      //stores the popup surface ID that is received by Init()
      typedef std::map<ScreenBroker::UInt32, ScreenBroker::SurfaceState::Enum> SurfaceStatesMap;

      hmibase::util::CriticalSection _mCs;

      ReverseViewHashMap _mReverseViewHashMap;
      const IAppViewSettings* _mAppViewSettings;

      NoTouchSurfaceVec _mNoTouchSurfaceVec;
      SurfaceStatesMap _mPopupSurfaceStatesMap;
      SurfaceStatesMap _mMainSurfacesMap;
      SurfaceStatesMap _mVideoSurfacesMap;

      bool _mInitialized;
      ScreenBroker::UInt32 _mAppId;
      ScreenBroker::UInt32 _mMainSurfaceId;
      ScreenBroker::UInt32 _mStatusBarSurfaceId;
      ScreenBroker::UInt32 _mFooterBarSurfaceId;
      ScreenBroker::UInt32 _mVideoSurfaceId; // this surface can be used for map or video
      ScreenBroker::Scaling::Enum _mScaling;

      ScreenBroker::SurfaceState::Enum _mVideoSurfaceState;
      ScreenBroker::SurfaceState::Enum _mStatusBarSurfaceState;
      ScreenBroker::SurfaceState::Enum _mFooterBarSurfaceState;

#if defined VARIANT_S_FTR_ENABLE_IGNORE_DISPLAYID_IN_SINGLEDISPLAY_SETUP
      bool isMultiDisplaySetup;
#endif

#ifdef VARIANT_S_FTR_ENABLE_SBCLIENT_MOCK
      bool hasPopup(unsigned int surfaceId);
      void updateSurfaceState(ScreenBroker::UInt32 surfaceId, int state, const ::hmibase::view::ViewIdentifier& viewId);

      std::map<unsigned int, std::vector<::hmibase::view::ViewIdentifier>> popupsOnSurface;
      Util::Timer _popupTimer;
#endif

      WindowInputEventHook* _mInputEventHook;
      WindowInputEventHook* getWindowInputEventHook()
      {
         return _mInputEventHook;
      }

      ///
      ScreenBrokerClient();

      ///
      void ShowSurface(ScreenBroker::UInt32 surfaceId,
                       ScreenBroker::UInt32 userData,
                       ScreenBroker::UInt32 requestId,
                       ScreenBroker::UInt32 entryCustomAnimationType,//AnimationType::None
                       ScreenBroker::UInt32 exitCustomAnimationType) const; //AnimationType::None)

      ///
      void HideSurface(ScreenBroker::UInt32 surfaceId,
                       ScreenBroker::UInt32 userData,
                       ScreenBroker::UInt32 requestId,
                       ScreenBroker::UInt32 entryCustomAnimationType,//AnimationType::None
                       ScreenBroker::UInt32 exitCustomAnimationType) const; //AnimationType::None)

      // use suffix 'Private'for the following method,
      // as the same names are already used for static methods in the base class
      ///
      void ShowPopupPrivate(ScreenBroker::UInt32 surfaceId,
                            ScreenBroker::UInt32 userData,
                            ScreenBroker::UInt32 requestId,
                            const ScreenBroker::PopupPresentationArg& popupPresentationArg) const;

      // use suffix 'Private'for the following method,
      // as the same names are already used for static methods in the base class
      ///
      void HidePopupPrivate(ScreenBroker::UInt32 surfaceId,
                            ScreenBroker::UInt32 userData,
                            ScreenBroker::UInt32 requestId) const;

      /** Sets a popup filter, at the moment only two kinds of filters are possible:
       * Disable all popups: set disableAll = true
       * Disable all popups < a certain priority, therefore set filter.SetPriority(prio)
       */
      // use suffix 'Private'for the following method,
      // as the same names are already used for static methods in the base class
      void SetPopupFilterPrivate(bool disableAll,
                                 ScreenBroker::UInt32 displayID,
                                 const ScreenBroker::PopupPresentationArg& filter = ScreenBroker::PopupPresentationArg());

      bool DisableTouch(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 surfaceId, bool animRunning);

      //This API is not thread-safe by intend. Reason is that it can only be called by Screen Broker (indirectly via OnSurfaceStateChanged()
      //and SetTouchAcceptance()). No other component than Screen Broker may call this method.
      ///disables touch while an animation is running for main view surfaces
      bool SetTouchAcceptance(ScreenBroker::UInt32 surfaceId, bool enable);

      bool SendPopupAnimStatus(const ::hmibase::view::ViewIdentifier& viewId, bool animRunning);

      //checks if the surface is in the popup surface map, which means this surface is a popup surface
      bool IsInPopupSurfaceMap(ScreenBroker::UInt32 surfaceId);

      ::hmibase::view::ViewIdentifier& GetReverseViewHashMap(ScreenBroker::UInt32 userData);
      void SetReverseViewHashMap(const ::hmibase::view::ViewIdentifier& viewId, ScreenBroker::UInt32 lUserData);

      //map displayid for the surfaceid
      void SetDisplayID(ScreenBroker::UInt32 surfaceId, ScreenBroker::UInt32 displayId, ScreenBroker::UInt32 displayAlias);
};


#endif
