/* ***************************************************************************************
* FILE:          LcmHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  LcmHandler.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "sys_std_if.h"
#include "LcmHandler.h"
#include "spm_core_fi_typesConst.h"
#include "View/IMessageSystem.h"

using namespace ::org::genivi::NodeStateManager::Consumer;
using namespace ::org::genivi::NodeStateManager::LifeCycleConsumer;
using namespace ::org::genivi::NodeStateManager::LifecycleControl;
using namespace ::org::bosch::cm::lcm;
using namespace ::org::bosch::cm::lcm::lcmbaseappcomponent;
using namespace ::asf::core;

using namespace SPM_CORE_FI;
using namespace spm_core_fi_types;

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/LcmHandler.cpp.trc.h"
#endif

namespace hmibase {
namespace app {
namespace base {
namespace lcm {

LcmHandler::LcmHandler(const std::string& busName, const std::string& objName, LcmHandlerCallback* cb)
   : _lcmAppComponent(busName /*"org.bosch.cm.lcm.exampleapp.ExampleApp"*/,
                      objName /*"/org/genivi/NodeStateManager/LifeCycleConsumer/ExampleApplication"*/,
                      *this)
   , _spmProxy(SPM_CORE_FIProxy::createProxy("spmFiPort", *this))
   , _cb(cb)
   , _appReady(false)
{
   ETG_TRACE_USR1_THR(("LcmHandler called"));

   /* you can use the following function to get the command line arguments
   getCommandLineArguments()

   */
   /* call function setAppStartReady() whenever the application is fully started and dependend apps
      can be started by systemd.
      ATTENTION: This must not necessarily be done here. Any other place e.g. onAvailable might be
      more suitable for your application
   */
   //_lcmAppComponent.setAppStartReady();
}


LcmHandler::~LcmHandler()
{
   ETG_TRACE_USR1_THR(("~LcmHandler called"));
   _cb = 0;
}


// TimerCallbackIF implementation
void LcmHandler::onExpired(asf::core::Timer& /*timer*/, boost::shared_ptr<asf::core::TimerPayload> /*payload*/)
{
   ETG_TRACE_USR1_THR(("onExpired ..........."));
}


// LCM Interface implementation
/**
 **************************************************************************************************
 * \brief this function is called from the LcmBaseAppComponent if a new application mode is requested from LCM
 *        here the transition actions need to be done
 *        This function is mandatory
 *
 * \param[in] newAppMode - new application mode that is requested
 * \param[in] oldAppMode - previous application mode
 * \param[out] None
 * \param[inout] None
 * \return error code
 * typedef enum _NsmErrorStatus_e
 * {
 *    NsmErrorStatus_NotSet,               *< Initial value when error type is not set            *
 *    NsmErrorStatus_Ok,                   *< Value when no error occurred                        *
 *    NsmErrorStatus_Error,                *< A general, non-specific error occurred              *
 *    NsmErrorStatus_Dbus,                 *< Error in D-Bus communication                        *
 *    NsmErrorStatus_Internal,             *< Internal error (memory alloc. failed, etc.)         *
 *    NsmErrorStatus_Parameter,            *< A passed parameter was incorrect                    *
 *    NsmErrorStatus_WrongSession,         *< The requested session is unknown.                   *
 *    NsmErrorStatus_ResponsePending,      *< Command accepted, return value delivered asynch.    *
 *    NsmErrorStatus_Last                  *< Last error value to identify valid errors           *
 *} NsmErrorStatus_e;
 *
 **************************************************************************************************
 */

_NsmErrorStatus_e LcmHandler::OnAppModeChange(_Nsm_Shutdown_Type_e newAppMode, _Nsm_Shutdown_Type_e oldAppMode)
{
   _NsmErrorStatus_e returnVal = _NsmErrorStatus_e__NsmErrorStatus_NotSet;

   ETG_TRACE_USR1_THR(("OnAppModeChange - %i", newAppMode));

   switch (newAppMode)
   {
      case _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Runup:
      case _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Normal:
      case _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Fast:
         returnVal = _NsmErrorStatus_e__NsmErrorStatus_Ok;
         break;
      default:
         returnVal = _NsmErrorStatus_e__NsmErrorStatus_Error;
         break;
   }

   if (_cb)
   {
      if (_cb->OnAppModeChange(mapPowerState(newAppMode), mapPowerState(oldAppMode)) == false)
      {
         ETG_TRACE_USR1_THR(("LcmHandler::OnAppModeChange Response to LCM is pending by app %s", hmibase::trace::getAppName().c_str()));
         returnVal = _NsmErrorStatus_e__NsmErrorStatus_ResponsePending;
      }
   }
   else
   {
      ETG_TRACE_FATAL_THR(("LcmHandler::OnAppModeChange no callback component in app %s", hmibase::trace::getAppName().c_str()));
   }
   ETG_TRACE_USR1_THR(("LcmHandler::OnAppModeChange RC=%d, PID=%d", (unsigned)returnVal, static_cast<uint32>(getpid())));
   return returnVal;
}


ShutdownType LcmHandler::mapPowerState(_Nsm_Shutdown_Type_e mode)
{
   ShutdownType returnVal = SHUTDOWN_NOT;
   switch (mode)
   {
      case _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Runup:
         returnVal = SHUTDOWN_RUNUP;
         break;
      case _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Normal:
         returnVal = SHUTDOWN_NORMAL;
         break;
      case _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Fast:
         returnVal = SHUTDOWN_FAST;
         break;
      default:
         break;
   }
   return returnVal;
}


/**
 **************************************************************************************************
 * \brief this function is called from the LCMBaseApp if the application watchdog timer has run out
 *        This function is mandatory
 *
 * \param[in]
 * \param[out] None
 * \param[inout] None
 * \return TRUE  - all is OK
 *         FALSE - some inconsistency in the application state was detected
 *
 **************************************************************************************************
 */
bool LcmHandler::OnAppWatchdog()
{
   bool returnVal = false;
//   ETG_TRACE_USR1_THR(("OnAppWatchdog ..........."));
   /****************************
   *   Example How-To
      // check consistency of your app here
   *****************************/
   returnVal = true; //lint !e838 // only if all is OK
   return returnVal;
}


/**
 **************************************************************************************************
 * \brief this function is called from the LCMBaseApp if all LCM services have been registered
 *
 * \param[in]
 * \param[out] None
 * \param[inout] None
 * \return none
 *
 **************************************************************************************************
 */
void LcmHandler::OnLcmRegistered()
{
   ETG_TRACE_USR1_THR(("OnLcmRegistered ..........."));
   /****************************
   *   Example How-To
      // now all SPM services are available
   *****************************/
}


/**
 **************************************************************************************************
 * \brief this function is called from the LCMBaseApp before the App change request to RunUp
 *
 * \param[in]
 * \param[out] None
 * \param[inout] None
 * \return none
 *
 **************************************************************************************************
 */
void LcmHandler::onLoadPersistency()
{
   ETG_TRACE_USR1_THR(("onLoadPersistency ..........."));
   if (_cb)
   {
      _cb->onLoadPersistency();
   }
}


/**
 **************************************************************************************************
 * \brief this function is called from the LCMBaseApp after the App change request to Shutdown
 *
 * \param[in]
 * \param[out] None
 * \param[inout] None
 * \return none
 *
 **************************************************************************************************
 */
void LcmHandler::onSavePersistency()
{
   ETG_TRACE_USR1_THR(("onSavePersistency ..........."));
   if (_cb)
   {
      _cb->onSavePersistency();
   }
}


void LcmHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_spmProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("HmiStateHandler spmProxy available"));
      _spmProxy->sendHmiStateUpReg(*this);
      _spmProxy->sendHmiStateGet(*this);
   }
}


void LcmHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_spmProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("HmiStateHandler spmProxy unavailable"));
      _spmProxy->sendHmiStateRelUpRegAll();
   }
}


void LcmHandler::onHmiStateError(const ::boost::shared_ptr< SPM_CORE_FIProxy >& /*proxy*/, const boost::shared_ptr< HmiStateError >& /*error*/)
{
}


void LcmHandler::onHmiStateStatus(const ::boost::shared_ptr< SPM_CORE_FIProxy >& /*proxy*/, const boost::shared_ptr< HmiStateStatus >& status)
{
   sendHmiStateMessage(status->getHmiState());
}


void LcmHandler::sendHmiStateMessage()
{
   if (_spmProxy->hasHmiState())
   {
      sendHmiStateMessage(_spmProxy->getHmiState().getHmiState());
   }
   else
   {
      ETG_TRACE_FATAL_THR(("No HmiState available from SPM, service availability %d", _spmProxy->isAvailable()));
#if(defined (GEN4ARM) || defined(GEN4INTEL) || defined(GEN4LSIM))
      ETG_TRACE_FATAL_THR(("No HmiState available from SPM, service availability %d force HMI on", _spmProxy->isAvailable()));
      sendHmiStateMessage(T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI);
#endif
   }
}


void LcmHandler::sendHmiStateMessage(::spm_core_fi_types::T_SPM_e32_HMI_STATE state)
{
   hmibase::HMIStateEnum val = hmibase::HMISTATE_SHOW_HMI;
   switch (state)
   {
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF:
         val = hmibase::HMISTATE_SHOW_OFF;
         break;
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_CLOCK:
         val = hmibase::HMISTATE_SHOW_CLOCK;
         break;
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY_RESTRICTED:
         val = hmibase::HMISTATE_STANDBY_RESTRICTED;
         break;
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY:
         val = hmibase::HMISTATE_STANDBY;
         break;
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF_PHONE:
         val = hmibase::HMISTATE_PHONE;
         break;
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SECURE_GREEN:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SECURE_YELLOW:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SECURE_RED:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SECURE_WAIT:
         val = hmibase::HMISTATE_SECURE;
         break;
      default:
         break;
   }
   ETG_TRACE_USR4_THR(("LcmHandler::sendHmiStateMessage %d", ETG_CENUM(T_SPM_e32_HMI_STATE, state)));
   if (_appReady)
   {
      if (_cb)
      {
         _cb->onAppSubStateReady(val);
      }
   }
}


void LcmHandler::setAppStartReady()
{
   _lcmAppComponent.setAppStartReady();
   _appReady = true;
   sendHmiStateMessage();
}


}
}


}
}
