/* ***************************************************************************************
* FILE:          IApplicationSettings.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IApplicationSettings.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef _IApplicationSettings_H_
#define _IApplicationSettings_H_

#ifdef WIN32

class KeyMappingBase
{
};


class Keymapping : public KeyMappingBase
{
};


#else
#include "ScreenBrokerClient/KeyMappingBase.h"
#include "Gadget/SyncBlockProducerFactory.h"
#endif

#include "ScreenBrokerClient/IAppViewSettings.h"
#include "hmibase/util/TraceUtils.h"
#include "AppBase/ScreenBrokerClient/IUserInputObserver.h"

namespace hmibase {
struct DisplayParameters
{
   char id;
   int width;
   int height;
};


struct SurfaceIds
{
   std::vector<unsigned int> mainSurfaces;
   std::vector<unsigned int> popupSurfaces;
   std::vector<unsigned int> videoSurfaces;
   std::vector<unsigned int> permanentSurfaces;
};


typedef enum
{
   MAIN_SURFACE,
   VIDEO_SURFACE,
   POPUP_SURFACE,
   PERMANENT_SURFACE
} SurfaceType;
struct AppSettingsSurfaceIDs
{
   unsigned int surfaceID;
   SurfaceType st;
};


struct ScreenBrokerSetting
{
   SurfaceIds ids;
   ScreenBroker::Scaling::Enum scaling;
   const IAppViewSettings* appViewSettings;
};


}

class IApplicationSettings
{
   public:
      virtual ~IApplicationSettings() {}

      virtual hmibase::DisplayParameters& getDisplayParameters() const = 0;
      virtual hmibase::trace::sCGITraceGroup getCourierTraceGroup() const = 0;
      virtual hmibase::ScreenBrokerSetting& getScreenBrokerSettings() const = 0;
      virtual unsigned int getSyncBlockID() const = 0;
      virtual unsigned int getNoOfSyncBlockProducerInstances() const = 0;
      virtual ::hmibase::input::IUserInputObserver* getUserInputObserver() const = 0;
};


#ifdef WIN32

#define DEFAULT_APPSETTINGS(_mainSurfaceId)                    \
class AppSettings : public IApplicationSettings                \
{                                                              \
public:                                                        \
   AppSettings():_syncBlockId(0),_noOfSyncBlockProducerInstances(0), _uiObserver(0), _width(-1), _height(-1) \
   {                                                           \
   }                                                           \
                                                               \
                                                              \
   void setUserInputObserver(::hmibase::input::IUserInputObserver* uiObserver)   \
   {                                                           \
      _uiObserver = uiObserver;                                \
   }                                                           \
                                                               \
   ::hmibase::input::IUserInputObserver* getUserInputObserver() const            \
   {                                                           \
      return _uiObserver;                                      \
   }                                                           \
                                                               \
   hmibase::DisplayParameters& getDisplayParameters() const    \
   {                                                           \
      static hmibase::DisplayParameters displayParams;         \
      displayParams.id = 3;                                    \
      displayParams.width = _width;                            \
      displayParams.height = _height;                          \
                                                               \
      return displayParams;                                    \
   }                                                           \
                                                               \
   hmibase::trace::sCGITraceGroup getCourierTraceGroup() const   \
   {                                                           \
      return COURIER_APPLICATION_TRACE_GROUP;                  \
   }                                                           \
                                                               \
   hmibase::ScreenBrokerSetting& getScreenBrokerSettings() const  \
   {                                                           \
      static hmibase::ScreenBrokerSetting sbSettings;          \
      sbSettings.ids.mainSurfaces.push_back(_mainSurfaceId);   \
      return sbSettings;                                       \
   }                                                           \
                                                               \
   unsigned int getSyncBlockID() const                         \
   {                                                           \
      return _syncBlockId;                                     \
   }                                                           \
                                                               \
   void setSyncBlockID(unsigned int id)                        \
   {                                                           \
      _syncBlockId = id;                                       \
   }                                                           \
                                                               \
                                                               \
   void setResolution(Candera::Vector2 resolution)             \
   {                                                           \
      _width = (int)(resolution.GetX()+0.5f);         \
      _height = (int)(resolution.GetY()+0.5f);        \
   }                                                           \
   unsigned int getNoOfSyncBlockProducerInstances() const      \
   {                                                           \
      return _noOfSyncBlockProducerInstances;                  \
   }                                                           \
                                                               \
   void setNoOfSyncBlockProducerInstances(unsigned int no)     \
   {                                                           \
      _noOfSyncBlockProducerInstances = no;                    \
   }                                                           \
                                                               \
private:                                                       \
   unsigned int _syncBlockId;                                  \
   unsigned int _noOfSyncBlockProducerInstances;               \
   ::hmibase::input::IUserInputObserver* _uiObserver;          \
   int _width;                                                 \
   int _height;                                                \
};                                                             \
                                                               \
static AppSettings _appSettings;                               \

#define SYNC_BLOCK_CONSUMER_ID(x)  appSettings.setSyncBlockID(x);
#define SYNC_BLOCK_PROVIDER_INSTANCES(x) appSettings.setNoOfSyncBlockProducerInstances(x);
#define SYNC_BLOCK_CONNECTION(syncBlockId, producerInstanceId)

#else

#ifndef VARIANT_S_FTR_ENABLE_QT_MESSAGING
#define KEYMAPPING_CGI                                                                  \
   KeyMappingBase& getKeyMapping()                                          		    \
   {                                                                                    \
      static KeyMapping keyMapping;                                                     \
      return keyMapping;                                                                \
   }
#else
#define KEYMAPPING_CGI  // not used in Qt
#endif


#define DEFAULT_APPSETTINGS(_mainSurfaceId, _videoSurfaceId, ...)                       \
\
KEYMAPPING_CGI                                                                          \
\
class AppSettings : public IApplicationSettings                                         \
{                                                                                       \
public:                                                                                 \
   AppSettings():_syncBlockId(0),_noOfSyncBlockProducerInstances(0), _uiObserver(0) {}; \
   virtual ~AppSettings() { _uiObserver = 0; };                                         \
                                                                                        \
                                                                                        \
   void setUserInputObserver(::hmibase::input::IUserInputObserver* uiObserver)   \
   {                                                           \
      _uiObserver = uiObserver;                                \
   }                                                           \
                                                               \
   /*lint --e(1763) the object itself is still const*/         \
   ::hmibase::input::IUserInputObserver* getUserInputObserver() const            \
   {                                                           \
      return _uiObserver;                                      \
   }                                                           \
                                                               \
                                                                              \
   virtual hmibase::DisplayParameters& getDisplayParameters() const           \
   {                                                                          \
      static hmibase::DisplayParameters displayParams;                        \
      displayParams.id = hmi::main_display_config::id;                        \
      displayParams.width = -1;                                               \
      displayParams.height = -1;                                              \
                                                                              \
      return displayParams;                                                   \
   }                                                                          \
                                                                              \
   virtual hmibase::trace::sCGITraceGroup getCourierTraceGroup() const \
   {                                                                          \
      return COURIER_APPLICATION_TRACE_GROUP;                                 \
   }                                                                          \
                                                                              \
   virtual hmibase::ScreenBrokerSetting& getScreenBrokerSettings() const      \
   {                                                                          \
      unsigned int popupSurfaces[]={__VA_ARGS__};                             \
      static hmibase::ScreenBrokerSetting sbSettings;                         \
      std::vector<unsigned int> p;                                            \
      std::vector<unsigned int> permanent;                                    \
      int iSize = sizeof(popupSurfaces)/sizeof(unsigned int);                 \
      for(int i = 0; i < iSize; i++)                                          \
      {                                                                       \
    	if((popupSurfaces[i] > ::hmibase::SURFACEID_PERMANENTSURFACEBASE)             \
		    && (popupSurfaces[i] < ::hmibase::SURFACEID_PERMANENTSURFACEBASE_MAX))    \
		{                                                                             \
           permanent.push_back(popupSurfaces[i]);		                              \
		}                                                                       \
        else                                                                  \
		{                                                                       \
		   p.push_back(popupSurfaces[i]);		                                 \
		}                                                                       \
      }                                                                       \
      sbSettings.ids.permanentSurfaces = permanent;                           \
      sbSettings.ids.popupSurfaces = p;                                       \
      sbSettings.ids.mainSurfaces.push_back(_mainSurfaceId);                  \
      sbSettings.ids.videoSurfaces.push_back(_videoSurfaceId);                \
                                                                              \
      sbSettings.scaling = ScreenBroker::Scaling::None;                       \
                                                                              \
      sbSettings.appViewSettings = AppViewSettings::GetInstance();            \
                                                                              \
      return sbSettings;                                                      \
   }                                                                          \
                                                            \
   unsigned int getSyncBlockID() const                      \
   {                                                        \
      return _syncBlockId;                                  \
   }                                                        \
                                                            \
   void setSyncBlockID(unsigned int id)                     \
   {                                                        \
      _syncBlockId = id;                                    \
   }                                                        \
                                                            \
   unsigned int getNoOfSyncBlockProducerInstances() const   \
   {                                                        \
      return _noOfSyncBlockProducerInstances;               \
   }                                                        \
                                                            \
   void setNoOfSyncBlockProducerInstances(unsigned int no)  \
   {                                                        \
      _noOfSyncBlockProducerInstances = no;                 \
   }                                                        \
                                                            \
private:                                                    \
   unsigned int _syncBlockId;                               \
   unsigned int _noOfSyncBlockProducerInstances;            \
   ::hmibase::input::IUserInputObserver* _uiObserver;       \
};                                                          \
                                                            \
static AppSettings appSettings;                             \

#define SYNC_BLOCK_CONSUMER_ID(x)  appSettings.setSyncBlockID(x);
#define SYNC_BLOCK_PROVIDER_INSTANCES(x) appSettings.setNoOfSyncBlockProducerInstances(x);
#define SYNC_BLOCK_CONNECTION(syncBlockId, producerInstanceId)  hmibase::gadget::SyncBlockProducerFactory::GetInstance().AddIdTuple(syncBlockId, producerInstanceId);

#define APPSETTINGS_CREATE   															  \
class AppSettings : public IApplicationSettings                                           \
{                                                                                         \
public:                                                                                   \
   AppSettings():_syncBlockId(0),_noOfSyncBlockProducerInstances(0), _uiObserver(0) {};   \
   virtual ~AppSettings() { _uiObserver = 0; };                                           \
                                                                                          \
   virtual KeyMappingBase& getKeyMapping() const                                          \
   {                                                                                      \
      static KeyMapping keyMapping;                                                       \
      return keyMapping;                                                                  \
   }                                                                                      \
                                                                                          \
   void setUserInputObserver(::hmibase::input::IUserInputObserver* uiObserver)   \
   {                                                           \
      _uiObserver = uiObserver;                                \
   }                                                           \
                                                               \
   /*lint --e(1763) the object itself is still const*/         \
   ::hmibase::input::IUserInputObserver* getUserInputObserver() const            \
   {                                                           \
      return _uiObserver;                                      \
   }                                                           \
                                                               \
                                                                              \
   virtual hmibase::DisplayParameters& getDisplayParameters() const           \
   {                                                                          \
      static hmibase::DisplayParameters displayParams;                        \
      displayParams.id = hmi::main_display_config::id;                        \
      displayParams.width = -1;                                               \
      displayParams.height = -1;                                              \
                                                                              \
      return displayParams;                                                   \
   }                                                                          \
                                                                              \
   virtual hmibase::trace::sCGITraceGroup getCourierTraceGroup() const \
   {                                                                          \
      return COURIER_APPLICATION_TRACE_GROUP;                                 \
   }                                                                          \
                                                                              \
   virtual hmibase::ScreenBrokerSetting& getScreenBrokerSettings() const      \
   {                                                                          \
      static hmibase::ScreenBrokerSetting sbSettings;                         \
      std::vector<unsigned int> p;                                            \
      std::vector<unsigned int> m;                                            \
      std::vector<unsigned int> vi;                                           \
      std::vector<unsigned int> perm;                                           \
      int iSize = sizeof(_surfaceList)/sizeof(_surfaceList[0]);               \
                                                                              \
      for(int i = 0; i < iSize; i++)                                          \
      {                                                                       \
    	  if(_surfaceList[i].st == hmibase::POPUP_SURFACE)                    \
    	  { 																  \
    		  p.push_back(_surfaceList[i].surfaceID);                         \
    	  }																	  \
		  else if(_surfaceList[i].st == hmibase::MAIN_SURFACE)		          \
    	  {															  	      \
    		  m.push_back(_surfaceList[i].surfaceID);    			          \
    	  }																	  \
		  else if(_surfaceList[i].st == hmibase::VIDEO_SURFACE)     		  \
    	  {														        	  \
    	      vi.push_back(_surfaceList[i].surfaceID);    				      \
      	  }															          \
		  else if(_surfaceList[i].st == hmibase::PERMANENT_SURFACE)     		  \
    	  {														        	  \
    	      perm.push_back(_surfaceList[i].surfaceID);    				      \
      	  }															          \
      }                                                                       \
                                                                              \
      sbSettings.ids.popupSurfaces = p;                                       \
      sbSettings.ids.mainSurfaces = m;                                        \
      sbSettings.ids.videoSurfaces = vi;                                      \
      sbSettings.ids.permanentSurfaces = perm;                                \
                                                                              \
      sbSettings.scaling = ScreenBroker::Scaling::None;                       \
                                                                              \
      sbSettings.appViewSettings = AppViewSettings::GetInstance();            \
                                                                              \
      return sbSettings;                                                      \
   }                                                                          \
                                                                              \
   unsigned int getSyncBlockID() const                      \
   {                                                        \
      return _syncBlockId;                                  \
   }                                                        \
                                                            \
   void setSyncBlockID(unsigned int id)                     \
   {                                                        \
      _syncBlockId = id;                                    \
   }                                                        \
                                                            \
   unsigned int getNoOfSyncBlockProducerInstances() const   \
   {                                                        \
      return _noOfSyncBlockProducerInstances;               \
   }                                                        \
                                                            \
   void setNoOfSyncBlockProducerInstances(unsigned int no)  \
   {                                                        \
      _noOfSyncBlockProducerInstances = no;                 \
   }                                                        \
                                                            \
private:                                                    \
   unsigned int _syncBlockId;                               \
   unsigned int _noOfSyncBlockProducerInstances;            \
   ::hmibase::input::IUserInputObserver* _uiObserver;       \
};                                                          \
                                                            \
static AppSettings appSettings;                             \

#define APPSETTINGS_BEGIN()  \
   KEYMAPPING_CGI               \
   hmibase::AppSettingsSurfaceIDs _surfaceList[] = {
#define APPSETTINGS_ADD_MAINSURFACE(surfaceid)     {surfaceid, hmibase::MAIN_SURFACE}
#define APPSETTINGS_ADD_POPUPSURFACE(surfaceid)   {surfaceid, hmibase::POPUP_SURFACE}
#define APPSETTINGS_ADD_VIDEOSURFACE(surfaceid)   {surfaceid, hmibase::VIDEO_SURFACE}
#define APPSETTINGS_ADD_PERMANENTSURFACE(surfaceid)   {surfaceid, hmibase::PERMANENT_SURFACE}
#define APPSETTINGS_END() }; \
		APPSETTINGS_CREATE
#endif
#endif
