/* ***************************************************************************************
* FILE:          GuiComponentBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GuiComponentBase.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "GuiComponentBase.h"
#include "asf/threading/Thread.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#include "AppBase/ScreenBrokerClient/IAppViewSettings.h"
#include "Trace/StartupInvestigation.h"

#include "View/CGI/GuiBase/ICgiApp.h"
#include "View/CGI/CgiExtensions/CourierMessageReceiverThread.h"
#include "View/CGI/GuiBase/Platform/Genivi/AppEnvironmentPlatform.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GuiComponentBase.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace app {
namespace base {

GuiComponentBase* GuiComponentBase::_theInstance = 0;

GuiComponentBase::GuiComponentBase(const char* appName, const IApplicationSettings& appSettings):
   HmiAsfComponentBase(appName)
   , _appSettings(appSettings)
   , _hallTunnelProxy(CourierTunnelService::CourierMessageReceiver::CourierMessageReceiverProxy::createProxy("courierMessageReceiverPort", *this))
   , _hmiThread(0)
   , _cgiApp(0)
{
   vInitPlatformEtg();
   SystemConfiguration::s_initialize();
#  ifdef BUILDVERSION_DISPLAY_STARTUP_TRACE_ENABLED
   BuildVersion_vDisplayStartupMessage(getAppName());
#  endif
   ETG_TRACE_USR4_THR(("GuiComponentBase Constructor"));
   _theInstance = this;
   hmibase::trace::StartupInvestigation::s_initialize();
   HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_CONSTRUCTOR_GUI);
}


GuiComponentBase::~GuiComponentBase()
{
   // we will never come here as processes will be killed (hard but real)
   if (_hmiThread)
   {
      _hmiThread->join();
      delete _hmiThread;
      _hmiThread = 0;
   }
   if (_cgiApp != 0)
   {
      delete _cgiApp;
      _cgiApp = 0;
   }
}


void GuiComponentBase::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _hallTunnelProxy)
   {
      /*since client has to be initialized and run in main thread in ASF, client connection is done in GUI main thread
      If child thread - thread bridge needs to be done */
      _courierMessageReceiverAdapter.setProxy(_hallTunnelProxy);
      bool _isInitialized = ScreenBrokerClient::GetInstance().InitClient(this);
      if (!_isInitialized)
      {
         ETG_TRACE_ERR_THR(("ScreenBrokerClient not initialized!"));
      }
      startGuiMainThread();
   }
}


void GuiComponentBase::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


void GuiComponentBase::triggerHall()
{
   sendLocalMessage(_courierMessageReceiverAdapter);
}


void GuiComponentBase::waitUntilTunnelServiceReady()
{
}


GuiComponentBase::SurfaceParameter::SurfaceParameter():
   mainSurfaceId(0),
   applicationPopupSurfaceId(0),
   globalPopupSurfaceId(0),
   videoSurfaceId(0),
   appViewSettings(0)
{
}


// static
GuiComponentBase& GuiComponentBase::getInstance()
{
   return *_theInstance;
}


void GuiComponentBase::startGuiMainThread()
{
   if (_hmiThread == 0)
   {
      // create and start GUI main thread
      std::string threadName = "GUI_";
      std::string suffAppName = GuiComponentBase::getAppName();
      suffAppName.erase(0, 7);
      threadName.append(suffAppName);
      _hmiThread = new asf::threading::Thread(threadName, threadFunc, (void*)this);
   }
}


int GuiComponentBase::run()
{
   preRun();

   ETG_TRACE_USR4_THR(("GuiComponentBase Initializing GUI thread ..."));

   if (Courier::Init())
   {
      //ETG_TRACE_USR4_CLS_THR((getDefaultTraceClass(), "*** %s Main started ***", getAppName()));

      ETG_TRACE_USR4_THR(("GuiComponentBase Courier Initialized"));

      setupCgiInstance();
      HMI_APP_ASSERT(getCgiApp() != 0);

      AppPlatform::Impl::AppEnvironment lAppEnvironment(_appSettings);

      if (getCgiApp()->Init(&lAppEnvironment, this))
      {
         ETG_TRACE_USR4_THR(("CGI Initialization for %25s", hmibase::trace::getAppName().c_str()));
         ::hmibase::view::CourierMessageReceiverThread courierReceiveThread;
         courierReceiveThread.SetName("Model_MsgReceiver");
         courierReceiveThread.Attach(*this);
         courierReceiveThread.Activate();

         if (courierReceiveThread.Run() == false)
         {
            // todo : problems with trace ETG_TRACE_FATAL_THR(("pid %d, thread %s not started", getpid(), courierReceiveThread.GetName()))
            ETG_TRACE_FATAL_THR(("GuiComponentBase thread %s not started", courierReceiveThread.GetName()))
            HMI_APP_ASSERT_ALWAYS();
         }

         getCgiApp()->Run();

         postRun();
      }
      else
      {
         ETG_TRACE_FATAL_THR(("GuiComponentBase CGI Init failed"));
         HMI_APP_ASSERT_ALWAYS();
      }
      getCgiApp()->Finalize();
      //ET_TRACE_CLOSE;   // end trace
   }
   return 0;
}


void GuiComponentBase::setCgiApp(class ICgiApp* ptr)
{
   _cgiApp = ptr;
}


class ICgiApp* GuiComponentBase::getCgiApp()
{
      return _cgiApp;
}


// overwrite these e.g. to initialize and read/write persistent DP values
void GuiComponentBase::preRun()
{
}


// overwrite these e.g. to initialize and read/write persistent DP values
void GuiComponentBase::postRun()
{
}


// static
void* GuiComponentBase::threadFunc(void* args)
{
   GuiComponentBase* obj = reinterpret_cast<GuiComponentBase*>(args);
   if (obj != 0)
   {
      obj->run();
   }
   return NULL;
}


}
}


}
