/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     EchoDBusClientComponent.h
 *
 *\author   CM-AI/PJ-G31
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#include "asf/core/BaseComponent.h"
#include "asf/core/Logger.h"
#include "example/echoDBus/EchoDBusServiceProxy.h"

namespace example { namespace echoCca { namespace dbusAdapterClient {

using namespace ::example::echoDBus::EchoDBusService;
using namespace ::asf::core;

class DBusAdapterClientComponent : public BaseComponent,
	ServiceAvailableIF,
	EchoCallbackIF,
	EchoResponseSentCallbackIF,
	EchoCountCallbackIF
{
public:
	
	DBusAdapterClientComponent() :
		_echoProxy(EchoDBusServiceProxy::createProxy("echoPort", *this)),
		echoReceived(false),
		echoCountReceived(false) {
	}
	
	virtual ~DBusAdapterClientComponent() {
	}
	
	// ServiceAvailableIF implementation
	virtual void onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange &stateChange) {
		LOG_INFO("onAvailable, _echoProxy %p is connected", &proxy);
		_echoProxy->sendEchoResponseSentRegister(*this);
		_echoProxy->sendEchoCountRegister(*this);
		_echoProxy->sendEchoRequest(*this, "First ASF DBus client");
	}

	virtual void onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange &stateChange) {
		_echoProxy->sendEchoResponseSentDeregisterAll();
		_echoProxy->sendEchoCountDeregisterAll();
	}
	
	// EchoCallbackIF implementation
	virtual void onEchoResponse(const ::boost::shared_ptr< EchoDBusServiceProxy >& proxy,
			const boost::shared_ptr< EchoResponse >& response) {
		LOG_INFO("Received onEchoReply=\"%s\"", response->getMessage().c_str());
		echoReceived = true;
		checkEndOfTest();
	}
	
	virtual void onEchoError(const ::boost::shared_ptr< EchoDBusServiceProxy >& proxy,
			const boost::shared_ptr<EchoError>& error) {
		LOG_ERROR("Received error: onEchoError %s, %s", error->getName().c_str(), error->getMessage().c_str());
	}
	
	// EchoResponseSentCallbackIF implementation
	virtual void onEchoResponseSentSignal(const ::boost::shared_ptr< EchoDBusServiceProxy >& proxy,
	const ::boost::shared_ptr< EchoResponseSentSignal >& signal) {
		LOG_INFO("Received echoResponseSent, message=\"%s\"", signal->getMessage().c_str());
	}
	
	virtual void onEchoResponseSentError(const ::boost::shared_ptr< EchoDBusServiceProxy >& proxy,
	const ::boost::shared_ptr< EchoResponseSentError >& error) {
		LOG_ERROR("Received error: onEchoResponseSentError %s, %s", error->getName().c_str(), error->getMessage().c_str());
	}
	
	// EchoCountCallbackIF implementation
	virtual void onEchoCountUpdate(const ::boost::shared_ptr< EchoDBusServiceProxy >& proxy,
			const ::boost::shared_ptr< EchoCountUpdate >& update) {
		LOG_INFO("Received onEchoCount %d", update->getEchoCount());
		echoCountReceived=true;
		checkEndOfTest();
	}
	
	virtual void onEchoCountError(const ::boost::shared_ptr< EchoDBusServiceProxy >& proxy,
			const ::boost::shared_ptr< EchoCountError >& error) {
		LOG_ERROR("Received error: onEchoCountError %s, %s", error->getName().c_str(), error->getMessage().c_str());
	}
	
private:

	void checkEndOfTest() {
		if(echoReceived && echoCountReceived) {
			LOG_INFO("Test: OK");
		}
	}
	
	::boost::shared_ptr< EchoDBusServiceProxy > _echoProxy;
	bool echoReceived;
	bool echoCountReceived;
	
	DECLARE_CLASS_LOGGER ();
};

DEFINE_CLASS_LOGGER_AND_LEVEL ("example/echoCca/dbusAdapterClient", DBusAdapterClientComponent, Info);

}}}
