/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     DBusAdapterComponent.h
 *\brief	Example component which maps a cca service to a dbus interface
 *
 *\author   gaurav.chadha@in.bosch.com CM-AI/PJ-G31
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#include "asf/core/Application.h"
#include "asf/core/BaseComponent.h"
#include "asf/core/Logger.h"

#include "example/echoDBus/EchoDBusServiceStub.h"
#include "example/echoCca/EchoFIProxy.h"

#include <map>

using namespace std;

using namespace ::asf::core;
using namespace ::example::echoDBus::EchoDBusService;

namespace example {
namespace echoCca {
namespace dbusAdapter{

class DBusAdapterEchoDBusServiceStub : public EchoDBusServiceStub,
::example::echoCca::EchoFI::EchoCallbackIF,
::example::echoCca::EchoFI::EchoCountCallbackIF {
public:

	DBusAdapterEchoDBusServiceStub(const ::boost::shared_ptr< asf::core::Proxy >& proxy) :
		EchoDBusServiceStub("echoPort"),
		_echofi( ::boost::dynamic_pointer_cast <example::echoCca::EchoFI::EchoFIProxy >(proxy) ) {
		_echofi->sendEchoCountUpReg(*this);
	}

	//EchoDBusServiceStub
	virtual void onEchoRequest (const ::boost::shared_ptr< EchoRequest >& request) {
		LOG_INFO("Request message is = \"%s\"",request->getMessage().c_str());
		act_t ccaAct = _echofi->sendEchoStart (*this, request->getMessage());
		_actMap.insert(pair<act_t,act_t>(ccaAct, request->getAct()));
	}

	//EchoCallbackIF
	virtual void onEchoResult(const ::boost::shared_ptr< example::echoCca::EchoFI::EchoFIProxy >& proxy,
			const boost::shared_ptr< example::echoCca::EchoFI::EchoResult >& result) {
		LOG_INFO("Result message to send reply is = \"%s\"",
				result->getResult().c_str());
		ActMap::iterator actPtr = _actMap.find(result->getAct());
		if (actPtr != _actMap.end()) {
			LOG_DEBUG("key is %u and value is %u ", actPtr->first, actPtr->second);
			sendEchoResponse(result->getResult(), actPtr->second);
		} else {
			LOG_FATAL("ACT received does not matches any key");
		}
		sendEchoResponseSentSignal(result->getResult());
	}

	virtual void onEchoError(const ::boost::shared_ptr< example::echoCca::EchoFI::EchoFIProxy >& proxy,
			const boost::shared_ptr< example::echoCca::EchoFI::EchoError >& error) {
		LOG_INFO("Received error: onEchoError");
		ActMap::iterator actPtr = _actMap.find(error->getAct());
		if (actPtr != _actMap.end()){
			sendEchoResponse("Method result error", actPtr->second);
		}
	}

	//EchoCountCallbackIF
	virtual void onEchoCountError(const ::boost::shared_ptr< example::echoCca::EchoFI::EchoFIProxy >& proxy,
			const boost::shared_ptr< example::echoCca::EchoFI::EchoCountError >& error) {
		LOG_INFO("Received error: onEchoCountError");
	}

	virtual void onEchoCountStatus(const ::boost::shared_ptr< example::echoCca::EchoFI::EchoFIProxy >& proxy,
			const boost::shared_ptr< example::echoCca::EchoFI::EchoCountStatus >& status) {
		LOG_INFO("Received Property count status");
		setEchoCount(status->getEchoCount());
	}

private:

	typedef map<act_t,act_t> ActMap;

	ActMap _actMap;

	boost::shared_ptr< example::echoCca::EchoFI::EchoFIProxy > _echofi;

	DECLARE_CLASS_LOGGER();

};


class DBusAdapterComponent : public BaseComponent,
	ServiceAvailableIF
{
public:
	DBusAdapterComponent() :
		_echofi(example::echoCca::EchoFI::EchoFIProxy::createProxy ("echoFiPort", *this)),
		_echoStub()
	{
	}

	~DBusAdapterComponent() {
//		if (_echoStub) {
//			delete _echoStub;
//		}
	}

	//ServiceAvailableIF
	virtual void onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange &stateChange) {
		LOG_INFO("Service is available");
		if ( _echofi == proxy ) {
			LOG_INFO("starting dbus stub");
			_echoStub.reset(new DBusAdapterEchoDBusServiceStub(proxy));
		}
	}

	virtual void onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange &stateChange) {
		LOG_INFO("Service is unavailable");
//		if (_echoStub) {
//			delete _echoStub;
//			_echoStub = 0;
//		}
	}

private:

	boost::shared_ptr< example::echoCca::EchoFI::EchoFIProxy > _echofi;


	boost::shared_ptr< DBusAdapterEchoDBusServiceStub > _echoStub;
	//DBusAdapterEchoDBusServiceStub* _echoStub;

	DECLARE_CLASS_LOGGER();
};

DEFINE_CLASS_LOGGER_AND_LEVEL("example/echoCca/dbusAdapter", DBusAdapterEchoDBusServiceStub, Info);
DEFINE_CLASS_LOGGER_AND_LEVEL("example/echoCca/dbusAdapter", DBusAdapterComponent, Info);

}
}
}
