/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     DoIP_Protocol.h
 *\brief    this is the main protocol handler which implements the UDP
 *          part of DoIP and handles TCP connection requests.
 *          All TCP connections are forked out into DoIP_Connection
 *          entities from here.
 *
 *\author   CM-CI1/ENP1 - Losch
 *
 *\par Copyright:
 *(c) 2018 Robert Bosch Car Multimedia GmbH
 *
 *\par History:
 * See history of revision control system
 ***************************************************************************/

#ifndef DOIP_PROTOCOLL_H
#define DOIP_PROTOCOLL_H

#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>

#include "DoIP_ServerCfg.h"
#include "ethernetbus.h"

#include <list>
#include <map>


class DoIP_Connection;
class DoIP_Server;
class DoIP_Factory;
class DoIP_Socket;
class DoIP_Protocoll
{
   // Must not be copied or assigned to prevent multiple close calls
   // for iSocket_* members in destructor.
    DoIP_Protocoll ( const DoIP_Protocoll& obj ) = delete;                                                           \
    DoIP_Protocoll& operator = ( const DoIP_Protocoll& obj ) = delete;
public:
   DoIP_Protocoll(DoIP_Factory *poDoipFactory, tclEthernetBus* poBus, tU16 u16Port, tU16 u16SourceAddr, tU8* au8EID, tU8* au8GID);
   virtual ~DoIP_Protocoll();

   MOCK_VIRTUAL void vTraceStackDump(void);

   MOCK_VIRTUAL tU16 getNodeAddress() {
      return cu16NodeAddress;
   }
   const tU16 cu16NodeAddress;

   MOCK_VIRTUAL void vCommonTimerTick(void);

   MOCK_VIRTUAL void vGetEIDGID(tU8* au8EID, tU8* au8GID);
   MOCK_VIRTUAL void vSetVIN(tU8 const acVIN[], tU8 u8VINlen);
   MOCK_VIRTUAL DoIP_TesterCfg* poAddTesterCfg(tU16 u16TesterAddr, tU16 u16NumBytesAckNack);
   MOCK_VIRTUAL std::map<tU16, DoIP_TesterCfg>::iterator rGetTesterCfg(tU16 u16TesterAddr);
   MOCK_VIRTUAL bool bRemove(DoIP_Connection* poConn);

   MOCK_VIRTUAL bool bRegister(DoIP_Server* poServer);
   MOCK_VIRTUAL bool bRemove(DoIP_Server* poServer);
   MOCK_VIRTUAL DoIP_Server* poGetDoIPServer();

   MOCK_VIRTUAL void vLocalIpAddrAssignmentChg(bool bConnected, sockaddr_in IPv4, sockaddr_in IPv4_bcast, sockaddr_in6 IPv6); // called by EthernetBus on IP change or if-up / if-down
   //   MOCK_VIRTUAL void vCloseTCPSocket(int iSocket);

   MOCK_VIRTUAL void vPowerModeResponse(tU8 powerState);
   MOCK_VIRTUAL void vRoutingActivationAuthenticationResp(bool Authentified, tU8* AuthenticationResData);

   MOCK_VIRTUAL bool bIsSourceAddressRegisteredToOtherConnection(DoIP_Connection* poThis, tU16 sourceAddr, DoIP_Connection** poOther);
   MOCK_VIRTUAL bool bIsSourceAddressNotYetRegistered(tU16 sourceAddr);

   MOCK_VIRTUAL void vMemCopyWithEndianessConversion(void* destination, const void* source, tU32 length);

   MOCK_VIRTUAL bool bCheckHeader(tU8* buffer, tU8* respCode, bool isTP);
   MOCK_VIRTUAL bool bCheckHeader(uint8_t protVer, uint8_t invProtVer, uint16_t payloadType, uint32_t payloadLength, tU8* respCode, bool isTP);

   MOCK_VIRTUAL void vPrepareResponse(tU8* buffer, tU8 responseCode);

protected:
   void vConnectionCloseAll(void);

   void vOpenUDP();
   void vCloseUDP();
   void vHandleTimer();
   void vHandleMessages();
   void vSendMessage(tU8 queueIdx);
   void vEntityStatus_HandleRequest(tU8 queueIdx);
   void vPowermode_HandleRequest(tU8 queueIdx);
   void vVehicleIdentification_HandleRequest(tU8 queueIdx);
   void vVehicleIdentification_HandleEIDRequest(tU8 queueIdx);
   void vVehicleIdentification_HandleVINRequest(tU8 queueIdx);
   void vVehicleIdentification_PrepareMessage(tU8* buffer);
   tU8  u8VehicleIdentification_GetGIDVINStatusByte(void);
   tU8  u8VehicleIdentification_GetFurtherActionByte(void);

   static bool bCheckVersionInfo(tU8 protVer, tU8 invProtVer, tU16 payloadType);
   static bool bCheckPayloadType(tU16 payloadType, bool isTP);
   static bool bCheckPayloadLength(tU16 payloadType, tU32 payloadLength);
   void vRxIndication(tU8 au8UdpBuffer[], tU32 u32RecvBytes, sockaddr_in& remoteaddr);

   DoIP_Connection* FindConnection(int iSocket, bool bRemove = false);

   static const tU8  cu8UDPThreadPriority     = 40;
   static const tU8  cu8TCPThreadPriority     = 41;
   static const tU32 TCPSOCKET_SERVER_STACKSIZE    = (6*1024);


private:
   tclEthernetBus* poEthBus;
   DoIP_Factory *poFactory;
   sockaddr_in     IPv4;
   sockaddr_in     IPv4_bcast;
   sockaddr_in6    IPv6;

   std::list<DoIP_Connection*>  oDoIP_Connections; // list of possible testers and their instances including routing config

   DoIP_Server* poDoIP_Server; // active DoIP server from local application, may be NULL, if none exists

   DoIP_ConfigType DoIPConfig;

   // UDP
   void vStartUDPServer();
   void vStartUDPBroadcastServer();
   DoIP_Socket *iSocket_UDP; /*Socket file descriptor UDP connection */
   DoIP_Socket *iSocket_UDP_bcast; /*Socket file descriptor UDP broadcast connection */

   tU8 udpState;

   tU32 timerVehicleAnnouncement;
   bool isAnnouncementMessageReady;
   bool VehicleIdentificationResponeMessage_b;
   tU8  DoIPUDPDiscoveryCounter;

   typedef struct
   {
      tU8 buffer[DOIP_CFG_cu32TCPBufferSize ];
      tU8 udpQueueState;
      sockaddr_in remoteAddr;
   } tUDPQueue;
   tUDPQueue arUDPQueue[DOIP_CFG_MaxUDPQueueSize];


   // TCP
   void vStartTCPServer();
   DoIP_Socket *iSocket_TCP_Server;

   pthread_t rTCPServerThread; /* Thread to handle the Ethernet server*/
   pthread_t rUDPReadThread; /* Thread to handle the Ethernet read,write,Rx and Tx states*/
   pthread_t rUDPBroadcastThread; /* Thread to handle the Ethernet read,write,Rx and Tx states*/

   bool bReadNextFrameUDP(DoIP_Socket *socket);
   bool bAcceptNextConnectionTCP(void);
   static void* pvTCPServerThread(void* pvParent);
   static void* pvUDPReadThread(void* pvParent);
   static void* pvUDPBroadcastThread(void* pvParent);
};


#endif
