/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
***************************************************************************/

/*!
*\file     DoIP_Protocoll.cpp
*\brief
*
*\author   CM-CI3/ESW-Losch
*
*\par Copyright:
*(c) 2017 Robert Bosch Car Multimedia GmbH
*
*\par History:
* See history of revision control system
***************************************************************************/

/****************************************************************************
| includes
|--------------------------------------------------------------------------*/
#include "DoIP_Server.h"
#include "DoIP_Protocoll.h"
#include "DoIP_Connection.h"
#include "DoIP_Socket.h"
#include "DoIP_Factory.h"
#include "doipcommon/DoIP_RoutingActivation.h"

#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <string.h>

//#define LOOPBACKSUPPORT_ENABLED
//#define LOOPBACKDISCOVERYPORT_DISABLED

DoIP_Protocoll::DoIP_Protocoll(DoIP_Factory *poDoIPFactory, tclEthernetBus* poBus, tU16 u16Port, tU16 u16SourceAddr, tU8* au8EID, tU8* au8GID)
 : cu16NodeAddress(u16SourceAddr),
   poEthBus(poBus),
   poFactory(poDoIPFactory),
   poDoIP_Server(NULL),
   iSocket_UDP(new DoIP_Socket),
   iSocket_UDP_bcast(new DoIP_Socket),
   iSocket_TCP_Server(new DoIP_Socket),
   rTCPServerThread(-1),
   rUDPReadThread(-1),
   rUDPBroadcastThread(-1)
{
   printf("DoIP_Protocoll::DoIP_Protocoll()\n");

   memset(&IPv4, 0, sizeof(sockaddr_in));
   memset(&IPv4_bcast, 0, sizeof(sockaddr_in));
   memset(&IPv6, 0, sizeof(sockaddr_in6));

   DoIPConfig.portNumber = u16Port;

   /* initialize VIN, EID and GID with default values and copy possible config */

   DoIPConfig.bValidEID = false;
   DoIPConfig.acEID[0] = DOIP_EID_BYTE0;
   DoIPConfig.acEID[1] = DOIP_EID_BYTE1;
   DoIPConfig.acEID[2] = DOIP_EID_BYTE2;
   DoIPConfig.acEID[3] = DOIP_EID_BYTE3;
   DoIPConfig.acEID[4] = DOIP_EID_BYTE4;
   DoIPConfig.acEID[5] = DOIP_EID_BYTE5;

   if(au8EID != NULL)
   {
      memcpy(DoIPConfig.acEID, au8EID, sizeof(DoIPConfig.acEID));
      DoIPConfig.bValidEID = true;
   }

   DoIPConfig.bValidGID = false;
   DoIPConfig.acGID[0] = DOIP_GID_BYTE0;
   DoIPConfig.acGID[1] = DOIP_GID_BYTE1;
   DoIPConfig.acGID[2] = DOIP_GID_BYTE2;
   DoIPConfig.acGID[3] = DOIP_GID_BYTE3;
   DoIPConfig.acGID[4] = DOIP_GID_BYTE4;
   DoIPConfig.acGID[5] = DOIP_GID_BYTE5;

   if(au8GID != NULL)
   {
      memcpy(DoIPConfig.acGID, au8GID, sizeof(DoIPConfig.acGID));
      DoIPConfig.bValidGID = true;
   }

   memset(DoIPConfig.acVIN, DOIP_CFG_VinInvalidityPattern, sizeof(DoIPConfig.acVIN));
   DoIPConfig.bValidVIN = false;

   // UDP
   udpState = DOIP_UDP_IDLE;
   timerVehicleAnnouncement = 0;
   isAnnouncementMessageReady = FALSE;
   VehicleIdentificationResponeMessage_b = FALSE;
   DoIPUDPDiscoveryCounter = 0;

   for (tU16 queueIdx = 0; queueIdx < DOIP_CFG_MaxUDPQueueSize; queueIdx++)
   {
      arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_EMPTY;
   }

}

DoIP_Protocoll::~DoIP_Protocoll()
{
   vConnectionCloseAll();
}

void DoIP_Protocoll::vTraceStackDump(void)
{
   DOIP_TRACE_NOTICE("\t\tDoIP: port=%u EID=%02X%02X%02X%02X%02X%02X GID=%02X%02X%02X%02X%02X%02X addr=0x%02X conns=%u",
                     DoIPConfig.portNumber,
                     DoIPConfig.acEID[0], DoIPConfig.acEID[1], DoIPConfig.acEID[2], DoIPConfig.acEID[3], DoIPConfig.acEID[4], DoIPConfig.acEID[5],
                     DoIPConfig.acGID[0], DoIPConfig.acGID[2], DoIPConfig.acGID[2], DoIPConfig.acGID[3], DoIPConfig.acGID[4], DoIPConfig.acGID[5],
                     cu16NodeAddress, oDoIP_Connections.size());

   std::map<tU16, DoIP_TesterCfg>::iterator tester;
   for(tester = DoIPConfig.oTesters.begin(); tester != DoIPConfig.oTesters.end(); tester++)
   {
      DOIP_TRACE_NOTICE("\t\t\tTesterCfg: addr=0x%02X numByteDiagAckNack=%u", tester->first, tester->second.numByteDiagAckNack);

      std::map<tU8, DoIP_RoutingActivationCfg>::iterator route;
      for(route = tester->second.oRouteActCfg.begin(); route != tester->second.oRouteActCfg.end(); route++)
      {
         DOIP_TRACE_NOTICE("\t\t\t\tRouteCfg: actNum=0x%02X authReq=%u", route->first, route->second.bAuthRequ);

         std::list<DoIP_RoutingActivationTarget>::iterator target;
         for(target = route->second.targetCfg.begin(); target != route->second.targetCfg.end(); target++)
         {
            DOIP_TRACE_NOTICE("\t\t\t\t\tTargetCfg: type=%u bus=%u addr=0x%02X baseAddr=0x%lX addrExt=0x%X",
                              target->type, target->busNumber, target->targetSA, target->targetBaseAddr, target->targetAddrExt);
         }
      }
   }

   std::list<DoIP_Connection*>::iterator conn;
   for(conn = oDoIP_Connections.begin(); conn != oDoIP_Connections.end(); conn++)
   {
      (*conn)->vTraceStackDump();
   }
}


DoIP_TesterCfg*  DoIP_Protocoll::poAddTesterCfg(tU16 u16TesterAddr, tU16 u16NumBytesAckNack)
{
   DOIP_TRACE_NOTICE("DoIP_Protocoll::poAddTesterCfg: tester=0x%02X numbytesacknack=%u", u16TesterAddr, u16NumBytesAckNack);

   DoIP_TesterCfg rTesterCfg = { .numByteDiagAckNack = u16NumBytesAckNack, .oRouteActCfg = std::map<tU8, DoIP_RoutingActivationCfg>() };
   DoIPConfig.oTesters.insert(std::map<tU16, DoIP_TesterCfg>::value_type(u16TesterAddr, rTesterCfg));
   return &DoIPConfig.oTesters.find(u16TesterAddr)->second;
}

std::map<tU16, DoIP_TesterCfg>::iterator DoIP_Protocoll::rGetTesterCfg(tU16 u16TesterAddr)
{
   return DoIPConfig.oTesters.find(u16TesterAddr);
}


bool DoIP_Protocoll::bRemove(DoIP_Connection* poConn)
{
   poEthBus->bEnterCriticalSection();
   std::list<DoIP_Connection*>::iterator conn;
   for(conn = oDoIP_Connections.begin(); conn != oDoIP_Connections.end(); conn++)
   {
      if((*conn) == poConn)
      {
         DOIP_TRACE_NOTICE("DoIP_Protocoll::bRemove: tester=0x%02X", poConn->u16GetTesterAddress());

         oDoIP_Connections.erase(conn);
         poEthBus->bLeaveCriticalSection();
         return true;
      }
   }
   poEthBus->bLeaveCriticalSection();
   return false;
}

void DoIP_Protocoll::vCommonTimerTick(void)
{
   /* UDP */
   vHandleTimer();
   vHandleMessages();

   poEthBus->bEnterCriticalSection();
   std::list<DoIP_Connection*>::iterator conn;
   DoIP_Connection* poCurrent;
   for (conn = oDoIP_Connections.begin(); conn != oDoIP_Connections.end(); )
   {
      // move the iterator before executing the tick, because the subsequent calls may remove this one and invalidate
      poCurrent = *conn;
      conn++;
      poCurrent->vCommonTimerTick();
   }
   poEthBus->bLeaveCriticalSection();
}


void DoIP_Protocoll::vGetEIDGID(tU8* au8EID, tU8* au8GID)
{
   if (DoIPConfig.bValidEID)
   {
      memcpy(au8EID, DoIPConfig.acEID, 6);
   }
   else if(poEthBus->bHasValidMacAddress())
   {
      /* MAC shall be used as EID */
      memcpy(au8EID, poEthBus->pu8GetMacAddress(), 6);
   }

   if ( DoIPConfig.bValidEID && !DoIPConfig.bValidGID )
   {
      /* EID should be used as GID */
      memcpy(au8GID, au8EID, 6);
   }
   else
   {
      /* Check if GID is configured */
      if ( DoIPConfig.bValidGID )
      {
         memcpy(au8GID, DoIPConfig.acGID, 6);
      }
   }
}

void DoIP_Protocoll::vSetVIN(tU8 const acVIN[], tU8 u8VINlen)
{
   if(u8VINlen != 0)
   {
      memcpy(DoIPConfig.acVIN, acVIN, u8VINlen);
      DoIPConfig.bValidVIN = true;
   }
}

bool DoIP_Protocoll::bRegister(DoIP_Server* poServer)
{
   if(poDoIP_Server == NULL && poServer->cu16LocalAddr == cu16NodeAddress)
   {
      poDoIP_Server = poServer;
      return true;
   }
   return false;
}

bool DoIP_Protocoll::bRemove(DoIP_Server* poServer)
{
   if(poDoIP_Server == poServer)
   {
      poDoIP_Server = NULL;
      return true;
   }
   return false;
}

DoIP_Server* DoIP_Protocoll::poGetDoIPServer()
{
   return poDoIP_Server;
}

void DoIP_Protocoll::vLocalIpAddrAssignmentChg(bool bConnected, sockaddr_in rIPv4, sockaddr_in rIPv4_bcast, sockaddr_in6 rIPv6)
{
   {
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &rIPv4.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   char addressBufferBcast[INET_ADDRSTRLEN] = "!!!";
   inet_ntop(AF_INET, &rIPv4_bcast.sin_addr, addressBufferBcast, INET_ADDRSTRLEN);
   DOIP_TRACE_WARNING("DoIP_Protocoll::vLocalIpAddrAssignmentChg: connected=%u IPv4=%s IPv4_bcast=%s",
                      bConnected, addressBuffer, addressBufferBcast);
   }

   if(bConnected)
   {
      this->IPv4 = rIPv4;
      this->IPv4_bcast = rIPv4_bcast;
      this->IPv6 = rIPv6;

      /* Open TCP and UDP Connection */
      vStartUDPServer();
      vStartUDPBroadcastServer(); // an additional socket for incoming broadcasts is required - Linux requires separate instances

      vStartTCPServer();

      vOpenUDP();
   }
   else
   {
      vConnectionCloseAll();
   }
}




void DoIP_Protocoll::vConnectionCloseAll(void)
{
   DOIP_TRACE_WARNING("DoIP_Protocoll::vConnectionCloseAll");

   poEthBus->bEnterCriticalSection();
   std::list<DoIP_Connection*>::iterator conn;
   for (conn = oDoIP_Connections.begin(); conn != oDoIP_Connections.end(); conn++)
   {
      (*conn)->vConnectionClose(FALSE);
   }
   poEthBus->bLeaveCriticalSection();
   vCloseUDP();

   iSocket_TCP_Server->close();
   //iSocket_TCP_Server = -1;

   iSocket_UDP->close();
   //iSocket_UDP = -1;

   iSocket_UDP_bcast->close();
   //iSocket_UDP_bcast = -1;

   delete iSocket_TCP_Server;
   iSocket_TCP_Server = nullptr;
   delete iSocket_UDP;
   iSocket_UDP = nullptr;
   delete iSocket_UDP_bcast;
   iSocket_UDP_bcast  = nullptr;

}


void DoIP_Protocoll::vOpenUDP()
{
   /* At startup prepare socket for vehicle discovery message */
   udpState = DOIP_UDP_DISCOVERY;
   /* Reserve the second location of the queue for the vehicle announcement message*/
   memset(&arUDPQueue[1].remoteAddr, 0, sizeof(sockaddr_in));
   arUDPQueue[1].remoteAddr.sin_family = AF_INET;
   arUDPQueue[1].remoteAddr.sin_addr.s_addr = INADDR_BROADCAST;
   arUDPQueue[1].remoteAddr.sin_port = htons(DoIPConfig.portNumber);
   arUDPQueue[1].udpQueueState = DOIP_UDP_DATA_PROCESSED;

   /* start the initial announcements now */
   DoIPUDPDiscoveryCounter = 0;
   timerVehicleAnnouncement = tclEthernetBus::cu32CommonTimerStep + DOIP_CFG_InitialVehicleAnnouncementTime;
}

void DoIP_Protocoll::vCloseUDP()
{
   udpState = DOIP_UDP_IDLE;
   timerVehicleAnnouncement = 0u;
}

void DoIP_Protocoll::vHandleTimer()
{
   if (VehicleIdentificationResponeMessage_b == TRUE)
   {
      isAnnouncementMessageReady = TRUE;
      VehicleIdentificationResponeMessage_b = false;
   }

   tU32 u32NextTime = timerVehicleAnnouncement > tclEthernetBus::cu32CommonTimerStep ?
                          timerVehicleAnnouncement - tclEthernetBus::cu32CommonTimerStep : 0;

   if(timerVehicleAnnouncement > 0 && u32NextTime == 0)
   {
      if (DoIPUDPDiscoveryCounter < DOIP_CFG_VehicleAnnouncementRepetition)
      {
         DOIP_TRACE_INFO("DoIP_Protocoll::vHandleTimer: vehicle announcement #%u", DoIPUDPDiscoveryCounter);
         timerVehicleAnnouncement = DOIP_CFG_VehicleAnnouncementInterval;
         isAnnouncementMessageReady = TRUE;
         DoIPUDPDiscoveryCounter++;
      }
      else
      {
         DOIP_TRACE_NOTICE("DoIP_Protocoll::vHandleTimer: last vehicle announcement -> UDP online");
         /* Last Vehicle Announcement during Vehicle Discovery phase has been transmitted, switch UDP Connection to ONLINE now */
         timerVehicleAnnouncement = 0;
         udpState = DOIP_UDP_IDLE;
      }
   }
   else
   {
      timerVehicleAnnouncement = u32NextTime;
   }
}

void DoIP_Protocoll::vHandleMessages()
{
    tU16 payLoadType;

    /* Loop over all queue elements */
    for (tU8 queueIdx = 0; queueIdx < DOIP_CFG_MaxUDPQueueSize; queueIdx++)
    {
       /* Check if new message was received */
        if (arUDPQueue[queueIdx].udpQueueState == DOIP_UDP_DATA_RECEIVED)
        {
            /* Handle data in queue according to payload */
            memcpy(&payLoadType, &arUDPQueue[queueIdx].buffer[DOIP_MSG_POS_PAYLOAD_TYPE], DOIP_MSG_SIZE_PAYLOAD_TYPE);

            switch (payLoadType)
            {
             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_REQ:
                 vVehicleIdentification_HandleRequest(queueIdx);
                 break;
             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_EID_REQ:
                 vVehicleIdentification_HandleEIDRequest(queueIdx);
                 break;
             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_VIN_REQ:
                 vVehicleIdentification_HandleVINRequest(queueIdx);
                 break;
             case DOIP_PAYLOAD_TYPE_ENTITY_STATUS_REQ:
                 vEntityStatus_HandleRequest(queueIdx);
                 break;
             case DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_REQ:
                 vPowermode_HandleRequest(queueIdx);
                 break;
             default:
                 break;
            }
        }

        /* Announcement and VehicleID response are send with delay */
        if (arUDPQueue[queueIdx].udpQueueState == DOIP_UDP_DATA_PROCESSED)
        {
            /* If timer elapsed, set data in queue to ready */
            if (isAnnouncementMessageReady == TRUE)
            {
                vVehicleIdentification_PrepareMessage(&(arUDPQueue[queueIdx].buffer[0]));
                arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_READY;
            }
        }

        /* Send message if ready */
        if (arUDPQueue[queueIdx].udpQueueState == DOIP_UDP_DATA_READY)
        {
            vSendMessage(queueIdx);
        }
    }
    if (isAnnouncementMessageReady == TRUE)
    {
        isAnnouncementMessageReady = FALSE;
    }
}

void DoIP_Protocoll::vSendMessage(tU8 queueIdx)
{
   tU32 payLoadLength;

   vMemCopyWithEndianessConversion(&payLoadLength,
         &arUDPQueue[queueIdx].buffer[DOIP_MSG_POS_PAYLOAD_LENGTH],
         DOIP_MSG_SIZE_PAYLOAD_LENGTH);

   {
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &arUDPQueue[queueIdx].remoteAddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   DOIP_TRACE_INFO("DoIP_Protocoll::vSendMessage[%s]: payloadlength=%u", addressBuffer, payLoadLength);
   }

   if (iSocket_UDP->sendto(arUDPQueue[queueIdx].buffer, (payLoadLength + DOIP_HEADER_SIZE),
              0, (sockaddr*)&arUDPQueue[queueIdx].remoteAddr, sizeof(sockaddr_in)) >= 0)
   {
      if ((udpState == DOIP_UDP_DISCOVERY) && (queueIdx == 1))
      {
         /* Reserve the second location of the queue for the vehicle announcement message during discovery phase*/
         arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_PROCESSED;
      }
      else
      {
         /* Clear the queue*/
         arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_EMPTY;
      }
   }
}

void DoIP_Protocoll::vEntityStatus_HandleRequest(tU8 queueIdx)
{
   tU16 payLoadType;
   tU32 payLoadLength;

   arUDPQueue[queueIdx].buffer[0] = DOIP_PROTOCOL_VERSION;
   arUDPQueue[queueIdx].buffer[1] = DOIP_PROTOCOL_VERSION_INV;
   payLoadType = DOIP_PAYLOAD_TYPE_ENTITY_STATUS_RESP;
   vMemCopyWithEndianessConversion(&arUDPQueue[queueIdx].buffer[2], &payLoadType, 2);
   payLoadLength = DOIP_PAYLOAD_LENGTH_ENTITY_STATUS_RESP;
   vMemCopyWithEndianessConversion(&arUDPQueue[queueIdx].buffer[4], &payLoadLength, 4);
   arUDPQueue[queueIdx].buffer[8] = DOIP_CFG_NodeType;
   arUDPQueue[queueIdx].buffer[9] = (tU8)std::min((int)DoIPConfig.oTesters.size(), 255);
   arUDPQueue[queueIdx].buffer[10] = (tU8)std::min((int)oDoIP_Connections.size(), 255);
   #if ( DOIP_CFG_EntityStatusMaxByteFieldUse != FALSE )
   tU32 maxSize = DOIP_CFG_cu32UDPBufferSize;
   vMemCopyWithEndianessConversion(&arUDPQueue[queueIdx].buffer[11], &maxSize, 4);
   #endif
   arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_READY;

   {
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &arUDPQueue[queueIdx].remoteAddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   DOIP_TRACE_NOTICE("DoIP_Protocoll::vEntityStatus_HandleRequest: remote=%s", addressBuffer);
   }
}

void DoIP_Protocoll::vPowermode_HandleRequest(tU8 queueIdx)
{
   arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_WAITING_POWER_MODE;

   {
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &arUDPQueue[queueIdx].remoteAddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   DOIP_TRACE_NOTICE("DoIP_Protocoll::vPowermode_HandleRequest: remote=%s", addressBuffer);
   }

   if(poDoIP_Server != NULL)
   {
      // ask application how to respond
      poDoIP_Server->vPowerModeRequest();
      //vPowerModeResponse(DOIP_READY);
   }
   else
   {
      vPowerModeResponse(DOIP_NOT_SUPPORTED);
   }
}

void DoIP_Protocoll::vPowerModeResponse(tU8 powerState)
{
   // this is PowerMode response from application
   tU16 payLoadType;
   tU32 payLoadLength;
   tU8  queueIdx;

   DOIP_TRACE_NOTICE("DoIP_Protocoll::vPowerModeResponse: powerState=%u", powerState);

   for(queueIdx = 0; queueIdx < DOIP_CFG_MaxUDPQueueSize; queueIdx++)
   {
      if(arUDPQueue[queueIdx].udpQueueState == DOIP_UDP_DATA_WAITING_POWER_MODE)
      {
         arUDPQueue[queueIdx].buffer[0] = DOIP_PROTOCOL_VERSION;
         arUDPQueue[queueIdx].buffer[1] = DOIP_PROTOCOL_VERSION_INV;
         payLoadType = DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_RESP;
         vMemCopyWithEndianessConversion(&arUDPQueue[queueIdx].buffer[2], &payLoadType, 2);
         payLoadLength = DOIP_PAYLOAD_LENGTH_DIAG_POWERMODE_RESP;
         vMemCopyWithEndianessConversion(&arUDPQueue[queueIdx].buffer[4], &payLoadLength, 4);

         vMemCopyWithEndianessConversion(&arUDPQueue[queueIdx].buffer[8], &powerState, 1);
         arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_READY;
      }
   }
}

void DoIP_Protocoll::vRoutingActivationAuthenticationResp(bool Authentified, tU8* AuthenticationResData)
{
   std::list<DoIP_Connection*>::iterator conn;
   for(conn = oDoIP_Connections.begin(); conn != oDoIP_Connections.end(); conn++)
   {
      (*conn)->vRoutingActivationAuthenticationResp(Authentified, AuthenticationResData);
   }
}

void DoIP_Protocoll::vVehicleIdentification_HandleRequest(tU8 queueIdx)
{
   {
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &arUDPQueue[queueIdx].remoteAddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   DOIP_TRACE_NOTICE("DoIP_Protocoll::vVehicleIdentification_HandleRequest: remote=%s", addressBuffer);
   }

   /* Only one timer per udp connection, therefore if counter already set because of other request, nothing to do */
   if (timerVehicleAnnouncement == 0)
   {
      timerVehicleAnnouncement = DOIP_CFG_InitialVehicleAnnouncementTime;
      VehicleIdentificationResponeMessage_b = true;
   }
   arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_PROCESSED;
}

void DoIP_Protocoll::vVehicleIdentification_HandleEIDRequest(tU8 queueIdx)
{
   {
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &arUDPQueue[queueIdx].remoteAddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   DOIP_TRACE_NOTICE("DoIP_Protocoll::vVehicleIdentification_HandleEIDRequest: remote=%s", addressBuffer);
   }

    tU8 EID[6];

    if (DoIPConfig.bValidEID)
    {
        memcpy(EID, DoIPConfig.acEID, sizeof(EID));
    }
    else if(poEthBus->bHasValidMacAddress())
    {
        memcpy(EID, poEthBus->pu8GetMacAddress(), sizeof(EID));
    }
    else
    {
        EID[0] = DOIP_EID_BYTE0;
        EID[1] = DOIP_EID_BYTE1;
        EID[2] = DOIP_EID_BYTE2;
        EID[3] = DOIP_EID_BYTE3;
        EID[4] = DOIP_EID_BYTE4;
        EID[5] = DOIP_EID_BYTE5;
    }

    /*If the elements compared are equal, memory comparision function returns false, else a non zero value */
    if (memcmp(EID, &arUDPQueue[queueIdx].buffer[DOIP_HEADER_SIZE], 6) == 0x0)
    {
        /* Frame Vehicle Announcement Message */
        vVehicleIdentification_PrepareMessage(arUDPQueue[queueIdx].buffer);
        arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_READY;
    }
    else
    {
        /* Ignore this request as it is not meant for this node to respond */
        arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_EMPTY;
    }
}

void DoIP_Protocoll::vVehicleIdentification_HandleVINRequest(tU8 queueIdx)
{
   {
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &arUDPQueue[queueIdx].remoteAddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   DOIP_TRACE_NOTICE("DoIP_Protocoll::vVehicleIdentification_HandleVINRequest: remote=%s", addressBuffer);
   }

    /*If the elements compared are equal, memory comparision function returns false, else a non zero value */
    if (memcmp(DoIPConfig.acVIN, &arUDPQueue[queueIdx].buffer[DOIP_HEADER_SIZE], 17) == 0x0)
    {
        /* Frame Vehicle Announcement Message */
        vVehicleIdentification_PrepareMessage(&(arUDPQueue[queueIdx].buffer[0]));
        arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_READY;
    }
    else
    {
        /* Ignore this request as it is not meant for this node to respond */
        arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_EMPTY;
    }
}

void DoIP_Protocoll::vVehicleIdentification_PrepareMessage( tU8* buffer)
{
    tU16 payLoadType;
    tU32 payLoadLength;
    tU8 EID[6];
    tU8 GID[6];

    EID[0] = DOIP_EID_BYTE0;
    EID[1] = DOIP_EID_BYTE1;
    EID[2] = DOIP_EID_BYTE2;
    EID[3] = DOIP_EID_BYTE3;
    EID[4] = DOIP_EID_BYTE4;
    EID[5] = DOIP_EID_BYTE5;

    GID[0] = DOIP_GID_BYTE0;
    GID[1] = DOIP_GID_BYTE1;
    GID[2] = DOIP_GID_BYTE2;
    GID[3] = DOIP_GID_BYTE3;
    GID[4] = DOIP_GID_BYTE4;
    GID[5] = DOIP_GID_BYTE5;

    /* Header */
    buffer[DOIP_MSG_POS_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION;
    buffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION_INV;

    /* PayloadType */
    payLoadType = DOIP_PAYLOAD_TYPE_VEHICLE_ID_RESP;
    vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_TYPE], &payLoadType, DOIP_MSG_SIZE_PAYLOAD_TYPE);

    /* PayloadLength */
    payLoadLength = DOIP_PAYLOAD_LENGTH_VEHICLE_ID_RESP;
    vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_LENGTH], &payLoadLength, DOIP_MSG_SIZE_PAYLOAD_LENGTH);

    /* Content */
    /* Vin */
    memcpy(&buffer[DOIP_MSG_POS_CONTENT], DoIPConfig.acVIN, 17);

    vMemCopyWithEndianessConversion(&buffer[25], &cu16NodeAddress, 2);//DOIP_CFG_LogicalAddress;

    if (DoIPConfig.bValidEID)
    {
        memcpy(&EID[0], DoIPConfig.acEID, 6);
    }
    else if(poEthBus->bHasValidMacAddress())
    {
        /* MAC shall be used as EID */
        memcpy(EID, poEthBus->pu8GetMacAddress(), sizeof(EID));
    }

    if ( DoIPConfig.bValidEID && !DoIPConfig.bValidGID )
    {
        /* EID should be used as GID */
        memcpy(&GID[0], &EID[0], 6);
    }
    else
    {
        /* Check if GID is configured */
        if ( DoIPConfig.bValidGID )
        {
            memcpy(&GID[0], DoIPConfig.acGID, 6);
        }
    }

    memcpy(&buffer[27], &EID[0], 6);
    memcpy(&buffer[33], &GID[0], 6);

    /* Further Action byte */
    buffer[39] = u8VehicleIdentification_GetFurtherActionByte();

    /* VIN / GID Synchronisation Status Code */
    buffer[40] = u8VehicleIdentification_GetGIDVINStatusByte();
}

tU8 DoIP_Protocoll::u8VehicleIdentification_GetGIDVINStatusByte(void)
{
    tU8 statusVINSync = DOIP_VIN_GID_NOT_SYNCHRONIZED;

    /* Check if Dcm_GetVIN was successful */
    if (DoIPConfig.bValidVIN)
    {
        statusVINSync = DOIP_VIN_GID_SYNCHRONIZED;
    }
    else
    {
        /* Check if this node it GID-master */
        if ( DoIPConfig.bValidGID )
        {
            statusVINSync = DOIP_VIN_GID_SYNCHRONIZED;
        }
        else
        {
            /* Return already set to DOIP_VIN_GID_NOT_SYNCHRONIZED */
        }
    }
    return statusVINSync;
}

/* Function returns the further action byte value in vehicle announcement message*/
tU8 DoIP_Protocoll::u8VehicleIdentification_GetFurtherActionByte(void)
{
   /* If central security is not configured, further action byte value is set to 0x00*/
   tU8 furtheraction_u8 = DOIP_ROUTEACTIVE_NO_FURTHER_ACTION;

   std::map<tU16, DoIP_TesterCfg>::iterator tester;
   for(tester = DoIPConfig.oTesters.begin(); tester != DoIPConfig.oTesters.end(); tester++)
   {
      std::map<tU8, DoIP_RoutingActivationCfg>::iterator route;
      for(route = tester->second.oRouteActCfg.begin(); route != tester->second.oRouteActCfg.end(); route++)
      {
          /* Check whether at least one routing activation with central security is possible*/
          if (route->first == DOIP_CENTRAL_SECURITY_ID)
          {
              furtheraction_u8 = DOIP_ROUTENOTACTIVE_INIT_CENTRAL_SECURITY;
              break;
          }
      }
   }

   return furtheraction_u8;
}





/*Function for UDP Connections */
void DoIP_Protocoll::vRxIndication(tU8 au8UdpBuffer[], tU32 u32RecvBytes, sockaddr_in& remoteaddr)
{
    bool isInvalidRequest = FALSE;
    tU8 respCode = 0;
    tU8 queueIdx = 1;
    tU16 payLoadType;
    tU32 payLoadLength;
    tU32 totCopiedLength = 0;

     /* Loop through the queue till,
      * the queue is not parsed completely and if all the requests are not copied completely and no error is found*/
     for (queueIdx = 1; queueIdx < DOIP_CFG_MaxUDPQueueSize; queueIdx++)
     {
         if ((totCopiedLength < u32RecvBytes) && (isInvalidRequest == FALSE))
         {
             if (arUDPQueue[queueIdx].udpQueueState == DOIP_UDP_DATA_EMPTY)
             {
                 /* Parse the received UDP request here and also trigger the transmission of response from here */
                 if (u32RecvBytes >= DOIP_HEADER_SIZE)
                 {
                    /*coping the files to corresponding buffer positionen  */
                    arUDPQueue[queueIdx].remoteAddr = remoteaddr;

                      vMemCopyWithEndianessConversion(
                             &arUDPQueue[queueIdx].buffer[DOIP_MSG_POS_PROTOCOL_VERSION],
                             &au8UdpBuffer[totCopiedLength + DOIP_MSG_POS_PROTOCOL_VERSION],
                             DOIP_MSG_SIZE_PROTOCOL_VERSION);
                     vMemCopyWithEndianessConversion(
                             &arUDPQueue[queueIdx].buffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION],
                             &au8UdpBuffer[totCopiedLength + DOIP_MSG_POS_INV_PROTOCOL_VERSION],
                             DOIP_MSG_SIZE_INV_PROTOCOL_VERSION);
                     vMemCopyWithEndianessConversion(
                             &arUDPQueue[queueIdx].buffer[DOIP_MSG_POS_PAYLOAD_TYPE],
                             &au8UdpBuffer[totCopiedLength + DOIP_MSG_POS_PAYLOAD_TYPE],
                             DOIP_MSG_SIZE_PAYLOAD_TYPE);
                     vMemCopyWithEndianessConversion(
                             &arUDPQueue[queueIdx].buffer[DOIP_MSG_POS_PAYLOAD_LENGTH],
                             &au8UdpBuffer[totCopiedLength + DOIP_MSG_POS_PAYLOAD_LENGTH],
                             DOIP_MSG_SIZE_PAYLOAD_LENGTH);

                     if (!bCheckHeader(arUDPQueue[queueIdx].buffer, &respCode, FALSE))
                     {
                         vPrepareResponse(arUDPQueue[queueIdx].buffer, respCode);
                         arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_READY;
                         isInvalidRequest = TRUE;
                     }
                     else
                     {
                         vMemCopyWithEndianessConversion(&payLoadType,
                                 &au8UdpBuffer[totCopiedLength + DOIP_MSG_POS_PAYLOAD_TYPE],
                                 DOIP_MSG_SIZE_PAYLOAD_TYPE);
                         vMemCopyWithEndianessConversion(&payLoadLength,
                                 &au8UdpBuffer[totCopiedLength + DOIP_MSG_SIZE_PAYLOAD_LENGTH],
                                 DOIP_MSG_SIZE_PAYLOAD_LENGTH);

                         switch (payLoadType)
                         {
                             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_REQ:
                             case DOIP_PAYLOAD_TYPE_ENTITY_STATUS_REQ:
                             case DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_REQ:
                                 arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_RECEIVED;
                                 /* Payload length is 0*/
                                 totCopiedLength += DOIP_HEADER_SIZE;
                                 break;
                             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_VIN_REQ:
                             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_EID_REQ:
                                 arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_RECEIVED;
                                 memcpy(&(arUDPQueue[queueIdx].buffer[DOIP_HEADER_SIZE]),
                                         &(au8UdpBuffer[totCopiedLength + DOIP_HEADER_SIZE]),
                                         payLoadLength);
                                 totCopiedLength += (payLoadLength + DOIP_HEADER_SIZE);
                                 break;
                                 /* Silently ignore if requests other than UDP is received*/
                             case DOIP_PAYLOAD_TYPE_GENERIC_HEADER_NACK:
                             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_RESP:
                             case DOIP_PAYLOAD_TYPE_ENTITY_STATUS_RESP:
                             case DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_RESP:
                             default:
                                 break;
                         }
                     }
                 }
                 else
                 {
                     /* Any UDP frame is expected to be at least 8 bytes - Protocol Version (1 byte), Inverse Protocol Version (1 byte), Payload Type (2 bytes) and Payload Length (4 bytes) */
                     vPrepareResponse(&(arUDPQueue[queueIdx].buffer[0]), DOIP_NACK_PROTOCOL_INFO_INCORRECT);
                     arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_READY;
                     isInvalidRequest = TRUE;
                 }
             }
         }
         else
         {
             break;
         }

     }
     /* If buffer is full and the message is not copied completly*/
     if ((totCopiedLength < u32RecvBytes) && (queueIdx == DOIP_CFG_MaxUDPQueueSize) && (isInvalidRequest == FALSE))
     {
         /* 0th position of the queue is used for storing the NACK buffer overflow*/
         vPrepareResponse(&(arUDPQueue[0].buffer[0]), DOIP_NACK_OUT_OF_MEMORY);
         arUDPQueue[0].udpQueueState = DOIP_UDP_DATA_READY;
     }
}



bool DoIP_Protocoll::bIsSourceAddressRegisteredToOtherConnection(DoIP_Connection* poThis, tU16 sourceAddr, DoIP_Connection** poOther)
{
   bool retVal = FALSE;

   std::list<DoIP_Connection*>::iterator conn;
   for (conn = oDoIP_Connections.begin(); conn != oDoIP_Connections.end(); conn++)
   {
        if ((*conn) != poThis)
        {
            if ((*conn)->bConnectionIsActive()
                 && ((*conn)->u16GetTesterAddress() == sourceAddr)
                 && (*conn)->bCheckRoutingActivationRegistrationStatus(DOIP_RA_STATUS_ROUTE_REGISTERED))
            {
                retVal = TRUE;
                *poOther = (*conn);
                break;
            }
        }
    }

    return retVal;
}

bool DoIP_Protocoll::bIsSourceAddressNotYetRegistered(tU16 sourceAddr)
{
   bool retVal = TRUE;

   std::list<DoIP_Connection*>::iterator conn;
   for (conn = oDoIP_Connections.begin(); conn != oDoIP_Connections.end(); conn++)
   {
      if ((*conn)->bConnectionIsActive() && ((*conn)->u16GetTesterAddress() == sourceAddr)
             && (*conn)->bCheckRoutingActivationRegistrationStatus(DOIP_RA_STATUS_ROUTE_REGISTERED))
      {
         retVal = FALSE;
         break;
      }
   }

   return retVal;
}

void DoIP_Protocoll::vMemCopyWithEndianessConversion(void* destination, const void* source, tU32 length)
{
   uint16_t      tmp16;
   uint32_t      tmp32;
   const uint8_t *src=(const uint8_t *)source;

   if (1 == length)
   {
      *(uint8_t *)destination=src[0];
   }
   else if (2 == length)
   {
      tmp16 = (uint16_t)(src[0] << 8);
      tmp16 = (uint16_t)(tmp16 | src[1]);
      memcpy(destination, &tmp16, 2);
   }
   else if (4 == length)
   {
      tmp32=(uint32_t)src[0]<<24;
      tmp32|=(uint32_t)src[1]<<16;
      tmp32|=(uint32_t)src[2]<<8;
      tmp32|=(uint32_t)src[3];
      memcpy(destination, &tmp32, 4);
   }
}

bool DoIP_Protocoll::bCheckVersionInfo(tU8 protVer, tU8 invProtVer, tU16 payloadType)
{
    bool retVal = TRUE;

    /* Check: Protocol information is correct */
    if (protVer != (tU8) (~invProtVer))
    {
        retVal = FALSE;
    }
    else
    {
        if ((payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_REQ) || (payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_EID_REQ)
                || (payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_VIN_REQ))
        {
            /* For vehicle identification request also default header is allowed */
            if ((protVer != DOIP_PROTOCOL_VERSION) && (protVer != DOIP_PROTOCOL_VERSION_DEFAULT))
            {
                retVal = FALSE;
            }
        }
        else
        {
            if (protVer != DOIP_PROTOCOL_VERSION)
            {
                retVal = FALSE;
            }

        }
    }
    return retVal;
}

bool DoIP_Protocoll::bCheckPayloadType(tU16 payloadType, bool isTP)
{
    bool retVal = TRUE;
    if (isTP)
    {
        if (!((payloadType == DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_REQ)
                || (payloadType == DOIP_PAYLOAD_TYPE_ALIVE_CHECK_RESP) || (payloadType == DOIP_PAYLOAD_TYPE_DIAG_MSG)))
        {
            retVal = FALSE;
        }
    }
    else
    {
        /* Allowed messages for UDP */
        if (!((payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_REQ) || (payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_EID_REQ)
                || (payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_VIN_REQ)
                || (payloadType == DOIP_PAYLOAD_TYPE_ENTITY_STATUS_REQ)
                || (payloadType == DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_REQ)))
        {
           /* Should invalid UDP messages be ignored */
           if(DOIP_CFG_GenericHeaderIgnoreInvalidUDP)
           {
              if (!((payloadType == DOIP_PAYLOAD_TYPE_GENERIC_HEADER_NACK)
                   || (payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_RESP)
                   || (payloadType == DOIP_PAYLOAD_TYPE_ENTITY_STATUS_RESP)
                   || (payloadType == DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_RESP)))
              {
                 retVal = FALSE;
              }
           }
           else
           {
              retVal = FALSE;
           }
        }
    }
    return retVal;
}

bool DoIP_Protocoll::bCheckPayloadLength(tU16 payloadType, tU32 payloadLength)
{
    bool retVal = TRUE;
    /* check: payload length is not valid for payloadType: nack 0x04, close socket */
    if (((payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_REQ) && (payloadLength != DOIP_PAYLOAD_LENGTH_VEHICLE_ID_REQ))
            || ((payloadType == DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_REQ)
                    && (payloadLength != DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_7)
                    && (payloadLength != DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_11))
            || ((payloadType == DOIP_PAYLOAD_TYPE_ALIVE_CHECK_RESP)
                    && (payloadLength != DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_RESP))
            || ((payloadType == DOIP_PAYLOAD_TYPE_DIAG_MSG) && (payloadLength < DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN))
            || ((payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_EID_REQ)
                    && (payloadLength != DOIP_PAYLOAD_LENGTH_VEHICLE_ID_EID_REQ))
            || ((payloadType == DOIP_PAYLOAD_TYPE_VEHICLE_ID_VIN_REQ)
                    && (payloadLength != DOIP_PAYLOAD_LENGTH_VEHICLE_ID_VIN_REQ))
            || ((payloadType == DOIP_PAYLOAD_TYPE_ENTITY_STATUS_REQ)
                    && (payloadLength != DOIP_PAYLOAD_LENGTH_ENTITY_STATUS_REQ))
            || ((payloadType == DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_REQ)
                    && (payloadLength != DOIP_PAYLOAD_LENGTH_DIAG_POWERMODE_REQ)))
    {
        retVal = FALSE;
    }
    return retVal;
}


 bool DoIP_Protocoll::bCheckHeader(uint8_t protVer, uint8_t invProtVer, uint16_t payloadType, uint32_t payloadLength, tU8* respCode, bool isTP) {
    bool retVal = FALSE;

    if (!bCheckVersionInfo(protVer, invProtVer, payloadType))
    {
        DOIP_TRACE_NOTICE("%s: Wrong version", __PRETTY_FUNCTION__);
        *respCode = DOIP_GA_RESPCODE_INVALID_PROTOCOL;
    }
    //check: payload type is not supported: nack 0x01, discard
    //only check allowed incoming payloads
    else if (!bCheckPayloadType(payloadType, isTP))
    {
        DOIP_TRACE_NOTICE("%s: Unsupported payload", __PRETTY_FUNCTION__);
        *respCode = DOIP_GA_RESPCODE_PAYLOAD_TYPE_UNSUPPORTED;
    }
    //check: payload exceeds DoIPMaxRequestBytes: nack 0x02, discard
    else if (payloadLength > (isTP ? (DOIP_CFG_cu32TCPMaxRequestSize) : DOIP_CFG_cu32UDPBufferSize))
    {
        DOIP_TRACE_NOTICE("%s: Payload too long", __PRETTY_FUNCTION__);
        *respCode = DOIP_GA_RESPCODE_EXCEEDS_MAX_REQ_BYTES;
    }
    else if (!bCheckPayloadLength(payloadType, payloadLength))
    {
        DOIP_TRACE_NOTICE("%s: Payload has wrong length", __PRETTY_FUNCTION__);
        *respCode = DOIP_GA_RESPCODE_INVALID_PAYLOAD_LENGTH_TYPE;
    }
    else
    {
        retVal = TRUE;
    }
    //check: start of reception buffer size exceeds doip buffer: nack 0x03, discard
    //uncheckable because we don't have a buffer

    return retVal;

}


bool DoIP_Protocoll::bCheckHeader(tU8* buffer, tU8* respCode, bool isTP)
{
    tU16 payloadType;
    tU32 payloadLength;
    tU8 protVer;
    tU8 invProtVer;
    
    
   memcpy(&payloadType, &buffer[DOIP_MSG_POS_PAYLOAD_TYPE], DOIP_MSG_SIZE_PAYLOAD_TYPE);
   memcpy(&payloadLength, &buffer[DOIP_MSG_POS_PAYLOAD_LENGTH], DOIP_MSG_SIZE_PAYLOAD_LENGTH);
   memcpy(&protVer, &buffer[DOIP_MSG_POS_PROTOCOL_VERSION], DOIP_MSG_SIZE_PROTOCOL_VERSION);
   memcpy(&invProtVer, &buffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION], DOIP_MSG_SIZE_INV_PROTOCOL_VERSION);

   return bCheckHeader(protVer, invProtVer, payloadType, payloadLength, respCode, isTP);
   
}

void DoIP_Protocoll::vPrepareResponse(tU8* buffer, tU8 responseCode)
{
    tU16 payloadType = DOIP_PAYLOAD_TYPE_GENERIC_HEADER_NACK;
    tU32 payloadLength = DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_NACK;

    /* DoIP Header */
    buffer[DOIP_MSG_POS_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION;
    buffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION_INV;
    /* buffer[2], buffer[3] */
    vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_TYPE], &payloadType, DOIP_MSG_SIZE_PAYLOAD_TYPE);
    /* buffer[4], buffer[5], buffer[6], buffer[7] */
    vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_LENGTH], &payloadLength, DOIP_MSG_SIZE_PAYLOAD_LENGTH);

    /* buffer[8] - responde code */
    buffer[DOIP_MSG_POS_CONTENT] = responseCode;
}







DoIP_Connection* DoIP_Protocoll::FindConnection(int iSocket, bool bRemove)
{
   DoIP_Connection* poConn = NULL;
   // first search for a connection that uses this socket already
   std::list<DoIP_Connection*>::iterator conn;
   for (conn = oDoIP_Connections.begin(); (iSocket != -1) && (conn != oDoIP_Connections.end()); conn++)
   {
      if((*conn)->iGetSocket() == iSocket)
      {
         poConn = (*conn); // found it!
         if(bRemove) oDoIP_Connections.erase(conn);
         break;
      }
   }

   return poConn;
}




void DoIP_Protocoll::vStartUDPServer()
{
   /*open a UDP unicast server connection */

   /* Open UDP socket for receiving data */
   if(iSocket_UDP->create(AF_INET,SOCK_DGRAM,IPPROTO_UDP) < 0)
   {
      DOIP_TRACE_CRIT("DoIP_Protocoll::vStartUDPServer: sockfdUDP1 = socket(PF_INET,SOCK_DGRAM,IPPROTO_UDP))< 0");
   }
   else
   {
      //Socket option : allow multiple sockets to use the same ports
      int optval = 1;
      if ( iSocket_UDP->setsockopt(SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0 )
      {
         DOIP_TRACE_WARNING("DoIP_Protocoll::vStartUDPServer: It is not able to change SO_REUSEADDR");
      }

      //Set Broadcast options on
      int broadcastEnable = 1;
      if ( iSocket_UDP->setsockopt(SOL_SOCKET, SO_BROADCAST, &broadcastEnable, sizeof(broadcastEnable)) < 0 )
      {
         DOIP_TRACE_ERR("DoIP_Protocoll::vStartUDPServer: It is not able to change SO_BROADCAST");
      }


      struct sockaddr_in address = IPv4;
      address.sin_port = htons(DoIPConfig.portNumber);

      if(iSocket_UDP->bind( (struct sockaddr*)&address, sizeof(address)) < 0)
      {
         /*Do here the Error handling */
         DOIP_TRACE_CRIT("DoIP_Protocoll::vStartUDPServer: It is not possible to open an UDP socket");
      }
      else
      {
         //sched_param rUDPSchedAttr;
         pthread_attr_t rUDPReadAttr;
         pthread_attr_init(&rUDPReadAttr);
         (void)pthread_attr_setstacksize(&rUDPReadAttr, TCPSOCKET_SERVER_STACKSIZE);
         //pthread_attr_setschedpolicy(&rUDPReadAttr,SCHED_FIFO);
         //pthread_attr_setinheritsched(&rUDPReadAttr,PTHREAD_EXPLICIT_SCHED);
         //pthread_attr_getschedparam(&rUDPReadAttr,&rUDPSchedAttr);
         //rUDPSchedAttr.sched_priority = cu8UDPThreadPriority;
         //pthread_attr_setschedparam (&rUDPReadAttr, &rUDPSchedAttr);

         if(pthread_create( &rUDPReadThread, &rUDPReadAttr, DoIP_Protocoll::pvUDPReadThread, this) == 0)
         {
            pthread_setname_np(rUDPReadThread, "DoIP@UDP");
         }
         else
         {
            DOIP_TRACE_CRIT("DoIP_Protocoll::vStartUDPServer: DoIP UDP Thread is not started");
         }
      }
   }
}

void DoIP_Protocoll::vStartUDPBroadcastServer()
{
   /*open a UDP broadcast server connection */

   /* Open UDP socket for receiving data */
   if(iSocket_UDP_bcast->create(AF_INET,SOCK_DGRAM,IPPROTO_UDP) < 0)
   {
      DOIP_TRACE_CRIT("DoIP_Protocoll::vStartUDPBroadcastServer: sockfdUDP1 = socket(PF_INET,SOCK_DGRAM,IPPROTO_UDP))< 0");
   }
   else
   {
      //Socket option : allow multiple sockets to use the same ports
      int optval = 1;
      if ( iSocket_UDP_bcast->setsockopt(SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0 )
      {
         DOIP_TRACE_WARNING("DoIP_Protocoll::vStartUDPBroadcastServer: It is not able to change SO_REUSEADDR");
      }

      //Set Broadcast options on
      int broadcastEnable = 1;
      if ( iSocket_UDP_bcast->setsockopt(SOL_SOCKET, SO_BROADCAST, &broadcastEnable, sizeof(broadcastEnable)) < 0 )
      {
         DOIP_TRACE_ERR("DoIP_Protocoll::vStartUDPBroadcastServer: It is not able to change SO_BROADCAST");
      }


      struct sockaddr_in address = IPv4_bcast;
      address.sin_port = htons(DoIPConfig.portNumber);

      if(iSocket_UDP_bcast->bind((struct sockaddr*)&address, sizeof(address)) < 0)
      {
         /*Do here the Error handling */
         DOIP_TRACE_CRIT("DoIP_Protocoll::vStartUDPBroadcastServer: It is not possible to open an UDP socket");
      }
      else
      {
         //sched_param rUDPSchedAttr;
         pthread_attr_t rUDPReadAttr;
         pthread_attr_init(&rUDPReadAttr);
         (void)pthread_attr_setstacksize(&rUDPReadAttr, TCPSOCKET_SERVER_STACKSIZE);
         //pthread_attr_setschedpolicy(&rUDPReadAttr,SCHED_FIFO);
         //pthread_attr_setinheritsched(&rUDPReadAttr,PTHREAD_EXPLICIT_SCHED);
         //pthread_attr_getschedparam(&rUDPReadAttr,&rUDPSchedAttr);
         //rUDPSchedAttr.sched_priority = cu8UDPThreadPriority;
         //pthread_attr_setschedparam (&rUDPReadAttr, &rUDPSchedAttr);

         if(pthread_create( &rUDPBroadcastThread, &rUDPReadAttr, DoIP_Protocoll::pvUDPBroadcastThread, this) == 0)
         {
            pthread_setname_np(rUDPBroadcastThread, "DoIP@UDP_bcast");
         }
         else
         {
            DOIP_TRACE_CRIT("DoIP_Protocoll::vStartUDPBroadcastServer: DoIP UDP Thread is not started");
         }
      }
   }
}

void DoIP_Protocoll::vStartTCPServer()
{
   if((iSocket_TCP_Server->create(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
   {
      /*Do here the Error handling */
      DOIP_TRACE_CRIT("DoIP_Protocoll::vStartTCPServer: It is not possible to open an TCP socket");
   }
   else
   {
      //allow multiple sockets to use the same port
      int optval = 1;
      if (iSocket_TCP_Server->setsockopt(SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0)
      {
         DOIP_TRACE_WARNING("DoIP_Protocoll::vStartTCPServer: Can not change SO_REUSEADDR");
      }
      optval = 1;
      if (iSocket_TCP_Server->setsockopt(IPPROTO_TCP, TCP_NODELAY, &optval, sizeof(optval)) < 0)
      {
         DOIP_TRACE_WARNING("DoIP_Protocoll::vStartTCPServer: Can not change TCP_NODELAY");
      }

      struct sockaddr_in address = IPv4;
      address.sin_port = htons(DoIPConfig.portNumber);

      if(iSocket_TCP_Server->bind((struct sockaddr*)&address, sizeof(address)) < 0)
      {
         /*Do here the Error handling */
         DOIP_TRACE_CRIT("DoIP_Protocoll::vStartTCPServer: It is not possible to open an TCP socket");
         iSocket_TCP_Server->close();
      }
      else
      {
         if (iSocket_TCP_Server->listen(3) == -1)
         {
            DOIP_TRACE_CRIT("DoIP_Protocoll::vStartTCPServer: It is not possible to listen an TCP socket");
         }
         else
         {
            /* start the reading thread for the TCP sockets */
            //sched_param rTCPSchedAttr;
            pthread_attr_t rTCPReadAttr;
            pthread_attr_init(&rTCPReadAttr);
            (void)pthread_attr_setstacksize(&rTCPReadAttr, TCPSOCKET_SERVER_STACKSIZE);
            //pthread_attr_setschedpolicy(&rTCPReadAttr,SCHED_FIFO);
            //pthread_attr_setinheritsched(&rTCPReadAttr,PTHREAD_EXPLICIT_SCHED);
            //pthread_attr_getschedparam(&rTCPReadAttr,&rTCPSchedAttr);
            //rTCPSchedAttr.sched_priority = cu8TCPThreadPriority; //sched_get_priority_min( SCHED_FIFO );
            //pthread_attr_setschedparam (&rTCPReadAttr, &rTCPSchedAttr);

            if(pthread_create( &rTCPServerThread, &rTCPReadAttr, DoIP_Protocoll::pvTCPServerThread, this) == 0)
            {
               pthread_setname_np(rTCPServerThread, "DoIP@TCPServer");
            }
            else
            {
               DOIP_TRACE_CRIT("DoIP_Protocoll::vStartTCPServer: DoIP TCP Thread is not started");
            }
         }
      }
   }
}


bool DoIP_Protocoll::bAcceptNextConnectionTCP(void)
{
   //handle new connections
   struct sockaddr_in remoteaddr; //client address
   socklen_t socketaddrlen = sizeof remoteaddr;
   int newfd = iSocket_TCP_Server->accept((struct sockaddr*)&remoteaddr,&socketaddrlen);
   if (newfd != -1)
   {
      {
      char addressBuffer[INET_ADDRSTRLEN];
      inet_ntop(AF_INET, &remoteaddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
      DOIP_TRACE_NOTICE("DoIP_Protocoll::bAcceptNextConnectionTCP: new connection=%s socket=%i", addressBuffer, newfd);
      }

      int optval = 1;
      if (iSocket_TCP_Server->setsockopt(IPPROTO_TCP, TCP_NODELAY, &optval, sizeof(optval)) < 0)
      {
         DOIP_TRACE_WARNING("DoIP_Protocoll::vStartTCPServer: Can not change TCP_NODELAY");
      }

      DoIP_Connection* poConn = poFactory->createConnection(this, new DoIP_Socket(newfd));
      if(poConn != NULL)
      {
         poEthBus->bEnterCriticalSection();
         oDoIP_Connections.push_back(poConn);
         poConn->vConnectionOpen();
         poEthBus->bLeaveCriticalSection();
      }
      else
      {
         close(newfd);
      }
   }
   else
   {
      iSocket_TCP_Server->setFd(-1);
      return false;
   }

   return true;
}

bool DoIP_Protocoll::bReadNextFrameUDP(DoIP_Socket *socket)
{
   struct sockaddr_in remoteaddr; //client address
   socklen_t slen = sizeof(remoteaddr);

   tU8 au8UdpBuffer[DOIP_CFG_cu32UDPBufferSize];

   size_t iRecvBytes = socket->recvfrom(au8UdpBuffer, DOIP_CFG_cu32UDPBufferSize , 0, (struct sockaddr *)&remoteaddr, &slen);

   {
   char addressBuffer[INET_ADDRSTRLEN];
   inet_ntop(AF_INET, &remoteaddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   DOIP_TRACE_NOTICE("DoIP_Protocoll::bReadNextFrameUDP[%s][%i]: [%lu] %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X",
                     addressBuffer, socket->getFd(), iRecvBytes,
                     au8UdpBuffer[0], au8UdpBuffer[1], au8UdpBuffer[2], au8UdpBuffer[3], au8UdpBuffer[4],
                     au8UdpBuffer[5], au8UdpBuffer[6], au8UdpBuffer[7], au8UdpBuffer[8], au8UdpBuffer[9]);
   }

   if(iRecvBytes > 0)
   {
      if (remoteaddr.sin_addr.s_addr != IPv4.sin_addr.s_addr) // avoid loopbacks
      {
         vRxIndication(au8UdpBuffer, (tU32)iRecvBytes, remoteaddr);
      }
#ifdef LOOPBACKSUPPORT_ENABLED
      else
      {
          /* local communication */

          /* UDP server port range */
          if (ntohs(remoteaddr.sin_port) >= DOIP_UDP_PORT_MIN)
          {
              DOIP_TRACE_NOTICE("DoIP_Protocoll::bReadNextFrameUDP ALLOW loopback for receive ports starting with %d, remote %s:%d local %s:%d", DOIP_UDP_PORT_MIN, inet_ntoa(remoteaddr.sin_addr),ntohs(remoteaddr.sin_port), inet_ntoa(IPv4.sin_addr),ntohs(IPv4.sin_port));
              vRxIndication(au8UdpBuffer, (tU32)iRecvBytes, remoteaddr);
          }
          /* UDP DISCOVERY PORT */
#ifdef LOOPBACKDISCOVERYPORT_ENABLED
          else if(ntohs(remoteaddr.sin_port) == DOIP_UDP_PORT_DISCOVERY)
          {
              //DOIP_TRACE_NOTICE("DoIP_Protocoll::bReadNextFrameUDP ALLOW possible loopback for RECOVERY port %d, remote %s:%d local %s:%d", DOIP_UDP_PORT_DISCOVERY, inet_ntoa(remoteaddr.sin_addr),ntohs(remoteaddr.sin_port), inet_ntoa(IPv4.sin_addr),ntohs(IPv4.sin_port));
              vRxIndication(au8UdpBuffer, (tU32)iRecvBytes, remoteaddr);
          }
#endif
          else
          {
              DOIP_TRACE_NOTICE("DoIP_Protocoll::bReadNextFrameUDP AVOID possible unacceptable loopback, remote %s:%d local %s:%d", inet_ntoa(remoteaddr.sin_addr),ntohs(remoteaddr.sin_port), inet_ntoa(IPv4.sin_addr),ntohs(IPv4.sin_port));
          }
       }
#endif
   }
   else
   {
      return false;
   }

   return true;
}

void* DoIP_Protocoll::pvTCPServerThread(void* poPar)
{
   DoIP_Protocoll* poParent = (DoIP_Protocoll*)poPar;

   while(poParent->bAcceptNextConnectionTCP()) ;

   DOIP_TRACE_WARNING("DoIP_Protocoll::pvTCPServerThread: thread has ended");
   return NULL;
}

void* DoIP_Protocoll::pvUDPReadThread(void* poPar)
{
   DoIP_Protocoll* poParent = (DoIP_Protocoll*)poPar;

   while(poParent->bReadNextFrameUDP(poParent->iSocket_UDP));

   DOIP_TRACE_WARNING("DoIP_Protocoll::pvUDPReadThread: thread has ended");
   return NULL;
}

void* DoIP_Protocoll::pvUDPBroadcastThread(void* poPar)
{
   DoIP_Protocoll* poParent = (DoIP_Protocoll*)poPar;

   while(poParent->bReadNextFrameUDP(poParent->iSocket_UDP_bcast));

   DOIP_TRACE_WARNING("DoIP_Protocoll::pvUDPBroadcastThread: thread has ended");
   return NULL;
}


