/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
***************************************************************************/

/*!
*\file     DoIP_Connection.cpp
*\brief
*
*\author   CM-CI3/ESW-Losch
*
*\par Copyright:
*(c) 2017 Robert Bosch Car Multimedia GmbH
*
*\par History:
* See history of revision control system
***************************************************************************/

/****************************************************************************
| includes
|--------------------------------------------------------------------------*/
#include "DoIP_Connection.h"
#include "DoIP_Channel.h"
#include "DoIP_Factory.h"
#include "DoIP_Server.h"
#include "DoIP_Socket.h"
#include "doipcommon/DoIP_RoutingActivation.h"

#include <string.h>
#include <errno.h>


DoIP_Connection::DoIP_Connection(DoIP_Factory *poFactory, DoIP_Protocoll* poServer, DoIP_Socket *psocket)
 : rActiveTesterCfg(poServer->rGetTesterCfg(cu16TesterAddrInit)),
   rActiveRouteCfg(poServer->rGetTesterCfg(cu16TesterAddrInit)->second.oRouteActCfg.end()),
   poCurrentChannel(NULL),
   state(DOIP_CON_OFFLINE),
   timerInactivity(0),
   rTCPReadThread(-1),
   poDoIPProtocol(poServer),
   mpoFactory(poFactory),
   poSocket(psocket)
{

   memset(&remoteAddr, 0, sizeof(sockaddr_in));
   memset(&remoteAddrIpv6, 0, sizeof(sockaddr_in6));

   // TCP
   routingActivation.rx = {};
   routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_IDLE;
   routingActivation.routingActivationStatus = (tU8) DOIP_RA_STATUS_INIT;
   routingActivation.routingActivationSA = cu16TesterAddrInit;
   routingActivation.routingActivationType = 0xFF;
   RX.state = DOIP_TCP_IDLE;
   RX.payloadLength = 0;
   RX.payloadType = 0;
   RX.cnt=0;
   aliveCheck.aliveCheckState = DOIP_ALIVECHECK_IDLE;
   aliveCheck.aliveCheckTimer = 0;
   timerInactivity = 0;

}

DoIP_Connection::~DoIP_Connection()
{
   DOIP_TRACE_NOTICE("DoIP_Connection::~DoIP_Connection()\n");
   poDoIPProtocol->bRemove(this);

   vConnectionClose(TRUE);
   delete poSocket;
   poSocket=0;
}

void DoIP_Connection::vTraceStackDump(void)
{
   DOIP_TRACE_NOTICE("DoIP_Connection::vTraceStackDump:\n");
   if (rActiveTesterCfg==poDoIPProtocol->rGetTesterCfg(cu16TesterAddrInit)) {
      DOIP_TRACE_NOTICE("\t\t\tDoIP_Connection: rActiveTesterCfg invalid");
   }
   else { 
      tU16 u16ActiveRoute = (rActiveRouteCfg != rActiveTesterCfg->second.oRouteActCfg.end()) ? rActiveRouteCfg->first : 0x100;
      
      DOIP_TRACE_NOTICE("\t\t\tDoIP_Connection: addr=0x%02X actroute=0x%02X actchans=%u",
                        rActiveTesterCfg->first, u16ActiveRoute, oActiveChannels.size());
      
      std::map<tU8, DoIP_RoutingActivationCfg>::iterator cfg;
      for(cfg = rActiveTesterCfg->second.oRouteActCfg.begin(); cfg != rActiveTesterCfg->second.oRouteActCfg.end(); cfg++)
      {
         DOIP_TRACE_NOTICE("\t\t\t\tRoute: actnum=0x%02X authrequ=%u targets=%u",
                           cfg->first, cfg->second.bAuthRequ, cfg->second.targetCfg.size());
      }
   }

   std::list<DoIP_Channel*>::iterator channel;
   for(channel = oActiveChannels.begin(); channel != oActiveChannels.end(); channel++)
   {
      (*channel)->vTraceStackDump();
   }

   DOIP_TRACE_NOTICE("\t\t\tstate=%u", state);

   DOIP_TRACE_NOTICE("\t\t\troutingActivation.routingActivationState=%u",  routingActivation.routingActivationState);
   DOIP_TRACE_NOTICE("\t\t\troutingActivation.routingActivationStatus=%u", routingActivation.routingActivationStatus); 
   DOIP_TRACE_NOTICE("\t\t\troutingActivation.routingActivationSA=0x%04x", routingActivation.routingActivationSA);
   DOIP_TRACE_NOTICE("\t\t\troutingActivation.routingActivationType=%u", routingActivation.routingActivationType);
   DOIP_TRACE_NOTICE("\t\t\tRX.state = %u", RX.state);;
   DOIP_TRACE_NOTICE("\t\t\tRX.payloadLength = %u", RX.payloadLength);
   DOIP_TRACE_NOTICE("\t\t\tRX.payloadType = %u", RX.payloadType);;
   DOIP_TRACE_NOTICE("\t\t\tRX.cnt=%u", RX.cnt);
   DOIP_TRACE_NOTICE("\t\t\taliveCheck.aliveCheckState=%u", aliveCheck.aliveCheckState);
   DOIP_TRACE_NOTICE("\t\t\taliveCheck.aliveCheckTimer = %u", aliveCheck.aliveCheckTimer);
   DOIP_TRACE_NOTICE("\t\t\ttimerInactivity = %u", timerInactivity);;

}

DoIP_Server* DoIP_Connection::poGetDoIPServer()
{
   return poDoIPProtocol->poGetDoIPServer();
}

bool DoIP_Connection::bRemove(DoIP_Channel* poChan)
{
   DOIP_TRACE_NOTICE("DoIP_Connection::bRemove: channel=0x%02X", poChan->cu16TargetAddress);

   std::list<DoIP_Channel*>::iterator channel;
   for(channel = oActiveChannels.begin(); channel != oActiveChannels.end(); channel++)
   {
      if(*channel == poChan)
      {
         oActiveChannels.erase(channel);
         return true;
      }
   }
   return false;
}


void DoIP_Connection::vCommonTimerTick(void)
{
   /* TCP */
   /* Timer */
   if (bConnectionIsActive())
   {
      vHandleTimer();
   }

   /* Routing Activation */
   /* Is connection still active ? */
   if (bConnectionIsActive())
   {
      vRoutingActivationMainFunction();
   }

   /* AliveCheck */
   /* Is connection still active ? */
   if (bConnectionIsActive())
   {
      vAliveCheckMainFunction();
   }

   /* Messages */
   /* Is connection still active ? */
   if (bConnectionIsActive())
   {
      vHandleMessages();
   }
}


void DoIP_Connection::vConnectionOpen()
{
   DOIP_TRACE_NOTICE("DoIP_Connection::vConnectionOpen: state=%u socket=%p socketfd=%d", state, poSocket, poSocket->getFd());

   if (state == DOIP_CON_OFFLINE)
   {
      int error_code;
      socklen_t error_code_size = sizeof(error_code);

      if (poSocket->getsockopt(SOL_SOCKET,SO_ERROR, &error_code,&error_code_size) == 0)
      {
         int optval = 1;
         if (poSocket->setsockopt(IPPROTO_TCP, TCP_NODELAY, &optval, sizeof(optval)) < 0)
         {
            DOIP_TRACE_WARNING("DoIP_Connection::vConnectionOpen: Cannot change TCP_NODELAY");
         }

         /* start the reading thread for the TCP sockets */
         //sched_param rTCPSchedAttr;
#ifdef UNIT_TEST
         DOIP_TRACE_NOTICE("UNIT_TEST defined, dont start thread\n");
         vOpenTCP();
         state = DOIP_CON_ONLINE;
#else
         pthread_attr_t rTCPReadAttr;
         pthread_attr_init(&rTCPReadAttr);
         (void)pthread_attr_setstacksize(&rTCPReadAttr, TCPSOCKET_LISTENER_STACKSIZE + DOIP_CFG_cu32TCPBufferSize);
         //pthread_attr_setschedpolicy(&rTCPReadAttr,SCHED_FIFO);
         //pthread_attr_setinheritsched(&rTCPReadAttr,PTHREAD_EXPLICIT_SCHED);
         //pthread_attr_getschedparam(&rTCPReadAttr,&rTCPSchedAttr);
         //rTCPSchedAttr.sched_priority = cu8TCPThreadPriority; //sched_get_priority_min( SCHED_FIFO );
         //pthread_attr_setschedparam (&rTCPReadAttr, &rTCPSchedAttr);

         if(pthread_create( &rTCPReadThread, &rTCPReadAttr, DoIP_Connection::pvTCPReadThread, this) == 0)
         {
            pthread_setname_np(rTCPReadThread, "DoIP@TCP");

            vOpenTCP();
            state = DOIP_CON_ONLINE;
         }
         else
         {
            DOIP_TRACE_CRIT("DoIP_Protocoll::vStartTCPServer: DoIP TCP Thread is not started");
         }
#endif
      }
      else
      {
         DOIP_TRACE_CRIT("DoIP_Protocoll::vStartTCPServer: DoIP TCP socket=%d has errors", poSocket->getFd());
      }
   }
}

bool DoIP_Connection::bConnectionIsActive()
{
   return state == DOIP_CON_ONLINE;
}


void DoIP_Connection::vConnectionClose(bool isSoConModeChg)
{
   DOIP_TRACE_WARNING("DoIP_Connection::vConnectionClose: socket=%d modechg=%u state=%u", poSocket->getFd(), isSoConModeChg, state);

   /* Socket is online, close could result for 2 reasons */
   /* 1) a close could be requested internally */
   /* 2) a close could be closed from SoAd (error) */
   if (state == DOIP_CON_ONLINE)
   {
      state = DOIP_CON_ONLINE_CLOSE_REQUESTED;

      /* Connection (in SoAd) may still be open, but from DoIP point of view there closed and internal states are reset */
      vChannelCloseAll();
      vCloseTCP();

      if(!isSoConModeChg)
      {
         /* Socket in SoAd is open, send close-request to SoAd */
         poSocket->shutdown(SHUT_RDWR);
      }
   }

   /* Close for socket was requested and now is closed in SoAd (read thread failed) */
   if ((isSoConModeChg) && (state == DOIP_CON_ONLINE_CLOSE_REQUESTED))
   {
      state = DOIP_CON_OFFLINE;
      poSocket->close();
   }
}












void DoIP_Connection::vOpenTCP()
{
   if (!bConnectionIsActive())
   {
       /* Initialize TCP socket */
       routingActivation.routingActivationStatus = (tU8) DOIP_RA_STATUS_INIT;
       routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_IDLE;
       routingActivation.routingActivationSA = cu16TesterAddrInit;
       aliveCheck.aliveCheckState = DOIP_ALIVECHECK_IDLE;
       aliveCheck.aliveCheckTimer = 0;
       /* Init RX */
       RX.payloadType = 0;
       RX.payloadLength = 0;
       RX.state = DOIP_TCP_IDLE;

       vSetInitialInactivity();
   }
}

void DoIP_Connection::vCloseTCP()
{
   /* Reset TCP socket */
   if(poCurrentChannel != NULL)
   {
      poCurrentChannel->vTxConfirmation(false);
      poCurrentChannel = NULL;
   }
   rActiveTesterCfg = poDoIPProtocol->rGetTesterCfg(cu16TesterAddrInit);
   rActiveRouteCfg = poDoIPProtocol->rGetTesterCfg(cu16TesterAddrInit)->second.oRouteActCfg.end();
   timerInactivity = 0;
   routingActivation.routingActivationStatus = (tU8) DOIP_RA_STATUS_INIT;
   routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_IDLE;
   routingActivation.routingActivationSA = cu16TesterAddrInit;
   aliveCheck.aliveCheckState = DOIP_ALIVECHECK_IDLE;
   aliveCheck.aliveCheckTimer = 0;
   RX.state = DOIP_TCP_IDLE;
}

void DoIP_Connection::vSetGeneralInactivity()
{
   if (bRoutingActivationIsValid(routingActivation.routingActivationSA))
   {
      timerInactivity = DOIP_CFG_GeneralInactivityTime;
   }
}

void DoIP_Connection::vSetInitialInactivity()
{
   timerInactivity = DOIP_CFG_InitialInactivityTime;
}

void DoIP_Connection::vHandleTimer()
{
   /* Alive check timer */
   tU32 u32NextTime = aliveCheck.aliveCheckTimer > tclEthernetBus::cu32CommonTimerStep ?
                        aliveCheck.aliveCheckTimer - tclEthernetBus::cu32CommonTimerStep : 0;

   if(aliveCheck.aliveCheckTimer > 0 && u32NextTime == 0)
   {
      DOIP_TRACE_WARNING("DoIP_Connection::vHandleTimer: alive check timeout RX.cnt=%u", RX.cnt);

      aliveCheck.aliveCheckTimer = 0;
      vConnectionClose(FALSE);
      return; // immediately fall out of this function, no further actions required
   }
   else
   {
      aliveCheck.aliveCheckTimer = u32NextTime;
   }


   /* Inactivity timer */
   u32NextTime = timerInactivity > tclEthernetBus::cu32CommonTimerStep ?
                   timerInactivity - tclEthernetBus::cu32CommonTimerStep : 0;

   if(timerInactivity > 0 && u32NextTime == 0)
   {
      DOIP_TRACE_WARNING("DoIP_Connection::vHandleTimer: inactivity timeout RX.cnt=%u", RX.cnt);

      timerInactivity = 0;
      vConnectionClose(FALSE);
   }
   else
   {
      timerInactivity = u32NextTime;
   }
}

void DoIP_Connection::vHandleMessages()
{

   /* First check RX (GenericAcknowledge, DM-Ack and NAck) */
   if (RX.state == DOIP_TCP_RX_READY2SEND)
   {
      RX.state = DOIP_TCP_RX_WAIT4CONFIRMATION;

      DOIP_TRACE_NOTICE("DoIP_Connection::vHandleMessages: send RX.ackBuffer");
      if (bSendMessage(RX.ackBuffer, FALSE))
      {
         if (RX.nextReadLen) {
            RX.state = DOIP_TCP_RX_RECEIVING;
            RX.payloadType=DOIP_PAYLOAD_TYPE_GENERIC_HEADER_NACK;
         }
         else {
            RX.state = DOIP_TCP_IDLE;

         }
      }
      else
      {
         RX.state = DOIP_TCP_RX_READY2SEND; // sending failed, try again at next cycle
      }
      DOIP_TRACE_NOTICE("DoIP_Connection::vHandleMessages: set RX.state to 0x%02x", RX.state);
   }

   /* Check App Tx Queue*/
   if (TX.txState == DOIP_CHANNEL_TX_READY2SEND)
   {
      DOIP_TRACE_NOTICE("DoIP_Connection::vHandleMessages: send TX.txBuffer");
      if(bSendMessage(TX.txBuffer, TRUE))
      {
         //TX.txState = DOIP_CHANNEL_TX_SENDING;
         TX.txState = DOIP_CHANNEL_TX_IDLE;

         vSetGeneralInactivity(); // keep connection alive for some further time
      }
      DOIP_TRACE_NOTICE("DoIP_Connection::vHandleMessages: set TX.txState to 0x%02x", TX.txState);
   }

}

bool DoIP_Connection::bSendMessage(tU8* data, bool isDiagMessage)
{
   bool isExecuted = FALSE;

   /* Copy values from buffer */
   tU16 payloadType;
   tU32 payloadLength;
   payloadType=getU16(&data[DOIP_MSG_POS_PAYLOAD_TYPE]);
   payloadLength=getU32(&data[DOIP_MSG_POS_PAYLOAD_LENGTH]);

   tS32 s32BufferLen = (payloadLength + DOIP_HEADER_SIZE);

   DOIP_TRACE_WARNING("DoIP_Connection::bSendMessage: payloadType=%04x payloadLength=%u", payloadType, payloadLength);
   if (poSocket->send(data, s32BufferLen) != s32BufferLen)
   {
      DOIP_TRACE_WARNING("DoIP_Connection::bSendMessage: failed (isDiag=%u)", isDiagMessage);

      if (isDiagMessage)
      {
         RX.state = DOIP_TCP_IDLE;
         if(poCurrentChannel != NULL)
         {
            poCurrentChannel->vTxConfirmation(false);
            poCurrentChannel = NULL;
         }
      }
      else
      {
         vConnectionClose(FALSE);
      }
   }
   else
   {
      /* Set inactivity timer once TxConfirmation for any transmition is received */
      vSetGeneralInactivity();

      switch (payloadType)
      {
      case DOIP_PAYLOAD_TYPE_ALIVE_CHECK_REQ:
         vAliveCheckTxConfirmation();
         break;
      case DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_RESP:
         vRoutingActivationTxConfirmation();
         break;
      case DOIP_PAYLOAD_TYPE_GENERIC_HEADER_NACK:
         vGenericAcknowledgeTxConfirmation();
         break;
      case DOIP_PAYLOAD_TYPE_DIAG_MSG_ACK: /*fallthrough*/
      case DOIP_PAYLOAD_TYPE_DIAG_MSG_NACK:
         vDiagnosticMessageAcknowledgeTxConfirmation();
         break;
      case DOIP_PAYLOAD_TYPE_DIAG_MSG:
         vDiagnosticMessageTxConfirmation();
         break;
      default:
         DOIP_TRACE_NOTICE("DoIP_Connection::vTxConfirmation: DO nothing");
         break;
      }

      isExecuted = TRUE;
   }

   return isExecuted;
}



bool DoIP_Connection::bChannel_Transmit(DoIP_Channel* poChannel, tU8 const au8Data[], tU32 u32Len)
{
   DOIP_TRACE_NOTICE("DoIP_Connection::u8Channel_Transmit[0x%02X]: TX.txstate=%u [%lu] 0x%02X%02X%02X%02X%02X...",\
                     poChannel->cu16TargetAddress, TX.txState, u32Len,
                     au8Data[0], au8Data[1], au8Data[2], au8Data[3], au8Data[4]);

   if (TX.txState == DOIP_CHANNEL_TX_IDLE && u32Len <= DOIP_CFG_cu32TCPMaxRequestSize)
   {
      TX.txState = DOIP_CHANNEL_TX_LOCK;

      if(poCurrentChannel != NULL && poCurrentChannel != poChannel)
      {
         DOIP_TRACE_WARNING("DoIP_Connection::u8Channel_Transmit[0x%02X]: other application [0x%02X] Tx request during processing!",
                            poCurrentChannel->cu16TargetAddress, poChannel->cu16TargetAddress);
      }
      poCurrentChannel = poChannel;

      /* DoIP Header */
      TX.txBuffer[DOIP_MSG_POS_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION;
      TX.txBuffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION_INV;
      /* PayloadType */
      setU16( &TX.txBuffer[DOIP_MSG_POS_PAYLOAD_TYPE], (tU16)DOIP_PAYLOAD_TYPE_DIAG_MSG);
      /* PayloadLength (SduLength + SA + TA */
      setU32( &TX.txBuffer[DOIP_MSG_POS_PAYLOAD_LENGTH], (tU32)(u32Len + DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN));
      /* LogicalAddressDoIP */
      setU16( &TX.txBuffer[DOIP_MSG_POS_CONTENT+0u], poChannel->cu16TargetAddress ); //SA: DoIP node address
      /* LogicalAddressTester */
      setU16( &TX.txBuffer[DOIP_MSG_POS_CONTENT+2u], rActiveTesterCfg->first ); //TA: tester address is TA for sending
      /* copy the payload behind the header */
      memcpy(&TX.txBuffer[DOIP_MSG_POS_CONTENT+DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN], au8Data, u32Len);

      TX.txState = DOIP_CHANNEL_TX_READY2SEND;

      if(cbDiagMsgFastProcessing)
      {
         // directly try to send this message now or retry from main loop
         vHandleMessages();
      }
      // will be sent from main loop

      return true;
   }

   DOIP_TRACE_WARNING("DoIP_Connection::u8Channel_Transmit[0x%02X]: failed (TX.txstate=%u)", poChannel->cu16TargetAddress, TX.txState);
   return false;
}




/*
  check DoIp-Header.
  basic handling: eval size of payload
  error-handling:
  - in case of "fatal" errors, send nack and drop connection
  - for other errors: do error-handling after complete message is received


 */

void DoIP_Connection::setDoipHeader(uint16_t payloadType, uint32_t payloadLength, uint8_t *doipHeader)
{
   doipHeader[DOIP_MSG_POS_PROTOCOL_VERSION]=DOIP_PROTOCOL_VERSION;
   doipHeader[DOIP_MSG_POS_INV_PROTOCOL_VERSION]=DOIP_PROTOCOL_VERSION_INV;
   setU16(&doipHeader[DOIP_MSG_POS_PAYLOAD_TYPE], payloadType);
   setU32(&doipHeader[DOIP_MSG_POS_PAYLOAD_LENGTH], payloadLength);

}

void DoIP_Connection::vHandleRxDoipHeader()
{
   DOIP_TRACE_INFO("DoIP_Connection::vRxDoipHeader: connection=0x%02X RX.state=%u", rActiveTesterCfg->first, RX.state);

   RX.payloadType=getU16(&RX.recvBuffer[DOIP_MSG_POS_PAYLOAD_TYPE]);
   RX.payloadLength=getU32(&RX.recvBuffer[DOIP_MSG_POS_PAYLOAD_LENGTH]);
   //header check
   RX.nextReadLen=0;
   uint8_t responseCode=0;
   RX.state = DOIP_TCP_RX_RECEIVING;
   RX.payloadCounter =0;
   RX.nextReadLen=std::min((uint32_t)(RX.payloadLength-RX.payloadCounter), (uint32_t)DOIP_CFG_cu32TCPBufferSize);

   if (!poDoIPProtocol->bCheckHeader(RX.recvBuffer[DOIP_MSG_POS_PROTOCOL_VERSION],
                                     RX.recvBuffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION],
                                     RX.payloadType, RX.payloadLength, &responseCode, TRUE))
   {
      // Send negativ Acknowledge, because header Data is wrong
      setDoipHeader(DOIP_PAYLOAD_TYPE_GENERIC_HEADER_NACK, DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_NACK, RX.ackBuffer);
      RX.ackBuffer[DOIP_MSG_POS_CONTENT] = responseCode;
      RX.state = DOIP_TCP_RX_READY2SEND;

   }

   if(cbDiagMsgFastProcessing)
   {
      DOIP_TRACE_NOTICE( "DoIP_Connection::vRxDoipHeader, responseCode=0x%02x", responseCode);
      // directly proceed with ACKNACK and consecutive message indication to application
      vHandleMessages();
   }
   return;

}


void DoIP_Connection::vHandleRxPayloadIndication()
{
   DOIP_TRACE_INFO("DoIP_Connection::bRxIndication START: connection=0x%02X len=%u RX.state=%u payloadCounter=%u RX.nextReadLen=%u RX.payloadLength=%u RX.cnt=%u",
                   rActiveTesterCfg->first, RX.recvBufferUsed, RX.state, RX.payloadCounter, RX.nextReadLen, RX.payloadLength, RX.cnt);
   assert(RX.recvBufferUsed == RX.nextReadLen);

   memcpy(RX.payloadBuffer + RX.payloadCounter, RX.recvBuffer, RX.recvBufferUsed);
   RX.payloadCounter+=RX.recvBufferUsed;
   RX.nextReadLen=std::min((uint32_t)(RX.payloadLength-RX.payloadCounter), (uint32_t)DOIP_CFG_cu32TCPBufferSize);

   DOIP_TRACE_INFO("DoIP_Connection::bRxIndication: END RX.nextReadLen=%u", RX.nextReadLen);
}

void DoIP_Connection::vHandleRxMsgComplete()
{
   DOIP_TRACE_INFO("DoIP_Connection::vHandleRxMsgComplete: connection=0x%02X RX.state=%u RX.payloadType=%u START",
                   rActiveTesterCfg->first, RX.state, RX.payloadType);

   RX.state = DOIP_TCP_RX_LOCK;
   switch (RX.payloadType)
   {
      case DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_REQ:
      {
         vHandleRxRoutingActivationRequest();
      }
      break;

      case DOIP_PAYLOAD_TYPE_ALIVE_CHECK_RESP:
      {
         vHandleRxAliveCheckResp();
      }
      break;

      case DOIP_PAYLOAD_TYPE_DIAG_MSG:
      {
         vHandleRxDiagMsg();
      }
      break;
      default:
         // Response has already been sent, no action here
         break;
   }

   RX.state = DOIP_TCP_IDLE;

}






void DoIP_Connection::vHandleRxRoutingActivationRequest() {
   DOIP_TRACE_INFO("DoIP_Connection::vHandleRxRoutingActivationRequest: DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_REQ: actState=%d len=%u", routingActivation.routingActivationState, RX.payloadLength);
   if (
       ((RX.payloadLength == DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_11) || (RX.payloadLength == DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_7))
       && // TODO RA: Remove below check when adapting RA handling
       (routingActivation.routingActivationState == DOIP_ROUTINGACTIVATION_IDLE)
       )
   {
      if (RX.payloadLength == DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_11) {
         routingActivation.rx.hasOem=true;
      }
      else {
         routingActivation.rx.hasOem=false;
      }

      routingActivation.rx.testerAddress=getU16(&RX.payloadBuffer[0u]);
      routingActivation.rx.activationType=getU8(&RX.payloadBuffer[2u]);

      memcpy(&(routingActivation.rx.iso), &RX.payloadBuffer[3u], 4);
      if (routingActivation.rx.hasOem)
      {
         //don't change endianess/byte order of the OEM field
         memcpy( &(routingActivation.rx.oem), &RX.payloadBuffer[7u], 4u); //oem specific
      }
      DOIP_TRACE_NOTICE("DoIP_Connection::vHandleRxRoutingActivationRequest: received u16TesterAddr=0x%04x", routingActivation.rx.testerAddress);
      DOIP_TRACE_NOTICE("DoIP_Connection::vHandleRxRoutingActivationRequest: rActiveTesterCfg->first=0x%04x", rActiveTesterCfg->first);

      if(rActiveTesterCfg->first != routingActivation.rx.testerAddress)
      {
         // this connection was just instantiated and is virgin, load the routing activation config
         rActiveTesterCfg = poDoIPProtocol->rGetTesterCfg(routingActivation.rx.testerAddress);
         DOIP_TRACE_NOTICE("DoIP_Connection::bRxIndication: tester 0x%02X identified (>>0x%02X)", routingActivation.rx.testerAddress, rActiveTesterCfg->first);
      }
      routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_RECEIVED;
      RX.payloadCounter = 0;
   }
   else {
      DOIP_TRACE_NOTICE("DoIP_Connection::bRxIndication:len or state not ok");
   }
}

void DoIP_Connection::vHandleRxAliveCheckResp() {
   DOIP_TRACE_INFO("DoIP_Connection::bRxIndication: DOIP_PAYLOAD_TYPE_ALIVE_CHECK_RESP");
   if (RX.payloadLength == DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_RESP)
   {
      copyU16(&aliveCheck.aliveCheckBuffer[DOIP_MSG_POS_CONTENT], &RX.payloadBuffer[0]); //SA
      aliveCheck.aliveCheckState = DOIP_ALIVECHECK_RECEIVED;
      RX.payloadCounter = 0;
   }
}


/*
  handle the complete diag-message
 */
void DoIP_Connection::vHandleRxDiagMsg() {
   DOIP_TRACE_INFO("DoIP_Connection::vHandleRxDiagMsg");

   //prepare response / acknowledment message
   RX.diagMsgSourceAddress=getU16(&RX.payloadBuffer[0u]); //SA
   RX.diagMsgTargetAddress=getU16(&RX.payloadBuffer[2u]); //TA

   //perform checks
   uint8_t respCode = u8CheckDiagMsgHeader();
   do {
      if (respCode != DOIP_DM_ACKCODE_SUCCESS)
      {
         vPrepareDiagnosticAcknowledgeMessage(respCode);
         break;
      }
      poCurrentChannel = poFindChannel(RX.diagMsgTargetAddress);
      if(poCurrentChannel == NULL)
      {
         DOIP_TRACE_WARNING( "DoIP_Connection::bRxIndication: channel 0x%02X not found", RX.diagMsgTargetAddress);
         vPrepareDiagnosticAcknowledgeMessage(DOIP_DM_NACKCODE_TA_UNREACHABLE);
         break;
      }
      tU32 sduLength=RX.payloadLength - DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN;
      //min of sdu length or payload length -> transmit what we have but max a single message
      if (poCurrentChannel->u32RxStartOfReception(sduLength) < sduLength) {
         // the request's PDU size exceeds the target channel's max PDU size
         vPrepareDiagnosticAcknowledgeMessage(DOIP_DM_NACKCODE_MSG_TOO_LARGE);
         break;
      }

      vPrepareDiagnosticAcknowledgeMessage(DOIP_DM_ACKCODE_SUCCESS);
      DOIP_TRACE_NOTICE( "DoIP_Connection::bRxIndication: send SUCCESS response");
   } while (0);

   if(cbDiagMsgFastProcessing)
   {
      // directly proceed with ACKNACK and consecutive message indication to application
      vHandleMessages();
   }
}

tU16 DoIP_Connection::u16GetAckNackResponseLength()
{
   //For negativ response the configuration of the tester (numByteDiagAckNack) is relevant
    DOIP_TRACE_INFO("DoIP_Connection::u16GetAckNackResponseLength config=%u", rActiveTesterCfg->second.numByteDiagAckNack);
    DOIP_TRACE_INFO("DoIP_Connection::u16GetAckNackResponseLength want=%u",
                    DOIP_CFG_cu32TCPBufferSize - (DOIP_HEADER_SIZE+DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN+1u));

   tU16 res= DOIP_MIN(rActiveTesterCfg->second.numByteDiagAckNack, (DOIP_CFG_cu32TCPBufferSize - (DOIP_HEADER_SIZE+DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN+1u)));
    DOIP_TRACE_INFO("DoIP_Connection::u16GetAckNackResponseLength res=%u", res);
    return res;
}

void DoIP_Connection::vPrepareDiagnosticAcknowledgeMessage(tU8 responseCode)
{
    uint16_t payloadType = (responseCode == DOIP_DM_ACKCODE_SUCCESS) ?
    DOIP_PAYLOAD_TYPE_DIAG_MSG_ACK : DOIP_PAYLOAD_TYPE_DIAG_MSG_NACK;

    uint32_t ackLen=DOIP_MIN(RX.payloadLength, u16GetAckNackResponseLength());
    DOIP_TRACE_INFO("DoIP_Connection::vPrepareDiagnosticAcknowledgeMessage responseCode=%02x ackLen=%u", responseCode, ackLen);
    //DoIP Header

    setDoipHeader(payloadType, DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN + 1 + ackLen, RX.ackBuffer);

    //Diag Msg Resp
    //exchange addresses
    setU16(&RX.ackBuffer[8], RX.diagMsgTargetAddress);
    setU16(&RX.ackBuffer[10], RX.diagMsgSourceAddress);
    //DoIP_LibMemCopy(&pConn->buffer[10], &tempAddr, 2);

    //save resp code
    RX.ackBuffer[12] = responseCode;

    memcpy(&RX.ackBuffer[13], RX.payloadBuffer, ackLen);

    //acknowledgment data already saved when received
    RX.state = DOIP_TCP_RX_READY2SEND;
}

tU8 DoIP_Connection::u8CheckDiagMsgHeader()
{
    tU8 retVal = DOIP_DM_NACKCODE_UNDEFINED;

    do {
       //check tester SA is registered to established connection -> nack code 0x02, close socket
       if (!bRoutingActivationIsValid(RX.diagMsgSourceAddress))
       {
          retVal = DOIP_DM_NACKCODE_INVALID_SA;
          break;
       }
       //check TA & SA are connected with current activation -> nack 0x03, discard msg
       if (!bCheckTaConfiguration())
       {
          retVal = DOIP_DM_NACKCODE_UNKNOWN_TA;
          break;
       }
       //check payload length is bigger than doIPMaxrequestBytes-4 -> nack 0x04, discard msg (changed to -4 according DoIP_00125
       if (RX.payloadLength > (DOIP_CFG_cu32TCPMaxRequestSize - 4u))
       {
          retVal = DOIP_DM_NACKCODE_MSG_TOO_LARGE;
          break;
       }
       //check buffer size is not big enough -> nack 0x05, discard msg
       /*
         else if( doip buffer < pPduInfoType->SduLength )
         {
         retVal = DOIP_DM_NACKCODE_OUT_OF_MEM;
         }
       */
       //check TA is activated -> nack 0x06, discard msg
       if (!bCheckTaAtConfiguration())
       {
          retVal = DOIP_DM_NACKCODE_TA_UNREACHABLE;
          break;
       }
       
       retVal = DOIP_DM_ACKCODE_SUCCESS;

    } while (0);

    return retVal;
}

bool DoIP_Connection::bCheckTaConfiguration()
{
   bool retVal = FALSE;

   std::list<DoIP_RoutingActivationTarget>::iterator cfg=rActiveRouteCfg->second.targetCfg.begin();
   if(!rActiveRouteCfg->second.targetCfg.size())
   {
      DOIP_TRACE_INFO("%s: No active routes", __PRETTY_FUNCTION__);
   }
   for(cfg = rActiveRouteCfg->second.targetCfg.begin(); !retVal && cfg != rActiveRouteCfg->second.targetCfg.end(); cfg++)
   {
      if( cfg->targetSA == RX.diagMsgTargetAddress )
      {
         retVal = TRUE;
      }
   }

   if(!retVal)
   {
      DOIP_TRACE_INFO("%s: No route to target Address found", __PRETTY_FUNCTION__);
   }
   return retVal;
}

bool DoIP_Connection::bCheckTaAtConfiguration()
{
   bool retVal = FALSE;

   std::list<DoIP_Channel*>::iterator channel;
   if(!oActiveChannels.size())
   {
      DOIP_TRACE_INFO("%s: No active channels", __PRETTY_FUNCTION__);
   }
   for(channel = oActiveChannels.begin(); !retVal && channel != oActiveChannels.end(); channel++)
   {
      if( (*channel)->cu16TargetAddress == RX.diagMsgTargetAddress )
      {
         retVal = TRUE;
      }
   }

   return retVal;
}

void DoIP_Connection::vDiagnosticMessageAcknowledgeTxConfirmation()
{
   tU8 responseCode = RX.ackBuffer[DOIP_MSG_POS_CONTENT + 4u]; //pos 12

   switch (responseCode)
   {
   case DOIP_DM_NACKCODE_INVALID_SA:
      vConnectionClose(FALSE);
      break;
   case DOIP_DM_ACKCODE_SUCCESS:
      if(poCurrentChannel != NULL)
      {
         poCurrentChannel->vRxIndication(RX.payloadBuffer + DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN, RX.payloadLength - DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN);
         poCurrentChannel = NULL;
      }
      break;
   case DOIP_DM_NACKCODE_TP_ERROR:
      if(poCurrentChannel != NULL)
      {
         //vRxIndication(pdurPdu, NTFRSLT_E_NOT_OK);
         poCurrentChannel->vRxError(0);
         poCurrentChannel = NULL;
      }
      break;
   default:
      break;
   }

   if (!RX.nextReadLen) {
      RX.state = DOIP_TCP_IDLE;
   }
}

void DoIP_Connection::vDiagnosticMessageTxConfirmation()
{
   if(poCurrentChannel != NULL)
   {
      poCurrentChannel->vTxConfirmation(true);
      poCurrentChannel = NULL;
   }
}

void DoIP_Connection::vGenericAcknowledgeTxConfirmation()
{
   tU8 responseCode = RX.ackBuffer[DOIP_MSG_POS_CONTENT];

   switch (responseCode)
   {
   case DOIP_GA_RESPCODE_INVALID_PROTOCOL:
   case DOIP_GA_RESPCODE_INVALID_PAYLOAD_LENGTH_TYPE:
      vConnectionClose(FALSE);
      break;
   default:
      if (!RX.nextReadLen) {
         RX.state = DOIP_TCP_IDLE;
      }
      break;
   }
}

bool DoIP_Connection::bRoutingActivationIsValid(tU16 sourceAddr)
{
    bool ret_val = FALSE;
    /*
      check if we have an activated routingActivation and that sourceAddr (tester) of received message matches 
     */
    if ((routingActivation.routingActivationSA == sourceAddr)
         && ((routingActivation.routingActivationStatus & DOIP_RA_STATUS_ROUTE_REG_MASK) == DOIP_RA_STATUS_ROUTE_REGISTERED)
         && ((routingActivation.routingActivationStatus & DOIP_RA_STATUS_ROUTE_ACTIVE_MASK) == DOIP_RA_STATUS_ROUTE_ACTIVE))
    {
        ret_val = TRUE;
    }

    return ret_val;
}

void DoIP_Connection::vRoutingActivationMainFunction()
{
    /* Either new RA-request was received or an alivecheck for another connection is pending */
    if ((routingActivation.routingActivationState == DOIP_ROUTINGACTIVATION_RECEIVED)
         || (routingActivation.routingActivationState == DOIP_ROUTINGACTIVATION_WAIT4ALIVECHECK))
    {
        vRoutingActivationRxHandleRequest();
    }

    /* Routing Confirmation is pending */
    if (routingActivation.routingActivationState == DOIP_ROUTINGACTIVATION_PENDING)
    {
        vRoutingActivationPollResults();
    }

    /* Ready to send the RA status */
    if (routingActivation.routingActivationState == DOIP_ROUTINGACTIVATION_READY2SEND)
    {
       routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_WAIT4CONFIRMATION;
        if (!bSendMessage(routingActivation.routingActivationBuffer, FALSE))
        {
           
            routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_READY2SEND;
        }
    }
}

void DoIP_Connection::vRoutingActivationRxHandleRequest()
{
    tU32 payloadLength = 0;
    tU16 sourceAddr = 0;
    tU8 activationType = 0;
    DoIP_Connection* poOtherConn = NULL;
    
    DoIP_RoutingActivationMessage const &rxMsg=routingActivation.rx;
    memcpy(&payloadLength, &routingActivation.routingActivationBuffer[DOIP_MSG_POS_PAYLOAD_LENGTH], DOIP_MSG_SIZE_PAYLOAD_LENGTH);
    memcpy( &sourceAddr, &routingActivation.routingActivationBuffer[DOIP_MSG_POS_CONTENT+0u], 2u );
    memcpy( &activationType, &routingActivation.routingActivationBuffer[DOIP_MSG_POS_CONTENT+2u], 1u );

    DOIP_TRACE_NOTICE("DoIP_Connection::vRoutingActivationRxHandleRequest[0x%02X]: srcaddr=0x%02X acttype=0x%02X hasOem=%d",
        rActiveTesterCfg->first, rxMsg.testerAddress, rxMsg.activationType, rxMsg.hasOem);

    //check if SA is correct
    if (rxMsg.testerAddress != rActiveTesterCfg->first)
    {
        //if not, replay with response code 0x00
        vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_UNKNOWN_SA, (tU8*)NULL);
        //close socket after tx
    }
    //check if activation type is supported
    else if (!bIsRoutingActivationTypeSupported(rxMsg.activationType))
    {
        //if not, replay with response code 0x06
        vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_UNKNWON_TYPE, (tU8*) NULL);
        //close socket after tx
    }
    //check if our connection is registered to another SA
    else if (bIsConnectionAlreadyAssginedToOtherSourceAddress(rxMsg.testerAddress))
    {
        //if it is, reply with response code 0x02
        vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_CONNECTION_IN_USE, (tU8*)NULL);
        //close socket after tx
    }
    //check if SA is used in another connection
    //hint: if an alive check timeout occurred we will not enter this part again as the socket gets closed and registration is freed
    else if (poDoIPProtocol->bIsSourceAddressRegisteredToOtherConnection(this, rxMsg.testerAddress, &poOtherConn))
    {
        if (routingActivation.routingActivationState == DOIP_ROUTINGACTIVATION_RECEIVED)
        {
            vAliveCheckRequest();
            routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_WAIT4ALIVECHECK;
        }
        else if (routingActivation.routingActivationState == DOIP_ROUTINGACTIVATION_WAIT4ALIVECHECK)
        {
            //if other socket is still online although the alive check has run -> won't get the routing activation
            if (!bAliveCheckIsActive())
            {
                vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_SA_IN_USE, (tU8*)NULL);
            }
        }
        else
        {
           // Do Error Handling here
           // DOIP_DET(DOIP_DET_APIID_ROUTINGACTIVATION, DOIP_E_NOT_SUPPORTED);
        }
    }
    else
    {
        //check if SA is used in own/current connection or if SA is free
        if (bIsSourceAddressRegistered(rxMsg.testerAddress) ||
            poDoIPProtocol->bIsSourceAddressNotYetRegistered(rxMsg.testerAddress))
        {
            tU8 tempAuth;
            tU8 tempConf;

            //activation Type changed -> reset activation state
            if (bIsSourceAddressRegisteredToConnectionWithAnotherActivationType(rxMsg.testerAddress, rxMsg.activationType))
            {
               DOIP_TRACE_WARNING("DoIP_Connection::vRoutingActivationRxHandleRequest: activation type change from 0x%02X to 0x%02X",
                                  routingActivation.routingActivationType, rxMsg.activationType);
                routingActivation.routingActivationStatus = (tU8) DOIP_RA_STATUS_INIT;
                vChannelCloseAll();
            }

            //save oem bytes for later use
            if (rxMsg.hasOem)
            {
                memcpy(routingActivation.routingActivationOemBufferReq,rxMsg.oem, 4u);
                memset(routingActivation.routingActivationOemBufferResp, 0, 4);
            }
            else
            {
                memset(routingActivation.routingActivationOemBufferReq, 0, 4);
                memset(routingActivation.routingActivationOemBufferResp, 0, 4);
            }

            tempAuth = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_AUTH_MASK);
            tempConf = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_CONF_MASK);

            //ask for authentication
            if ((tempAuth == DOIP_RA_STATUS_AUTH_NOT_CHECKED) || (tempAuth == DOIP_RA_STATUS_AUTH_PENDING))
            {
                vCheckRoutingAuthentication(rxMsg.testerAddress, rxMsg.activationType);
            }
            //update temp variable
            tempAuth = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_AUTH_MASK);

            if (tempAuth == DOIP_RA_STATUS_AUTH_SUCCESS)
            {
                //ask for confirmation
                if ((tempConf == DOIP_RA_STATUS_CONF_NOT_CHECKED) || (tempConf == DOIP_RA_STATUS_CONF_PENDING))
                {
                    vCheckRoutingConfirmation(rxMsg.testerAddress, rxMsg.activationType);
                }
                //update temp variable
                tempConf = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_CONF_MASK);
            }

            if (tempAuth == DOIP_RA_STATUS_AUTH_FAILED)
            {
                //no routing activation
                //response code 0x04 DOIP_RA_RESPCODE_AUTH_FAILED
                //route not active
                //route registered

                DOIP_TRACE_NOTICE("DoIP_Connection::vRoutingActivationRxHandleRequest: authentication failed");

                vSetRoutingActivationStatus(FALSE, TRUE, rxMsg.testerAddress, rxMsg.activationType);
                vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_AUTH_FAILED, routingActivation.routingActivationOemBufferResp);
            }
            else if (tempConf == DOIP_RA_STATUS_CONF_FAILED)
            {
                //no routing activation
                //response code 0x05 DOIP_RA_RESPCODE_CONF_FAILED
                //route not active
                //route not registered
                //close socket -> handled in DoIP_HandleRoutingActivationRespAfterTransmission

                DOIP_TRACE_NOTICE("DoIP_Connection::vRoutingActivationRxHandleRequest: confirmation failed");

                vSetRoutingActivationStatus(FALSE, FALSE, rxMsg.testerAddress, rxMsg.activationType);
                vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_CONF_FAILED, routingActivation.routingActivationOemBufferResp);
            }
            else if ((tempAuth == DOIP_RA_STATUS_AUTH_PENDING) || (tempConf == DOIP_RA_STATUS_CONF_PENDING))
            {
                //routing activation is still pending
                //response code 0x11 DOIP_RA_RESPCODE_CONF_PENDING
                //route not active
                //route registered

                vSetRoutingActivationStatus(FALSE, TRUE, rxMsg.testerAddress, rxMsg.activationType);
                vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_CONF_PENDING, routingActivation.routingActivationOemBufferResp);
            }
            else if ((tempAuth == DOIP_RA_STATUS_AUTH_SUCCESS) && (tempConf == DOIP_RA_STATUS_CONF_SUCCESS))
            {
                //routing activation successful
                //response code 0x10
                //route active
                //route registered

                DOIP_TRACE_NOTICE("DoIP_Connection::vRoutingActivationRxHandleRequest: activation successful");

                vSetRoutingActivationStatus(TRUE, TRUE, rxMsg.testerAddress, rxMsg.activationType);
                vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_SUCCESS, routingActivation.routingActivationOemBufferResp);
            }
            else
            {
                //should never be reached
            }
        }
    }
}

void DoIP_Connection::vRoutingActivationTxConfirmation()
{
    tU8 responseCode = routingActivation.routingActivationBuffer[DOIP_HEADER_SIZE + 4u];
    switch (responseCode)
    {
        case DOIP_RA_RESPCODE_AUTH_FAILED:
        case DOIP_RA_RESPCODE_CONF_FAILED:
        case DOIP_RA_RESPCODE_SUCCESS:
            routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_IDLE;
            break;
        case DOIP_RA_RESPCODE_CONF_PENDING:
            routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_PENDING;
            break;
        case DOIP_RA_RESPCODE_UNKNOWN_SA:
        case DOIP_RA_RESPCODE_NO_FREE_SOCKET:
        case DOIP_RA_RESPCODE_CONNECTION_IN_USE:
        case DOIP_RA_RESPCODE_SA_IN_USE:
        case DOIP_RA_RESPCODE_UNKNWON_TYPE:
        default:
            vConnectionClose(FALSE);
            break;
    }
}

void DoIP_Connection::vRoutingActivationPollResults()
{
    tU8 tempAuth = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_AUTH_MASK);
    tU8 tempConf = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_CONF_MASK);

    if (tempAuth == DOIP_RA_STATUS_AUTH_PENDING)
    {
        vCheckRoutingAuthentication(routingActivation.routingActivationSA, routingActivation.routingActivationType);
        tempAuth = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_AUTH_MASK);
    }

    if (tempAuth == DOIP_RA_STATUS_AUTH_SUCCESS)
    {
        if (bIsRoutingActivationTypeSupported(routingActivation.routingActivationType)) //possible that it has not yet been checked
        {
            if ((tempConf == DOIP_RA_STATUS_CONF_NOT_CHECKED) || (tempConf == DOIP_RA_STATUS_CONF_PENDING))
            {
                vCheckRoutingConfirmation(routingActivation.routingActivationSA, routingActivation.routingActivationType);
            }
        }
        else
        {
            vSetRoutingActivationConfirmationStatus(DOIP_RA_STATUS_CONF_SUCCESS);
        }
        tempConf = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_CONF_MASK);
    }

    if (tempAuth == DOIP_RA_STATUS_AUTH_FAILED)
    {
        //no routing activation
        //response code 0x04 DOIP_RA_RESPCODE_AUTH_FAILED
        //route not active
        //route registered

        DOIP_TRACE_NOTICE("DoIP_Connection::vRoutingActivationPollResults: authentication failed");

        vUpdateRoutingActivationStatus(FALSE, TRUE);
        vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_AUTH_FAILED, routingActivation.routingActivationOemBufferResp);
    }
    else if (tempConf == DOIP_RA_STATUS_CONF_FAILED)
    {
        //no routing activation
        //response code 0x05 DOIP_RA_RESPCODE_CONF_FAILED
        //route not active
        //route not registered
        //close socket -> handled in DoIP_HandleRoutingActivationRespAfterTransmission

        DOIP_TRACE_NOTICE("DoIP_Connection::vRoutingActivationPollResults: confirmation failed");

        vUpdateRoutingActivationStatus(FALSE, FALSE);
        vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_CONF_FAILED, routingActivation.routingActivationOemBufferResp);
    }
    else if ((tempAuth == DOIP_RA_STATUS_AUTH_SUCCESS) && (tempConf == DOIP_RA_STATUS_CONF_SUCCESS))
    {
        //routing activation successful
        //response code 0x10
        //route active
        //route registered

        DOIP_TRACE_NOTICE("DoIP_Connection::vRoutingActivationPollResults: activation successful");

        vUpdateRoutingActivationStatus(TRUE, TRUE);
        vRoutingActivationPrepareResponse(DOIP_RA_RESPCODE_SUCCESS, routingActivation.routingActivationOemBufferResp);
    }
    else
    {
        //can be reached by pending
    }
}

bool DoIP_Connection::bIsSourceAddressRegistered(tU16 sourceAddr)
{
    bool retVal = FALSE;

    if ((routingActivation.routingActivationSA == sourceAddr)
            && ((routingActivation.routingActivationStatus & DOIP_RA_STATUS_ROUTE_REG_MASK) == DOIP_RA_STATUS_ROUTE_REGISTERED))
    {
        retVal = TRUE;
    }

    return retVal;
}

void DoIP_Connection::vRoutingActivationPrepareResponse(tU8 responseCode, const tU8 *oem)
{
   // after this method, routingActivationBuffer holds  response
    //DoIP Header
    routingActivation.routingActivationBuffer[0] = DOIP_PROTOCOL_VERSION;
    routingActivation.routingActivationBuffer[1] = DOIP_PROTOCOL_VERSION_INV;
    /* PayloadType */
    setU16(&routingActivation.routingActivationBuffer[2], (tU16)DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_RESP);
    /* PayloadLength */
    setU32(&routingActivation.routingActivationBuffer[4], (tU32)DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_RESP_13);

    /* Routing Activation Response */
    /* LogicalAddressTester */
    setU16(&routingActivation.routingActivationBuffer[8], (tU16)rActiveTesterCfg->first);//routingActivation.routingActivationSA;
    setU16(&routingActivation.routingActivationBuffer[10], poDoIPProtocol->getNodeAddress());//DOIP_CFG_LogicalAddress;

    routingActivation.routingActivationBuffer[12] = responseCode;
    routingActivation.routingActivationBuffer[13] = 0x00; //ISO
    routingActivation.routingActivationBuffer[14] = 0x00; //ISO
    routingActivation.routingActivationBuffer[15] = 0x00; //ISO
    routingActivation.routingActivationBuffer[16] = 0x00; //ISO
    if (oem == NULL)
    {
        routingActivation.routingActivationBuffer[17] = 0x00; //OEM
        routingActivation.routingActivationBuffer[18] = 0x00; //OEM
        routingActivation.routingActivationBuffer[19] = 0x00; //OEM
        routingActivation.routingActivationBuffer[20] = 0x00; //OEM
    }
    else
    {
        //don't change endianess/byte order of the OEM field
        memcpy(&routingActivation.routingActivationBuffer[17], oem, 4);
    }

    /* Set inactivity timer once routing activation request is received */
    vSetGeneralInactivity();
    routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_READY2SEND;
}

bool DoIP_Connection::bIsRoutingActivationTypeSupported(tU8 activationType)
{
   bool retVal = FALSE;
   std::map<tU8, DoIP_RoutingActivationCfg>::iterator config;
   for (config = rActiveTesterCfg->second.oRouteActCfg.begin(); config != rActiveTesterCfg->second.oRouteActCfg.end(); config++)
   {
      if(config->first == activationType)
      {
         retVal = TRUE;
         break;
      }
   }

   return retVal;
}

bool DoIP_Connection::bIsConnectionAlreadyAssginedToOtherSourceAddress(tU16 sourceAddress)
{
   return (routingActivation.routingActivationSA != cu16TesterAddrInit) && (routingActivation.routingActivationSA != sourceAddress);
}

bool DoIP_Connection::bIsActivationTypeActive(tU8 activationType)
{
   return routingActivation.routingActivationType == activationType;
}

bool DoIP_Connection::bIsSourceAddressRegisteredToConnectionWithAnotherActivationType(tU16 sourceAddr, tU8 activationType)
{
    return ((routingActivation.routingActivationSA == sourceAddr)
            && (routingActivation.routingActivationType != activationType)
            && ((routingActivation.routingActivationStatus & DOIP_RA_STATUS_ROUTE_REG_MASK) == DOIP_RA_STATUS_ROUTE_REGISTERED));
}

void DoIP_Connection::vCheckRoutingAuthentication(tU16 sourceAddr, tU8 activationType)
{
   tU8 tempAuth = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_AUTH_MASK);

   if(tempAuth == DOIP_RA_STATUS_AUTH_NOT_CHECKED)
   {
      vSetRoutingActivationAuthenticationStatus(DOIP_RA_STATUS_AUTH_PENDING);

      std::map<tU8, DoIP_RoutingActivationCfg>::iterator config = rActiveTesterCfg->second.oRouteActCfg.find(activationType);

      if(config != rActiveTesterCfg->second.oRouteActCfg.end())
      {
         if(config->second.bAuthRequ)
         {
            if(poDoIPProtocol->poGetDoIPServer() != NULL)
            {
               // extend the inactivity timeout for application response
               vSetInitialInactivity();

               DOIP_TRACE_NOTICE("DoIP_Connection::vCheckRoutingAuthentication: asking application");
               poDoIPProtocol->poGetDoIPServer()->vRoutingActivationAuthenticationReq(sourceAddr, activationType, routingActivation.routingActivationOemBufferReq);
            }
            else
            {
               DOIP_TRACE_NOTICE("DoIP_Connection::vCheckRoutingAuthentication: no application present");
               vRoutingActivationAuthenticationResp(false, routingActivation.routingActivationOemBufferReq);
            }
         }
         else
         {
            DOIP_TRACE_NOTICE("DoIP_Connection::vCheckRoutingAuthentication: no authentication required");
            // there is no authentification required -> accept it
            vRoutingActivationAuthenticationResp(true, routingActivation.routingActivationOemBufferReq);
         }
      }
   }
}

void DoIP_Connection::vRoutingActivationAuthenticationResp(bool Authentified, tU8* AuthenticationResData)
{
   tU8 tempAuth = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_AUTH_MASK);

   if(tempAuth == DOIP_RA_STATUS_AUTH_PENDING)
   {
      memcpy(routingActivation.routingActivationOemBufferResp, AuthenticationResData, 4);

      DOIP_TRACE_NOTICE("DoIP_Connection::vRoutingActivationAuthenticationResp: authentified=%u", Authentified);

      if (Authentified)
      {
         vSetRoutingActivationAuthenticationStatus(DOIP_RA_STATUS_AUTH_SUCCESS);
      }
      else
      {
         vSetRoutingActivationAuthenticationStatus(DOIP_RA_STATUS_AUTH_FAILED);
      }
   }
}

void DoIP_Connection::vCheckRoutingConfirmation(tU16 /*sourceAddr*/, tU8 activationType)
{
   tU8 tempConf = (tU8) (routingActivation.routingActivationStatus & DOIP_RA_STATUS_CONF_MASK);

   if(tempConf == DOIP_RA_STATUS_CONF_NOT_CHECKED)
   {
      vSetRoutingActivationConfirmationStatus(DOIP_RA_STATUS_CONF_PENDING);

      std::map<tU8, DoIP_RoutingActivationCfg>::iterator rRouteCfg = rActiveTesterCfg->second.oRouteActCfg.find(activationType);

      if(rRouteCfg != rActiveTesterCfg->second.oRouteActCfg.end())
      {
         DOIP_TRACE_NOTICE("DoIP_Connection::vCheckRoutingConfirmation: allocating %u channels", rRouteCfg->second.targetCfg.size());

         // try to activate each channel now
         std::list<DoIP_RoutingActivationTarget>::iterator channel;
         for(channel = rRouteCfg->second.targetCfg.begin(); channel != rRouteCfg->second.targetCfg.end(); channel++)
         {
            DOIP_TRACE_INFO("DoIP_Connection::vCheckRoutingConfirmation: type=%u bus=%u tester=0x%04X target=0x%04X base=%0x04X ext=0x%04X",
                            channel->type, channel->busNumber, rActiveTesterCfg->first, channel->targetSA, channel->targetBaseAddr, channel->targetAddrExt);

            DoIP_Channel* poRoute=nullptr;
            if (mpoFactory) {
               poRoute =mpoFactory->createChannel(this, channel->type, channel->busNumber,
                                                  rActiveTesterCfg->first, channel->targetSA, 
                                                  channel->targetBaseAddr, (tU8)channel->targetAddrExt);
            }
            if(poRoute != NULL)
            {
               oActiveChannels.push_back(poRoute);
            }
            else
            {
               DOIP_TRACE_INFO("%s: No channel created", __PRETTY_FUNCTION__);
               // TODO how to react when a channel fails???
            }
         }

         #if 0
         if(poDoIPProtocol->poGetDoIPServer() != NULL)
         {
            poDoIPProtocol->poGetDoIPServer()->vRoutingActivationConfirmationReq(sourceAddr, activationType, routingActivation.routingActivationOemBufferReq);
         }
         else
         #endif
         {
            vSetRoutingActivationConfirmationStatus(DOIP_RA_STATUS_CONF_SUCCESS);
         }
      }
      else
      {
         DOIP_TRACE_WARNING("DoIP_Connection::vCheckRoutingConfirmation: route 0x%02X not found", activationType);
         vSetRoutingActivationConfirmationStatus(DOIP_RA_STATUS_CONF_FAILED);
      }
   }
}

void DoIP_Connection::vSetRoutingActivationAuthenticationStatus(tU8 authenticationStatus)
{
    routingActivation.routingActivationStatus &= (tU8) (~(DOIP_RA_STATUS_AUTH_MASK));
    routingActivation.routingActivationStatus = (tU8) (routingActivation.routingActivationStatus |(authenticationStatus & DOIP_RA_STATUS_AUTH_MASK));
}

bool DoIP_Connection::bCheckRoutingActivationAuthenticationStatus(tU8 authenticationStatus)
{
   return (routingActivation.routingActivationStatus & DOIP_RA_STATUS_ROUTE_ACTIVE_MASK) == authenticationStatus;
}

int DoIP_Connection::iGetSocket() {
   return poSocket->getFd();
}


bool DoIP_Connection::bCheckRoutingActivationRegistrationStatus(tU8 registrationStatus)
{
   return (routingActivation.routingActivationStatus & DOIP_RA_STATUS_ROUTE_REG_MASK) == registrationStatus;
}

void DoIP_Connection::vSetRoutingActivationConfirmationStatus(tU8 confirmationStatus)
{
    routingActivation.routingActivationStatus &= (tU8) (~(DOIP_RA_STATUS_CONF_MASK));
    routingActivation.routingActivationStatus =(tU8) (routingActivation.routingActivationStatus |  (confirmationStatus & DOIP_RA_STATUS_CONF_MASK));
}

void DoIP_Connection::vSetRoutingActivationStatus(bool active, bool registered, tU16 sourceAddr, tU8 activationType)
{
    routingActivation.routingActivationStatus &= (tU8)(~(DOIP_RA_STATUS_ROUTE_REG_MASK|DOIP_RA_STATUS_ROUTE_ACTIVE_MASK));

    if (active)
    {
        routingActivation.routingActivationStatus |= (tU8) DOIP_RA_STATUS_ROUTE_ACTIVE;
    }
    else
    {
        routingActivation.routingActivationStatus |= (tU8) DOIP_RA_STATUS_ROUTE_NOT_ACTIVE;
    }

    if (registered)
    {
        routingActivation.routingActivationStatus |= (tU8) DOIP_RA_STATUS_ROUTE_REGISTERED;
        routingActivation.routingActivationSA = sourceAddr;
        routingActivation.routingActivationType = activationType;
        if(rActiveTesterCfg != poDoIPProtocol->rGetTesterCfg(cu16TesterAddrInit))
        {
           rActiveRouteCfg = rActiveTesterCfg->second.oRouteActCfg.find(routingActivation.routingActivationType);
        }
    }
    else
    {
        routingActivation.routingActivationStatus |= (tU8) DOIP_RA_STATUS_ROUTE_NOT_REG;
    }
}

void DoIP_Connection::vUpdateRoutingActivationStatus(bool active, bool registered)
{
    routingActivation.routingActivationStatus &= (tU8)(~(DOIP_RA_STATUS_ROUTE_REG_MASK|DOIP_RA_STATUS_ROUTE_ACTIVE_MASK));

    if (active)
    {
        routingActivation.routingActivationStatus |= (tU8) DOIP_RA_STATUS_ROUTE_ACTIVE;
    }
    else
    {
        routingActivation.routingActivationStatus |= (tU8) DOIP_RA_STATUS_ROUTE_NOT_ACTIVE;
    }

    if (registered)
    {
        routingActivation.routingActivationStatus |= (tU8) DOIP_RA_STATUS_ROUTE_REGISTERED;
    }
    else
    {
        routingActivation.routingActivationStatus |= (tU8) DOIP_RA_STATUS_ROUTE_NOT_REG;
    }
}

void DoIP_Connection::vAliveCheckMainFunction()
{
    /* Sending of alive check was requested by a routingactivation of another TCP-Connection*/
    if (aliveCheck.aliveCheckState == DOIP_ALIVECHECK_REQUESTED)
    {
        vAliveCheckTxPrepareRequest();
        aliveCheck.aliveCheckState = DOIP_ALIVECHECK_READY2SEND;
    }

    /* AliveBuffer is prepared, try to send message */
    if (aliveCheck.aliveCheckState == DOIP_ALIVECHECK_READY2SEND)
    {
        if (bSendMessage(aliveCheck.aliveCheckBuffer, FALSE))
        {
            aliveCheck.aliveCheckState = DOIP_ALIVECHECK_WAIT4CONFIRMATION;
        }
    }

    /* Tester-Response was received, check values in AliveBuffer */
    if (aliveCheck.aliveCheckState == DOIP_ALIVECHECK_RECEIVED)
    {
        vAliveCheckRxHandle();
    }
}

void DoIP_Connection::vAliveCheckTxPrepareRequest()
{
    tU16 payLoadType = DOIP_PAYLOAD_TYPE_ALIVE_CHECK_REQ;
    tU32 payLoadLength = DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_REQ;

    //DoIP Header
    aliveCheck.aliveCheckBuffer[0] = DOIP_PROTOCOL_VERSION;
    aliveCheck.aliveCheckBuffer[1] = DOIP_PROTOCOL_VERSION_INV;
    setU16( &aliveCheck.aliveCheckBuffer[2], payLoadType );
    //pConn->buffer[2], pConn->buffer[3]
    setU32( &aliveCheck.aliveCheckBuffer[4], payLoadLength );
    //pConn->buffer[4], pConn->buffer[5], pConn->buffer[6], pConn->buffer[7]
}

void DoIP_Connection::vAliveCheckTxConfirmation()
{
    /* Set the timer to the configured timeout */
    aliveCheck.aliveCheckTimer = DOIP_CFG_AliveCheckResponseTimeout;
    /* Set state to waiting for tester response */
    aliveCheck.aliveCheckState = DOIP_ALIVECHECK_WAIT4RESPONSE;
}

void DoIP_Connection::vAliveCheckRxHandle()
{
   tU16 sourceAddr;
   memcpy( &sourceAddr, &aliveCheck.aliveCheckBuffer[DOIP_MSG_POS_CONTENT+0u], 2u );

   if (bIsSourceAddressRegistered(sourceAddr))
   {
      /* Set inactivity timer once alive check response is received */
      aliveCheck.aliveCheckTimer = 0;
      vSetGeneralInactivity();
      aliveCheck.aliveCheckState = DOIP_ALIVECHECK_IDLE;
   }
   else
   {
      vConnectionClose(FALSE);
   }
}

bool DoIP_Connection::bAliveCheckIsActive()
{
    return ((aliveCheck.aliveCheckState != DOIP_ALIVECHECK_IDLE));
}

void DoIP_Connection::vAliveCheckRequest()
{
    if (!bAliveCheckIsActive())
    {
        aliveCheck.aliveCheckState = DOIP_ALIVECHECK_REQUESTED;
    }
}

void DoIP_Connection::vChannelCloseAll()
{
   rActiveRouteCfg = poDoIPProtocol->rGetTesterCfg(cu16TesterAddrInit)->second.oRouteActCfg.end();
   poCurrentChannel = NULL;

   while(!oActiveChannels.empty())
   {
      DoIP_Channel* poTemp = oActiveChannels.front();
      oActiveChannels.pop_front(); //remove from list first before deleting
      delete poTemp;
   }
}

DoIP_Channel* DoIP_Connection::poFindChannel(tU16 targetAddr)
{
   DoIP_Channel* poChannel = NULL;

   std::list<DoIP_Channel*>::iterator channel;
   for(channel = oActiveChannels.begin(); channel != oActiveChannels.end(); channel++)
   {
      if((*channel)->cu16TargetAddress == targetAddr)
      {
         poChannel = *channel;
         break;
      }
   }
   return poChannel;
}


ssize_t DoIP_Connection::readNBytes(uint8_t *buffer, uint32_t numBytesRequested) {
  uint32_t bytesLeft = numBytesRequested;

  while (true) {
     ssize_t bytesRead = poSocket->read(buffer + numBytesRequested - bytesLeft, bytesLeft);
     if (bytesRead == -1) {
        if (errno == EINTR) {
           DOIP_TRACE_INFO("DoIP_Connection::readNBytes(numBytesRequested=%u bytesLeft=%u): got EINTR",
                           numBytesRequested, bytesLeft);
           continue;
        }
        else {
           DOIP_TRACE_INFO("DoIP_Connection::readNBytes(numBytesRequested=%u bytesLeft=%u): FAILED:%s",
                           numBytesRequested, bytesLeft, strerror(errno));
           return -1;
        }
     }
     else if (numBytesRequested && (bytesRead == 0) ) {
        DOIP_TRACE_INFO("DoIP_Connection::readNBytes(numBytesRequested=%u bytesLeft=%u): return 0, peer closed connection",
                        numBytesRequested, bytesLeft);
        return -1;
     }

     bytesLeft-=(uint32_t)bytesRead;
     if (!bytesLeft) {
        break;
     }
  }
  return numBytesRequested;
}


bool DoIP_Connection::bReadNextFrameTCP(void)
{
   size_t bufOffset = 0;
   RX.cnt++;
   if (RX.state == DOIP_TCP_IDLE) {
      RX.nextReadLen=DOIP_HEADER_SIZE;
   }
   DOIP_TRACE_NOTICE("DoIP_Connection::bReadNextFrameTCP START RX.nextReadLen=%u cnt=%u", RX.nextReadLen, RX.cnt);
   ssize_t iRecvBytes=readNBytes(RX.recvBuffer + bufOffset, RX.nextReadLen);
   if ((uint32_t)iRecvBytes != (uint32_t)RX.nextReadLen) {
      // general error-handling will close socket, here we will detect it and exit.
      // got error or connection closed by client
      DOIP_TRACE_WARNING("DoIP_Connection::bReadNextFrameTCP: closing connection socket=%u nextReadLen=%u iRecvBytes=%i",
                         poSocket->getFd(), RX.nextReadLen, (int32_t)iRecvBytes);

      //poDoIPProtocol->poEthBus->bEnterCriticalSection();
      delete this;
      //poDoIPProtocol->poEthBus->bLeaveCriticalSection();

      return false;
   }

   DOIP_TRACE_NOTICE("DoIP_Connection::bReadNextFrameTCP cnt=%u: [%u] %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X",
                     RX.cnt,
                     (tU32)iRecvBytes,
                     RX.recvBuffer[0], RX.recvBuffer[1], RX.recvBuffer[2], RX.recvBuffer[3], RX.recvBuffer[4],
                     RX.recvBuffer[5], RX.recvBuffer[6], RX.recvBuffer[7], RX.recvBuffer[8], RX.recvBuffer[9],
                     RX.recvBuffer[10],RX.recvBuffer[11]);

   RX.recvBufferUsed=(tU32)iRecvBytes;

   if (RX.state == DOIP_TCP_IDLE) {
      vHandleRxDoipHeader();
   }
   else if (RX.state == DOIP_TCP_RX_RECEIVING) {
      vHandleRxPayloadIndication();
   }

   if (RX.nextReadLen==0) {
      vHandleRxMsgComplete();
   }

   return true;
}


void* DoIP_Connection::pvTCPReadThread(void* poPar)
{
   DoIP_Connection* poParent = (DoIP_Connection*)poPar;

   while(poParent->bReadNextFrameTCP()) ;

   DOIP_TRACE_WARNING("DoIP_Connection::pvTCPReadThread: thread has ended");
   return NULL;
}

