/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     DoIP_Tester.cpp
 *\brief
 *
 *\author   CM-CI1/ENP1 - Seidler
 *
 *\par Copyright:
 *(c) 2018 Robert Bosch Car Multimedia GmbH
 *
 *\par History:
 * See history of revision control system
 ***************************************************************************/

/****************************************************************************
| includes
|--------------------------------------------------------------------------*/
#include "DoIP_Tester.h"

#include <string.h>
#include <cstdarg>
#include <cerrno>
#include <ifaddrs.h>
#include <arpa/inet.h>
#include <linux/if_packet.h>
//#include "doipcommon/dia_DoIPDefines.h"
#include <sys/types.h>
#include <net/if.h>

DoIP_Tester::DoIP_Tester(const char* pszDevice, tU16 u16Port, tU16 testerAddress, DoIP_Cfg &doipCfg)
: 
  busNumber(0xFF),
  testerAddr(testerAddress),
  pszDeviceName(NULL),
  cu16Port(u16Port),
  bValidMacAddress(false),
  bIPv4allocated(false),
  iSocket_UDP_Client(-1),
  iSocket_UDP_Client_bcast(-1),
  iSocket_UDP_Client2(-1),
  iSocket_UDP_Client2_bcast(-1),
  udpState(DOIP_UDP_IDLE),
  rUDPClientThread(-1),
  rUDPBroadcastClientThread(-1),
  rUDPBroadcastClient2Thread(-1),
  rUDPClient2Thread(-1),
  queueIdx(0),
  mDoipCfg(doipCfg),
  mUtils(doipCfg),
  node(NULL)
{
   DOIP_TRACE_NOTICE("DoIP_Tester::DoIP_Tester -- %s", pszDevice);
   pszDeviceName = new char[strlen(pszDevice)+1];
   strcpy(pszDeviceName, pszDevice);

   memset(&au8MacAddress,0,sizeof(au8MacAddress));
   memset(&arUDPQueue,0, sizeof(arUDPQueue));
   memset(&IPv4, 0, sizeof(sockaddr_in));

   pthread_mutex_init(&mutDoIP_Nodes, NULL);
}

DoIP_Tester::~DoIP_Tester()
{
    if(pszDeviceName != NULL)
    {
       delete[] pszDeviceName;
    }

    delete node;
    node = NULL;
}

void DoIP_Tester::vTraceStackDump(void)
{
    char addressBuffer[INET_ADDRSTRLEN] = "???";
    inet_ntop(AF_INET, &IPv4.sin_addr, addressBuffer, INET_ADDRSTRLEN);
    pthread_mutex_lock(&mutDoIP_Nodes);
    DOIP_TRACE_NOTICE("DoIP_Tester::vTraceStackDump: Tester (internal test equipment) Device: %s MAC=%02X:%02X:%02X:%02X:%02X:%02X IP=%s",
                         ((pszDeviceName != NULL) ? pszDeviceName : ""),
                         au8MacAddress[0], au8MacAddress[1], au8MacAddress[2], au8MacAddress[3],
                         au8MacAddress[4], au8MacAddress[5], addressBuffer);

    DOIP_TRACE_NOTICE("DoIP entities: %i", oDoIP_Nodes.size());
    std::list<DoIP_Node*>::iterator itNode;

    for (itNode=oDoIP_Nodes.begin(); itNode != oDoIP_Nodes.end(); itNode++)
    {
        (*itNode)-> vTraceStackDump();
    }
    pthread_mutex_unlock(&mutDoIP_Nodes);
}



tDoIPResult 
DoIP_Tester::Start(void)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::Start");
    pUpdateLocalIPAddress();

    //vStartUDPClient();
    //vStartUDPBroadcastClient();
    //vStartUDPClient2();
    //vStartUDPBroadcastClient2();
    return (((DOIP_E_NOERROR == vStartUDPClient()) && (DOIP_E_NOERROR == vStartUDPBroadcastClient())) ? DOIP_SUCCESS : DOIP_FAILED) ;
}

tDoIPResult
DoIP_Tester::Stop(void)
{
  DOIP_TRACE_NOTICE("DoIP_Tester::Stop");
  tDoIPResult retVal = DOIP_FAILED;

  if(true == vConnectionCloseAll())
  {
    DOIP_TRACE_NOTICE("DoIP_Tester:: UDP Clients closed");
    retVal = DOIP_SUCCESS;
  }

 return retVal;
}

void DoIP_Tester::vCommonTimerTick(void)
{
    /* Timer */
#if 0
    if (bConnectionIsActive())
    {
       vHandleTimer();
    }

    /* Routing Activation */
    /* Is connection still active ? */
    if (bConnectionIsActive())
    {
       vRoutingActivationMainFunction();
    }

    /* AliveCheck */
    /* Is connection still active ? */
    if (bConnectionIsActive())
    {
       vAliveCheckMainFunction();
    }
#endif
    /* Messages */
    /* Is connection still active ? */
    if (true)//bConnectionIsActive())
    {
       vHandleMessages();
    }
}


void DoIP_Tester::pUpdateLocalIPAddress()
{
    DOIP_TRACE_NOTICE("DoIP_Tester::pUpdateLocalIPAddress");
    struct ifaddrs * ifAddrStruct=NULL;
    struct ifaddrs * ifa=NULL;
    bool   bIPfound = false;

    if (getifaddrs(&ifAddrStruct)>=0)
    {
        for (ifa = ifAddrStruct; ifa != NULL; ifa = ifa->ifa_next)
        {
          if(ifa->ifa_addr != NULL)
          {
             if (ifa->ifa_addr->sa_family == AF_INET)
             {
                 if(pszDeviceName != NULL && strcmp(ifa->ifa_name, pszDeviceName) == 0)
                 {
                     this->IPv4 = *((struct sockaddr_in *)ifa->ifa_addr);
                     bIPv4allocated = true;
                     bIPfound = true;
                 }
             }
             else if (ifa->ifa_addr->sa_family == AF_PACKET)
             {
                if(pszDeviceName != NULL && strcmp(ifa->ifa_name, pszDeviceName) == 0)
                {
                   struct sockaddr_ll *s = (struct sockaddr_ll*)ifa->ifa_addr;
                   memcpy(au8MacAddress, s->sll_addr, (s->sll_halen > 6) ? 6 : s->sll_halen);
                   bValidMacAddress = true;
                }
             }
          }
        }
    }
    else
    {
       DOIP_TRACE_NOTICE("DoIP_Tester::pUpdateLocalIPAddress getifaddrs error: %s", strerror(errno));
    }
    if (bIPfound)
    {
        char   addressBuffer[INET6_ADDRSTRLEN] = "???";
        inet_ntop(AF_INET, &IPv4.sin_addr, addressBuffer, INET_ADDRSTRLEN);
        DOIP_TRACE_NOTICE("DoIP_Tester::pUpdateLocalIPAddress for device: %s ip: %s", pszDeviceName, addressBuffer);
    }
    else
    {
        DOIP_TRACE_NOTICE("DoIP_Tester::pUpdateLocalIPAddress failed for device: %s", ((pszDeviceName != NULL) ? pszDeviceName : ""));
    }
}

tDoIPResult 
DoIP_Tester::vStartUDPClient()
{
   /* open a UDP broadcast client connection */
    DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient");

    tDoIPResult retVal = DOIP_SUCCESS;

   /* Open IP4 DGRAM UDP socket for sending data */
   if((iSocket_UDP_Client = socket(AF_INET,SOCK_DGRAM,0)) < 0) // protocol 0 or IPPROTO_UDP
   {
       DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: unable to open an UDP socket");
   }
   else
   {
      //Socket option : allow multiple sockets to use the same ports
      int optval = 1;
      if ( setsockopt(iSocket_UDP_Client, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0 )
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: It is not able to change SO_REUSEADDR");
      }

      //disable usage of Broadcast option
      int broadcastEnable = 0;
      if ( setsockopt(iSocket_UDP_Client, SOL_SOCKET, SO_BROADCAST, &broadcastEnable, sizeof(broadcastEnable)) < 0 )
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: It is not able to change SO_BROADCAST");
      }

#if 0
      if (strlen(pszDeviceName)>1)
      {
          if ( setsockopt(iSocket_UDP_Client, SOL_SOCKET, SO_BINDTODEVICE, pszDeviceName, sizeof(strlen(pszDeviceName)))   < 0){
              DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: It is not able to SO_BINDTODEVICE on %s", pszDeviceName);
          }
          else
          {
              DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: SO_BINDTODEVICE on %s", pszDeviceName);
          }
      }
      else
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: no interface specified SO_BINDTODEVICE will not take place");
      }
#endif

      // bind to fix port usage for now (even for client), listen on UDP_TEST_EQUIPMENT_REQUEST
      struct sockaddr_in servaddr = IPv4;
      //memset(&servaddr, 0, sizeof(servaddr));
      servaddr.sin_port = htons(cu16Port);

      if(bind(iSocket_UDP_Client, (struct sockaddr*)&servaddr, sizeof(servaddr)) < 0)
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: It is not possible to bind UDP socket to PORT: DOIP_UDP_PORT_DISCOVERY");
      }
      else
      {
         DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: UDP bound to Port %i, socket id %i", cu16Port, iSocket_UDP_Client);

         pthread_attr_t rUDPReadAttr;
         pthread_attr_init(&rUDPReadAttr);
         (void)pthread_attr_setstacksize(&rUDPReadAttr, DOIP_CFG_UDPSOCKET_LISTENER_STACKSIZE);

         if(pthread_create( &rUDPClientThread, &rUDPReadAttr, DoIP_Tester::pvUDPClientThread, this) == 0)
         {
            pthread_setname_np(rUDPClientThread, "DoIP@UDP_Client");
         }
         else
         {
             DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient: DoIP UDP Thread is not started");
             retVal = DOIP_FAILED;
         }
      }
   }
   return retVal;
}

tDoIPResult 
DoIP_Tester::vStartUDPBroadcastClient()
{
   /* open a UDP broadcast client connection */
    DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient");

    tDoIPResult retVal = DOIP_SUCCESS;

   /* Open IP4 DGRAM UDP socket for sending data */
   if((iSocket_UDP_Client_bcast = socket(AF_INET,SOCK_DGRAM,0)) < 0) // protocol 0 or IPPROTO_UDP
   {
       DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: unable to open an UDP socket");
   }
   else
   {
      //Socket option : allow multiple sockets to use the same ports
      int optval = 1;
      if ( setsockopt(iSocket_UDP_Client_bcast, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0 )
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: It is not able to change SO_REUSEADDR");
      }

      //enable usage of Broadcast option
      int broadcastEnable = 1;
      if ( setsockopt(iSocket_UDP_Client_bcast, SOL_SOCKET, SO_BROADCAST, &broadcastEnable, sizeof(broadcastEnable)) < 0 )
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: It is not able to change SO_BROADCAST");
      }

      if (strlen(pszDeviceName)>1)
      {
		  DOIP_TRACE_NOTICE("Interface Index :  %d", if_nametoindex(pszDeviceName));
			  
		  struct ifreq ifr;
		  memset(&ifr, 0, sizeof(ifr));
		  snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), pszDeviceName); 
				
          if ( setsockopt(iSocket_UDP_Client_bcast, SOL_SOCKET, SO_BINDTODEVICE, (void *)&ifr, sizeof(ifr)) < 0)
		  {
              DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: It is not able to SO_BINDTODEVICE on %s", pszDeviceName);
			  DOIP_TRACE_NOTICE("Because %s(%d)", strerror(errno), errno);
          }
          else
          {
              DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: SO_BINDTODEVICE on %s", pszDeviceName);
          }
      }
      else
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: no interface specified SO_BINDTODEVICE will not take place");
      }

      // bind to fix port usage for now (even for client), listen on UDP_TEST_EQUIPMENT_REQUEST
      struct sockaddr_in servaddr;
      memset(&servaddr, 0, sizeof(servaddr));
      servaddr.sin_family = AF_INET;
      servaddr.sin_port = htons(cu16Port);
      servaddr.sin_addr.s_addr = htonl (INADDR_BROADCAST);

      if(bind(iSocket_UDP_Client_bcast, (struct sockaddr*)&servaddr, sizeof(servaddr)) < 0)
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: It is not possible to bind UDP socket to PORT: DOIP_UDP_PORT_DISCOVERY");
      }
      else
      {
         DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: UDP bound to Port %i, socket id %i", cu16Port, iSocket_UDP_Client_bcast);

         pthread_attr_t rUDPReadAttr;
         pthread_attr_init(&rUDPReadAttr);
         (void)pthread_attr_setstacksize(&rUDPReadAttr, DOIP_CFG_UDPSOCKET_LISTENER_STACKSIZE);

         if(pthread_create( &rUDPBroadcastClientThread, &rUDPReadAttr, DoIP_Tester::pvUDPBroadcastClientThread, this) == 0)
         {
            pthread_setname_np(rUDPBroadcastClientThread, "DoIP@UDP_bcast_Client");
         }
         else
         {
             DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient: DoIP UDP Thread is not started");
             retVal = DOIP_FAILED;
         }
      }
   }

   return retVal;
}




void DoIP_Tester::vStartUDPClient2()
{
   /* open a UDP broadcast client connection */
    DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2");

   /* Open IP4 DGRAM UDP socket for sending data */
   if((iSocket_UDP_Client2 = socket(AF_INET,SOCK_DGRAM,0)) < 0) // protocol 0 or IPPROTO_UDP
   {
       DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: unable to open an UDP socket");
   }
   else
   {
      //Socket option : allow multiple sockets to use the same ports
      int optval = 1;
      if ( setsockopt(iSocket_UDP_Client2, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0 )
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: It is not able to change SO_REUSEADDR");
      }

      //enable usage of Broadcast option
      int broadcastEnable = 1;
      if ( setsockopt(iSocket_UDP_Client2, SOL_SOCKET, SO_BROADCAST, &broadcastEnable, sizeof(broadcastEnable)) < 0 )
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: It is not able to change SO_BROADCAST");
      }

      if (strlen(pszDeviceName)>1)
      {
          if ( setsockopt(iSocket_UDP_Client2, SOL_SOCKET, SO_BINDTODEVICE, pszDeviceName, sizeof(strlen(pszDeviceName)))  < 0){
              DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: It is not able to SO_BINDTODEVICE on %s", pszDeviceName);
          }
          else
          {
              DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: SO_BINDTODEVICE on %s", pszDeviceName);
          }
      }
      else
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: no interface specified SO_BINDTODEVICE will not take place");
      }

      // bind to fix port usage for now (even for client), listen on DOIP_UDP_PORT_DISCOVERY
      struct sockaddr_in servaddr = IPv4;
      memset(&servaddr, 0, sizeof(servaddr));
      servaddr.sin_port = htons(DOIP_UDP_PORT_DISCOVERY);

      if(bind(iSocket_UDP_Client2, (struct sockaddr*)&servaddr, sizeof(servaddr)) < 0)
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: It is not possible to bind UDP socket to PORT: DOIP_UDP_PORT_DISCOVERY");
      }
      else
      {
         DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: UDP bound to Port %i, socket id %i", DOIP_UDP_PORT_DISCOVERY, iSocket_UDP_Client2);

         pthread_attr_t rUDPReadAttr;
         pthread_attr_init(&rUDPReadAttr);
         (void)pthread_attr_setstacksize(&rUDPReadAttr, DOIP_CFG_UDPSOCKET_LISTENER_STACKSIZE);

         if(pthread_create( &rUDPClient2Thread, &rUDPReadAttr, DoIP_Tester::pvUDPClient2Thread, this) == 0)
         {
            pthread_setname_np(rUDPClient2Thread, "DoIP@UDP_Client2");
         }
         else
         {
             DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPClient2: DoIP UDP Thread is not started");
         }
      }
   }
}

void DoIP_Tester::vStartUDPBroadcastClient2()
{
   /* open a UDP broadcast client connection */
    DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2");

   /* Open IP4 DGRAM UDP socket for sending data */
   if((iSocket_UDP_Client2_bcast = socket(AF_INET,SOCK_DGRAM,0)) < 0) // protocol 0 or IPPROTO_UDP
   {
       DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: unable to open an UDP socket");
   }
   else
   {
      //Socket option : allow multiple sockets to use the same ports
      int optval = 1;
      if ( setsockopt(iSocket_UDP_Client2_bcast, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0 )
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: It is not able to change SO_REUSEADDR");
      }

      //enable usage of Broadcast option
      int broadcastEnable = 1;
      if ( setsockopt(iSocket_UDP_Client2_bcast, SOL_SOCKET, SO_BROADCAST, &broadcastEnable, sizeof(broadcastEnable)) < 0 )
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: It is not able to change SO_BROADCAST");
      }

      if (strlen(pszDeviceName)>1)
      {
          if ( setsockopt(iSocket_UDP_Client2_bcast, SOL_SOCKET, SO_BINDTODEVICE, pszDeviceName, sizeof(strlen(pszDeviceName)))   < 0){
              DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: It is not able to SO_BINDTODEVICE on %s", pszDeviceName);
          }
          else
          {
              DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: SO_BINDTODEVICE on %s", pszDeviceName);
          }
      }
      else
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: no interface specified SO_BINDTODEVICE will not take place");
      }

      // bind to fix port usage for now (even for client), listen on DOIP_UDP_PORT_DISCOVERY
      struct sockaddr_in servaddr;
      memset(&servaddr, 0, sizeof(servaddr));
      servaddr.sin_family = AF_INET;
      servaddr.sin_port = htons(DOIP_UDP_PORT_DISCOVERY);
      servaddr.sin_addr.s_addr = htonl (INADDR_BROADCAST);

      if(bind(iSocket_UDP_Client2_bcast, (struct sockaddr*)&servaddr, sizeof(servaddr)) < 0)
      {
          DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: It is not possible to bind UDP socket to PORT: DOIP_UDP_PORT_DISCOVERY");
      }
      else
      {
         DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: UDP bound to Port %i, socket id %i", DOIP_UDP_PORT_DISCOVERY, iSocket_UDP_Client2_bcast);

         pthread_attr_t rUDPReadAttr;
         pthread_attr_init(&rUDPReadAttr);
         (void)pthread_attr_setstacksize(&rUDPReadAttr, DOIP_CFG_UDPSOCKET_LISTENER_STACKSIZE);

         if(pthread_create( &rUDPBroadcastClient2Thread, &rUDPReadAttr, DoIP_Tester::pvUDPBroadcastClient2Thread, this) == 0)
         {
            pthread_setname_np(rUDPBroadcastClient2Thread, "DoIP@UDP_bcast_Client2");
         }
         else
         {
             DOIP_TRACE_NOTICE("DoIP_Tester::vStartUDPBroadcastClient2: DoIP UDP Thread is not started");
         }
      }
   }
}

void* DoIP_Tester::pvUDPClientThread(void* poPar)
{
   DoIP_Tester* poTester = (DoIP_Tester*)poPar;

   while(poTester->bReadNextFrameUDP(poTester->iSocket_UDP_Client));

   return NULL;
}

void* DoIP_Tester::pvUDPBroadcastClientThread(void* poPar)
{
   DoIP_Tester* poTester = (DoIP_Tester*)poPar;

   while(poTester->bReadNextFrameUDP(poTester->iSocket_UDP_Client_bcast));

   return NULL;
}

void* DoIP_Tester::pvUDPClient2Thread(void* poPar)
{
   DoIP_Tester* poTester = (DoIP_Tester*)poPar;

   while(poTester->bReadNextFrameUDP(poTester->iSocket_UDP_Client2));

   return NULL;
}

void* DoIP_Tester::pvUDPBroadcastClient2Thread(void* poPar)
{
   DoIP_Tester* poTester = (DoIP_Tester*)poPar;

   while(poTester->bReadNextFrameUDP(poTester->iSocket_UDP_Client2_bcast));

   return NULL;
}

bool DoIP_Tester::bReadNextFrameUDP(int socket)
{

  	DOIP_TRACE_NOTICE("DoIP_Tester::bReadNextFrameUDP: socket=%u", socket);
    bool res = false;
    struct sockaddr_in remoteaddr; //client address
    socklen_t slen = sizeof(remoteaddr);
    tU8 au8UdpBuffer[DOIP_CFG_cu32UDPBufferSize];
    size_t iRecvBytes = recvfrom(socket, au8UdpBuffer, DOIP_CFG_cu32UDPBufferSize , 0, (struct sockaddr *)&remoteaddr, &slen);

    {
        char addressBuffer[INET_ADDRSTRLEN];
        inet_ntop(AF_INET, &remoteaddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
        DOIP_TRACE_NOTICE("DoIP_Tester::bReadNextFrameUDP IP[%s] socket[%i]: length[%lu] %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X",
            addressBuffer, socket, iRecvBytes,
            au8UdpBuffer[0], au8UdpBuffer[1], au8UdpBuffer[2], au8UdpBuffer[3], au8UdpBuffer[4],
            au8UdpBuffer[5], au8UdpBuffer[6], au8UdpBuffer[7], au8UdpBuffer[8], au8UdpBuffer[9]); 
    }

    if(iRecvBytes > 0)
    {
        //  if (remoteaddr.sin_addr.s_addr != IPv4.sin_addr.s_addr) // avoid loopbacks
        //  {
        vRxIndication(au8UdpBuffer, (tU32)iRecvBytes, remoteaddr);
        //}
        res = true;
    }

    return res;
}

/*Function for UDP Connections */
void DoIP_Tester::vRxIndication(tU8 au8UdpBuffer[], tU32 u32RecvBytes, sockaddr_in& remoteaddr)
{
    tU16 payLoadType;
    tU8 respCode = 0;

    DOIP_TRACE_NOTICE("DoIP_Tester::vRxIndication: len=%u", u32RecvBytes);

    if (au8UdpBuffer != NULL && u32RecvBytes >= DOIP_HEADER_SIZE)
    {
        tU8 buffer[DOIP_CFG_cu32UDPBufferSize];


       mUtils.vMemCopyWithEndianessConversion(
               &buffer[DOIP_MSG_POS_PROTOCOL_VERSION],
               &au8UdpBuffer[DOIP_MSG_POS_PROTOCOL_VERSION],
               DOIP_MSG_SIZE_PROTOCOL_VERSION);
       mUtils.vMemCopyWithEndianessConversion(
               &buffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION],
               &au8UdpBuffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION],
               DOIP_MSG_SIZE_INV_PROTOCOL_VERSION);
       mUtils.vMemCopyWithEndianessConversion(
               &buffer[DOIP_MSG_POS_PAYLOAD_TYPE],
               &au8UdpBuffer[DOIP_MSG_POS_PAYLOAD_TYPE],
               DOIP_MSG_SIZE_PAYLOAD_TYPE);
       mUtils.vMemCopyWithEndianessConversion(
               &buffer[DOIP_MSG_POS_PAYLOAD_LENGTH],
               &au8UdpBuffer[DOIP_MSG_POS_PAYLOAD_LENGTH],
               DOIP_MSG_SIZE_PAYLOAD_LENGTH);


        // check for DoIP header
        if (!mUtils.bCheckHeader(buffer, &respCode, FALSE))
        {
            DOIP_TRACE_NOTICE("DoIP_Tester::vRxIndication DoIP Header Error: %i", respCode);
            /* Send negativ Acknowledge, because header Data is wrong */
            //tbd lDoIP_Tester::vPrepareResponse(&buffer[0], respCode);
        }
        // check for payload type
        mUtils.vMemCopyWithEndianessConversion(&payLoadType,
                                         &au8UdpBuffer[0 + DOIP_MSG_POS_PAYLOAD_TYPE],
                                         DOIP_MSG_SIZE_PAYLOAD_TYPE);
        switch (payLoadType)
        {
            case DOIP_PAYLOAD_TYPE_ENTITY_STATUS_RESP:
                {
                    DOIP_TRACE_NOTICE("DoIP_Tester::vRxIndication DOIP_PAYLOAD_TYPE_ENTITY_STATUS_RESP");

                    std::list<DoIP_Node*>::iterator itNodeStat;
                    pthread_mutex_lock(&mutDoIP_Nodes);
                    for (itNodeStat=oDoIP_Nodes.begin(); itNodeStat != oDoIP_Nodes.end(); itNodeStat++)
                    {
                        //(*itNodeStat)->entityStatusInd();

                        (*itNodeStat)-> entityStatusResp(au8UdpBuffer, remoteaddr);

                    }
                    pthread_mutex_unlock(&mutDoIP_Nodes);
                }
                break;
            case DOIP_PAYLOAD_TYPE_VEHICLE_ID_RESP:
                {
                    DOIP_TRACE_NOTICE("DoIP_Tester::vRxIndication DOIP_PAYLOAD_TYPE_VEHICLE_ID_RESP");
                    // checked in header later  if (u32RecvBytes ==  DOIP_HEADER_SIZE + DOIP_PAYLOAD_LENGTH_VEHICLE_ID_RESP)
                    tU8  acVIN[17];
                    tU16 logAddr;
                    tU8  acEID[6];
                    tU8  acGID[6];
                    tU8  actrequ;
                    tU8  VINGIDstatus;

                    memcpy(&acVIN,          &au8UdpBuffer[DOIP_MSG_POS_CONTENT], 17);
                    mUtils.vMemCopyWithEndianessConversion(&logAddr, &au8UdpBuffer[DOIP_MSG_POS_CONTENT + 17], 2);
                    memcpy(&acEID,          &au8UdpBuffer[DOIP_MSG_POS_CONTENT + 19], 6);
                    memcpy(&acGID,          &au8UdpBuffer[DOIP_MSG_POS_CONTENT + 25], 6);
                    memcpy(&actrequ,        &au8UdpBuffer[DOIP_MSG_POS_CONTENT + 31], 1);
                    memcpy(&VINGIDstatus,   &au8UdpBuffer[DOIP_MSG_POS_CONTENT + 32], 1);

                    node = pdiscoverEntityInd(remoteaddr, logAddr, acEID, acGID, actrequ, VINGIDstatus, acVIN);
                    if (node != NULL)
                    {
                        //node->entityStatusReq(iSocket_UDP_Client);

                        //node->powermodeRequest(iSocket_UDP_Client);
                    }
                }
                break;
            case DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_RESP:
                {
                    DOIP_TRACE_NOTICE("DoIP_Tester::vRxIndication DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_RESP");

                    tU8 pwrMode = au8UdpBuffer[DOIP_MSG_POS_CONTENT];

                    std::list<DoIP_Node*>::iterator itNodeStat;
                    pthread_mutex_lock(&mutDoIP_Nodes);
                    for (itNodeStat=oDoIP_Nodes.begin(); itNodeStat != oDoIP_Nodes.end(); itNodeStat++)
                    {
						pthread_mutex_unlock(&mutDoIP_Nodes);
                        (*itNodeStat)-> powermodeResp(pwrMode, remoteaddr);
						pthread_mutex_lock(&mutDoIP_Nodes);
                    }
                    pthread_mutex_unlock(&mutDoIP_Nodes);
                }
                break;
            default:
                DOIP_TRACE_NOTICE("DoIP_Tester::vRxIndication unknown or Tester Cmd Payload Type: %i", payLoadType);
                break;
        }
    }
}


/* send basic VehicleIdentificationRequest message to DoIP entity's */
void DoIP_Tester::vdiscoverEntities(const char* in_adr)
{
	DOIP_TRACE_NOTICE("DoIP_Tester::vdiscoverEntities");
    int res;
    struct sockaddr_in servaddr;

    res = inet_pton(AF_INET, in_adr, &servaddr.sin_addr);
    if(res <= 0)
    {
        if (res == 0)
        {
            DOIP_TRACE_NOTICE("DoIP_Tester::vdiscoverEntities: VehicleIdentificationRequest input IP not in presentation format");
        }
        else
        {
            DOIP_TRACE_NOTICE("DoIP_Tester::vdiscoverEntities: VehicleIdentificationRequest input IP error %i", errno);
        }
    }
    else
    {
        DOIP_TRACE_NOTICE("DoIP_Tester::vdiscoverEntities: VehicleIdentificationRequest (direct msg) to %s", in_adr);

        tU16 payLoadType;
        tU32 payLoadLength;
        tU8 au8UdpBuffer[DOIP_HEADER_SIZE+DOIP_PAYLOAD_LENGTH_VEHICLE_ID_REQ];

        au8UdpBuffer[0] = DOIP_PROTOCOL_VERSION;
        au8UdpBuffer[1] = DOIP_PROTOCOL_VERSION_INV;
        payLoadType = DOIP_PAYLOAD_TYPE_VEHICLE_ID_REQ;
        mUtils.vMemCopyWithEndianessConversion(&au8UdpBuffer[DOIP_MSG_POS_PAYLOAD_TYPE], &payLoadType, DOIP_MSG_SIZE_PAYLOAD_TYPE);
        payLoadLength = DOIP_PAYLOAD_LENGTH_VEHICLE_ID_REQ;
        mUtils.vMemCopyWithEndianessConversion(&au8UdpBuffer[DOIP_MSG_POS_PAYLOAD_LENGTH], &payLoadLength, DOIP_MSG_SIZE_PAYLOAD_LENGTH);

        servaddr.sin_family = AF_INET;
        servaddr.sin_port = htons(DOIP_UDP_PORT_DISCOVERY); // send destination port, source port bind on socket fd

        ssize_t ret = 0;
        ret = sendto(iSocket_UDP_Client_bcast, au8UdpBuffer, (payLoadLength + DOIP_HEADER_SIZE),
                   0, (sockaddr*)&servaddr, sizeof(sockaddr_in));
        if (ret==-1) {
           DOIP_TRACE_NOTICE("DoIP_Tester::vdiscoverEntities: sendto() failed");
        }
    }
}

DoIP_Node* DoIP_Tester::pdiscoverEntityInd(sockaddr_in& IP, tU16 logAddr, const tU8* EID, const tU8* GID, tU8 actrequ, tU8 VINGIDstatus, const tU8* VIN)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::pdiscoverEntityInd: : logAddr:%u EID:%02X%02X%02X%02X%02X%02X GID:%02X%02X%02X%02X%02X%02X further action required:%02X VIN/GID sync status:%02X VIN:%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X"
            ,logAddr
            ,EID[0],EID[1],EID[2],EID[3],EID[4],EID[5]
            ,GID[0],GID[1],GID[2],GID[3],GID[4],GID[5]
            ,actrequ
            ,VINGIDstatus
            ,VIN[0],VIN[1],VIN[2],VIN[3],VIN[4],VIN[5],VIN[6],VIN[7],VIN[8],VIN[9],VIN[10],VIN[11],VIN[12],VIN[13],VIN[14],VIN[15],VIN[16]
            );
    DoIP_Node* resNode = NULL;

    /* DoIP Node already available? */
    std::list<DoIP_Node*>::iterator itNode;
    pthread_mutex_lock(&mutDoIP_Nodes);
    for (itNode=oDoIP_Nodes.begin(); itNode != oDoIP_Nodes.end(); itNode++)
    {
        /* check on MAC/EID for now
         * nodes could respond with multiple logAddr
         * IP could also change due to external DHCP handling
         * VIN could be reconfigured
         *
         * logAddr handling (multiple logAddr per IP) will be done later
         * */

        if( memcmp ((*itNode)-> getEID(),EID, 6) == 0 )
        {
            DOIP_TRACE_NOTICE("DoIP_Tester::pdiscoverEntityInd: Node already present (EID check)");
            resNode = (*itNode);
            break;
        }
    }
    pthread_mutex_unlock(&mutDoIP_Nodes);
    if (resNode == NULL)
    {
        tU32 maxDataSize = 0;
        DOIP_TRACE_NOTICE("DoIP_Tester::pdiscoverEntityInd: new Node created and returned");
        resNode = vAddNode(testerAddr, IP, logAddr, EID, GID, VIN, DOIP_NODE, maxDataSize);
    }
    return resNode;
}


DoIP_Node* DoIP_Tester::vGetNodeByAddr(tU16 logicalAddr)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::vGetNodeByAddr %i", logicalAddr);
    std::list<DoIP_Node*>::iterator itNode;
    DoIP_Node* resNode = NULL;
    pthread_mutex_lock(&mutDoIP_Nodes);
    for (itNode=oDoIP_Nodes.begin(); itNode != oDoIP_Nodes.end(); itNode++)
    {
        if((*itNode)->logicalAddr == logicalAddr)
        {
            //stored node found
            resNode = (*itNode);
            break;
        }
    }
    pthread_mutex_unlock(&mutDoIP_Nodes);
    return resNode;
}

DoIP_Node* DoIP_Tester::vGetNodeByIP(tU32 ipAddress)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::vGetNodeByIP 0x%4x", ipAddress);

    std::list<DoIP_Node*>::iterator itNode;
    DoIP_Node* resNode = NULL;
    pthread_mutex_lock(&mutDoIP_Nodes);
    for (itNode=oDoIP_Nodes.begin(); itNode != oDoIP_Nodes.end(); itNode++)
    {
        //DOIP_TRACE_NOTICE("DoIP_Tester::vGetNodeByIP --> 0x%4x", (*itNode)->getIPAddr());
        if((*itNode)->getIPAddr() == ipAddress)
        {
            //stored node found
            DOIP_TRACE_NOTICE("DoIP_Tester::vGetNodeByIP --> Node found");
            resNode = (*itNode);
            break;
        }
    }
    pthread_mutex_unlock(&mutDoIP_Nodes);

    return resNode;
}

tDoIPResult 
DoIP_Tester::doesNodeExist(tU32 ipAddress)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::doesNodeExist 0x%4x", ipAddress);

    std::list<DoIP_Node*>::iterator itNode;
    tDoIPResult result = DOIP_FAILED;
    pthread_mutex_lock(&mutDoIP_Nodes);
    for (itNode=oDoIP_Nodes.begin(); itNode != oDoIP_Nodes.end(); itNode++)
    {
        if((*itNode)->getIPAddr() == ipAddress)
        {
            //stored node found
          DOIP_TRACE_NOTICE("DoIP_Tester::doesNodeExist --> Node found");
            result = DOIP_SUCCESS;
            break;
        }
    }
    pthread_mutex_unlock(&mutDoIP_Nodes);
    return result;
}

void DoIP_Tester::vHandleMessages()
{

    tU16 payLoadType;
    /* Loop over all queue elements */
    for (tU8 queueIdx = 0; queueIdx < DOIP_CFG_MaxUDPQueueSize; queueIdx++)
    {
    /* Check if new message was received */
        if (arUDPQueue[queueIdx].udpQueueState == DOIP_UDP_DATA_RECEIVED)
        {
            /* Handle data in queue according to payload */
            memcpy(&payLoadType, &arUDPQueue[queueIdx].buffer[DOIP_MSG_POS_PAYLOAD_TYPE], DOIP_MSG_SIZE_PAYLOAD_TYPE);

            switch (payLoadType)
            {
             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_REQ:
                 //vVehicleIdentification_HandleRequest(queueIdx);
                 break;
             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_EID_REQ:
                 //vVehicleIdentification_HandleEIDRequest(queueIdx);
                 break;
             case DOIP_PAYLOAD_TYPE_VEHICLE_ID_VIN_REQ:
                 //vVehicleIdentification_HandleVINRequest(queueIdx);
                 break;
             case DOIP_PAYLOAD_TYPE_ENTITY_STATUS_REQ:
                 //vEntityStatus_HandleRequest(queueIdx);
                 break;
             case DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_REQ:
                 //vPowermode_HandleRequest(queueIdx);
                 break;
             default:
                 break;
            }
        }

        /* Send message if ready */
        if (arUDPQueue[queueIdx].udpQueueState == DOIP_UDP_DATA_READY)
        {
            vSendMessage(queueIdx);
        }

     }
}

void DoIP_Tester::vSendMessage(tU8 queueIdx)
{
   tU32 payLoadLength;

   mUtils.vMemCopyWithEndianessConversion(&payLoadLength,
         &arUDPQueue[queueIdx].buffer[DOIP_MSG_POS_PAYLOAD_LENGTH],
         DOIP_MSG_SIZE_PAYLOAD_LENGTH);

   {
   char addressBuffer[INET_ADDRSTRLEN] = "???";
   inet_ntop(AF_INET, &arUDPQueue[queueIdx].remoteAddr.sin_addr, addressBuffer, INET_ADDRSTRLEN);
   DOIP_TRACE_NOTICE("DoIP_Tester::vSendMessage[%s]: payloadlength=%u", addressBuffer, payLoadLength);
   }


   ssize_t ret = 0;

   ret = sendto(iSocket_UDP_Client_bcast, arUDPQueue[queueIdx].buffer, (payLoadLength + DOIP_HEADER_SIZE),
              0, (sockaddr*)&arUDPQueue[queueIdx].remoteAddr, sizeof(sockaddr_in));

   if (ret >= 0)
   {
       DOIP_TRACE_NOTICE("DoIP_Tester::vSendMessage length: %d bytes\n", ret);
       if ((udpState == DOIP_UDP_DISCOVERY) && (queueIdx == 1))
      {
         /* Reserve the second location of the queue for the vehicle announcement message during discovery phase*/
         arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_PROCESSED;
      }
      else
      {
         /* Clear the queue*/
         arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_EMPTY;
      }
   }
   else
   {
       /* disable repetitive transmission for debugging */
#if 1
       arUDPQueue[queueIdx].udpQueueState = DOIP_UDP_DATA_EMPTY;
#endif
       DOIP_TRACE_NOTICE("DoIP_Tester::vSendMessage Error sending msg: %s\n", strerror(errno));
   }
}

bool DoIP_Tester::vConnectionCloseAll(void)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::vConnectionCloseAll");
    (void) close(iSocket_UDP_Client);
    (void) close(iSocket_UDP_Client_bcast);
    //(void) close(iSocket_UDP_Client2);
    //(void) close(iSocket_UDP_Client2_bcast);

    return true;

}

/* for test */
std::list<DoIP_Node*> DoIP_Tester::getDoIPNodes(void)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::getDoIPNodes");
    return oDoIP_Nodes;
}

/* for test */
void DoIP_Tester::powermodeReqestAll(void)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::powermodeReqestAll");

    pthread_mutex_lock(&mutDoIP_Nodes);
    if (!oDoIP_Nodes.empty())
    {
        for (std::list<DoIP_Node*>::iterator itNodeStat=oDoIP_Nodes.begin(); itNodeStat != oDoIP_Nodes.end(); itNodeStat++)
        {
            (*itNodeStat)-> powermodeRequest(iSocket_UDP_Client);
        }
    }
    pthread_mutex_unlock(&mutDoIP_Nodes);
}

/* for test */
void DoIP_Tester::entityStatusReqAll(void)
{
    DOIP_TRACE_NOTICE("DoIP_Tester::entityStatusReqAll");

    pthread_mutex_lock(&mutDoIP_Nodes);
    if (!oDoIP_Nodes.empty())
    {
        for (std::list<DoIP_Node*>::iterator itNodeStat=oDoIP_Nodes.begin(); itNodeStat != oDoIP_Nodes.end(); itNodeStat++)
        {
            (*itNodeStat)-> entityStatusReq(iSocket_UDP_Client);
        }
    }
    pthread_mutex_unlock(&mutDoIP_Nodes);
}

