/*
 * DoIP_Node.cpp
 *
 *  Created on: Aug 14, 2018
 *      Author: sdn2hi
 */

#include "DoIP_Node.h"
#include "DoIP_Utils.h"

#include <stdio.h>
#include <string.h>

#include <sys/socket.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
//#include "doipcommon/dia_DoIPDefines.h"


DoIP_Node::DoIP_Node(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, tU32 maxDataSize, DoIP_Cfg &doipCfg)
 :
    testerAddr(testerAddress),
    logicalAddr(logAddr),
    nodeType(DOIP_NODE),
    mds(maxDataSize),
	  mpDoIPCallback(nullptr),
    iSocket_TCP_Client(-1),
    IPv4(IP),
    powerMode(DOIP_NOT_READY),
    actType(DOIP_NODE),
    mcts(10),
    ncts(0),
    state(DOIP_CON_OFFLINE),
    timerInactivity(0),
    rTCPClientThread(-1),
    mDoipCfg(doipCfg),
    mUtils(doipCfg)
{
    memset(&acVIN, 0, sizeof(acVIN));
    memset(&acEID, 0, sizeof(acEID));
    memset(&acGID, 0, sizeof(acGID));

    // TCP
    routingActivation.rx = {};
    routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_IDLE;
    routingActivation.routingActivationStatus = (tU8)DOIP_RA_STATUS_INIT;
    routingActivation.routingActivationSA = testerAddress;
    routingActivation.routingActivationType = 0xFF;

    RX.reset();
}


DoIP_Node::DoIP_Node(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, const tU8* EID, const tU8* GID, const tU8* VIN, DoIP_NodeType nodeType, tU32 maxDataSize, DoIP_Cfg &doipCfg)
 :
  testerAddr(testerAddress),
  logicalAddr(logAddr),
  nodeType(nodeType),
  mds(maxDataSize),
  mpDoIPCallback(nullptr),
  iSocket_TCP_Client(-1),
  IPv4(IP),
  powerMode(DOIP_NOT_READY),
  actType(DOIP_NODE),
  mcts(10),
  ncts(0),
  state(DOIP_CON_OFFLINE),
  timerInactivity(0),
  rTCPClientThread(-1),
  mDoipCfg(doipCfg),
  mUtils(doipCfg)
{
	DOIP_TRACE_NOTICE("DoIP_Node::DoIP_Node");
    memcpy(&acVIN, VIN, sizeof(acVIN));
    memcpy(&acEID, EID, sizeof(acEID));
    memcpy(&acGID, GID, sizeof(acGID));

    routingActivation.rx = {};
    routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_IDLE;
    routingActivation.routingActivationStatus = (tU8)DOIP_RA_STATUS_INIT;
    routingActivation.routingActivationSA = testerAddress;
    routingActivation.routingActivationType = 0xFF;

    RX.reset();
}


DoIP_Node::~DoIP_Node()
{
    /* and finally close the socket */
    if(iSocket_TCP_Client >= 0)
    {
        (void) close(iSocket_TCP_Client);
        //iSocket = -1;
    }

}

const tU8* DoIP_Node::getEID()
{
    return acEID;
}

const tU8* DoIP_Node::getGID()
{
    return acGID;
}

void DoIP_Node::vTraceStackDump(void){

    DOIP_TRACE_NOTICE("DoIP_Node::vTraceStackDump: DoIP entity: logicalAddr:%i nodeType: %i power Mode: %i"
            ", VIN=%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X EID=%02X%02X%02X%02X%02X%02X GID=%02X%02X%02X%02X%02X%02X "
            "MCTS: %i NCTS: %i MDS: %i"
            ,logicalAddr
            ,nodeType
            ,powerMode
            ,acVIN[0], acVIN[1], acVIN[2], acVIN[3], acVIN[4], acVIN[5],acVIN[6], acVIN[7], acVIN[8], acVIN[9]
            ,acVIN[10], acVIN[11], acVIN[12], acVIN[13], acVIN[14], acVIN[15],acVIN[16]
            ,acEID[0], acEID[1], acEID[2], acEID[3], acEID[4], acEID[5]
            ,acGID[0], acGID[1], acGID[2], acGID[3],acGID[4], acGID[5]
            ,mcts
            ,ncts
            ,mds
    );
}

bool DoIP_Node::createTCPSocket()
{
    DOIP_TRACE_NOTICE("DoIP_Node::createTCPSocket");
    bool res=true;
    /* Open IP4 DGRAM UDP socket for sending data */
    if((iSocket_TCP_Client = socket(AF_INET,SOCK_STREAM,IPPROTO_TCP)) < 0) // protocol 0 or IPPROTO_UDP
    {
        DOIP_TRACE_NOTICE("DoIP_Node::createTCPSocket: unable to open an TCP socket");
        res=false;
    }
    else
    {
        //Socket option : allow multiple sockets to use the same ports
        int optval = 1;
        if ( setsockopt(iSocket_TCP_Client, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval)) < 0 )
        {
            DOIP_TRACE_NOTICE("DoIP_Node::createTCPSocket: It is not able to change SO_REUSEADDR");
            res=false;
        }
        optval = 1;
        if (setsockopt(iSocket_TCP_Client, IPPROTO_TCP, TCP_NODELAY, &optval, sizeof(optval)) < 0)
        {
            DOIP_TRACE_WARNING("DoIP_Node::createTCPSocket: cannot change TCP_NODELAY");
            res=false;
        }
    }
    return res;
}

bool DoIP_Node::connectTCP(void)
{
    bool res = false;
    /* open a UDP broadcast client connection */
    DOIP_TRACE_NOTICE("DoIP_Node::connect");

    if(iSocket_TCP_Client >= 0)
    {
        struct sockaddr_in serv_addr;
        memset(&serv_addr, 0, sizeof(serv_addr));
        serv_addr.sin_family = AF_INET;
        serv_addr.sin_addr.s_addr = IPv4.sin_addr.s_addr;
        serv_addr.sin_port = htons(DOIP_TCP_PORT_DATA);

        char ip[INET_ADDRSTRLEN];
        inet_ntop(AF_INET, &(serv_addr.sin_addr), ip, INET_ADDRSTRLEN);
        DOIP_TRACE_NOTICE("DoIP_Node try connect to  %s (%u)", ip, DOIP_TCP_PORT_DATA);

        int ret = 0;
        /* Open IP4 DGRAM UDP socket for sending data */
        ret = connect(iSocket_TCP_Client, (struct sockaddr *) &serv_addr, sizeof(serv_addr));
        if (ret<0)
        {
            DOIP_TRACE_NOTICE("DoIP_Node::connect failed: %i", ret);

            (void) close(iSocket_TCP_Client);
            /*
            if (opTimer)
            {
               opTimer->vStart(iSocketConnectRetryTime); // try again in some time
            }*/
        }
        else
        {
            char addressBuffer[INET_ADDRSTRLEN];
            inet_ntop(AF_INET, &serv_addr.sin_addr.s_addr, addressBuffer, INET_ADDRSTRLEN);
            DOIP_TRACE_NOTICE("DoIP_Node::connect: connected to=%s Port: %d socket=%i", addressBuffer,ntohs(serv_addr.sin_port), iSocket_TCP_Client);

            /* start the reading thread for the TCP sockets */
            pthread_attr_t rTCPReadAttr;
            pthread_attr_init(&rTCPReadAttr);
            (void)pthread_attr_setstacksize(&rTCPReadAttr, DOIP_CFG_TCPSOCKET_CLIENT_STACKSIZE);

            if(pthread_create( &rTCPClientThread, &rTCPReadAttr, pvTCPClientThread, this) == 0)
            {
                //pthread_setname_np(rTCPClientThread, "DoIP@TCPClient");
                DOIP_TRACE_NOTICE("DoIP_Node::connectTCP: DoIP TCP Thread created successfully");
                res=true;
            }
            else
            {
                DOIP_TRACE_NOTICE("DoIP_Node::connectTCP: DoIP TCP Thread is not started");
            }
        }
    }
    else
    {
        DOIP_TRACE_NOTICE("DoIP_Node::connect please setup socked handler first");
    }
    return res;
}

bool DoIP_Node::closeTCPSocket(void)
{
  if(0 == close(iSocket_TCP_Client))
  {
    return true;
  }

  return false;
}

void* DoIP_Node::pvTCPClientThread(void* poPar)
{
   DoIP_Node* poNode = (DoIP_Node*)poPar;

   while(poNode->bReadNextFrameTCP());

   return NULL;
}


// keeps reading from socket until required length is received
ssize_t DoIP_Node::readNBytes(int fd, uint8_t *buffer, uint32_t numBytesRequested) {
  uint32_t bytesLeft = numBytesRequested;

  while (true) {
     ssize_t bytesRead = read(fd, buffer + numBytesRequested - bytesLeft, bytesLeft);
     if (bytesRead == -1) {
        if (errno == EINTR) {
           DOIP_TRACE_INFO("DoIP_Node::readNBytes(numBytesRequested=%u bytesLeft=%u): got EINTR",
                           numBytesRequested, bytesLeft);
           continue;
        }
        else {
           DOIP_TRACE_INFO("DoIP_Node::readNBytes(numBytesRequested=%u bytesLeft=%u): FAILED:%s",
                           numBytesRequested, bytesLeft, strerror(errno));
           return -1;
        }
     }
     else if (numBytesRequested && (bytesRead == 0) ) {
        DOIP_TRACE_INFO("DoIP_Node::readNBytes(numBytesRequested=%u bytesLeft=%u): return 0, peer closed connection",
                               numBytesRequested, bytesLeft);
        return -1;
     }

     bytesLeft-=(uint32_t)bytesRead;
     if (!bytesLeft) {
        break;
     }
  }

  DOIP_TRACE_INFO("DoIP_Node::readNBytes:numBytesRequested=%u:\n", numBytesRequested);
  mUtils.logHex("DoIP_Node::readNBytes", buffer, numBytesRequested);

  return numBytesRequested;
}

bool DoIP_Node::sendTcp(uint8_t *buffer, uint32_t numBytes) {
   mUtils.logHex("DoIP_Node::sendTcp", buffer, numBytes);

   ssize_t numBytesSent=send(iSocket_TCP_Client, buffer, numBytes, 0);
   if (numBytesSent != (ssize_t)numBytes)
   {
      DOIP_TRACE_WARNING("DoIP_Node::sendTcp: failed numBytes=%u numBytesSent=%u",
                                numBytes, (uint32_t)numBytesSent);
      return false;
   }
   return true;
}



bool DoIP_Node::bReadNextFrameTCP(void)
{
   if (RX.state == DOIP_TCP_IDLE) {
      RX.reset();
      RX.nextReadLen=DOIP_HEADER_SIZE;
   }
   DOIP_TRACE_NOTICE("DoIP_Node::bReadNextFrameTCP recv on socket %i", iSocket_TCP_Client);
   ssize_t iRecvBytes=readNBytes(iSocket_TCP_Client, RX.recvBuffer, RX.nextReadLen);
   if ((uint32_t)iRecvBytes != RX.nextReadLen) {
      // general error-handling will close socket, here we will detect it and exit.
      // got error or connection closed by client
      DOIP_TRACE_WARNING("DoIP_Node::bReadNextFrameTCP: closing connection socket=%u nextReadLen=%p iRecvBytes=%i",
                         iSocket_TCP_Client, RX.recvBuffer, RX.nextReadLen);
      //      delete this;
      return false;
   }
   
   RX.recvBufferUsed=(tU32)iRecvBytes;
   if (RX.state == DOIP_TCP_IDLE) {
      vHandleRxDoipHeader();
   }
   else if (RX.state == DOIP_TCP_RX_RECEIVING) {
      vHandleRxPayload();
   } 

   if (RX.state == DOIP_TCP_RX_LOCK) {
      (void)bHandleRxMsgComplete();
   }

   return true;
}

void DoIP_Node::vHandleRxDoipHeader() {
   DOIP_TRACE_INFO("DoIP_Node::vRxDoipHeader:RX.state=%u", RX.state);
   mUtils.logHex("vHandleRxDoipHeader", RX.recvBuffer, 8);

   mUtils.vMemCopyWithEndianessConversion(&RX.payloadType, &RX.recvBuffer[DOIP_MSG_POS_PAYLOAD_TYPE], DOIP_MSG_SIZE_PAYLOAD_TYPE);
   mUtils.vMemCopyWithEndianessConversion(&RX.payloadLength, &RX.recvBuffer[DOIP_MSG_POS_PAYLOAD_LENGTH], DOIP_MSG_SIZE_PAYLOAD_LENGTH);

   DOIP_TRACE_INFO("DoIP_Node::vRxDoipHeader RX.payloadType=%u", RX.payloadType);
   DOIP_TRACE_INFO("DoIP_Node::vRxDoipHeader RX.payloadLength=%u", RX.payloadLength);

   if (!bCheckPayloadLength()) {
      return;
   }

   //header check
   RX.nextReadLen=0;
   uint8_t responseCode=0;
   RX.nextReadLen=std::min((uint32_t)(RX.payloadLength-RX.payloadCounter), (uint32_t)DOIP_CFG_cu32TCPBufferSize);
   if (!mUtils.bCheckHeader(RX.recvBuffer, &responseCode, TRUE))
   {
      /* skip, because header Data is wrong */
      DOIP_TRACE_WARNING("DoIP_Node::vHandleRxDoipHeader: header error responseCode %i", responseCode);
      // todo: check if response to server is needed
      RX.reset();
   }
      DOIP_TRACE_INFO("DoIP_Node::vRxDoipHeader done:RX.nextReadLen=%u", RX.nextReadLen);

   if (RX.nextReadLen) {
      RX.state = DOIP_TCP_RX_RECEIVING;
   } else {
      RX.state = DOIP_TCP_RX_LOCK;
   }

}

void DoIP_Node::vHandleRxPayload()
{
   DOIP_TRACE_INFO("DoIP_Node::vHandleRxPayload START: len=%u RX.state=%u payloadCounter=%u RX.nextReadLen=%u RX.payloadLength=%u RX.cnt=%u",
                   RX.recvBufferUsed, RX.state, RX.payloadCounter, RX.nextReadLen, RX.payloadLength, RX.cnt);
   if(RX.recvBufferUsed != RX.nextReadLen) {
      DOIP_TRACE_NOTICE("DoIP_Node::vHandleRxPayload() wrong len: RX.recvBufferUsed %u  != RX.nextReadLen %u",
                                 (uint32_t)RX.recvBufferUsed, (uint32_t)RX.nextReadLen);
      return;
   }

   memcpy(RX.payloadBuffer + RX.payloadCounter, RX.recvBuffer, RX.recvBufferUsed);
   RX.payloadCounter+=RX.recvBufferUsed;
   RX.nextReadLen=std::min((uint32_t)(RX.payloadLength-RX.payloadCounter), (uint32_t)DOIP_CFG_cu32TCPBufferSize);
   mUtils.logHex("DoIP_Node::vHandleRxPayload: ", RX.payloadBuffer, RX.payloadCounter);
   if (!RX.nextReadLen) {
      RX.state=DOIP_TCP_RX_LOCK;
   }
   DOIP_TRACE_INFO("DoIP_Node::vHandleRxPayload: END RX.nextReadLen=%u", RX.nextReadLen);
}


bool DoIP_Node::bHandleRoutingActivationResponse() {
   DOIP_TRACE_WARNING("DoIP_Node::bHandleRoutingActivationResponse: DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_RESP: actState=%d len=%u",
                      routingActivation.routingActivationState, RX.payloadLength);
   bool ret_val = false;
   /* TODO RA: Remove below check when adapting RA handling */
   //if (routingActivation.routingActivationState == DOIP_ROUTINGACTIVATION_IDLE)
   {
      tU8 responseCode=RX.payloadBuffer[4u];
      
      DOIP_TRACE_WARNING("DoIP_Node::bHandleRoutingActivationResponse: DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_RESP: actState=%d len=%u routing rescode=%i",
                         routingActivation.routingActivationState, RX.payloadLength, responseCode);
      // Update Client address?
      routingActivation.routingActivationState = DOIP_ROUTINGACTIVATION_RECEIVED;
      activationResp(responseCode);
      ret_val = true;
   }
   return ret_val;
}

bool DoIP_Node::bHandleRxAliveCheckRequest() {
   DOIP_TRACE_WARNING("DoIP_Node::bHandleRxAliveCheckRequest: DOIP_PAYLOAD_TYPE_ALIVE_CHECK_REQ");
      
   bool retVal = false;
   retVal = aliveCheckResp();
   
   /* DoIP messages that are used to determine whether an open TCP_DATA socket is still in use by external test equipment.
	  The alive check response message can also be used by the external test equipment to keep a currently idle
	  connection alive, i.e. it can be sent by the external test equipment even if it has not previously received an alive check
	  request from a DoIP entity.
   */
   //DOIP_TRACE_WARNING("DoIP_Node::vHandleRxDoipHeader: Call the routing activation handler with status as success!");
   //if( mpDoIPCallback ) mpDoIPCallback->onRoutingActivationResponse(DOIP_RA_RESPCODE_SUCCESS);
   
   return retVal;
}

bool DoIP_Node::bHandleRxDiagMsg() {
   DOIP_TRACE_WARNING("DoIP_Node::bHandleRxDiagMsg: DOIP_PAYLOAD_TYPE_DIAG_MSG RX.payloadLength=%u", RX.payloadLength);
   // DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN already checked in bCheckPayloadLength()
  mUtils.logHex("DoIP_Node::bHandleRxDiagMsg", RX.payloadBuffer, RX.payloadLength);
  
   // buffer SA, TA
   mUtils.vMemCopyWithEndianessConversion(&RX.diagMsgSourceAddress, &RX.payloadBuffer[0u], 2u); //SA
   mUtils.vMemCopyWithEndianessConversion(&RX.diagMsgTargetAddress, &RX.payloadBuffer[2u], 2u); //TA

   //perform checks
   if (testerAddr != RX.diagMsgTargetAddress) {
      DOIP_TRACE_INFO("DoIP_Node::bHandleRxDiagMsg: wrong target-addr: want=0x%04x got=0x%04x",
                      testerAddr, RX.diagMsgTargetAddress);
      return false;
   }

   if (logicalAddr != RX.diagMsgSourceAddress) {
      DOIP_TRACE_INFO("DoIP_Node::bHandleRxDiagMsg: wrong source-addr: want=0x%04x got=0x%04x",
                      logicalAddr, RX.diagMsgSourceAddress);
      return false;
   }
   DOIP_TRACE_INFO("DoIP_Node::bHandleRxDiagMsg: forward to diagnosticResp");
   diagnosticResp(RX.payloadBuffer + DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN, RX.payloadLength - DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN);
   return true;
}


bool DoIP_Node::bHandleRxDiagMsgAck() {
   DOIP_TRACE_NOTICE("DoIP_Node::bHandleRxDiagMsgAck");
   mUtils.vMemCopyWithEndianessConversion(&RX.diagMsgSourceAddress, &RX.payloadBuffer[0u], 2u); //SA
   mUtils.vMemCopyWithEndianessConversion(&RX.diagMsgTargetAddress, &RX.payloadBuffer[2u], 2u); //TA
   tU8 ackCode=RX.payloadBuffer[4u];

   //perform checks
   if (testerAddr != RX.diagMsgTargetAddress) {
      DOIP_TRACE_INFO("DoIP_Node::bHandleRxDiagMsgAck: wrong target-addr: want=0x%04x got=0x%04x",
                      testerAddr, RX.diagMsgTargetAddress);
      return false;
   }

   if (logicalAddr != RX.diagMsgSourceAddress) {
      DOIP_TRACE_INFO("DoIP_Node::bHandleRxDiagMsgAck: wrong source-addr: want=0x%04x got=0x%04x",
                      logicalAddr, RX.diagMsgSourceAddress);
      return false;
   }
   
   uint32_t len = RX.payloadLength-DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_ACK_MIN;
   if (len) {
      diagnosticACK(ackCode,
                    &RX.payloadBuffer[DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_ACK_MIN],
                    len);
   }
   else {
      diagnosticACK(ackCode);
   }


   return true;
}


bool DoIP_Node::bHandleRxDiagMsgNack() {
   DOIP_TRACE_NOTICE("DoIP_Node::bHandleRxDiagMsgNack");
   mUtils.vMemCopyWithEndianessConversion(&RX.diagMsgSourceAddress, &RX.payloadBuffer[0u], 2u); //SA
   mUtils.vMemCopyWithEndianessConversion(&RX.diagMsgTargetAddress, &RX.payloadBuffer[2u], 2u); //TA
   tU8 ackCode=RX.payloadBuffer[4u];

   //perform checks
   if (testerAddr != RX.diagMsgTargetAddress) {
      DOIP_TRACE_INFO("DoIP_Node::bHandleRxDiagMsgNack: wrong target-addr: want=0x%04x got=0x%04x",
                      testerAddr, RX.diagMsgTargetAddress);
      return false;
   }

   if (logicalAddr != RX.diagMsgSourceAddress) {
      DOIP_TRACE_INFO("DoIP_Node::bHandleRxDiagMsgNack: wrong source-addr: want=0x%04x got=0x%04x",
                      logicalAddr, RX.diagMsgSourceAddress);
      return false;
   }
   
   tU32 len = RX.payloadLength-DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_ACK_MIN;

   if (len) {
      diagnosticACK(ackCode,
                    &RX.payloadBuffer[DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_ACK_MIN],
                    len);
   }
   else {
      diagnosticACK(ackCode);
   }

   return true;
}


bool DoIP_Node::bCheckPayloadLength()
{
    bool retVal = false;
    if (RX.payloadLength > sizeof(RX.payloadBuffer)) {
       return false;
    }

    switch (RX.payloadType) {
       case DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_RESP:
          retVal=( (RX.payloadLength == DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_RESP_9)
                   || (RX.payloadLength == DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_RESP_13) );
          break;
       case DOIP_PAYLOAD_TYPE_ALIVE_CHECK_REQ:
          retVal=(RX.payloadLength == DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_REQ);
          break;
       case DOIP_PAYLOAD_TYPE_DIAG_MSG:
          retVal=(RX.payloadLength >= DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN);
          break;
       case DOIP_PAYLOAD_TYPE_DIAG_MSG_ACK:
          retVal=(RX.payloadLength >= DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_ACK_MIN);
          break;
       case DOIP_PAYLOAD_TYPE_DIAG_MSG_NACK:
          retVal=(RX.payloadLength >= DOIP_PAYLOAD_LENGTH_GENERIC_HEADER_NACK_MIN);
          break;
       default:
          retVal=false;
    }
    return retVal;
}


bool DoIP_Node::bHandleRxMsgComplete()
{
    DOIP_TRACE_WARNING("DoIP_Node::bHandleRxMsgComplete: RX.payloadLength=%u",
                RX.payloadLength);

    bool ret_val = false;

    RX.state = DOIP_TCP_RX_LOCK;


    switch (RX.payloadType)
    {
       case DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_RESP:
       {
          ret_val=bHandleRoutingActivationResponse();
       }
       break;

       case DOIP_PAYLOAD_TYPE_ALIVE_CHECK_REQ:
       {
          ret_val=bHandleRxAliveCheckRequest();
       }
       break;

       case DOIP_PAYLOAD_TYPE_DIAG_MSG:
       {
          ret_val=bHandleRxDiagMsg();
       }
       break;

       case DOIP_PAYLOAD_TYPE_DIAG_MSG_ACK:
       {
          ret_val=bHandleRxDiagMsgAck();
       }
       break;

       case DOIP_PAYLOAD_TYPE_DIAG_MSG_NACK:
       {
          ret_val=bHandleRxDiagMsgNack();
       }
       break;


       default:
       {
          //unknown TCP request -> discard
          ret_val = false;
       }
       break;
    }

    //Reset state
    if (RX.state == DOIP_TCP_RX_LOCK)
    {
       RX.reset();
    }
    return ret_val;
}

// Diagnostic power mode information request to DoIp entity
void DoIP_Node::powermodeRequest(int sockfd)
{
    DOIP_TRACE_NOTICE("DoIP_Node::powermodeRequest to EID: %02X%02X%02X%02X%02X%02X", acEID[0], acEID[1], acEID[2], acEID[3], acEID[4], acEID[5]);

    tU8 buffer[DOIP_HEADER_SIZE+DOIP_PAYLOAD_LENGTH_DIAG_POWERMODE_REQ];
    tU16 payLoadType;
    tU32 payLoadLength;

    buffer[0] = DOIP_PROTOCOL_VERSION;
    buffer[1] = DOIP_PROTOCOL_VERSION_INV;
    payLoadType = DOIP_PAYLOAD_TYPE_DIAG_POWERMODE_REQ;
    mUtils.vMemCopyWithEndianessConversion(&buffer[2], &payLoadType, 2);
    payLoadLength = DOIP_PAYLOAD_LENGTH_DIAG_POWERMODE_REQ;
    mUtils.vMemCopyWithEndianessConversion(&buffer[4], &payLoadLength, 4);

    struct sockaddr_in servaddr = IPv4;
    servaddr.sin_port = htons(DOIP_UDP_PORT_DISCOVERY); // send destination port, source port bind on socket fd

    (void) sendto(sockfd, buffer, (payLoadLength + DOIP_HEADER_SIZE),
                  0, (sockaddr*)&servaddr, sizeof(servaddr));
}

// Diagnostic power mode information request to DoIp entity
void DoIP_Node::powermodeResp(const tU8 pwrMode, sockaddr_in& remoteaddr)
{
    if((IPv4.sin_addr.s_addr == remoteaddr.sin_addr.s_addr) && (IPv4.sin_port == remoteaddr.sin_port))
    {
        powerMode = pwrMode;
        DOIP_TRACE_NOTICE("DoIP_Node::powermodeResp Node Adrr: %i  IP: 0x%8x Power Mode: %s", logicalAddr, remoteaddr.sin_addr.s_addr,  ((powerMode == 1) ? "ready" : "not ready OR not supported"));
    }
	
	//if( mpDoIPCallback ) mpDoIPCallback->onPowerModeRequestResponse(pwrMode);
	if( mpDoIPCallback )
		{
			DOIP_TRACE_NOTICE("DoIP_Node::powermodeResp Doip Client present ");
			mpDoIPCallback->onPowerModeRequestResponse(pwrMode);
		}
		else
		{
			DOIP_TRACE_NOTICE("DoIP_Node::powermodeResp Doip Client absent");
		}
	
}

/* status request to DoIP entity, it will respond with DoIP entity status response */
void DoIP_Node::entityStatusReq(int sockfd)
{
    DOIP_TRACE_NOTICE("DoIP_Node::entityStatusReq");

    tU8 buffer[DOIP_HEADER_SIZE+DOIP_PAYLOAD_LENGTH_ENTITY_STATUS_REQ];
    tU16 payLoadType;
    tU32 payLoadLength;

    buffer[0] = DOIP_PROTOCOL_VERSION;
    buffer[1] = DOIP_PROTOCOL_VERSION_INV;
    payLoadType = DOIP_PAYLOAD_TYPE_ENTITY_STATUS_REQ;
    mUtils.vMemCopyWithEndianessConversion(&buffer[2], &payLoadType, 2);
    payLoadLength = DOIP_PAYLOAD_LENGTH_ENTITY_STATUS_REQ;
    mUtils.vMemCopyWithEndianessConversion(&buffer[4], &payLoadLength, 4);

    struct sockaddr_in servaddr = IPv4;
    servaddr.sin_port = htons(DOIP_UDP_PORT_DISCOVERY); // send destination port, source port bind on socket fd

    (void) sendto(sockfd, buffer, (payLoadLength + DOIP_HEADER_SIZE),
                  0, (sockaddr*)&servaddr, sizeof(servaddr));
}



void DoIP_Node::entityStatusResp (const tU8 au8UdpBuffer[], sockaddr_in& remoteaddr)
{
    if((IPv4.sin_addr.s_addr == remoteaddr.sin_addr.s_addr) && (IPv4.sin_port == remoteaddr.sin_port))
    {
        tU8  lactType;
        tU8  ltestersSize;
        tU8  lconnectionsSize;
        tU32 lmaxSize;

        lactType         = au8UdpBuffer[DOIP_MSG_POS_CONTENT];
        ltestersSize     = au8UdpBuffer[DOIP_MSG_POS_CONTENT+1];
        lconnectionsSize = au8UdpBuffer[DOIP_MSG_POS_CONTENT+2];
        lmaxSize = 0;
        if (mDoipCfg.entityStatusMaxByteFieldUse) {
           mUtils.vMemCopyWithEndianessConversion(&lmaxSize, &au8UdpBuffer[DOIP_MSG_POS_CONTENT+3], 4);
           mds = lmaxSize;
        }
        DOIP_TRACE_NOTICE("DoIP_Node::entityStatusResp NodeType: %d testersSize: %d connectionsSize: %d, max Size: %d", lactType, ltestersSize, lconnectionsSize, lmaxSize);

        actType = lactType;
        mcts = ltestersSize;
        ncts = lconnectionsSize;
    }
    else
    {
        DOIP_TRACE_NOTICE("DoIP_Node::entityStatusResp, not valid adress");
    }
}

bool DoIP_Node::routingActivationReq(tU8 actNum, tU8 oem[4])
{
    DOIP_TRACE_NOTICE("DoIP_Node::routingActivationReq actNum=0x%02x", actNum);
    bool res = true;
    tU32 tempBuffer;
    tU32 u32Len;
    /* DoIP Header */
    buffer[DOIP_MSG_POS_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION;
    buffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION_INV;
    /* PayloadType */
    tempBuffer = DOIP_PAYLOAD_TYPE_ROUTING_ACTIVATION_REQ;
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_TYPE], &tempBuffer, DOIP_MSG_SIZE_PAYLOAD_TYPE);
    /* PayloadLength (SudLength + SA + TA */

    if (oem == NULL)
    {
        u32Len = (DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_7);
    }
    else
    {
        u32Len = (DOIP_PAYLOAD_LENGTH_ROUTING_ACTIVATION_REQ_11);
    }
    tempBuffer = u32Len;
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_LENGTH], &tempBuffer, DOIP_MSG_SIZE_PAYLOAD_LENGTH);

    /* external test equipment address information */
    tempBuffer = testerAddr;
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_CONTENT + 0u], &tempBuffer, 2u); //SA: Address of the external test equipment

    /* routing activation type */
    buffer[DOIP_MSG_POS_CONTENT + 2u] = actNum;

    /* reserved data */
    buffer[DOIP_MSG_POS_CONTENT + 3u] = 0;
    buffer[DOIP_MSG_POS_CONTENT + 4u] = 0;
    buffer[DOIP_MSG_POS_CONTENT + 5u] = 0;
    buffer[DOIP_MSG_POS_CONTENT + 6u] = 0;

    /* OEM specific data */
    if (oem != NULL)
    {
        memcpy(&buffer[DOIP_MSG_POS_CONTENT + 7], &oem[0], 4);
    }
    tS32 s32BufferLen = (u32Len + DOIP_HEADER_SIZE);
    if (!sendTcp(buffer, s32BufferLen)) {
        DOIP_TRACE_WARNING("DoIP_Node::routingActivationReq: failed");
        res = false;
    }
    return res;
}


bool DoIP_Node::diagnosticReq(tU8 const * au8Data, tU32 u32Len)
{
    DOIP_TRACE_NOTICE("DoIP_Node::diagnosticReq tgtAddr(default): %i len: %i:", this->logicalAddr, u32Len);

    return DoIP_Node::diagnosticReq(this->logicalAddr, au8Data, u32Len);
}

bool DoIP_Node::diagnosticReq(tU16 tgtAddr, tU8 const * au8Data, tU32 u32Len)
{
    bool res=true;
    DOIP_TRACE_NOTICE("DoIP_Node::diagnosticReq tgtAddr: %i len: %i:", tgtAddr, u32Len);

    /* DoIP Header 8 bytes */
    tU32 tempBuffer;
    buffer[DOIP_MSG_POS_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION;
    buffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION_INV;
    /* PayloadType */
    tempBuffer = DOIP_PAYLOAD_TYPE_DIAG_MSG;
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_TYPE], &tempBuffer, DOIP_MSG_SIZE_PAYLOAD_TYPE);
    /* PayloadLength (SudLength + SA + TA */
    tempBuffer = (DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN + u32Len);
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_LENGTH], &tempBuffer, DOIP_MSG_SIZE_PAYLOAD_LENGTH);

    /* diagnostic message 2+2+UD bytes */
    tempBuffer = testerAddr;
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_CONTENT + 0u], &tempBuffer, 2u); //SA: sender address (tester entity)
    if (tgtAddr == NULL)
    {
        tempBuffer = this->logicalAddr;
    }
    else
    {
        tempBuffer = tgtAddr;
    }
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_CONTENT + 2u], &tempBuffer, 2u); //TA: receiver address (doIP entity)
    /* UD (user data) diagnostic data (e.g. ISO 14229-1 diagnostic request)  */
    memcpy(&buffer[DOIP_MSG_POS_CONTENT + DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN], &au8Data[0], u32Len);

    tS32 s32BufferLen = (DOIP_HEADER_SIZE + DOIP_PAYLOAD_LENGTH_DIAG_MSG_MIN + u32Len);
    if (!sendTcp(buffer, s32BufferLen)) {
        DOIP_TRACE_WARNING("DoIP_Node::diagnosticReq: failed");
        res=false;
    }
    return res;
}

void DoIP_Node::activationResp(tU8 respType)
{
    DOIP_TRACE_NOTICE("DoIP_Node::activationResp respType %i", respType);
	
	if( mpDoIPCallback )
	{
		DOIP_TRACE_NOTICE("DoIP_Node::activationResp Doip Client present ");
		mpDoIPCallback->onRoutingActivationResponse(respType);
	}
	else
	{
		DOIP_TRACE_NOTICE("DoIP_Node::activationResp Doip Client absent");
	}
}

void DoIP_Node::diagnosticResp(tU8 const * data, tU32 U32len)
{
    if (U32len > 0)
    {
        DOIP_TRACE_NOTICE("DoIP_Node::diagnosticResp diagnostic message data received with len %i: ", U32len);

        if (&data[0]!=NULL)
        {
            for (tU32 i = 0; i<U32len; i++)
            {
                DOIP_TRACE_NOTICE("DoIP_Node::diagnosticResp diagnostic message data [%i]: %02x ", i, data[i]);
            }

            if( mpDoIPCallback ) mpDoIPCallback->onNodeDiagnosticResponse(data, U32len);
        }
     }
}

void DoIP_Node::diagnosticACK(tU8 ackCode, tU8* data /* = NULL*/, tU32 U32len /* = 0*/)
{
    DOIP_TRACE_NOTICE("DoIP_Node::diagnosticACK ACK code: %i %s", ackCode,((ackCode == DOIP_DM_ACKCODE_SUCCESS) ? " , diagnostic message correctly received" : ""));
    if (U32len > 0)
    {
        DOIP_TRACE_NOTICE("DoIP_Node::diagnosticACK previous diagnostic message data received with len %i: ", U32len);

        if (&data[0]!=NULL)
        {
            for (tU32 i = 0; i<U32len; i++)
            {
                DOIP_TRACE_NOTICE("DoIP_Node::diagnosticACK previous diagnostic message data [%i]: %02x ", i, data[i]);
            }
        }
    }
}

void DoIP_Node::diagnosticNACK(tU8 nackCode, tU8* data /* = NULL */, tU32 U32len /* = 0*/)
{
    DOIP_TRACE_NOTICE("DoIP_Node::diagnosticNACK NACK code: %i", nackCode);

    if (U32len > 0)
    {
        DOIP_TRACE_NOTICE("DoIP_Node::diagnosticNACK previous diagnostic message data received with len %i: ", U32len);

        if (&data[0]!=NULL)
        {
            for (tU32 i = 0; i<U32len; i++)
            {
                DOIP_TRACE_NOTICE("DoIP_Node::diagnosticNACK previous diagnostic message data [%i]: %02x ", i, data[i]);
            }
        }
     }
}

/* alive check response, can be send actively to keep a TCP_DATA connection to the doip_server alive */
bool DoIP_Node::aliveCheckResp(tU16 SrcAddr)
{
    bool res = false;
    DOIP_TRACE_NOTICE("DoIP_Node::aliveCheckResp Source address : %i", testerAddr);

    tU32 tempBuffer;
    /* DoIP Header */
    buffer[DOIP_MSG_POS_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION;
    buffer[DOIP_MSG_POS_INV_PROTOCOL_VERSION] = DOIP_PROTOCOL_VERSION_INV;
    /* PayloadType */
    tempBuffer = DOIP_PAYLOAD_TYPE_ALIVE_CHECK_RESP;
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_TYPE], &tempBuffer, DOIP_MSG_SIZE_PAYLOAD_TYPE);
    /* PayloadLength (SudLength + SA + TA */
    tempBuffer = DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_RESP;
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_PAYLOAD_LENGTH], &tempBuffer, DOIP_MSG_SIZE_PAYLOAD_LENGTH);

    tempBuffer = 0;
    if (SrcAddr == 0)
    {
        if (testerAddr != NULL)
        {
            tempBuffer = testerAddr;
        }
    }
    else
    {
        tempBuffer = SrcAddr;
    }
    mUtils.vMemCopyWithEndianessConversion(&buffer[DOIP_MSG_POS_CONTENT + 0u], &tempBuffer, 2u); //SA: source address
                                                                                                      // Contains the logical address of the external test
                                                                                                      // equipment that is currently active on this TCP_DATA socket.
    tS32 s32BufferLen = (DOIP_HEADER_SIZE + DOIP_PAYLOAD_LENGTH_ALIVE_CHECK_RESP);
    if (send(iSocket_TCP_Client, buffer, s32BufferLen, 0) != s32BufferLen)
    {
        DOIP_TRACE_WARNING("DoIP_Node::diagnosticReq: failed");
    }
    else
    {
        DOIP_TRACE_WARNING("DoIP_Node::diagnosticReq send successfully, size: %i (incl. 8 byte header)",s32BufferLen);
        res = true;
    }
    return res;
}
