//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_can_if.cpp
// PROJECT:      NISSAN
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog CAN interface
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 18.05.15  | BSOT Plischke                | Initial version
// 28.12.18  | mky6kor						| Extended data configuration(RTC-434184)
//////////////////////////////////////////////////////////////////////////
// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

   
#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_can_if
   #include "vd_diaglog_can_if.h"
#endif


#ifndef __INCLUDED_CSM_IF__
 #define CSM_S_IMPORT_INTERFACE_GENERIC_USER
 #define CSM_C_PROJECT_USERSPACE_EVOBUS
 #include "csm_if.h"
#endif


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_can_if.cpp.trc.h"
#endif

#define DIAGLOG_MILEAGE_TIMER_INTERVAL  1000   /*[ms]*/
#define DIAGLOG_MILE_TO_KM_CONV_RATIO   (1.609344f)

#define CAN_CSM_DISTANCE_TOTALIZER_SIGNAL_RESOLUTION (0.125)

vdl_tclCanIf* vdl_tclCanIf::pCanIfTrigger = NULL;
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclCanIf::vdl_tclCanIf()
// 
// DESCRIPTION: constructor
//
// PARAMETER:  
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclCanIf::vdl_tclCanIf():
   m_pCsmAccessUser(NULL),
   m_mileage(0),
   m_mileageUpdateAllowed(true),
   m_OdometerValue(0x00)
{
   ETG_TRACE_COMP_THR(( "--- vdl_tclCanIf::vdl_tclCanIf => PlugIn:$%08x",this));
   ETG_TRACE_USR1_THR(( "--- vdl_tclDTCTriggerBase::vOnInit => Create TIMER m_mileageTimerID"));

   try
   {
      m_mileageTimerID = s32CreateTimer();
   }
   catch ( ... )
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclCanIf::vdl_tclCanIf => catch"));
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclCanIf::vdl_tclCanIf()
// 
// DESCRIPTION: destructor
//
// PARAMETER:  
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclCanIf::~vdl_tclCanIf ()
{
   try
   {
      vClose();
   }
   catch ( ... )
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclCanIf::~vdl_tclCanIf => catch"));
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclCanIf::vOnInit()
// 
// DESCRIPTION: Update the TriggerStatus
//
// PARAMETER:  
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//

tVoid vdl_tclCanIf::vOnInit() 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclCanIf::vOnInit"));

	m_pCsmAccessUser = new csm_tclCsmAccessUser();
	if(m_pCsmAccessUser != NULL)
	{
		pCanIfTrigger = this;
		m_pCsmAccessUser->vApplCallbackPreInit();
	}

   ETG_TRACE_USR3_THR(( "<-- vdl_tclCanIf::vOnInit"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclCanIf::vClose()
// 
// DESCRIPTION: Update the TriggerStatus
//
// PARAMETER:  
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclCanIf::vClose() 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclCanIf::vClose"));

   if(m_pCsmAccessUser)
   {
      delete m_pCsmAccessUser;
      m_pCsmAccessUser = NULL;
   }

   // stop Timer
   if( bSetTimer(m_mileageTimerID,0) == false)
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclCanIf::getMileage => TIMER CREATTION FAILED m_mileageTimerID:%d",m_mileageTimerID));
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclCanIf::vClose"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:     vdl_tclCanIf::vTimer
//
// DESCRIPTION:  Timer event
//             
// PARAMETER:    TimerId: unique ID of the Timer
//
// RETURNVALUE:  None
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclCanIf::vTimer(tS32 s32TimerId)
{
   if(s32TimerId == m_mileageTimerID)
   {
      // timer run out, allow new update
      ETG_TRACE_USR2_THR(( "--- vdl_tclCanIf::vTimer => m_u32TriggerTimerId called"));
      if( bSetTimer(m_mileageTimerID,0) == false)
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclCanIf::getMileage => TIMER CREATTION FAILED m_mileageTimerID:%d",m_mileageTimerID));
         NORMAL_M_ASSERT_ALWAYS();
      }
      m_mileageUpdateAllowed = true;
   }
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclCanIf::getMileage()
// 
// DESCRIPTION: return the current mileage
//
// PARAMETER:  none
//
// RETURNVALUE: mileage (km)
// 
///////////////////////////////////////////////////////////////////////////////////
//
tU32 vdl_tclCanIf::getMileage() 
{

   ETG_TRACE_USR3_THR(( "--> vdl_tclCanIf::getMileage => getMileage"));
   
//Task 239541
#if 0
   tS32 s32RetVal;
   tU8  signalData[8] = {0,0,0,0,0,0,0,0};
   tU32 signalStatus = 0;
   tU8 size;

   // only update DIAGLOG_MILEAGE_TIMER_INTERVAL
   if(m_mileageUpdateAllowed)
   {
	   if(m_pCsmAccessUser != NULL)
      {
         size = 4;
         s32RetVal = m_pCsmAccessUser->s32SignalRead( CSM_C_ASIG_RX_DistanceTotalizer, (void*)&signalData[0], size, &signalStatus );
         if(  (s32RetVal == 0)
            &&(  ((signalStatus & CSM_C_SIGNAL_ARRIVED_FIRST) == CSM_C_SIGNAL_ARRIVED_FIRST)
               ||((signalStatus & CSM_C_SIGNAL_ARRIVED) == CSM_C_SIGNAL_ARRIVED)
			  && ((signalStatus & CSM_C_SIGNAL_TIMEOUT) != CSM_C_SIGNAL_TIMEOUT)
              )
           )
         {
            ETG_TRACE_USR4_THR(("--- vdl_tclCanIf::getMileage => CANv5 signal CSM_C_ASIG_RX_DistanceTotalizer Data:%*x ", ETG_LIST_LEN(size), ETG_LIST_PTR_T8(signalData)));
            // signal valid 
            m_mileage = (((tU32)signalData[3]<< 24) + ((tU32)signalData[2]<< 16) + ((tU32)signalData[1]<< 8) + (tU32)signalData[0]) /100U; // received resolution 0.01

            // now get the unit
            size = 1;
            s32RetVal = m_pCsmAccessUser->s32SignalRead( CSM_C_ASIG_RX_DistanceUnit, (void*)&signalData[0], size, &signalStatus );
            if(  (s32RetVal == 0)
               &&(  ((signalStatus & CSM_C_SIGNAL_ARRIVED_FIRST) == CSM_C_SIGNAL_ARRIVED_FIRST)
                  ||((signalStatus & CSM_C_SIGNAL_ARRIVED) == CSM_C_SIGNAL_ARRIVED)
                 )
              )
            {
               ETG_TRACE_USR4_THR(("--- vdl_tclCanIf::getMileage => CANv5 signal CSM_C_ASIG_RX_DistanceUnit Data:%*x ", ETG_LIST_LEN(size), ETG_LIST_PTR_T8(signalData)));
               if(signalData[0]==0)
               {
                  // km
                  // do nothing
               }
               else
               {
                  //convert mile-to-km and round to the nearest integral
                  m_mileage = (tU32)(((float)m_mileage)*DIAGLOG_MILE_TO_KM_CONV_RATIO + 0.5f); //lint !e524: prio2: Loss of precision (assignment) (double to unsigned long)
               }
               ETG_TRACE_USR1_THR(("--- vdl_tclCanIf::getMileage => CANg5 signal m_mileage:%d km", m_mileage));
            }// if(  (s32RetVal == 0)
            else
            {
               ETG_TRACE_ERR_THR(("!!! vdl_tclCanIf::getMileage => CSM_C_ASIG_RX_DistanceUnit signal read error: %d, signalStatus: 0x%x", s32RetVal, signalStatus));
               // This happens mainly on CANGen 3 / Slot 6
               ETG_TRACE_USR1_THR(("--- vdl_tclCanIf::getMileage => CANg3 signal m_mileage:%d km", m_mileage));
            }// else // if(  (s32RetVal == 0)

         }// if(  (s32RetVal == 0)...
         else
         {
            ETG_TRACE_ERR_THR(("!!! vdl_tclCanIf::getMileage => CSM_C_ASIG_RX_DistanceTotalizer signal read error: %d, signalStatus: 0x%x", s32RetVal, signalStatus));
			   m_mileage = 0xFFFFFF;
         }
      }// if(m_pCsmAccessUser != NULL)
      else
      {
         ETG_TRACE_ERR_THR(( "!!! vdl_tclCanIf::getMileage => m_pCsmAccessUser == NULL"));
      }

      // block update for the next DIAGLOG_MILEAGE_TIMER_INTERVAL
      if( bSetTimer(m_mileageTimerID,DIAGLOG_MILEAGE_TIMER_INTERVAL) == false)
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclCanIf::getMileage => TIMER CREATTION FAILED m_mileageTimerID:%d",m_mileageTimerID));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         m_mileageUpdateAllowed = false;
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclCanIf::getMileage => getMileage"));
#endif
   return m_mileage;
}

//-----------------------------------------------------------------------------

bool vdl_tclCanIf::bTransmitDM1TriggerMsg(tU8* dm1frame) const
{

	ETG_TRACE_USR3_THR(( "--> vdl_tclCanIf::bTransmitDM1TriggerMsg => bTransmitDM1TriggerMsg"));
	tU8 byteposition=0;
	tU8 dm2buffer[8] = {0};
	tU8 dmindex=0;
	while( byteposition < 8)
	{
	   ETG_TRACE_COMP_THR(("DM[%d]= %x",byteposition,dm1frame[byteposition]));//just to check DM1 message content
	   byteposition++;
	   //dm1frame++;
	}
	
	/* Sending the DM1 message on CAN */

	for(dmindex=0;dmindex<8;dmindex++)
	{
		dm2buffer[dmindex] = dm1frame[dmindex];

	}
	tS32 lRetCode = m_pCsmAccessUser->s32SignalWrite( CSM_C_SIG_TX_HUS_DM1,dm2buffer, sizeof(dm2buffer), CSM_C_TX_AS_CONFIGURED );
	if(lRetCode == CSM_C_NO_ERROR)
	{
		return true;
	}
	else
	{
		return false;
	}
}
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclCanIf::getOdometerValue()
//
// DESCRIPTION: return the current odometer
//
// PARAMETER:  none
//
// RETURNVALUE: odometer (km)
//
///////////////////////////////////////////////////////////////////////////////////
//
tU16 vdl_tclCanIf::getOdometerValue()
{

   ETG_TRACE_USR3_THR(( "--> vdl_tclCanIf::getOdometerValue => getOdometerValue"));

   tS32 s32RetVal=0;
   tU32 signalData[8] = {0,0,0,0,0,0,0,0};
   tU32 signalStatus = 0;
   tU8 size;

   if(m_pCsmAccessUser != NULL)
   {
	 size = 4;
	 s32RetVal = m_pCsmAccessUser->s32SignalRead( CSM_C_SIG_RX_TotalVehDist_Cval_DIAG_CGW_C1, (void*)&signalData[0], size, &signalStatus );
	 if(s32RetVal == 0)
	 {
	    for(int i=0; i<size; i++)
	      {
		  ETG_TRACE_USR3_THR(("signalData[%d]: 0x%02X : Decimal [%d]", i, signalData[i],signalData[i]));
	      }
              m_OdometerValue = tU16 ( (signalData[0])*CAN_CSM_DISTANCE_TOTALIZER_SIGNAL_RESOLUTION );
	      ETG_TRACE_USR3_THR(("getOdometerValue: odometer: %d", m_OdometerValue));
	 }
	 else
	 {
            ETG_TRACE_ERR_THR(("!!! vdl_tclCanIf::getOdometerValue => CSM_C_SIG_RX_TotalVehDist_Cval_DIAG_CGW_C1 signal read error: %d, signalStatus: 0x%x", s32RetVal, signalStatus));
	    m_OdometerValue = 0xFFFFFF;
	 }
   }
   else
   {
	 ETG_TRACE_ERR_THR(( "!!! vdl_tclCanIf::getOdometerValue => m_pCsmAccessUser == NULL"));
   }
   return m_OdometerValue;
}

   // only update DIAGLOG_MILEAGE_TIMER_INTERVAL
#if 0
   if(m_pCsmAccessUser != NULL)
   {
	 size = 4;
	 //s32RetVal = m_pCsmAccessUser->s32SignalRead( CSM_C_ASIG_RX_DistanceTotalizer, (void*)&signalData[0], size, &signalStatus );
	 if(  (s32RetVal == 0)
		&&(  ((signalStatus & CSM_C_SIGNAL_ARRIVED_FIRST) == CSM_C_SIGNAL_ARRIVED_FIRST)
		   ||((signalStatus & CSM_C_SIGNAL_ARRIVED) == CSM_C_SIGNAL_ARRIVED)
		  && ((signalStatus & CSM_C_SIGNAL_TIMEOUT) != CSM_C_SIGNAL_TIMEOUT)
		  )
	   )
	 {
		ETG_TRACE_USR4_THR(("--- vdl_tclCanIf::getOdometerValue => CANv5 signal CSM_C_ASIG_RX_DistanceTotalizer Data:%*x ", ETG_LIST_LEN(size), ETG_LIST_PTR_T8(signalData)));
		// signal valid
		m_mileage = (((tU32)signalData[3]<< 24) + ((tU32)signalData[2]<< 16) + ((tU32)signalData[1]<< 8) + (tU32)signalData[0]) /100U; // received resolution 0.01

		// now get the unit
		size = 1;
		//s32RetVal = m_pCsmAccessUser->s32SignalRead( CSM_C_ASIG_RX_DistanceUnit, (void*)&signalData[0], size, &signalStatus );
		if(  (s32RetVal == 0)
		   &&(  ((signalStatus & CSM_C_SIGNAL_ARRIVED_FIRST) == CSM_C_SIGNAL_ARRIVED_FIRST)
			  ||((signalStatus & CSM_C_SIGNAL_ARRIVED) == CSM_C_SIGNAL_ARRIVED)
			 )
		  )
		{
		   ETG_TRACE_USR4_THR(("--- vdl_tclCanIf::getOdometerValue => CANv5 signal CSM_C_ASIG_RX_DistanceUnit Data:%*x ", ETG_LIST_LEN(size), ETG_LIST_PTR_T8(signalData)));
		   if(signalData[0]==0)
		   {
			  // km
			  // do nothing
		   }
		   else
		   {
			  //convert mile-to-km and round to the nearest integral
			  m_mileage = (tU32)(((float)m_mileage)*DIAGLOG_MILE_TO_KM_CONV_RATIO + 0.5f); //lint !e524: prio2: Loss of precision (assignment) (double to unsigned long)
		   }
		   ETG_TRACE_USR1_THR(("--- vdl_tclCanIf::getOdometerValue => CANg5 signal m_mileage:%d km", m_mileage));
		}// if(  (s32RetVal == 0)
		else
		{
		   ETG_TRACE_ERR_THR(("!!! vdl_tclCanIf::getOdometerValue => CSM_C_ASIG_RX_DistanceUnit signal read error: %d, signalStatus: 0x%x", s32RetVal, signalStatus));
		   // This happens mainly on CANGen 3 / Slot 6
		   ETG_TRACE_USR1_THR(("--- vdl_tclCanIf::getOdometerValue => CANg3 signal m_mileage:%d km", m_mileage));
		}// else // if(  (s32RetVal == 0)

	 }// if(  (s32RetVal == 0)...
	 else
	 {
		ETG_TRACE_ERR_THR(("!!! vdl_tclCanIf::getOdometerValue => CSM_C_ASIG_RX_DistanceTotalizer signal read error: %d, signalStatus: 0x%x", s32RetVal, signalStatus));
		   m_mileage = 0xFFFFFF;
	 }
  }// if(m_pCsmAccessUser != NULL)
  else
  {
	 ETG_TRACE_ERR_THR(( "!!! vdl_tclCanIf::getOdometerValue => m_pCsmAccessUser == NULL"));
  }

  // block update for the next DIAGLOG_MILEAGE_TIMER_INTERVAL
  if( bSetTimer(m_mileageTimerID,DIAGLOG_MILEAGE_TIMER_INTERVAL) == false)
  {
	 ETG_TRACE_ERRMEM(( "!!! vdl_tclCanIf::getOdometerValue => TIMER CREATTION FAILED m_mileageTimerID:%d",m_mileageTimerID));
	 NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {
	 m_mileageUpdateAllowed = false;
  }


   ETG_TRACE_USR3_THR(( "<-- vdl_tclCanIf::getOdometerValue => getMileage"));
   return m_mileage;
#endif
