/************************************************************************
* FILE:         vd_diaglog_service.cpp
* PROJECT:      common
* SW-COMPONENT: DiagLog
*----------------------------------------------------------------------
*
* DESCRIPTION: DiagLog Main Interface
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 17.10.05  | CM-DI/ESA2 Barber  | initial version
* 15.10.12  | BSOT Plischke      | new Diaglog
*
*************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#define VD_DIAGLOG_S_IMPORT_INTERFACE_GENERIC
#include <vd_diaglog_if.h>

using namespace vdl_nsTraceFunctions;

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_service
   #include "vd_diaglog_service.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_memory
   #include <common/framework/vd_diaglog_report_memory.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#include <common/framework/extendedData/ExtendedData.h>


#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_DBGVISITORS
#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_STDVISITORS
#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_FUNCTIONIDS
#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_SERVICEINFO
#define CFC_FI_S_IMPORT_INTERFACE_FI_TYPES
#include "cfc_fi_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_service.cpp.trc.h"
#endif

using namespace VDD;


/* +++
MESSAGE MAP:
enter the function IDs (FID) and the corresponding functions here.
the function will be called when a message with the corresponding FID arrives
+++ */
BEGIN_MSG_MAP(vdDiagLog_tclService, ahl_tclBaseWork)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT, vOnNewTestResult)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_SAVETESTRESULTDYNAMICALDTC, vOnNewTestResultDynamialDTC)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_CONTROL, vOnControl)
//ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_CONTROLDTCSETTING, vOnControlDtcSetting)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT, vOnSendNextTestResult)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_SETFACTORYSETTINGSNEW, vOnSetToDefaultRequest)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED, vOnDtcTrigger)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_PRODUCTIONCLEARREQUEST, vOnProductionClear)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION, vOnGetItcInformation)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK, vOnGetDtcByStatusMask)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC, vOnGetSupportedDTC)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC, vOnGetLastConfirmedDTC)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK, vOnGetNumberOfDTCByStatusMask)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC, vOnGetStatusByDTC)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC, vOnGetStatusByITC)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETEXTENDEDDATABYDTC, vOnGetExtendedDataByDTC)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTIDENTIFICATION, vOnGetSnapShotIdentification)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTRECORDBYDTCNUMBER, vOnGetSnapShotDataByDTC)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_CLEARDTC, vOnClearDTC)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST, vOnSendActiveDTCList)
ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_DTCSTATUSCHANGED, vOnDtcStatusChanged)
END_MSG_MAP()



/*************************************************************************
*
* FUNCTION: vdDiagLog_tclService::vdDiagLog_tclService(vdDiagLog_tclApp* poGWMainApp)
*
* DESCRIPTION: constructor, creates object vdDiagLog_tclService - object
*
* PARAMETER: vdDiagLog_tclApp* poGWMainApp: main - object of this application
*
* RETURNVALUE: none
*
*************************************************************************/
vdDiagLog_tclService::vdDiagLog_tclService(vdDiagLog_tclApp* poVdDiagLog_MainApp):
   m_poGWMain(NULL)
{
   /* ---
   this constructor calls ( impicit ) the constructor of the upper class ( framework )
   so it registers the combinations of FID and message handler with the framework
   --- */
   m_poGWMain = poVdDiagLog_MainApp;
}

/*************************************************************************
*
* FUNCTION:tVoid vdDiagLog_tclService::vOnUnknownMessage(amt_tclBaseMessage* poMessage)
*
* DESCRIPTION: handle unknown message
*
* PARAMETER:  unknown message
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdDiagLog_tclService::vOnUnknownMessage(amt_tclBaseMessage* /*poMessage*/)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclService::vOnUnknownMessage"));
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclService::vOnUnknownMessage"));
}
/*************************************************************************
*
* FUNCTION: tVoid vdDiagLog_tclService::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
*
* DESCRIPTION: handles state - change messages form the spm
*
* PARAMETER: old state, new state
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdDiagLog_tclService::vOnNewAppState(tU32 /*u32OldAppState*/, tU32 /*u32AppState*/) const
{
   // Nothing to be done
}


//-----------------------------------------------------------------------------
// handle functions
//-----------------------------------------------------------------------------
///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnNewTestResult
//
// DESCRIPTION: react on messages with FID = DIAGLOG_C_U16_FKTID_SEND_TEST_RESULT
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnNewTestResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclService::vOnNewTestResult"));
   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      ETG_TRACE_USR4(( "--- vdDiagLog_tclService::vOnNewTestResult => u16SourceAppID = %x",u16SourceAppID));
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();
      ETG_TRACE_USR4(( "--- vdDiagLog_tclService::vOnNewTestResult => u16TargetAppID = %x",u16TargetAppID));

      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT"));
      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));


      // Msg must come from a client
      NORMAL_M_ASSERT(FALSE == poMessage->bIsServerMessage());
      NORMAL_M_ASSERT(NULL != m_poGWMain);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT _ OPCODE-ERROR**********"));

                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                      // message send success - set the return value to TRUE

                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT"));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));

                   }
                   else
                   {
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT"));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending: u16TargetAppID = %x", u16SourceAppID));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
                   }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnNewTestResult => OSAL_s32SemaphoreWait(m_hReportMemSem)"));

                  midw_diaglogfi_tclMsgSaveTestResultMethodStart oData;
                  midw_diaglogfi_tclMsgSaveTestResultMethodResult oResultData;

                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);

                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {

                     NORMAL_M_ASSERT_ALWAYS ();
                  }

                  // process the test report
                  bool bResult = bProcessTestReport(oData.TestResultList);

                  if(true == bResult)
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }
                  else
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }


                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT _ OPCODE-MethodResult**********"));

                  ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnNewTestResult => OSAL_s32SemaphorePost(m_hReportMemSem)"));

                  // send reply
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {

                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END

            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclService::vOnNewTestResult"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnNewTestResultDynamialDTC
//
// DESCRIPTION: react on messages with FID = DIAGLOG_C_U16_FKTID_SEND_TEST_RESULT
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnNewTestResultDynamialDTC(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclService::vOnNewTestResultDynamialDTC"));
   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      ETG_TRACE_USR4(( "--- vdDiagLog_tclService::vOnNewTestResultDynamialDTC => u16SourceAppID = %x",u16SourceAppID));
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();
      ETG_TRACE_USR4(( "--- vdDiagLog_tclService::vOnNewTestResultDynamialDTC => u16TargetAppID = %x",u16TargetAppID));

      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_SAVETESTRESULTDYNAMICALDTC"));
      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));


      // Msg must come from a client
      NORMAL_M_ASSERT(FALSE == poMessage->bIsServerMessage());
      NORMAL_M_ASSERT(NULL != m_poGWMain);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SAVETESTRESULTDYNAMICALDTC _ OPCODE-ERROR"));

                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                      // message send success - set the return value to TRUE

                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SAVETESTRESULTDYNAMICALDTC"));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));

                   }
                   else
                   {
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_SAVETESTRESULTDYNAMICALDTC"));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending: u16TargetAppID = %x", u16SourceAppID));
                      ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
                   }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnNewTestResultDynamialDTC => OSAL_s32SemaphoreWait(m_hReportMemSem)"));

                  midw_diaglogfi_tclMsgSaveTestResultDynamicalDTCMethodStart oData;
                  midw_diaglogfi_tclMsgSaveTestResultDynamicalDTCMethodResult oResultData;

                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);

                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {

                     NORMAL_M_ASSERT_ALWAYS ();
                  }

                  // process the test report
                  bool bResult = bProcessTestReport(oData.TestResultList,true);

                  if(true == bResult)
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }
                  else
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }


                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SAVETESTRESULTDYNAMICALDTC _ OPCODE-MethodResult"));

                  ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnNewTestResultDynamialDTC => OSAL_s32SemaphorePost(m_hReportMemSem)"));

                  // send reply
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {

                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END

            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclService::vOnNewTestResultDynamialDTC"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vControl
//
// DESCRIPTION: react on messages with FID = DIAGLOG_C_U16_FKTID_CONTROL. Aim is to Enable or disable the
// DIA_LOG_BLOCKING_MODE_DIAGNOSIS
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnControl(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnControl"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_CONTROL"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_CONTROL _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_CONTROL"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnControl => Send Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnControl => Send to Server failed"));
                  }
               }// if(NULL != m_poGWMain)
            break;
         }

         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnControl => OSAL_s32SemaphoreWait(m_hReportMemSem)"));

                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsgControlMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // prepare method result data
                  midw_diaglogfi_tclMsgControlMethodResult oResultData;

                  if (NULL == m_poGWMain)
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }
                  else
                  {
                     switch(oData.ControlParameter.enType)
                     {
                        case midw_fi_tcl_e8_ControlParameter::FI_EN_DISABLELOGGING:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Disable Logging Temporary"));
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_DISABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_ENABLELOGGING:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Enabele Logging Temporary"));
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_ENABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_DISABLELOGGINPERSISTENT:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Disable Logging Persistant"));

                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_DISABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_ENABLELOGGINGPERSISTENT:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Enable Logging Persistant"));

                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_ENABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_ENABLECALIBRATIONBLOCKING:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControl => Enable Calibration Blocking"));
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_READ_DISABLE_BLOCKING_ENABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_DISABLECALIBRATIONBLOCKING:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControl => Disable Calibration Blocking"));
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_READ_DISABLE_BLOCKING_DISABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }
                        default:
                        {
                           // unkown command
                           NORMAL_M_ASSERT_ALWAYS();
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                           break;
                        }
                     }// switch(oData.ControlParameter.Status.enType)
                  }
                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_CONTROL _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnControl"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnControlDtcSetting
//
// DESCRIPTION: react on messages with FID = DIAGLOG_C_U16_FKTID_CONTROL. Aim is to Enable or disable the
// DIA_LOG_BLOCKING_MODE_DIAGNOSIS
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnControlDtcSetting(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnControlDtcSetting"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_CONTROL"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_CONTROL _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_CONTROL"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Send Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnControlDtcSetting => Send to Server failed"));
                  }
               }// if(NULL != m_poGWMain)
            break;
         }

         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => OSAL_s32SemaphoreWait(m_hReportMemSem)"));

                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsgControlDtcSettingMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // prepare method result data
                  midw_diaglogfi_tclMsgControlDtcSettingMethodResult oResultData;

                  if (NULL == m_poGWMain)
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }
                  else
                  {
                     // not supported yet
                     NORMAL_M_ASSERT_ALWAYS();
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                     /*
                     switch(oData.ControlParameter.enType)
                     {
                        case midw_fi_tcl_e8_ControlParameter::FI_EN_DISABLELOGGING:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Disable Logging Temporary"));
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_DISABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_ENABLELOGGING:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Enabele Logging Temporary"));
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_ENABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_DISABLELOGGINPERSISTENT:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Disable Logging Persistant"));

                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_DISABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_ENABLELOGGINGPERSISTENT:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControlDtcSetting => Enable Logging Persistant"));

                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_PERSIST_ENABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_ENABLECALIBRATIONBLOCKING:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControl => Enable Calibration Blocking"));
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_EOL_BLOCKING_ENABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }

                        case midw_fi_tcl_e8_ControlParameter::FI_EN_DISABLECALIBRATIONBLOCKING:
                        {
                           ETG_TRACE_USR1_THR(("--- vdDiagLog_tclService::vOnControl => Disable Calibration Blocking"));
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_DISABLE_READING_BLOCKING_DISABLE;
                           tDTCListArray oDTCList;

                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                           break;
                        }
                        default:
                        {
                           // unkown command
                           NORMAL_M_ASSERT_ALWAYS();
                           oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                           break;
                        }
                     }// switch(oData.ControlParameter.Status.enType)
                  */
                  }
                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_CONTROL _ OPCODE-MethodResult**********"));

                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnControlDtcSetting"));
}
///////////////////////////////////////////////////////////////////////////
//
// FUNCTION:   vdDiagLog_tclService::bUpreg
//
// DESCRIPTION:   handle an UpReg request
//
// PARAMETER:  upreg request message
//
// RETURNVALUE:   true  : Client reqistration successful
//                false : Client registration failed
//
///////////////////////////////////////////////////////////////////////////
//
bool vdDiagLog_tclService::bUpreg(amt_tclServiceData const* poMessage)const
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::bUpreg"));
   tBool bRegOpStatus = false;

   if(poMessage != NULL)
   {
      (tVoid) ahl_bEnterCritical(vdDiagLog_tclApp::m_hNotTableSem);

      // Add the client to notification table

      if(  (NULL != m_poGWMain)
         &&(NULL != m_poGWMain->m_poNotTable))
      {
         bRegOpStatus =  m_poGWMain->m_poNotTable->bAddNotification  (
                                                                     poMessage->u16GetFunctionID(),
                                                                     poMessage->u16GetSourceAppID(),
                                                                     poMessage->u16GetRegisterID(),
                                                                     1,
                                                                     poMessage->u16GetCmdCounter(),
                                                                     poMessage->u16GetSourceSubID()
                                                                     );
      }

      (tVoid) ahl_bReleaseCritical(vdDiagLog_tclApp::m_hNotTableSem);
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::bUpreg"));
   return ((bRegOpStatus)?true:false);
}
///////////////////////////////////////////////////////////////////////////
//
// FUNCTION:   vdDiagLog_tclService::bRelUpreg
//
// DESCRIPTION:   Removes the sender of the message from the NotifcationTable
//
// PARAMETER:  Relupreg request message
//
// RETURNVALUE:   true  : Client dereqistration successful
//                false : Client deregistration failed
//
///////////////////////////////////////////////////////////////////////////
//
bool vdDiagLog_tclService::bRelUpreg(amt_tclServiceData const* poMessage) const
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::bRelUpreg"));
   tBool bRegOpStatus = false;

   if(poMessage != NULL)
   {
      (tVoid) ahl_bEnterCritical(vdDiagLog_tclApp::m_hNotTableSem);

      // Delete the client from notification table
      if(NULL != m_poGWMain)
      {
         bRegOpStatus =  m_poGWMain->m_poNotTable->bRemoveNotification  (
                                                                        poMessage->u16GetFunctionID(),
                                                                        poMessage->u16GetSourceAppID(),
                                                                        poMessage->u16GetRegisterID(),
                                                                        1,
                                                                        poMessage->u16GetCmdCounter(),
                                                                        poMessage->u16GetSourceSubID()
                                                                        );
      }

      (tVoid) ahl_bReleaseCritical(vdDiagLog_tclApp::m_hNotTableSem);
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::bRelUpreg"));
   return ((bRegOpStatus)?true:false);
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vUpdateAllClients
//
//  DESCRIPTION: update all clients with a property status
//
//  PARAMETER:   oFIData - property data
//               u16FuncId - property ID
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vUpdateAllClients(fi_tclTypeBase const& oFIData, tU16 u16FuncId)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vUpdateAllClients"));
   if(NULL == m_poGWMain)
   {
      ETG_TRACE_ERR_THR (("!!!  vdDiagLog_tclService::vUpdateAllClients >= ERROR: NULL == m_poGWMain"));
      return;
   }

   typedef std::vector<fi_tclVisitorMessage> tclVisitorMsgVector;
   tclVisitorMsgVector clVisitorMsgVector;

   (tVoid) ahl_bEnterCritical(vdDiagLog_tclApp::m_hNotTableSem);

   // Run through the table to find all registered clients
   for (
       ahl_tNotification* pNot = m_poGWMain->m_poNotTable->poGetNotificationList(u16FuncId);
       pNot != OSAL_NULL;
       pNot = pNot->pNext
       )
   {

      // construct result message
      fi_tclVisitorMessage oResultMsg(oFIData);
      // initialise result message
      oResultMsg.vInitServiceData(
                                 CCA_C_U16_APP_DIAGLOG,              // source
                                 pNot->u16AppID,                     // Target
                                 AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                                 0,                                  // StreamCounter
                                 pNot->u16RegisterID,                // RegisterID
                                 pNot->u16CmdCounter,                // nCmdCounter,
                                 CCA_C_U16_SRV_DIAGLOG,              // nServiceID,
                                 u16FuncId,                          // function ID
                                 AMT_C_U8_CCAMSG_OPCODE_STATUS,
                                 AMT_C_U16_DEFAULT_NULL,
                                 pNot->u16SubID
                                 );
      // store to backup buffer, don't send in semaphore
      clVisitorMsgVector.push_back(oResultMsg);
   }//for (ahl_tNotification*
   (tVoid) ahl_bReleaseCritical(vdDiagLog_tclApp::m_hNotTableSem);

   // now send the stored requests to the Components
   for(tU32 i=0; i<clVisitorMsgVector.size();i++)
   {
      // send it
      if(NULL != m_poGWMain)
      {

         ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
         switch (u16FuncId)
         {
            case MIDW_DIAGLOGFI_C_U16_DTCACTIVE:
               ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_DTCACTIVE _ OPCODE-STATUS**********"));
               break;

            case MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT:
               ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT (vUpdateAllClients) _ OPCODE-STATUS**********"));
               break;

			case MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST:
               ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST (vUpdateAllClients) _ OPCODE-STATUS**********"));
               break;

            default:
               break;
         }// switch (u16FuncId)

         if (m_poGWMain->enPostMessage(&clVisitorMsgVector[i], TRUE) == AIL_EN_N_NO_ERROR)
         {
            // message send success - set the return value to TRUE

            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = %d", u16FuncId));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", clVisitorMsgVector[i].u16GetSourceAppID()));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", clVisitorMsgVector[i].u16GetTargetAppID()));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_STATUS"));

         }
         else
         {
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI = %d", u16FuncId));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG sending failed: u16SourceAppID = %x", clVisitorMsgVector[i].u16GetSourceAppID()));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG sending failed: u16TargetAppID = %x", clVisitorMsgVector[i].u16GetTargetAppID()));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_STATUS"));
         }

      }// if(NULL != m_poGWMain)
      else
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tU32 i=0; i<clVisitorMsgVector.size();i++)
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vUpdateAllClients"));
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendMsg
//
//  DESCRIPTION: sends a CCA message
//
//  PARAMETER:   poMessage - incoming request
//                oFIData - property data
//                u8OpCode - CCA op code
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vSendMsg(
                                    amt_tclServiceData const*  poMessage,
                                    fi_tclTypeBase const&      oFIData,
                                    tU8                        u8OpCode
                                    )
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendMsg"));
   // Check the validity of incoming data
   if (NULL == poMessage)
   {
      ETG_TRACE_ERR_THR (("!!!  vdDiagLog_tclService::vSendMsg >= ERROR: NULL == poMessage"));
      // Message received from framework is NULL!!!
      NORMAL_M_ASSERT (NULL != poMessage);
      return;
   }// if (NULL == poMessage)
   // construct result message
   fi_tclVisitorMessage oResultMsg(oFIData);
   // initialise result message
   oResultMsg.vInitServiceData(
                              CCA_C_U16_APP_DIAGLOG,            // source
                              poMessage->u16GetSourceAppID(),     // Target
                              AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                              0,                                  // StreamCounter
                              poMessage->u16GetRegisterID(),      // RegisterID
                              poMessage->u16GetCmdCounter(),      // nCmdCounter,
                              CCA_C_U16_SRV_DIAGLOG,              // nServiceID,
                              poMessage->u16GetFunctionID(),      // function ID
                              u8OpCode,
                              AMT_C_U16_DEFAULT_NULL,
                              poMessage->u16GetSourceSubID()
                              );
   NORMAL_M_ASSERT(NULL != m_poGWMain);
   // send it
   if(NULL != m_poGWMain)
   {
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Send: FI = %d",  poMessage->u16GetFunctionID()));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16TargetAppID = %x",  poMessage->u16GetSourceAppID()));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: Opcode = %d", u8OpCode));
      ail_tenCommunicationError enResult = m_poGWMain->enPostMessage(&oResultMsg, TRUE);
      //NORMAL_M_ASSERT(enResult == AIL_EN_N_NO_ERROR);

      if(enResult == AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vSendMsg => Send Successfully to Server"));
      }
      else
      {
         ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vSendMsg => Send to Server failed"));
      }
   }// if(NULL != m_poGWMain)
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendMsg"));
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendError
//
//  DESCRIPTION: sends a CCA message
//
//  PARAMETER:   poMessage - incoming request
//               u16ErrorCode - Error Code
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vSendError(
                                    amt_tclServiceData const*  poMessage,
                                    tU16                        u16ErrorCode
                                    )
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendError"));
   // Check the validity of incoming data
   if (NULL == poMessage)
   {
      ETG_TRACE_ERR_THR (("!!!  vdDiagLog_tclService::vSendError >= ERROR: NULL == poMessage"));
      // Message received from framework is NULL!!!
      NORMAL_M_ASSERT (NULL != poMessage);
      return;
   }// if (NULL == poMessage)
   // construct result message
   amt_tclServiceDataError oErrorMsg;
   // initialise result message
   oErrorMsg.vInitServiceData(
                              CCA_C_U16_APP_DIAGLOG,            // source
                              poMessage->u16GetSourceAppID(),     // Target
                              AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                              0,                                  // StreamCounter
                              poMessage->u16GetRegisterID(),      // RegisterID
                              poMessage->u16GetCmdCounter(),      // nCmdCounter,
                              CCA_C_U16_SRV_DIAGLOG,              // nServiceID,
                              poMessage->u16GetFunctionID(),      // function ID
                              AMT_C_U8_CCAMSG_OPCODE_ERROR,
                              AMT_C_U16_DEFAULT_NULL,
                              poMessage->u16GetSourceSubID()
                              );

    oErrorMsg.vSetErrorData(u16ErrorCode);

   NORMAL_M_ASSERT(NULL != m_poGWMain);
   // send it
   if(NULL != m_poGWMain)
   {
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Send: FI = %d",  poMessage->u16GetFunctionID()));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16TargetAppID = %x",  poMessage->u16GetSourceAppID()));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: Opcode = %d", AMT_C_U8_CCAMSG_OPCODE_ERROR));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: Error = %d", u16ErrorCode));
      ail_tenCommunicationError enResult = m_poGWMain->enPostMessage(&oErrorMsg, TRUE);
      //NORMAL_M_ASSERT(enResult == AIL_EN_N_NO_ERROR);

      if(enResult == AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vSendError => Send Successfully to Server"));
      }
      else
      {
         ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vSendError => Send to Server failed"));
      }
   }// if(NULL != m_poGWMain)
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendError"));
}


///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnSendNextTestResult
//
//  DESCRIPTION: handle a SendNextTestResult message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vOnSendNextTestResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnSendNextTestResult (amt_tclServiceData*)"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

      // the opcode tells what we should do
      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_UPREG"));
            if(!bUpreg(poMessage))
            {
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_UPREG_FAILURE);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Could not register you for MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT)"));

                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendNextTestResult => Send Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendNextTestResult => Send to Server failed"));
                  }
               }
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               // Upreg successful : send a status message.
               vSendNextTestResultStatus(poMessage);
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_RELUPREG"));
            if(!bRelUpreg(poMessage))
            {
               // Relupreg failed: send an error message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_RELUPREG_FAILURE);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Send: FI MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Could not RELUPREG you for MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendNextTestResult => Sent Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendNextTestResult => Sent to Server failed"));
                  }
               }
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
              // Relupreg successful : no action
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendNextTestResult => Sent Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendNextTestResult => Sent to Server failed"));
                  }
               }
            }
            break;

         default:
            // unknown opcode: just delete message
            break;
      }
      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnSendNextTestResult (amt_tclServiceData*)"));
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendNextTestResultStatus
//
//  DESCRIPTION: sends a SendNextTestResult status message if client is registered
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vSendNextTestResultStatus(amt_tclServiceData const* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendNextTestResultStatus(amt_tclServiceData const*)"));
   // Check the validity of incoming data
   if (NULL == poMessage)
   {
      ETG_TRACE_ERR_THR (("!!!  vdDiagLog_tclService::vSendNextTestResultStatus(amt_tclServiceData const*) >= ERROR: NULL == poMessage"));
      // Message received from framework is NULL!!!
      NORMAL_M_ASSERT (NULL != poMessage);
      return;
   }
   // create status result object
   midw_diaglogfi_tclMsgSendNextTestResultStatus oStatusData;
   // send status data
   oStatusData.NextTestResultStruct.Status = FALSE;
   oStatusData.Status = FALSE;
   // get diaglog data

   ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
   ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT _ OPCODE-Status (false)**********"));
   vSendMsg(poMessage, oStatusData, AMT_C_U8_CCAMSG_OPCODE_STATUS);

   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendNextTestResultStatus(amt_tclServiceData const*)"));
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendNextTestResultStatus
//
//  DESCRIPTION: sends a SendNextTestResult status message if client is registered
//
//  PARAMETER:   status
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vSendNextTestResultStatus(midw_diaglogfi_tclMsgSendNextTestResultStatus const& oStatusData)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendNextTestResultStatus(midw_diaglogfi_tclMsgSendNextTestResultStatus const&)"));

   // Run through the table to find all registered clients
   if(NULL != m_poGWMain)
   {
      vUpdateAllClients(oStatusData, MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT);
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendNextTestResultStatus(midw_diaglogfi_tclMsgSendNextTestResultStatus const&)"));
}


///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vOnSetToDefaultRequest
//
//  DESCRIPTION: receive a SetToDefault request
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////

tVoid vdDiagLog_tclService::vOnSetToDefaultRequest(amt_tclServiceData* poMessage)
{
   tBool bSendError = TRUE;

   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnSetToDefaultRequest(amt_tclServiceData*)"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_SETFACTORYSETTINGSNEW"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
         {
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_METHODSTART"));
            _BP_TRY_BEGIN
            {
               // get method start data into fi type
               fi_tclVisitorMessage oMsg(poMessage);
               midw_diaglogfi_tclMsgSetFactorySettingsNewMethodStart oData;
               if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
               {
                  NORMAL_M_ASSERT_ALWAYS ();
               }
               // prepare method result data
               midw_diaglogfi_tclMsgSetFactorySettingsNewMethodResult oResultData;

               if (NULL != m_poGWMain)
               {
                  bSendError = FALSE;

                  switch(oData.DefSetStateGroup.enType)
                  {
                     case (midw_fi_tcl_e8_DiaglogDefsetType::FI_EN_CLEAR_ERRMEM):
                        oResultData.DefSetStateGroup = oData.DefSetStateGroup;
                        oResultData.DefSetStateMode = oData.DefSetStateMode;
                        oResultData.DefSetStateSuccessful = TRUE;
                        oResultData.DefSetStateFirstDifferentParameterId = 0;
                        break;
                     case (midw_fi_tcl_e8_DiaglogDefsetType::FI_EN_ACTIVATE_DEACTIVATE_DTC):
                        // do other project specfic things
                        if(m_poGWMain->m_oMemoryMaster.bProcessSetToDefault())
                        {
                           oResultData.DefSetStateSuccessful = TRUE;
                        }
                        else
                        {
                           oResultData.DefSetStateSuccessful = FALSE;
                        }
                        // lock DTC control
                        {
                           vdl_teDiaglogControlMsg eControlMsg = DIAGLOG_CONTROL_CTRL_LOGGING_TEMP_DISABLE;
                           tDTCListArray oDTCList;
                           m_poGWMain->m_oMemoryMaster.bProcessControlCommand(eControlMsg, oDTCList);
                        }

                        // clear all memories
                        (tVoid) m_poGWMain->m_oMemoryMaster.processClearRequest(0xFF,0xFFFFFF);

                       oResultData.DefSetStateGroup = oData.DefSetStateGroup;
                       oResultData.DefSetStateMode = oData.DefSetStateMode;
                       oResultData.DefSetStateFirstDifferentParameterId = 0;
                       break;
                     default:
                       // reject
                       oResultData.DefSetStateGroup = oData.DefSetStateGroup;
                       oResultData.DefSetStateMode = oData.DefSetStateMode;
                       oResultData.DefSetStateSuccessful = FALSE;
                       oResultData.DefSetStateFirstDifferentParameterId = 0;
                       // not supported so assert here
                       NORMAL_M_ASSERT_ALWAYS();
                       break;

                  }
                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SETFACTORYSETTINGSNEW _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
            }
            _BP_CATCH_ALL
            {
            NORMAL_M_ASSERT_ALWAYS();
            }
            _BP_CATCH_END
            break;
         }// case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
         default:
         {
           ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Wrong Opcode"));
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(poMessage);
            oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
            if(NULL != m_poGWMain)
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SETFACTORYSETTINGSNEW _ OPCODE-ERROR**********"));
               if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
               {
                  // message send success - set the return value to TRUE

                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SETFACTORYSETTINGSNEW"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));

               }
               else
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI = MIDW_DIAGLOGFI_C_U16_SETFACTORYSETTINGSNEW"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
               }
            }// if(NULL != m_poGWMain)
            else
            {
               NORMAL_M_ASSERT_ALWAYS();
            }
            break;
         }// default:
      } // switch (poMessage->u8GetOpCode())

      if(TRUE == bSendError)
      {
         amt_tclServiceDataError oErrorMessage(poMessage);
         oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         if(NULL != m_poGWMain)
         {
            if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnDTCActive => UPREG Successfully send to Server"));
            }
            else
            {
               ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnDTCActive => UPREG send to Server failed"));
            }
         }
      }
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnSetToDefaultRequest(amt_tclServiceData*)"));
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendUpdateStatus
//
//  DESCRIPTION: send to all clients the Update trigger to retrigger all Errors again
//
//  PARAMETER:   NONE
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vSendUpdateStatus(tITCVector& ITCs)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendUpdateStatus"));
   // send update to all clients
   midw_diaglogfi_tclMsgSendNextTestResultStatus oStatus;
   oStatus.Status = TRUE;
   oStatus.NextTestResultStruct.Status = TRUE;
   oStatus.NextTestResultStruct.ITCList = ITCs;
   oStatus.NextTestResultStruct.SendNextTestResReason.enType = midw_fi_tcl_e8_SendNextTestResReason::FI_EN_CLEAR_CODE;

   vSendNextTestResultStatus(oStatus);
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendUpdateStatus"));
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnDtcTrigger
//
//  DESCRIPTION: handle a MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vOnDtcTrigger(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnDtcTrigger (amt_tclServiceData*)"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

      // the opcode tells what we should do
      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_UPREG"));
            if(!bUpreg(poMessage))
            {
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_UPREG_FAILURE);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Could not register you for MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED)"));

                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendNextTestResult => Send Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendNextTestResult => Send to Server failed"));
                  }
               }
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               // Upreg successful : send a status message.
               midw_diaglogfi_tclMsgDTCTriggeredStatus oStatus;

               oStatus.bDTCTriggeredFlag                       = false;
               oStatus.u8DTCSource                             = 0x00;
               oStatus.u16DTCNumber                            = 0x00;
               oStatus.u8DTCFailureType                        = 0x00;
               oStatus.bDTCWarningIndicatorRequestedStatus     = false;
               oStatus.bDTCTestFailedSincePowerUpStatus        = false;
               oStatus.bDTCTestNotPassedSincePowerUpStatus     = false;
               oStatus.bDTCHistoryStatus                       = false;
               oStatus.bDTCTestFailedSinceCodeClearedStatus    = false;
               oStatus.bDTCTestNotPassedSinceCodeClearedStatus = false;
               oStatus.bDTCCurrentStatus                       = false;
               oStatus.bDTCCodeSupported                       = false;
               oStatus.e8DTCFaultType.enType                   = (midw_fi_tcl_e8_GMLnGWDTCFaultType::tenType)0x00;

               vSendMsg(poMessage, oStatus, AMT_C_U8_CCAMSG_OPCODE_STATUS);
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_RELUPREG"));
            if(!bRelUpreg(poMessage))
            {
               // Relupreg failed: send an error message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_RELUPREG_FAILURE);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Send: FI MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Could not RELUPREG you for MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendNextTestResult => Sent Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendNextTestResult => Sent to Server failed"));
                  }
               }
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
              // Relupreg successful : no action
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_DTCTRIGGERED"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendNextTestResult => Sent Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendNextTestResult => Sent to Server failed"));
                  }
               }
            }
            break;

         default:
            // unknown opcode: just delete message
            break;
      }
      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnSendNextTestResult (amt_tclServiceData*)"));
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdDiagLog_tclService::bProcessTestReport
//
// DESCRIPTION: process test reports in worklist
//
// PARAMETER:  midw_fi_tcl_TestResultList const& roTestReport - test report(s)
//             midw_diaglogfi_tclMsgDTCActiveStatus   roActivationStatus - DTC activation status
//
// RETURNVALUE: bool - true if data was sucessfully process
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdDiagLog_tclService::bProcessTestReport(midw_fi_tcl_TestResultList&  roTestReport, bool isDynamicalDTC)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclService::bProcessTestReport"));

   bool bReturn = true;

   // check we have loaded the database at start up
   if(false == bCheckLoadStatus(roTestReport))
   {
      ETG_TRACE_ERR_THR(( "!!! vdDiagLog_tclService::bProcessTestReport=> bCheckLoadStatus(roTestReport) FAILED "));
      return false;
   }

   if(m_poGWMain != NULL)
   {
      ETG_TRACE_USR4(( "---vdDiagLog_tclService::bProcessTestReport: roTestReport.TestResultList.size() = %d", roTestReport.TestResultList.size()));

      // check over all received reports (could be more then one)
      for(::size_t i = 0; i < roTestReport.TestResultList.size(); ++i)
      {
         midw_fi_tcl_TestResult oTestResult = roTestReport.TestResultList[i];
         vdl_tsDiaglogTestResult oData;

         // first get Statustype
         vdl_teDiaglogReportMsg  enMsgReportStatus = DIAGLOG_REPORT_NONE;
         switch(oTestResult.Result.enType)
         {
            case midw_fi_tcl_e8_TestResult::FI_EN_PASSEDDIAGNOSTIC:
            case midw_fi_tcl_e8_TestResult::FI_EN_PASSED:
               enMsgReportStatus = DIAGLOG_REPORT_PASSED;
               break;

            case midw_fi_tcl_e8_TestResult::FI_EN_FAILEDDIAGNOSTIC:
            case midw_fi_tcl_e8_TestResult::FI_EN_FAILED:
               enMsgReportStatus = DIAGLOG_REPORT_FAILED;
               break;
            case midw_fi_tcl_e8_TestResult::FI_EN_DISABLE:
               enMsgReportStatus = DIAGLOG_REPORT_DISABLE;
               break;
            case midw_fi_tcl_e8_TestResult::FI_EN_ENABLE:
               enMsgReportStatus = DIAGLOG_REPORT_ENABLE;
               break;
            case midw_fi_tcl_e8_TestResult::FI_EN_NORESULTDIAGNOSTIC:
            case midw_fi_tcl_e8_TestResult::FI_EN_NORESULT:
               enMsgReportStatus = DIAGLOG_REPORT_NORESULT;
               break;
            default:
               enMsgReportStatus = DIAGLOG_REPORT_NONE;
               break;
         }
         if(isDynamicalDTC == true)
         {
            oData.u32ITC         = 0;
            oData.u32DTC         = oTestResult.TroubleCode;
            oData.bDynamicalDTC  = true;
         }
         else
         {
            oData.u32ITC         = oTestResult.TroubleCode;
            oData.u32DTC         = 0;
            oData.bDynamicalDTC  = false;
         }
         oData.eReportStatus  = enMsgReportStatus;
         oData.u32Timestamp   = 0;

         bool bResult = m_poGWMain->m_oMemoryMaster.bProcessTestReport(oData);
         if(bResult == false)
         {
            // at minimum one DTC in the list is invalid
            bReturn = false;
         }
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(( "!!! vdDiagLog_tclService::bProcessTestReport => m_poGWMain = NULL"));
      NORMAL_M_ASSERT_ALWAYS();
      bReturn = false;
   }

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclService::bProcessTestReport"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: bool vdDiagLog_tclService::bCheckLoadStatus
//
// DESCRIPTION: check load status, load and merge results
//
// PARAMETER:  midw_fi_tcl_TestResultList const& roTestReport - test report(s)
//
// RETURNVALUE: bool - true = loaded, false = not loaded
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdDiagLog_tclService::bCheckLoadStatus(midw_fi_tcl_TestResultList& roTestReport)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclService::bCheckLoadStatus"));
   bool bResult = false;
   // if not already loaded
   if(m_poGWMain != NULL)
   {
      if(false == m_poGWMain->m_oMemoryMaster.bCheckLoadStatus())
      {
            m_oReportCache.TestResultList.insert(
                                                m_oReportCache.TestResultList.end(),
                                                roTestReport.TestResultList.begin(),
                                                roTestReport.TestResultList.end()
                                                );
            ETG_TRACE_ERR_THR(( "!!! vdDiagLog_tclService::bCheckLoadStatus => load failed ADD TO CACHE"));
      }
      else
      {
            roTestReport.TestResultList.insert  (
                                                roTestReport.TestResultList.begin(),
                                                m_oReportCache.TestResultList.begin(),
                                                m_oReportCache.TestResultList.end()
                                                );
            bResult = true;
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(( "!!! vdDiagLog_tclService::bCheckLoadStatus => m_poGWMain = NULL"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclService::bCheckLoadStatus"));
   return bResult;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION:tVoid vdDiagLog_tclService::vUpdateAfterBlockingMode
//
// DESCRIPTION: Implementation of SendNextTestResult-Message
//
// PARAMETER:  u32LastBlockingMode: old Blocking Mode
//          u32BlockingMode: new Blocking Mode
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vUpdateAfterBlockingMode(tServiceUpdateMap updateMap)
{
   ETG_TRACE_USR3_THR(( "--> vdDiagLog_tclService::vUpdateAfterBlockingMode"));
   typedef std::vector<fi_tclVisitorMessage> tclVisitorMsgVector;
   tclVisitorMsgVector clVisitorMsgVector;

   (tVoid) ahl_bEnterCritical(vdDiagLog_tclApp::m_hNotTableSem);
   // Run through the table to find all registered clients to MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT but do not send while semphore locked!!!
   for (ahl_tNotification* pNot = (vdDiagLog_tclApp::m_poInstance)->m_poNotTable->poGetNotificationList(MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT);
      pNot != OSAL_NULL;
      pNot = pNot->pNext)
   {
      midw_diaglogfi_tclMsgSendNextTestResultStatus oStatus;

      if (NULL != m_poGWMain)
      {
         tU32 AppId = (pNot->u16AppID) + DIAGLOG_RETRIGGER_ID_CCA;
         tServiceUpdateMapIt iterMap = updateMap.find(AppId);
         if (iterMap != updateMap.end())
         {
            // found
            tITCVector itcList = iterMap->second;
            for (tITCVectorCIt iterVector = itcList.begin(); iterVector != itcList.end(); ++iterVector)
            {
               tU16 ITC = *iterVector;
               oStatus.NextTestResultStruct.ITCList.push_back(ITC);
            }
         }
      }

      if (oStatus.NextTestResultStruct.ITCList.size()>0)
      {
         //Send the Status true
         oStatus.Status = TRUE;
         oStatus.NextTestResultStruct.Status = TRUE;
         oStatus.NextTestResultStruct.SendNextTestResReason.enType = midw_fi_tcl_e8_SendNextTestResReason::FI_EN_BLOCKING_MODE_RESET;

         #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            tITCVector tempVektor= oStatus.NextTestResultStruct.ITCList;
            size_t size = tempVektor.size();
            tU16 aData[size];
            int i=0;

            for(tITCVectorIt data_iter = tempVektor.begin(); data_iter != tempVektor.end(); ++data_iter)
            {
               aData[i++]=*data_iter;
            }
            ETG_TRACE_COMP_THR(( "--- vdDiagLog_tclService::vUpdateAfterBlockingMode => Send Update to App:%04x Data:%*x",ETG_ENUM(ail_u16AppId, pNot->u16AppID),ETG_LIST_LEN((unsigned int)size), ETG_LIST_PTR_T16(aData)));//lint !e772 // Trace only
         #endif // #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN

         // construct result message
         fi_tclVisitorMessage oResultMsg(oStatus);
         oResultMsg.vInitServiceData(
                              CCA_C_U16_APP_DIAGLOG,                    // source
                              pNot->u16AppID,                           // Target
                              AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,        // StreamType
                              0,                                        // StreamCounter
                              pNot->u16RegisterID,                      // RegisterID
                              pNot->u16CmdCounter,                      // nCmdCounter,
                              CCA_C_U16_SRV_DIAGLOG,                    // nServiceID,
                              MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT,  // function ID
                              AMT_C_U8_CCAMSG_OPCODE_STATUS,
                              AMT_C_U16_DEFAULT_NULL,
                              pNot->u16SubID
                              );

         // store to backup buffer, don't send in semaphore
         clVisitorMsgVector.push_back(oResultMsg);
      }

   }//for (ahl_tNotification* pNot = m_poGWMain->m_poNotTable->poGetNotificationList(MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT);
   (tVoid) ahl_bReleaseCritical(vdDiagLog_tclApp::m_hNotTableSem);


   ETG_TRACE_USR1_THR(( "--- vdDiagLog_tclService::vUpdateAfterBlockingMode => send Update to the Components"));
   // now send the stored requests to the Components
   for(tU32 i=0; i<clVisitorMsgVector.size();i++)
   {
      if(NULL != m_poGWMain)
      {
         ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
         ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ FI MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT _ OPCODE-STATUS**********"));
         if ((m_poGWMain->enPostMessage(&clVisitorMsgVector[i], TRUE))== AIL_EN_N_NO_ERROR)      // send it
         {
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT"));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", clVisitorMsgVector[i].u16GetSourceAppID()));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", clVisitorMsgVector[i].u16GetTargetAppID()));
            ETG_TRACE_USR2_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_STATUS (Status = true)"));
         }
         else
         {
            ETG_TRACE_ERR_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***### CCA_MSG sending failed: FI MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT"));
            ETG_TRACE_ERR_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG sending failed: u16SourceAppID = %x", clVisitorMsgVector[i].u16GetSourceAppID()));
            ETG_TRACE_ERR_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG sending failed: u16TargetAppID = %x", clVisitorMsgVector[i].u16GetTargetAppID()));
            ETG_TRACE_ERR_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_STATUS (Status = true)"));
         }
      }// if(NULL != vdDiagLog_tclApp::m_poInstance)
      else
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// for(tU32 i=0; i<clVisitorMsgVector.size();i++)

   ETG_TRACE_USR3_THR(( "<-- vdDiagLog_tclService::vUpdateAfterBlockingMode"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetDtcByStatusMask
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetDtcByStatusMask(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnGetDtcByStatusMask"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsggetDTCbyStatusMaskMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // construct method result data
                  midw_diaglogfi_tclMsggetDTCbyStatusMaskMethodResult oResultData;
                  // check for valid objects
                  if (NULL != m_poGWMain)
                  {
                     // create List Object
                     tReadReportListArray oReadReportList;

                     // request Memory for Data
                     vdl_teDiaglogOptions   eOptions   = OPTION_NONE;  // use standard option
                     tU32 const             u32DTCMask = 0xFFFF;       // use no Mask
                     tU8                    u8StatusAvailabilityMask = 0x00;

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/oData.Group:%x", oData.Group));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/oData.StatusMask:%x", oData.StatusMask));

                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.vProcessReadRequest(u8StatusAvailabilityMask, oReadReportList, oData.Group, oData.StatusMask, u32DTCMask, eOptions);

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tStatusAvailabilityMask:%x", u8StatusAvailabilityMask));
                     // for each entry in list
                     tReadReportListArrayIt iPos = oReadReportList.begin();
                     while(iPos.operator!=(oReadReportList.end()))
                     {
                        midw_fi_tcl_ReadResultStructure sReadResultStruct;
                        sReadResultStruct.DTC    = (*iPos).u32DTC;
                        sReadResultStruct.StatusOfDtc = static_cast<tU8>((*iPos).u32Status);
                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tITC: %x, Status:%x", sReadResultStruct.DTC, sReadResultStruct.StatusOfDtc));
                        // copy Data to response Buffer
                        oResultData.ReadResultList.DTCList.push_back (sReadResultStruct);
                        ++iPos;
                     }
                     oResultData.StatusAvailabilityMask = u8StatusAvailabilityMask;
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }
                  else
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }

                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnGetDtcByStatusMask"));
}



///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetSupportedDTC
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetSupportedDTC(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnGetSupportedDTC"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsggetSupportedDTCMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // construct method result data
                  midw_diaglogfi_tclMsggetSupportedDTCMethodResult oResultData;
                  // check for valid objects
                  if (NULL != m_poGWMain)
                  {
                     // create List Object
                     tReadReportListArray oReadReportList;

                     // request Memory for Data
                     tU8                    u8StatusAvailabilityMask = 0x00;

                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.vProcessReadSupportedDTCs(u8StatusAvailabilityMask, oReadReportList, oData.Group);

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tStatusAvailabilityMask:%x", u8StatusAvailabilityMask));
                     // for each entry in list
                     tReadReportListArrayIt iPos = oReadReportList.begin();
                     while(iPos.operator!=(oReadReportList.end()))
                     {
                        midw_fi_tcl_ReadResultStructure sReadResultStruct;
                        sReadResultStruct.DTC    = (*iPos).u32DTC;
                        sReadResultStruct.StatusOfDtc = static_cast<tU8>((*iPos).u32Status);
                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tITC: %x, Status:%x", sReadResultStruct.DTC, sReadResultStruct.StatusOfDtc));
                        // copy Data to response Buffer
                        oResultData.ReadResultList.DTCList.push_back (sReadResultStruct);
                        ++iPos;
                     }
                     oResultData.StatusAvailabilityMask = u8StatusAvailabilityMask;
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }
                  else
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }

                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnGetSupportedDTC"));
}


///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetLastConfirmedDTC
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetLastConfirmedDTC(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnGetLastConfirmedDTC"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsggetLastConfirmedDTCMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // construct method result data
                  midw_diaglogfi_tclMsggetLastConfirmedDTCMethodResult oResultData;
                  // check for valid objects

                  if (NULL != m_poGWMain)
                  {
                     tU8                    u8StatusAvailabilityMask = 0x00;
                     // create List Object
                     tReadReportListArray oReadReportList;

                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.vProcessReadLastConfirmedDTC(u8StatusAvailabilityMask,oReadReportList,DIAGLOG_MEMORY_CUSTOMER);

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tStatusAvailabilityMask:%x", u8StatusAvailabilityMask));

                     // for each entry in list
                     tReadReportListArrayIt iPos = oReadReportList.begin();
                     while(iPos.operator!=(oReadReportList.end()))
                     {
                        midw_fi_tcl_ReadResultStructure sReadResultStruct;
                        sReadResultStruct.DTC    = (*iPos).u32DTC;
                        sReadResultStruct.StatusOfDtc = static_cast<tU8>((*iPos).u32Status);
                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tITC: %x, Status:%x", sReadResultStruct.DTC, sReadResultStruct.StatusOfDtc));
                        // copy Data to response Buffer
                        oResultData.ReadResultList.DTCList.push_back (sReadResultStruct);
                        ++iPos;
                     }
                     oResultData.StatusAvailabilityMask = u8StatusAvailabilityMask;
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }
                  else
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }

                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnGetLastConfirmedDTC"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetNumberOfDTCByStatusMask
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetNumberOfDTCByStatusMask(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnGetNumberOfDTCByStatusMask"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsggetNumberOfDTCByStatusMaskMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // construct method result data
                  midw_diaglogfi_tclMsggetNumberOfDTCByStatusMaskMethodResult oResultData;
                  // check for valid objects
                  if (NULL != m_poGWMain)
                  {

                     // creat Data Object
                     tU32 u32NumberOfDtc = 0;
                     tU8  u8StatusAvailabilityMask = 0x00;

                     // forward message to Memory
                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.vProcessReadNumberOfDtcByStatusMask(u8StatusAvailabilityMask, u32NumberOfDtc, oData.Group, oData.StatusMask);

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tStatusAvailabilityMask:%x NumberOfDtc:%x", u8StatusAvailabilityMask, u32NumberOfDtc));
                     // copy Data to response Buffer
                     oResultData.StatusAvailabilityMask = u8StatusAvailabilityMask;
                     oResultData.NumberOfDTC = u32NumberOfDtc;
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }
                  else
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }

                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnGetNumberOfDTCByStatusMask"));
}



///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetStatusByDTC
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetStatusByDTC(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnGetStatusByDTC"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsggetStatusByDTCMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // construct method result data
                  midw_diaglogfi_tclMsggetStatusByDTCMethodResult oResultData;
                  // check for valid objects
                  if (NULL != m_poGWMain)
                  {
                     // create List Object
                     tReadReportListArray oReadReportList;

                     // request Memory for Data
                     vdl_teDiaglogOptions   eOptions     = OPTION_DTC_EQUAL;  // use equal option
                     tU8 const              u8StatusMask = 0xFF;              // use no Mask
                     tU8                    u8StatusAvailabilityMask = 0x00;

                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.vProcessReadRequest(u8StatusAvailabilityMask, oReadReportList, oData.Group, u8StatusMask,oData.DTCMask,eOptions);

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tStatusAvailabilityMask:%x", u8StatusAvailabilityMask));
                     // for each entry in list
                     tReadReportListArrayIt iPos = oReadReportList.begin();
                     while(iPos.operator!=(oReadReportList.end()))
                     {
                        midw_fi_tcl_ReadResultStructure sReadResultStruct;
                        sReadResultStruct.DTC    = (*iPos).u32DTC;
                        sReadResultStruct.StatusOfDtc = static_cast<tU8>((*iPos).u32Status);
                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tITC: %x, Status:%x", sReadResultStruct.DTC, sReadResultStruct.StatusOfDtc));
                        // copy Data to response Buffer
                        oResultData.ReadResultList.DTCList.push_back (sReadResultStruct);
                        ++iPos;
                     }
                     oResultData.StatusAvailabilityMask = u8StatusAvailabilityMask;
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }
                  else
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }

                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnGetStatusByDTC"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetStatusByITC
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetStatusByITC(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnGetStatusByITC"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsggetStatusByITCMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // construct method result data
                  midw_diaglogfi_tclMsggetStatusByITCMethodResult oResultData;
                  // check for valid objects
                  if (NULL != m_poGWMain)
                  {
                     // create List Object
                     tReadReportListArray oReadReportList;

                     // request Memory for Data
                     vdl_teDiaglogOptions   eOptions     = OPTION_ITC_EQUAL;  // use equal option
                     tU8 const              u8StatusMask = 0xFF;              // use no Mask
                     tU8                    u8StatusAvailabilityMask = 0x13;

                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.vProcessReadRequest(u8StatusAvailabilityMask, oReadReportList, oData.Group, u8StatusMask,oData.ITCMask,eOptions);

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tStatusAvailabilityMask:%x", u8StatusAvailabilityMask));
                     // for each entry in list
                     tReadReportListArrayIt iPos = oReadReportList.begin();
                     while(iPos.operator!=(oReadReportList.end()))
                     {
                        midw_fi_tcl_ReadResultStructure sReadResultStruct;
                        sReadResultStruct.DTC    = (*iPos).u16ITC;
                        sReadResultStruct.StatusOfDtc = static_cast<tU8>((*iPos).u32Status);
                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tITC: %x, Status:%x", sReadResultStruct.DTC, sReadResultStruct.StatusOfDtc));
                        // copy Data to response Buffer
                        oResultData.ReadResultList.DTCList.push_back (sReadResultStruct);
                        ++iPos;
                     }
                     //oResultData.StatusAvailabilityMask = u8StatusAvailabilityMask;
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }
                  else
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }

                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnGetStatusByITC"));
}


///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetExtendedDataByDTC
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////

namespace VDD
{
struct ExD2FI
{
   ExD2FI();
   ExD2FI(midw_fi_tcl_FreezeFrameList* aFFList) : ffList(aFFList) {}
   void operator() (ExtendedDataStruct const& p) const
      {
      midw_fi_tcl_FreezeFrameData ffData;
      ffData.ID = p.recNo;
      ffData.Data = p.data;
      ffList->FreezeFrameDataList.push_back(ffData);
      }
private:
   midw_fi_tcl_FreezeFrameList* ffList;
};

}

tVoid vdDiagLog_tclService::vOnGetExtendedDataByDTC(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnGetExtendedDataByDTC"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsggetExtendedDataByDTCMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }

                  // construct method result data
                  midw_diaglogfi_tclMsggetExtendedDataByDTCMethodResult oResultData;
                  // check for valid objects
                  if (NULL != m_poGWMain)
                  {
                     tU8 group = oData.Group;
                     tU32 dtc = oData.DTC;
                     oResultData.DTC = dtc;
                     oResultData.StatusOfDtc = vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.getDtcStatus(group, dtc);
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                     tU8 recordNumber = oData.DTCExtDataRecordNumber;
                     ExtendedData const* pExtendedData = vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.getExtendedData(group, dtc, recordNumber);
                     if(pExtendedData)
                     {
                        //lint -e864 // Info 864: prio3: Expression involving variable 'frames' possibly depends on order of evaluation
                        tExtendedFrameList frames;
                        pExtendedData->getFrameList(frames);
                        for_each(frames.begin(), frames.end(), VDD::ExD2FI(&oResultData.FreezeFrameList));
                        //lint +e864 // Info 864: prio3: Expression involving variable 'frames' possibly depends on order of evaluation
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Reject: m_poGWMain == NULL"));
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }
                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;

         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnGetExtendedDataByDTC"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnClearDTC
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_CLEARDTC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnClearDTC(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnClearDTC"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();


      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_CLEARDTC"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_CLEARDTC _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_CLEARDTC"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnClearDTC => Send Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnClearDTC => Send to Server failed"));
                  }
               }

            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
                _BP_TRY_BEGIN
               {

                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsgclearDTCMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }
                  // construct method result data
                  midw_diaglogfi_tclMsgclearDTCMethodResult oResultData;
                  // check for valid objects
                  if (NULL == m_poGWMain)
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                     // send reply
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_CLEARDTC _ OPCODE-MethodResult**********"));
                     vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
                  }
                  else
                  {
                     tBool bReturn;
                     bReturn = m_poGWMain->m_oMemoryMaster.processClearRequest(oData.Group,oData.DTCMask);
                     if(bReturn == true)
                     {
                        oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                        // send reply
                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_CLEARDTC _ OPCODE-MethodResult**********"));
                        vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
                     }
                     else
                     {
                        ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnClearDTC => requested DTC not Found"));
                        vSendError(poMessage, AMT_C_U16_ERROR_PARAMETER_OUT_OF_RANGE);
                     }
                  }
                }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnClearDTC"));
}



///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetSnapShotIdentification
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetSnapShotIdentification(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("--> vdDiagLog_tclService::vOnGetSnapShotIdentification"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTIDENTIFICATION"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTIDENTIFICATION _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTIDENTIFICATION"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTIDENTIFICATION"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsgreportDTCSnapshotIdentificationMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }

                  // construct method result data
                  midw_diaglogfi_tclMsgreportDTCSnapshotIdentificationMethodResult oResultData;
                  // check for valid objects
                  if (NULL != m_poGWMain)
                  {
                     tU8 group         = oData.Group;

                     tSnapShotIdList oSnapShotIdList;

                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.getSnapShotId(group, oSnapShotIdList);

                     tSnapShotIdListIt iPos = oSnapShotIdList.begin();
                     while(iPos.operator!=(oSnapShotIdList.end()))
                     {
                        midw_fi_tcl_SnapshotIdentification  oFiSnapShotId;
                        oFiSnapShotId.DTC            = (*iPos).DTC;
                        oFiSnapShotId.RecordNumber   = (*iPos).RecordNumber;


                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t\tDTC: %06x, RecNo:%02x", oFiSnapShotId.DTC, oFiSnapShotId.RecordNumber));
                        // copy Data to response Buffer
                        oResultData.SnapshotIdentificationList.SnapshotIdentification.push_back (oFiSnapShotId);
                        ++iPos;
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Reject: m_poGWMain == NULL"));
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }
                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTIDENTIFICATION _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_tclService::vOnGetSnapShotIdentification"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetSnapShotDataByDTC
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetSnapShotDataByDTC(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("--> vdDiagLog_tclService::vOnGetSnapShotDataByDTC"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTRECORDBYDTCNUMBER"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTRECORDBYDTCNUMBER _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTRECORDBYDTCNUMBER"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTRECORDBYDTCNUMBER"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
               _BP_TRY_BEGIN
               {
                  // get method start data into fi type
                  fi_tclVisitorMessage oMsg(poMessage);
                  midw_diaglogfi_tclMsgreportDTCSnapshotRecordByDTCNumberMethodStart oData;
                  if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
                  {
                     NORMAL_M_ASSERT_ALWAYS ();
                  }

                  // construct method result data
                  midw_diaglogfi_tclMsgreportDTCSnapshotRecordByDTCNumberMethodResult oResultData;
                  // check for valid objects
                  if (NULL != vdDiagLog_tclApp::m_poInstance)
                  {
                     tU8 group         = oData.Group;
                     tU32 dtc          = oData.DTC;
                     tU8 recordNumber  = oData.DTCSnapshotRecordNumber;
                     tU8 status        = 0;
                     ETG_TRACE_USR1_THR (("--- vdDiagLog_tclService::vOnGetSnapShotDataByDTC => Group:%01d DTC:%06x RecordNumber:%02x",group,dtc,recordNumber));


                     tSnapShotDataList oSnapShotDataList;

                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.getSnapShotData(group, dtc, recordNumber, status, oSnapShotDataList);
                     oResultData.DTC = dtc;
                     oResultData.StatusOfDtc = status;
                     tSnapShotDataListIt iPos = oSnapShotDataList.begin();
                     while(iPos.operator!=(oSnapShotDataList.end()))
                     {
                        midw_fi_tcl_SnapshotData fiSnapShotStruct;
                        fiSnapShotStruct.RecordNumber = (*iPos).RecordNumber;
                        fiSnapShotStruct.NumberOfData = (*iPos).NumberOfDid;
                        fiSnapShotStruct.Data         = (*iPos).data;

                        // the following array is only for TTFIS
                        size_t size = (*iPos).data.size();
                        tU8 aData[size];
                        for (size_t i=0; i<size;i++)
                        {
                           aData[i]=(tU8)((*iPos).data[i]);
                        }

                        ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "/t/tDTC:%06x,Status:%02x, RecNo:%02x, Size:%02x, Data:%*x", dtc, status, fiSnapShotStruct.RecordNumber,fiSnapShotStruct.NumberOfData,ETG_LIST_LEN((unsigned int)size), ETG_LIST_PTR_T8(aData)));//lint !e772 // Trace only
                        // copy Data to response Buffer
                        oResultData.SnapshotDataList.SnapshotData.push_back (fiSnapShotStruct);

                        ++iPos;
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Reject: m_poGWMain == NULL"));
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }
                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTRECORDBYDTCNUMBER _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
               }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_tclService::vOnGetSnapShotDataByDTC"));
}


///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnProductionClear
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_PORDUCTIONCLEARREQUEST
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnProductionClear(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnProductionClear"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();


      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_PRODUCTIONCLEARREQUEST"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
            {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_PRODUCTIONCLEARREQUEST _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_PRODUCTIONCLEARREQUEST"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnProductionClear => Send Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnProductionClear => Send to Server failed"));
                  }
               }

            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
                _BP_TRY_BEGIN
               {
                  ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnProductionClear => OSAL_s32SemaphoreWait(m_hReportMemSem)"));
                  midw_diaglogfi_tclMsgProductionClearRequestMethodResult oResultData;

                  // check for loggin enabled and valid objects
                  if (NULL == m_poGWMain)
                  {
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
                  }
                  else
                  {
                     m_poGWMain->m_oMemoryMaster.processClearRequest(DIAGLOG_MEMORY_CUSTOMER, 0xFFFFFF);
                     oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
                  }

                  ETG_TRACE_USR2_THR(("--- vdDiagLog_tclService::vOnProductionClear => OSAL_s32SemaphorePost(m_hReportMemSem)"));

                  // send reply
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_PRODUCTIONCLEARREQUEST _ OPCODE-MethodResult**********"));
                  vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
                }
               _BP_CATCH_ALL
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
               _BP_CATCH_END
            }
            break;
         default:
            // unknown opcode: just delete message
            break;
      }

      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnProductionClear"));
}

///////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnGetItcInformation
//
// DESCRIPTION: react on messages with FID = MIDW_DIAGLOGFI_C_U16_GETEXDENDETDATABYDTC
//
// PARAMETER:   message to analyse
//
// RETURNVALUE: NONE
//
// History:
// InitialVersion
//
///////////////////////////////////////////////////////////////////////////////
tVoid vdDiagLog_tclService::vOnGetItcInformation(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("--> vdDiagLog_tclService::vOnGetItcInformation"));
   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      NORMAL_M_ASSERT(poMessage->bIsServerMessage() == FALSE);
      NORMAL_M_ASSERT(m_poGWMain != NULL);

      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION _ OPCODE-ERROR**********"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     // message send success - set the return value to TRUE

                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI = MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));

                  }
                  else
                  {
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sending failed: FI MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION"));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: u16TargetAppID = %x", u16SourceAppID));
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sending failed: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR  (Wrong Opcode)"));
                  }
               }
            break;
         }

         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
         {
             ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = MethodStart"));
            _BP_TRY_BEGIN
            {
               // get method start data into fi type
               fi_tclVisitorMessage oMsg(poMessage);
               midw_diaglogfi_tclMsggetItcInformationMethodStart oData;
               if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
               {
                  NORMAL_M_ASSERT_ALWAYS ();
               }

               // construct method result data
               midw_diaglogfi_tclMsggetItcInformationMethodResult oResultData;
               // check for valid objects
               if (NULL != vdDiagLog_tclApp::m_poInstance)
               {
                  tU8 group         = oData.Group;

                  for(::size_t i = 0; i < oData.ITCList.size(); ++i)
                  {
                     tU16 ITC = oData.ITCList[i];

                     vdl_tsDiaglogItcInformation itcInformation(ITC);
                     vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.getItcInformation(group, ITC, itcInformation);

                     midw_fi_tcl_ITC_Information   fiItcInformation;
                     fiItcInformation.DTC                = itcInformation.u32DTC;
                     fiItcInformation.ITC                = itcInformation.u16ITC;
                     fiItcInformation.StatusOfItc        = itcInformation.u8Status;
                     fiItcInformation.AgeingCounter      = itcInformation.u8AgeingCounter;
                     fiItcInformation.OccurenceCounter   = itcInformation.u8OccurenceCounter;
                     ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "********* ITC:%04x DTC:%06x Status:%02x AgeCtr:%d OccCtr:%d",itcInformation.u16ITC,itcInformation.u32DTC,itcInformation.u8Status,itcInformation.u8AgeingCounter,itcInformation.u8OccurenceCounter));

                     // copy Data to response Buffer
                     oResultData.ItcInformationList.push_back (fiItcInformation);
                  }// for(::size_t i = 0; i < oData.ITCList.size(); ++i)
                  oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED;
               }// if (NULL != vdDiagLog_tclApp::m_poInstance)
               else
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Reject: m_poGWMain == NULL"));
                  oResultData.Status.enType = midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED;
               }
               // send reply
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION _ OPCODE-MethodResult**********"));
               vSendMsg(poMessage, oResultData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
            }
            _BP_CATCH_ALL
            {
               NORMAL_M_ASSERT_ALWAYS();
            }
            _BP_CATCH_END
            break;
         }// case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:

         default:
         {
            // unknown opcode: just delete message
            break;
         }
      }// switch (poMessage->u8GetOpCode())
      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<-- vdDiagLog_tclService::vOnGetItcInformation"));
}

//////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnSendActiveDTCList
//
//  DESCRIPTION: handle a SendActiveDTCList message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vOnSendActiveDTCList(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnSendActiveDTCList (amt_tclServiceData*)"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

      // the opcode tells what we should do
      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_UPREG"));
            if(!bUpreg(poMessage))
            {
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_UPREG_FAILURE);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Could not register you for MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST)"));

                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendActiveDTCList => Send Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendActiveDTCList => Send to Server failed"));
                  }
               }
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               // Upreg successful : send a status message.
               vSendActiveDTCListStatus(poMessage);
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_RELUPREG"));
            if(!bRelUpreg(poMessage))
            {
               // Relupreg failed: send an error message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_RELUPREG_FAILURE);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Send: FI MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Could not RELUPREG you for MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendActiveDTCList => Sent Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendActiveDTCList => Sent to Server failed"));
                  }
               }
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
              // Relupreg successful : no action
            }
            break;
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnSendActiveDTCList => Sent Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnSendActiveDTCList => Sent to Server failed"));
                  }
               }
            }
            break;

         default:
            // unknown opcode: just delete message
            break;
      }
      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnSendActiveDTCList (amt_tclServiceData*)"));
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendActiveDTCListStatus
//
//  DESCRIPTION: sends a SendActiveDTCList status message if client is registered
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vSendActiveDTCListStatus(amt_tclServiceData const* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendActiveDTCListStatus(amt_tclServiceData const*)"));
   // Check the validity of incoming data
   if (NULL == poMessage)
   {
      ETG_TRACE_ERR_THR (("!!!  vdDiagLog_tclService::vSendActiveDTCListStatus(amt_tclServiceData const*) >= ERROR: NULL == poMessage"));
      // Message received from framework is NULL!!!
      NORMAL_M_ASSERT (NULL != poMessage);
      return;
   }
   // create status result object
   midw_diaglogfi_tclMsgSendActiveDTCListStatus oStatusData;
   // send status data
   std::vector<tU32> dtcList;
   std::vector<tU32>::iterator ItList;
   for(ItList = dtcList.begin();ItList!= dtcList.end();++ItList)
   {
      oStatusData.ActiveDTCList.DTCList.push_back(*ItList); //empty list
   }
   // get diaglog data

   ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
   ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST _ OPCODE-Status (false)**********"));
   vSendMsg(poMessage, oStatusData, AMT_C_U8_CCAMSG_OPCODE_STATUS);

   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendActiveDTCListStatus(amt_tclServiceData const*)"));
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendActiveDTCListStatus
//
//  DESCRIPTION: sends a SendActiveDTCList status message if client is registered
//
//  PARAMETER:   status
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
//tVoid vdDiagLog_tclService::vSendActiveDTCListStatus(midw_diaglogfi_tclMsgSendActiveDTCListStatus const& oStatusData)
tVoid vdDiagLog_tclService::vSendActiveDTCListStatus(std::vector<tU32> dtcList)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendActiveDTCListStatus(midw_diaglogfi_tclMsgSendActiveDTCListStatus(dtcList)"));

   // Run through the table to find all registered clients
   // create status result object

   midw_diaglogfi_tclMsgSendActiveDTCListStatus oStatusData;
   // send status data
   std::vector<tU32>::iterator ItList;
   for(ItList = dtcList.begin();ItList!= dtcList.end();++ItList)
   {
      oStatusData.ActiveDTCList.DTCList.push_back(*ItList);
   }

   if(NULL != m_poGWMain)
   {
      vUpdateAllClients(oStatusData, MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST);
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendActiveDTCListStatus(midw_diaglogfi_tclMsgSendActiveDTCListStatus(dtcList)"));
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendDtcStatusChanged
//
//  DESCRIPTION: sends a DTCStatusChanged status message if client is registered
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vSendDtcStatusChanged(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendDtcStatusChanged(amt_tclServiceData const*)"));
   // Check the validity of incoming data
   if (NULL == poMessage)
   {
      ETG_TRACE_ERR_THR (("!!!  vdDiagLog_tclService::vSendDtcStatusChanged(amt_tclServiceData const*) >= ERROR: NULL == poMessage"));
      // Message received from framework is NULL!!!
      NORMAL_M_ASSERT (NULL != poMessage);
      return;
   }

   fi_tclVisitorMessage oMsg(poMessage);
   midw_diaglogfi_tclMsgDtcStatusChangedUpReg oData;
   if (oMsg.s32GetData(oData, MIDW_DIAGLOGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
   {
      NORMAL_M_ASSERT_ALWAYS ();
      return;
   }

   // create status result object
   midw_diaglogfi_tclMsgDtcStatusChangedStatus oStatusData;
   oStatusData.DTCResponse.DTC = oData.DTCNumber;
   oStatusData.DTCResponse.StatusOfDtc = vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.getDtcStatus(oData.Group, oData.DTCNumber);

   ETG_TRACE_USR3_THR(("--- vdDiagLog_tclService::vSendDtcStatusChanged => DTC = 0x%x, Status = 0x%x",
         oStatusData.DTCResponse.DTC, oStatusData.DTCResponse.StatusOfDtc));

   // send status data

   ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
   ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_DTCSTATUSCHANGED _ OPCODE-Status (false)**********"));
   vSendMsg(poMessage, oStatusData, AMT_C_U8_CCAMSG_OPCODE_STATUS);

   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendDtcStatusChanged(amt_tclServiceData const*)"));
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vdDiagLog_tclService::vSendDTCStatusChanged
//
//  DESCRIPTION: sends a DTCStatusChanged message for specific DTC if client is registered
//
//  PARAMETER:   dtc
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vSendDtcStatusChanged(tU8 MemoryId, tU32 u32DTC)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vSendDtcStatusChanged(dtc)"));

   ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "vdDiagLog_tclService::vSendDtcStatusChanged: mem = %d, dtc = 0x%06X", MemoryId, u32DTC));

   // create status result object
   midw_diaglogfi_tclMsgDtcStatusChangedStatus oStatusData;
   // send status data

   oStatusData.DTCResponse.DTC = u32DTC;
   oStatusData.DTCResponse.StatusOfDtc = vdDiagLog_tclApp::m_poInstance->m_oMemoryMaster.getDtcStatus(MemoryId, u32DTC);

   if(NULL != m_poGWMain)
   {
      vUpdateAllClients(oStatusData, MIDW_DIAGLOGFI_C_U16_DTCSTATUSCHANGED);
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vSendDtcStatusChanged(dtc)"));
}

//////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vOnDtcStatusChanged
//
//  DESCRIPTION: handle a DTCStatusChanged
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid vdDiagLog_tclService::vOnDtcStatusChanged(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3_THR (("-->  vdDiagLog_tclService::vOnDtcStatusChanged (amt_tclServiceData*)"));

   if(poMessage != NULL)
   {
      tU16 u16SourceAppID = poMessage->u16GetSourceAppID();
      tU16 u16TargetAppID = poMessage->u16GetTargetAppID();

      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "CCA_MSG received: FI MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST"));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16SourceAppID = %x", u16SourceAppID));
      ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: u16TargetAppID = %x", u16TargetAppID));

      // Msg must come from a client
      OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

      // the opcode tells what we should do
      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         {
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_UPREG"));
            if(!bUpreg(poMessage))
            {
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_UPREG_FAILURE);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Could not register you for MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST)"));

                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnDtcStatusChanged => Send Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnDtcStatusChanged => Send to Server failed"));
                  }
               }
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               // Upreg successful : send a status message.
               vSendDtcStatusChanged(poMessage);
            }
            break;
         }

         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         {
            ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Opcode = AMT_C_U8_CCAMSG_OPCODE_RELUPREG"));
            if(!bRelUpreg(poMessage))
            {
               // Relupreg failed: send an error message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_RELUPREG_FAILURE);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Send: FI MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Send: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Could not RELUPREG you for MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnDtcStatusChanged => Sent Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnDtcStatusChanged => Sent to Server failed"));
                  }
               }
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
              // Relupreg successful : no action
            }
            break;
         }

         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
            {
               ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG received: Wrong Opcode"));
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(poMessage);
               oErrorMessage.vSetErrorData(AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               if(NULL != m_poGWMain)
               {
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "......................................................................................................"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "*********Trying to send CCA Msg _ MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST _ OPCODE-ERROR**********"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "***CCA_MSG Sent: FI MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST"));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16SourceAppID = %x", CCA_C_U16_APP_DIAGLOG));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: u16TargetAppID = %x", u16SourceAppID));
                  ETG_TRACE_USR4_CLS((TR_CLASS_DIAGLOG_CCA_MSG, "\t CCA_MSG Sent: Opcode = AMT_C_U8_CCAMSG_OPCODE_ERROR (Wrong Opcode)"));
                  if (m_poGWMain->enPostMessage(&oErrorMessage, TRUE) == AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_USR4_THR(("--- vdDiagLog_tclService::vOnDtcStatusChanged => Sent Successfully to Server"));
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("!!! vdDiagLog_tclService::vOnDtcStatusChanged => Sent to Server failed"));
                  }
               }
            }
            break;

         default:
            // unknown opcode: just delete message
            break;
      }
      (tVoid) poMessage->bDelete();
   }// if(poMessage != NULL)
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
   ETG_TRACE_USR3_THR (("<--  vdDiagLog_tclService::vOnDtcStatusChanged (amt_tclServiceData*)"));
}
