//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_storage_handler.cpp
// PROJECT:      common
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog Object class - generic base class
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 28.10.14  | BSOT Plischke                | Initial version
//
//////////////////////////////////////////////////////////////////////////

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_storage_handler
   #include <common/framework/vd_diaglog_storage_handler.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_object
   #include <common/framework/vd_diaglog_object.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_file_storage_interface
   #include <common/framework/vd_diaglog_storage_interface.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_storage_handler.cpp.trc.h"
#endif


OSAL_tTimerHandle vdl_tclStorageHandler::m_hSaveDataBaseTimer = OSAL_C_INVALID_HANDLE;

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclStorageHandler::vdl_tclStorageHandler
// 
// DESCRIPTION: default constructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclStorageHandler::vdl_tclStorageHandler() 
{
   vOnInit();
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclStorageHandler::~vdl_tclStorageHandler
// 
// DESCRIPTION: default denstructor
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclStorageHandler::~vdl_tclStorageHandler()
{
   _BP_TRY_BEGIN
   {
      vOnClose();
   }
   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclStorageHandler::~vdl_tclStorageHandler => _BP_CATCH_ALL"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
  
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclStorageHandler::vOnInit
// 
// DESCRIPTION: Init
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclStorageHandler::vOnInit() 
{
   // STORAGE create timer
   ETG_TRACE_USR1_THR(( "--- vdl_tclStorageHandler::bOnInit => Create STORAGE-TIMER m_hSaveDataBaseTimer"));
   tS32 s32Result = OSAL_s32TimerCreate(vSaveDataBaseTimerHandler, (tPVoid)this, &m_hSaveDataBaseTimer);    
   if(OSAL_ERROR == s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERR_THR(("!!! vdl_tclStorageHandler::vOnInit => ERROR: OSAL_s32TimerCreate  FAILED ErrorCode:%x",u32ErrorCode));
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_tclApp::vOnInit => OSAL_s32TimerCreate failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
      return;
   }
   // set timer period
   vStartMemoryBlockingTimer(DIAGLOG_STORAGE_STARTUP_TIME);
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclStorageHandler::vOnClose
// 
// DESCRIPTION: Init
//
// PARAMETER:  void
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclStorageHandler::vOnClose() const
{
   // destroy the timer now
   if(m_hSaveDataBaseTimer != OSAL_C_INVALID_HANDLE)
   {
      ETG_TRACE_USR1_THR(( "--- vdl_tclStorageHandler::vOnClose => Destroy STORAGE-TIMER m_hSaveDataBaseTimer"));
      tS32 s32Result = OSAL_s32TimerDelete(m_hSaveDataBaseTimer);
      m_hSaveDataBaseTimer = OSAL_C_INVALID_HANDLE;
      NORMAL_M_ASSERT(OSAL_ERROR != s32Result);
   }
}



///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: void vdl_tclStorageHandler::vSaveDataBaseTimerHandler
// 
// DESCRIPTION: timer callback handler (Storage in diaglog.log)
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclStorageHandler::vSaveDataBaseTimerHandler(tPVoid /*pvArg*/)
{
   if(vdDiagLog_tclApp::m_hEventHandle != OSAL_C_INVALID_HANDLE) 
   {
      tS32 s32Result = OSAL_s32EventPost(vdDiagLog_tclApp::m_hEventHandle, DIAGLOG_EVENT_TIMER_STORAGE_MASK, OSAL_EN_EVENTMASK_OR);
      NORMAL_M_ASSERT(OSAL_ERROR != s32Result);
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: OSAL_tpfCallback vdl_tclStorageHandler::vStartMemoryBlockingTimer
// 
// DESCRIPTION: restart the Memory Blocking Timer
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclStorageHandler::vStartMemoryBlockingTimer(tU32 timeMs) const
{
   ETG_TRACE_USR1_THR(("--- vdl_tclStorageHandler::vStartMemoryBlockingTimer => Time:%d[ms]",timeMs));
   tS32 s32Result = OSAL_s32TimerSetTime (m_hSaveDataBaseTimer, timeMs, 0);

   if(OSAL_ERROR == s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();

      ETG_TRACE_FATAL_THR(("!!! vdl_tclStorageHandler::vStartMemoryBlockingTimer => ERROR: OSAL_s32TimerSetTime  FAILED Time:%x ErrorCode:%x",timeMs,u32ErrorCode));
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdDiagLog_tclApp::vStartMemoryBlockingTimer => OSAL_s32TimerSetTime failed: Time=%x, Error=%x",
                               timeMs,
                               u32ErrorCode);

      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclStorageHandler::updateAfterStorageTimer
// 
// DESCRIPTION: storage Timer run out, storage now allowed again
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclStorageHandler::updateAfterStorageTimer() const
{
   ETG_TRACE_USR1_THR(("--- vdl_tclStorageHandler::updateAfterStorageTimer"));
   // over all MemoryObjects
   for(tMemoryObjectListCIt iPos = m_oMemoryObjectList.begin(); iPos != m_oMemoryObjectList.end(); ++iPos)
   {
      vdl_tclDiagLogObject* pMemoryObject = (*iPos); // get memory 
      if(NULL != pMemoryObject)
      {
         pMemoryObject->setCanSave(true);
         pMemoryObject->requestForSaveIfModified();
      }// if(NULL != pMemoryObject)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclStorageHandler::updateAfterStorageTimer => NULL == pMemoryObject"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }//  for(tMemoryObjectListIt iPos = m_oMemoryObjectList.begin( ...

}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclStorageHandler::saveDoneInformation
// 
// DESCRIPTION: storage Timer run out, storage now allowed again
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclStorageHandler::saveDoneInformation() const 
{
   vStartMemoryBlockingTimer(DIAGLOG_STORAGE_INTERVAL_TIME);
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclStorageHandler::creatNewStorageHandler
// 
// DESCRIPTION: storga Timer run out, storage now aloowed again
//
// PARAMETER: tVoid
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclDiagLogObject* vdl_tclStorageHandler::registerNewStorageObject(vdl_tclStorageUserInterface* const poInterface ,tCString const szFilename)
{

   for(tMemoryObjectListCIt iPos = m_oMemoryObjectList.begin(); iPos != m_oMemoryObjectList.end(); ++iPos)
   {
      vdl_tclDiagLogObject* pMemoryObject = (*iPos); // get memory 

      if(pMemoryObject != NULL)
      {
         if (strcmp(szFilename,pMemoryObject->getFileName())== 0)
         {
            ETG_TRACE_ERRMEM(("!!! vdl_tclStorageHandler::registerNewStorageObject => ERROR: FileName already exist in List %s",szFilename));
            return NULL;
         }
      }
      else
      {
         ETG_TRACE_ERRMEM(("!!! vdl_tclStorageHandler::registerNewStorageObject => ERROR: List invalid -> should never happen"));
      }
   }

   if(poInterface != NULL)
   {
      vdl_tclDiagLogObject* pNewMemoryObject = new vdl_tclDiagLogObject(this, poInterface, szFilename);
      ETG_TRACE_USR1_THR(("--- vdl_tclStorageHandler::registerNewStorageObject => register Object %x",pNewMemoryObject));
      m_oMemoryObjectList.push_back(pNewMemoryObject);
      return pNewMemoryObject;
   }
   else
   {
      return NULL;
   }
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclStorageHandler::unregisterNewStorageObject
// 
// DESCRIPTION: remove Storage Handler
//
// PARAMETER: const vdl_tclDiagLogObject* const poObject
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclStorageHandler::unregisterNewStorageObject(vdl_tclDiagLogObject* const poObject)
{

   for(tMemoryObjectListIt iPos = m_oMemoryObjectList.begin(); iPos != m_oMemoryObjectList.end(); ++iPos)
   {
      vdl_tclDiagLogObject* pMemoryObject = (*iPos); // get memory 

      if(pMemoryObject == poObject)
      {
         (tVoid) m_oMemoryObjectList.erase(iPos);  // delete from list
         ETG_TRACE_USR1_THR(("--- vdl_tclStorageHandler::unregisterNewStorageObject => Unregister Object %x",poObject));
         return ;
      }// if(pMemoryObject != poObject)
   }// for(tMemoryObjectListCIt iPos = m_oMemoryObjectList.begin(); iPos != m_oMemoryObjectList.end(); ++iPos)
}
